package com.ejianc.business.asset.service.impl;

import com.ejianc.business.asset.bean.AmortizeSetEntity;
import com.ejianc.business.asset.bean.AssetAmortizeEntity;
import com.ejianc.business.asset.mapper.AssetAmortizeMapper;
import com.ejianc.business.asset.service.IAmortizeSetService;
import com.ejianc.business.asset.service.IAssetAmortizeService;
import com.ejianc.business.asset.vo.AssetAmortizeDetailVO;
import com.ejianc.framework.core.util.ComputeUtil;
import com.ejianc.framework.skeleton.template.BaseServiceImpl;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import java.math.BigDecimal;
import java.text.SimpleDateFormat;
import java.util.*;
import java.util.stream.Collectors;

/**
 * 固定资产摊销
 * 
 * @author generator
 * 
 */
@Service("assetAmortizeService")
public class AssetAmortizeServiceImpl extends BaseServiceImpl<AssetAmortizeMapper, AssetAmortizeEntity> implements IAssetAmortizeService{


    @Autowired
    private IAmortizeSetService amortizeSetService;


    @Override
    public void getAmortizeSet(List<AssetAmortizeDetailVO> detailList) {
        Map<Long, Long> euipmentsMap = detailList.stream().collect(Collectors.toMap(AssetAmortizeDetailVO::getEquipmentId,AssetAmortizeDetailVO::getEquipmentTypeId));
        Map<Long, AmortizeSetEntity> amortizeSetMap = amortizeSetService.getSet(euipmentsMap);
        for (AssetAmortizeDetailVO detailVO : detailList) {
            Long equipmentId = detailVO.getEquipmentId();
            AmortizeSetEntity amortizeSetEntity = amortizeSetMap.get(equipmentId);
            if (amortizeSetEntity != null) {
                detailVO.setAmortizeSetId(amortizeSetEntity.getId());
                detailVO.setUsePeriod(amortizeSetEntity.getUsePeriod());
                detailVO.setResidualValueRate(amortizeSetEntity.getResidualValueRate());
            } else {
                detailVO.setErrMsg("未获取到该设备的摊销设置！");
            }
        }
    }

    /**
     * @param detailList
     * @param amortizeDate
     * @description: 计算本次摊销金额
     * @return: void
     * @author songlx
     * @date: 2022/7/19
     * 1、本次摊销比例：自动计算， 本次摊销金额/原值；
     * <p>
     * 2、本次摊销金额：
     * <p>
     * 2.1  根据后台设置的摊销方法自动计算。摊销周期数*原值/使用年限(月)；举例：1*（120/12）=10
     * <p>
     * 摊销周期：
     * <p>
     * 如果是之前没有摊销的设备，摊销周期数 = 当前摊销周期 - 入库月份+1；
     * <p>
     * 如果是已经摊销的设备，摊销周期数 = 当前摊销周期 - 上次摊销周期；
     * <p>
     * 2.2 弹框选择材料载入列表页面时，当选择材料的净值≤残值时，本期摊销金额 = 0。
     * <p>
     * 2.3 当摊销材料已经生效的摊销期数 + 本次摊销周期数  ≥  材料设置的使用年限(月)时：
     * <p>
     * 本期摊销金额 = 材料净值 - 材料残值。
     * <p>
     * 2.4 当摊销材料已经生效的摊销期数 + 本次摊销周期数 ＜ 材料设置的使用年限(月)时且根据公式计算的本次摊销金额＜材料净值 - 材料残值时：
     * <p>
     * 本期摊销金额 = 材料净值 - 材料残值。
     */
    @Override
    public void caclAmortizeMny(List<AssetAmortizeDetailVO> detailList, Date amortizeDate) {
        for (AssetAmortizeDetailVO detailVO : detailList) {
            Date lastAmortizeDate = detailVO.getLastAmortizeDate();
            Date startAmortizeDate = null;
            boolean isHasThis = false;
            if (lastAmortizeDate == null) {
                //首次摊销, 需要包含起始月
                isHasThis = true;
                startAmortizeDate = detailVO.getCheckDate();
            } else {
                startAmortizeDate = lastAmortizeDate;
            }
            //本次摊销周期数
            int thisAmortizeNum = getMonthBetweenDate(startAmortizeDate, amortizeDate, isHasThis);
            detailVO.setThisAmortizeNum(thisAmortizeNum);

            BigDecimal netWorthTax = detailVO.getNetWorthTax();
            BigDecimal residualValueTax = detailVO.getResidualValueTax();
            BigDecimal netWorth = detailVO.getNetWorth();
            BigDecimal residualValue = detailVO.getResidualValue();
            BigDecimal usedAmortizeNum = ComputeUtil.toBigDecimal(detailVO.getUsedAmortizeNum());
            BigDecimal usePeriod = ComputeUtil.toBigDecimal(detailVO.getUsePeriod());
            BigDecimal usedAmortizeNumHaveThis = ComputeUtil.safeAdd(ComputeUtil.toBigDecimal(detailVO.getThisAmortizeNum()), usedAmortizeNum);

            BigDecimal picOriginalValueTax = ComputeUtil.safeDiv(detailVO.getOriginalValueTax(), ComputeUtil.toBigDecimal(detailVO.getUsePeriod()));
            BigDecimal picOriginalValue = ComputeUtil.safeDiv(detailVO.getOriginalValue(), ComputeUtil.toBigDecimal(detailVO.getUsePeriod()));

            //计算出来的理论本期摊销金额
            BigDecimal thisAmortizeMnyTax = ComputeUtil.safeMultiply(ComputeUtil.toBigDecimal(thisAmortizeNum), picOriginalValueTax);
            BigDecimal thisAmortizeMny = ComputeUtil.safeMultiply(ComputeUtil.toBigDecimal(thisAmortizeNum), picOriginalValue);

            // 净值-残值 = 剩余可摊销金额
            BigDecimal maxAmortizeMnyTax = ComputeUtil.safeSub(netWorthTax, residualValueTax);
            BigDecimal maxAmortizeMny = ComputeUtil.safeSub(netWorth, residualValue);
            if (ComputeUtil.isLessOrEqual(maxAmortizeMnyTax, BigDecimal.ZERO)) {
                //1、净值≤残值时，本期摊销金额 = 0
                detailVO.setAmortizeMnyTax(BigDecimal.ZERO);
                detailVO.setAmortizeMny(BigDecimal.ZERO);
            } else if (ComputeUtil.isGreaterOrEqual(usedAmortizeNumHaveThis, usePeriod)) {
                //2、已经生效的摊销期数 + 本次摊销周期数  ≥  材料设置的使用年限(月)时：本期摊销金额 = 材料净值 - 材料残值
                detailVO.setAmortizeMnyTax(ComputeUtil.safeSub(netWorthTax, residualValueTax));
                detailVO.setAmortizeMny(ComputeUtil.safeSub(netWorth, residualValue));
            } else if (ComputeUtil.isGreaterThan(thisAmortizeMnyTax, maxAmortizeMnyTax)){
                detailVO.setAmortizeMnyTax(maxAmortizeMnyTax);
                detailVO.setAmortizeMny(maxAmortizeMny);
            } else {
                detailVO.setAmortizeMnyTax(thisAmortizeMnyTax);
                detailVO.setAmortizeMny(thisAmortizeMny);
            }
            //计算本次摊销比例
            BigDecimal amortizeRate = ComputeUtil.bigDecimalPercent(detailVO.getAmortizeMnyTax(), detailVO.getOriginalValueTax(), 2);
            detailVO.setAmortizeRate(amortizeRate);

        }
    }


    /**
     * 获取两个日期之间的所有月份 (年月)
     */
    public static int getMonthBetweenDate(Date startDate, Date endDate, boolean isHasThis) {
        SimpleDateFormat sdf = new SimpleDateFormat("yyyy-MM");
        // 声明保存日期集合
        List<String> list = new ArrayList<>();
        //用Calendar 进行日期比较判断
        Calendar calendar = Calendar.getInstance();
        while (startDate.getTime() <= endDate.getTime()) {
            // 把日期添加到集合
            if (isHasThis)
                list.add(sdf.format(startDate));
            // 设置日期
            calendar.setTime(startDate);
            //把日期增加一天
            calendar.add(Calendar.MONTH, 1);
            // 获取增加后的日期
            startDate = calendar.getTime();
        }

        return list.size();
    }

}
