package com.ejianc.business.asset.service.impl;

import com.ejianc.business.asset.bean.AmortizeSetEntity;
import com.ejianc.business.asset.bean.AssetAmortizeEntity;
import com.ejianc.business.asset.mapper.AssetAmortizeMapper;
import com.ejianc.business.asset.service.IAmortizeSetService;
import com.ejianc.business.asset.service.IAssetAmortizeService;
import com.ejianc.business.asset.vo.AssetAmortizeDetailVO;
import com.ejianc.framework.core.util.ComputeUtil;
import com.ejianc.framework.skeleton.template.BaseServiceImpl;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import java.math.BigDecimal;
import java.util.*;
import java.util.stream.Collectors;

/**
 * 固定资产摊销
 *
 * @author generator
 */
@Service("assetAmortizeService")
public class AssetAmortizeServiceImpl extends BaseServiceImpl<AssetAmortizeMapper, AssetAmortizeEntity> implements IAssetAmortizeService {


    @Autowired
    private IAmortizeSetService amortizeSetService;


    @Override
    public void getAmortizeSet(List<AssetAmortizeDetailVO> detailList, Long orgId) {
        Map<Long, Long> euipmentsMap = detailList.stream().collect(Collectors.toMap(AssetAmortizeDetailVO::getEquipmentId, AssetAmortizeDetailVO::getEquipmentTypeId, (key1, key2) -> key1));
        Map<Long, AmortizeSetEntity> amortizeSetMap = amortizeSetService.getSet(euipmentsMap, orgId);
        for (AssetAmortizeDetailVO detailVO : detailList) {
            Long equipmentId = detailVO.getEquipmentId();
            AmortizeSetEntity amortizeSetEntity = amortizeSetMap.get(equipmentId);
            if (amortizeSetEntity != null) {
                detailVO.setAmortizeSetId(amortizeSetEntity.getId());
                detailVO.setUsePeriod(amortizeSetEntity.getUsePeriod());
                detailVO.setResidualValueRate(amortizeSetEntity.getResidualValueRate());
            } else {
                detailVO.setErrMsg("未获取到该设备的摊销设置！");
            }
        }
    }

    /**
     * @param detailList
     * @param amortizeDate
     * @description: 计算本次摊销金额
     * @return: void
     * @author songlx
     * @date: 2022/7/19
     * 1、本次摊销比例：自动计算， 本次摊销金额/原值；
     * <p>
     * 2、本次摊销金额：
     * <p>
     * 2.1  根据后台设置的摊销方法自动计算。摊销周期数*原值/使用年限(月)；举例：1*（120/12）=10
     * <p>
     * 摊销周期：
     * <p>
     * 如果是之前没有摊销的设备，摊销周期数 = 当前摊销周期 - 入库月份+1；
     * <p>
     * 如果是已经摊销的设备，摊销周期数 = 当前摊销周期 - 上次摊销周期；
     * <p>
     * 2.2 弹框选择材料载入列表页面时，当选择材料的净值≤残值时，本期摊销金额 = 0。
     * <p>
     * 2.3 当摊销材料已经生效的摊销期数 + 本次摊销周期数  ≥  材料设置的使用年限(月)时：
     * <p>
     * 本期摊销金额 = 材料净值 - 材料残值。
     * <p>
     * 2.4 当摊销材料已经生效的摊销期数 + 本次摊销周期数 ＜ 材料设置的使用年限(月)时且根据公式计算的本次摊销金额＜材料净值 - 材料残值时：
     * <p>
     * 本期摊销金额 = 材料净值 - 材料残值。
     */
    @Override
    public void caclAmortizeMny(List<AssetAmortizeDetailVO> detailList, Date amortizeDate) {
        for (AssetAmortizeDetailVO detailVO : detailList) {
            //如果errMsg不为空，说明没有获取到摊销设置，不需要计算本次摊销金额和比例
            if(detailVO.getErrMsg() != null) continue;
            Date lastAmortizeDate = detailVO.getLastAmortizeDate();
            Date startAmortizeDate = null;
            boolean isHasThis = false;
            if (lastAmortizeDate == null) {
                //首次摊销, 需要包含起始月
                isHasThis = true;
                startAmortizeDate = detailVO.getCheckDate();
            } else {
                startAmortizeDate = lastAmortizeDate;
            }
            //本次摊销周期数
            int thisAmortizeNum = getMonthBetweenDate(startAmortizeDate, amortizeDate, isHasThis);
            detailVO.setThisAmortizeNum(thisAmortizeNum);

            BigDecimal netWorthTax = detailVO.getNetWorthTax();
            BigDecimal residualValueTax = detailVO.getResidualValueTax();
            BigDecimal netWorth = detailVO.getNetWorth();
            BigDecimal residualValue = detailVO.getResidualValue();
            BigDecimal usedAmortizeNum = ComputeUtil.toBigDecimal(detailVO.getUsedAmortizeNum());
            BigDecimal usePeriod = ComputeUtil.toBigDecimal(detailVO.getUsePeriod());
            BigDecimal usedAmortizeNumHaveThis = ComputeUtil.safeAdd(ComputeUtil.toBigDecimal(detailVO.getThisAmortizeNum()), usedAmortizeNum);

            BigDecimal picOriginalValueTax = ComputeUtil.safeDiv(detailVO.getOriginalValueTax(), ComputeUtil.toBigDecimal(detailVO.getUsePeriod()));
            BigDecimal picOriginalValue = ComputeUtil.safeDiv(detailVO.getOriginalValue(), ComputeUtil.toBigDecimal(detailVO.getUsePeriod()));

            //计算出来的理论本期摊销金额
            BigDecimal thisAmortizeMnyTax = ComputeUtil.safeMultiply(ComputeUtil.toBigDecimal(thisAmortizeNum), picOriginalValueTax);
            BigDecimal thisAmortizeMny = ComputeUtil.safeMultiply(ComputeUtil.toBigDecimal(thisAmortizeNum), picOriginalValue);

            // 净值-残值 = 剩余可摊销金额
            BigDecimal maxAmortizeMnyTax = ComputeUtil.safeSub(netWorthTax, residualValueTax);
            BigDecimal maxAmortizeMny = ComputeUtil.safeSub(netWorth, residualValue);
            if (ComputeUtil.isLessOrEqual(maxAmortizeMnyTax, BigDecimal.ZERO)) {
                //1、净值≤残值时，本期摊销金额 = 0
                detailVO.setAmortizeMnyTax(BigDecimal.ZERO);
                detailVO.setAmortizeMny(BigDecimal.ZERO);
            } else if (ComputeUtil.isGreaterOrEqual(usedAmortizeNumHaveThis, usePeriod)) {
                //2、已经生效的摊销期数 + 本次摊销周期数  ≥  材料设置的使用年限(月)时：本期摊销金额 = 材料净值 - 材料残值
                detailVO.setAmortizeMnyTax(ComputeUtil.safeSub(netWorthTax, residualValueTax));
                detailVO.setAmortizeMny(ComputeUtil.safeSub(netWorth, residualValue));
            } else if (ComputeUtil.isGreaterThan(thisAmortizeMnyTax, maxAmortizeMnyTax)) {
                detailVO.setAmortizeMnyTax(maxAmortizeMnyTax);
                detailVO.setAmortizeMny(maxAmortizeMny);
            } else {
                detailVO.setAmortizeMnyTax(thisAmortizeMnyTax);
                detailVO.setAmortizeMny(thisAmortizeMny);
            }
            //计算本次摊销比例
            BigDecimal amortizeRate = ComputeUtil.bigDecimalPercent(detailVO.getAmortizeMnyTax(), detailVO.getOriginalValueTax(), 2);
            detailVO.setAmortizeRate(amortizeRate);

        }
    }


    /**
     * 获取两个日期之间的所有月份 (年月)
     */
    public static int getMonthBetweenDate(Date d1, Date d2, boolean isHasThis) {
        Calendar c1 = Calendar.getInstance();
        Calendar c2 = Calendar.getInstance();
        c1.setTime(d1);
        c2.setTime(d2);
        int year1 = c1.get(Calendar.YEAR);
        int year2 = c2.get(Calendar.YEAR);
        int month1 = c1.get(Calendar.MONTH);
        int month2 = c2.get(Calendar.MONTH);
        int day1 = c1.get(Calendar.DAY_OF_MONTH);
        int day2 = c2.get(Calendar.DAY_OF_MONTH);
        // 获取年的差值
        int yearInterval = year1 - year2;
        // 如果 d1的 月-日 小于 d2的 月-日 那么 yearInterval-- 这样就得到了相差的年数
        if (month1 < month2 || month1 == month2 && day1 < day2) {
            yearInterval--;
        }
        // 获取月数差值
        int monthInterval = (month1 + 12) - month2;
        if (day1 < day2) {
            monthInterval--;
        }
        monthInterval %= 12;
        int monthsDiff = Math.abs(yearInterval * 12 + monthInterval);
        return isHasThis ? ++monthsDiff : monthsDiff;
    }

}
