package com.ejianc.business.asset.service.impl;

import com.alibaba.fastjson.JSONObject;
import com.baomidou.mybatisplus.core.conditions.update.LambdaUpdateWrapper;
import com.ejianc.business.asset.bean.AmortizeSetEntity;
import com.ejianc.business.asset.bean.AssetAllotOutDetailEntity;
import com.ejianc.business.asset.bean.AssetAmortizeDetailEntity;
import com.ejianc.business.asset.bean.AssetAmortizeEntity;
import com.ejianc.business.asset.mapper.AssetAmortizeMapper;
import com.ejianc.business.asset.service.IAmortizeSetService;
import com.ejianc.business.asset.service.IAssetAmortizeService;
import com.ejianc.business.asset.vo.AssetAmortizeDetailVO;
import com.ejianc.business.asset.vo.AssetAmortizeVO;
import com.ejianc.business.procost.api.ICostDetailApi;
import com.ejianc.business.procost.enums.SourceTypeEnum;
import com.ejianc.business.procost.vo.CostDetailVO;
import com.ejianc.framework.auth.session.SessionManager;
import com.ejianc.framework.core.exception.BusinessException;
import com.ejianc.framework.core.kit.collection.ListUtil;
import com.ejianc.framework.core.kit.mapper.BeanMapper;
import com.ejianc.framework.core.response.CommonResponse;
import com.ejianc.framework.core.util.ComputeUtil;
import com.ejianc.framework.skeleton.template.BaseServiceImpl;
import org.apache.commons.collections.CollectionUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import java.math.BigDecimal;
import java.util.*;
import java.util.stream.Collectors;

/**
 * 固定资产摊销
 *
 * @author generator
 */
@Service("assetAmortizeService")
public class AssetAmortizeServiceImpl extends BaseServiceImpl<AssetAmortizeMapper, AssetAmortizeEntity> implements IAssetAmortizeService {
    private Logger logger = LoggerFactory.getLogger(this.getClass());
    // 推送成本需要调用的接口 start
    @Autowired
    private ICostDetailApi costDetailApi;
    @Autowired
    private IAmortizeSetService amortizeSetService;
    @Autowired
    private SessionManager sessionManager;


    @Override
    public void getAmortizeSet(List<AssetAmortizeDetailVO> detailList, Long orgId) {
        Map<Long, Long> euipmentsMap = detailList.stream().collect(Collectors.toMap(AssetAmortizeDetailVO::getEquipmentId, AssetAmortizeDetailVO::getEquipmentTypeId, (key1, key2) -> key1));
        Map<Long, AmortizeSetEntity> amortizeSetMap = amortizeSetService.getSet(euipmentsMap, orgId);
        for (AssetAmortizeDetailVO detailVO : detailList) {
            Long equipmentId = detailVO.getEquipmentId();
            AmortizeSetEntity amortizeSetEntity = amortizeSetMap.get(equipmentId);
            if (amortizeSetEntity != null) {
                detailVO.setAmortizeSetId(amortizeSetEntity.getId());
                detailVO.setUsePeriod(amortizeSetEntity.getUsePeriod());
                detailVO.setResidualValueRate(amortizeSetEntity.getResidualValueRate());
            } else {
                detailVO.setErrMsg("未获取到该设备的摊销设置！");
            }
        }
    }

    /**
     * @param detailList
     * @param amortizeDate
     * @description: 计算本次摊销金额
     * @return: void
     * @author songlx
     * @date: 2022/7/19
     * 1、本次摊销比例：自动计算， 本次摊销金额/原值；
     * <p>
     * 2、本次摊销金额：
     * <p>
     * 2.1  根据后台设置的摊销方法自动计算。摊销周期数*原值/使用年限(月)；举例：1*（120/12）=10
     * <p>
     * 摊销周期：
     * <p>
     * 如果是之前没有摊销的设备，摊销周期数 = 当前摊销周期 - 入库月份+1；
     * <p>
     * 如果是已经摊销的设备，摊销周期数 = 当前摊销周期 - 上次摊销周期；
     * <p>
     * 2.2 弹框选择材料载入列表页面时，当选择材料的净值≤残值时，本期摊销金额 = 0。
     * <p>
     * 2.3 当摊销材料已经生效的摊销期数 + 本次摊销周期数  ≥  材料设置的使用年限(月)时：
     * <p>
     * 本期摊销金额 = 材料净值 - 材料残值。
     * <p>
     * 2.4 当摊销材料已经生效的摊销期数 + 本次摊销周期数 ＜ 材料设置的使用年限(月)时且根据公式计算的本次摊销金额＜材料净值 - 材料残值时：
     * <p>
     * 本期摊销金额 = 材料净值 - 材料残值。
     */
    @Override
    public void caclAmortizeMny(List<AssetAmortizeDetailVO> detailList, Date amortizeDate) {
        for (AssetAmortizeDetailVO detailVO : detailList) {
            //如果errMsg不为空，说明没有获取到摊销设置，不需要计算本次摊销金额和比例
            if(detailVO.getErrMsg() != null) continue;
            Date lastAmortizeDate = detailVO.getLastAmortizeDate();
            Date startAmortizeDate = null;
            boolean isHasThis = false;
            if (lastAmortizeDate == null) {
                //首次摊销, 需要包含起始月
                isHasThis = true;
                startAmortizeDate = detailVO.getCheckDate();
            } else {
                startAmortizeDate = lastAmortizeDate;
            }
            //本次摊销周期数
            int thisAmortizeNum = getMonthBetweenDate(startAmortizeDate, amortizeDate, isHasThis);
            detailVO.setThisAmortizeNum(thisAmortizeNum);

            BigDecimal netWorthTax = detailVO.getNetWorthTax();
            BigDecimal residualValueTax = detailVO.getResidualValueTax();
            BigDecimal netWorth = detailVO.getNetWorth();
            BigDecimal residualValue = detailVO.getResidualValue();
            BigDecimal usedAmortizeNum = ComputeUtil.toBigDecimal(detailVO.getUsedAmortizeNum());
            BigDecimal usePeriod = ComputeUtil.toBigDecimal(detailVO.getUsePeriod());
            BigDecimal usedAmortizeNumHaveThis = ComputeUtil.safeAdd(ComputeUtil.toBigDecimal(detailVO.getThisAmortizeNum()), usedAmortizeNum);

            BigDecimal picOriginalValueTax = ComputeUtil.safeDiv(detailVO.getOriginalValueTax(), ComputeUtil.toBigDecimal(detailVO.getUsePeriod()));
            BigDecimal picOriginalValue = ComputeUtil.safeDiv(detailVO.getOriginalValue(), ComputeUtil.toBigDecimal(detailVO.getUsePeriod()));

            //计算出来的理论本期摊销金额
            BigDecimal thisAmortizeMnyTax = ComputeUtil.safeMultiply(ComputeUtil.toBigDecimal(thisAmortizeNum), picOriginalValueTax);
            BigDecimal thisAmortizeMny = ComputeUtil.safeMultiply(ComputeUtil.toBigDecimal(thisAmortizeNum), picOriginalValue);

            // 净值-残值 = 剩余可摊销金额
            BigDecimal maxAmortizeMnyTax = ComputeUtil.safeSub(netWorthTax, residualValueTax);
            BigDecimal maxAmortizeMny = ComputeUtil.safeSub(netWorth, residualValue);
            if (ComputeUtil.isLessOrEqual(maxAmortizeMnyTax, BigDecimal.ZERO)) {
                //1、净值≤残值时，本期摊销金额 = 0
                detailVO.setAmortizeMnyTax(BigDecimal.ZERO);
                detailVO.setAmortizeMny(BigDecimal.ZERO);
            } else if (ComputeUtil.isGreaterOrEqual(usedAmortizeNumHaveThis, usePeriod)) {
                //2、已经生效的摊销期数 + 本次摊销周期数  ≥  材料设置的使用年限(月)时：本期摊销金额 = 材料净值 - 材料残值
                detailVO.setAmortizeMnyTax(ComputeUtil.safeSub(netWorthTax, residualValueTax));
                detailVO.setAmortizeMny(ComputeUtil.safeSub(netWorth, residualValue));
            } else if (ComputeUtil.isGreaterThan(thisAmortizeMnyTax, maxAmortizeMnyTax)) {
                detailVO.setAmortizeMnyTax(maxAmortizeMnyTax);
                detailVO.setAmortizeMny(maxAmortizeMny);
            } else {
                detailVO.setAmortizeMnyTax(thisAmortizeMnyTax);
                detailVO.setAmortizeMny(thisAmortizeMny);
            }
            //计算本次摊销比例
            BigDecimal amortizeRate = ComputeUtil.bigDecimalPercent(detailVO.getAmortizeMnyTax(), detailVO.getOriginalValueTax(), 2);
            detailVO.setAmortizeRate(amortizeRate);

        }
    }

    @Override
    public void costPush(AssetAmortizeEntity assetAmortizeEntity) {
        logger.info("开始costPush");
        List<AssetAmortizeDetailEntity> detailList = assetAmortizeEntity.getDetailList();
        String newRelationFlag = "1";
        if(CollectionUtils.isNotEmpty(detailList)){
            for (AssetAmortizeDetailEntity entity : detailList) {
                if (null == entity.getSubjectId() || null == entity.getWbsId()) {
                    newRelationFlag = "0";
                    logger.info("getSubjectId"+entity.getSubjectId());
                    logger.info("getWbsId"+entity.getWbsId());
                    break;
                }
            }
        }
        if (ListUtil.isEmpty(detailList)) {
            newRelationFlag = "0";
        }
        logger.info("newRelationFlag"+newRelationFlag);
        //更新是否关联
        LambdaUpdateWrapper<AssetAmortizeEntity> updateWrapper = new LambdaUpdateWrapper<>();
        updateWrapper.in(AssetAmortizeEntity::getId,assetAmortizeEntity.getId());
        updateWrapper.set(AssetAmortizeEntity::getRelationFlag,newRelationFlag);//(1:是，0：否)
        super.update(updateWrapper);
        //判断之前的单据是否关联
        String oldRelationFlag = assetAmortizeEntity.getRelationFlag();
        //之前已关联
        if ("1".equals(oldRelationFlag)) {
            if ("1".equals(newRelationFlag)) {
                logger.info("之前已关联"+newRelationFlag);
                saveCost(assetAmortizeEntity);
            }
            if (!"1".equals(newRelationFlag)) {
                //删除成本中心之前的数据
                logger.info("删除成本中心之前的数据-固定资产调出Id---{}",assetAmortizeEntity.getId());
                CommonResponse<String> commonResponse = costDetailApi.deleteSubject(assetAmortizeEntity.getId());
                logger.info("结果"+ JSONObject.toJSONString(commonResponse));
                if(!commonResponse.isSuccess()){
                    throw new BusinessException(commonResponse.getMsg());
                }
            }
        }
        //之前未关联
        if ("0".equals(oldRelationFlag)) {
            if ("1".equals(newRelationFlag)) {
                logger.info("之前未关联"+newRelationFlag);
                saveCost(assetAmortizeEntity);
            }
        }
    }

    private void saveCost(AssetAmortizeEntity assetAmortizeEntity){
        //明细
        List<CostDetailVO> costDetailVOList = new ArrayList<>();
        List<AssetAmortizeDetailEntity> detailList = assetAmortizeEntity.getDetailList();
        if(CollectionUtils.isNotEmpty(detailList)){
            for (AssetAmortizeDetailEntity entity : detailList) {
                CostDetailVO costDetailVO = new CostDetailVO();
                costDetailVO.setSubjectId(entity.getSubjectId());
                costDetailVO.setSubjectCode(entity.getSubjectCode());
                costDetailVO.setSubjectName(entity.getSubjectName());
                costDetailVO.setWbsId(entity.getWbsId());
                costDetailVO.setWbsCode(entity.getWbsCode());
                costDetailVO.setWbsName(entity.getWbsName());
                costDetailVO.setSourceId(entity.getPid());
                costDetailVO.setSourceDetailId(entity.getId());
                costDetailVO.setHappenTaxMny(entity.getAmortizeMnyTax());
                costDetailVO.setHappenMny(entity.getAmortizeMny());
                costDetailVO.setHappenDate(assetAmortizeEntity.getBillDate());
                costDetailVO.setCreateUserName(sessionManager.getUserContext().getUserName());
                costDetailVO.setSourceType("ASSET_AMORTIZE");
                costDetailVO.setSourceTabType("ASSET_AMORTIZE_SUB");
                costDetailVO.setProjectId(assetAmortizeEntity.getProjectId());
                //新加的
                costDetailVO.setSourceBillCode(assetAmortizeEntity.getBillCode());
                costDetailVO.setSourceBillName(SourceTypeEnum.固定资产摊销.getTypeName());
                costDetailVO.setSourceBillUrl("/ejc-proequipmentp-frontend/#/assetAmortize/card?id="+assetAmortizeEntity.getId());
                costDetailVOList.add(costDetailVO);
            }
        }

        //成本中心
        if (ListUtil.isNotEmpty(costDetailVOList)) {
            logger.info("推送数据--------"+JSONObject.toJSONString(costDetailVOList));
            CommonResponse<String> stringCommonResponse = costDetailApi.saveSubject(costDetailVOList);
            logger.info("推送结果--------"+JSONObject.toJSONString(stringCommonResponse));
            if (stringCommonResponse.isSuccess()) {
            } else {
                throw new BusinessException(stringCommonResponse.getMsg());
            }
        }
    }

    @Override
    public CommonResponse<AssetAmortizeVO> pushCost(AssetAmortizeVO assetAmortizeVO) {
        AssetAmortizeEntity assetAmortizeEntity = baseMapper.selectById(assetAmortizeVO.getId());
        if(CollectionUtils.isNotEmpty(assetAmortizeVO.getDetailList())){
            List<AssetAmortizeDetailEntity> assetAmortizeDetailEntities = BeanMapper.mapList(assetAmortizeEntity.getDetailList(), AssetAmortizeDetailEntity.class);
            assetAmortizeEntity.setDetailList(assetAmortizeDetailEntities);
        }
        super.saveOrUpdate(assetAmortizeEntity, false);
        //推送数据
        this.costPush(assetAmortizeEntity);
        return CommonResponse.success(BeanMapper.map(assetAmortizeEntity,AssetAmortizeVO.class));
    }


    /**
     * 获取两个日期之间的所有月份 (年月)
     */
    public static int getMonthBetweenDate(Date d1, Date d2, boolean isHasThis) {
        Calendar c1 = Calendar.getInstance();
        Calendar c2 = Calendar.getInstance();
        c1.setTime(d1);
        c2.setTime(d2);
        int year1 = c1.get(Calendar.YEAR);
        int year2 = c2.get(Calendar.YEAR);
        int month1 = c1.get(Calendar.MONTH);
        int month2 = c2.get(Calendar.MONTH);
        int day1 = c1.get(Calendar.DAY_OF_MONTH);
        int day2 = c2.get(Calendar.DAY_OF_MONTH);
        // 获取年的差值
        int yearInterval = year1 - year2;
        // 如果 d1的 月-日 小于 d2的 月-日 那么 yearInterval-- 这样就得到了相差的年数
        if (month1 < month2 || month1 == month2 && day1 < day2) {
            yearInterval--;
        }
        // 获取月数差值
        int monthInterval = (month1 + 12) - month2;
        if (day1 < day2) {
            monthInterval--;
        }
        monthInterval %= 12;
        int monthsDiff = Math.abs(yearInterval * 12 + monthInterval);
        return isHasThis ? ++monthsDiff : monthsDiff;
    }

}
