package com.ejianc.business.pricelib.service.impl;

import com.ejianc.business.contractbase.pool.enums.ContractPerformanceStateEnum;
import com.ejianc.business.pricelib.service.IPriceContractService;
import com.ejianc.business.pricelib.service.IPriceGuideDetailService;
import com.ejianc.business.pricelib.vo.PriceContractCollectVO;
import com.ejianc.business.pricelib.bean.PriceContractEntity;
import com.ejianc.business.pricelib.bean.PriceGuideDetailEntity;
import com.ejianc.framework.core.kit.mapper.BeanMapper;
import com.ejianc.framework.core.response.Parameter;
import com.ejianc.framework.core.response.QueryParam;
import org.apache.commons.collections.CollectionUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import com.ejianc.framework.skeleton.template.BaseServiceImpl;

import com.ejianc.business.pricelib.mapper.PriceContractCollectMapper;
import com.ejianc.business.pricelib.bean.PriceContractCollectEntity;
import com.ejianc.business.pricelib.service.IPriceContractCollectService;

import java.math.BigDecimal;
import java.util.*;
import java.util.function.Function;
import java.util.stream.Collectors;

/**
 * 设备购置价格库-根据清单id汇总表
 * 
 * @author generator
 * 
 */
@Service("priceContractCollectService")
public class PriceContractCollectServiceImpl extends BaseServiceImpl<PriceContractCollectMapper, PriceContractCollectEntity> implements IPriceContractCollectService{

    private Logger logger = LoggerFactory.getLogger(this.getClass());

    @Autowired
    private IPriceContractService priceContractService;
    @Autowired
    private IPriceGuideDetailService priceGuideDetailService;


    /**
     * 保存价格库内容到历史表
     */
    @Override
    public void savePriceContractToHistoryByTiming() {
        logger.info("---历史价格init开始---");
        //1.先执行全删除操作
        logger.info("---数据全删除操作开始---");
        baseMapper.delPriceContractCollectAll();
        logger.info("---数据插入操作开始---");
        //2.执行数据插入
        //查询价格库中未被移出的数据
        QueryParam queryParam = new QueryParam();
        queryParam.getParams().put("outFlag", new Parameter(QueryParam.EQ, 0));
        queryParam.getParams().put("performanceStatus", new Parameter(QueryParam.IN, Arrays.asList(ContractPerformanceStateEnum.履约中.getStateCode(), ContractPerformanceStateEnum.已终止.getStateCode(), ContractPerformanceStateEnum.已冻结.getStateCode(), ContractPerformanceStateEnum.已结束.getStateCode())));
        List<PriceContractEntity> priceContractList = priceContractService.queryList(queryParam);
        if (CollectionUtils.isEmpty(priceContractList)){
            return;
        }
        List<PriceContractCollectEntity> priceContractCollectList = new ArrayList<>();

        // 按照档案Id分组
        Map<Long, List<PriceContractEntity>> priceContractMap = priceContractList.stream()
                .collect(Collectors.groupingBy(PriceContractEntity::getDocId));
        // 汇总信息
        for (Map.Entry<Long, List<PriceContractEntity>> entry : priceContractMap.entrySet()) {

            List<PriceContractEntity> value = entry.getValue();
            List<BigDecimal> priceList = value.stream().map(PriceContractEntity::getContractPrice).collect(Collectors.toList());
            List<BigDecimal> taxPriceList = value.stream().map(PriceContractEntity::getContractTaxPrice).collect(Collectors.toList());

            PriceContractEntity priceContract = value.get(0);
            if (priceContract == null) continue;
            // 合同最高价(无税)
            BigDecimal maxPrice = Collections.max(priceList).divide(BigDecimal.ONE, 4, BigDecimal.ROUND_HALF_UP);
            // 合同最低价(无税)
            BigDecimal minPrice = Collections.min(priceList).divide(BigDecimal.ONE, 4, BigDecimal.ROUND_HALF_UP);
            String priceArea = minPrice + "-" + maxPrice; // 合同价格区间（无税）
            // 合同最高价（含税）
            BigDecimal taxMaxPrice = Collections.max(taxPriceList).divide(BigDecimal.ONE, 4, BigDecimal.ROUND_HALF_UP);
            // 合同最低价（含税）
            BigDecimal taxMinPrice = Collections.min(taxPriceList).divide(BigDecimal.ONE, 4, BigDecimal.ROUND_HALF_UP);
            String taxPriceArea = taxMinPrice + "-" + taxMaxPrice; // 合同价格区间（含税）

            // 基础字段赋值
            PriceContractCollectEntity priceContractCollectEntity;
            priceContractCollectEntity = new PriceContractCollectEntity();
            priceContractCollectEntity.setDocId(priceContract.getDocId());
            priceContractCollectEntity.setDocCode(priceContract.getDocCode());
            priceContractCollectEntity.setDocName(priceContract.getDocName());
            priceContractCollectEntity.setDocCategoryId(priceContract.getDocCategoryId());
            priceContractCollectEntity.setDocCategoryName(priceContract.getDocCategoryName());
            priceContractCollectEntity.setDocInnerCode(priceContract.getDocInnerCode());
            priceContractCollectEntity.setUnitId(priceContract.getUnitId());
            priceContractCollectEntity.setUnitName(priceContract.getUnitName());
            priceContractCollectEntity.setPriceTypeName(priceContract.getPriceTypeName());
            priceContractCollectEntity.setEquipmentHeight(priceContract.getEquipmentHeight());
            priceContractCollectEntity.setContractMaxPrice(maxPrice);
            priceContractCollectEntity.setContractMinPrice(minPrice);
            priceContractCollectEntity.setContractPriceArea(priceArea);
            priceContractCollectEntity.setContractTaxMaxPrice(taxMaxPrice);
            priceContractCollectEntity.setContractTaxMinPrice(taxMinPrice);
            priceContractCollectEntity.setContractTaxPriceArea(taxPriceArea);
            priceContractCollectList.add(priceContractCollectEntity);
        }

        logger.info("保存数据大小{}条",priceContractCollectList.size());
        super.saveOrUpdateBatch(priceContractCollectList);
        logger.info("---数据插入操作结束---");
    }

    /**
     * 根据档案id查询多个合同清单历史价格 和  指导价
     * @param docIds
     * @return
     */
    @Override
    public List<PriceContractCollectVO> getPriceContractByDocIds(List<Long> docIds) {

        QueryParam queryParam = new QueryParam();
        queryParam.getParams().put("docId", new Parameter(QueryParam.IN, docIds));
        List<PriceContractCollectEntity> priceContractCollectEntities = super.queryList(queryParam);
        List<PriceContractCollectVO> priceContractCollectVOList = null;
        if (CollectionUtils.isNotEmpty(priceContractCollectEntities)){
            priceContractCollectVOList = BeanMapper.mapList(priceContractCollectEntities, PriceContractCollectVO.class);
        }
        //设置指导价
        List<PriceContractCollectVO> guideDetailList = this.getGuidePriceAreaByDocIds(docIds);
        if (CollectionUtils.isNotEmpty(guideDetailList)){
            Map<Long, PriceContractCollectVO> guidePriceMap = guideDetailList.stream().collect(Collectors.toMap(PriceContractCollectVO::getDocId, Function.identity(), (key1, key2) -> key2));
            if (CollectionUtils.isNotEmpty(priceContractCollectVOList)){
                for (PriceContractCollectVO priceContractCollectVO : priceContractCollectVOList) {
                    if (guidePriceMap.containsKey(priceContractCollectVO.getDocId())){
                        priceContractCollectVO.setGuidePriceArea(guidePriceMap.get(priceContractCollectVO.getDocId()).getGuidePriceArea());
                        priceContractCollectVO.setGuideTaxPriceArea(guidePriceMap.get(priceContractCollectVO.getDocId()).getGuideTaxPriceArea());
                        guidePriceMap.remove(priceContractCollectVO.getDocId());
                    }
                }
            }
            if (guidePriceMap != null){
                List<PriceContractCollectVO> finalPriceContractCollectVOList = priceContractCollectVOList;
                guidePriceMap.keySet().stream().forEach(key -> {
                    PriceContractCollectVO priceContractCollectVO = new PriceContractCollectVO();
                    priceContractCollectVO.setDocId(guidePriceMap.get(key).getDocId());
                    priceContractCollectVO.setGuidePriceArea(guidePriceMap.get(key).getGuidePriceArea());
                    priceContractCollectVO.setGuideTaxPriceArea(guidePriceMap.get(key).getGuideTaxPriceArea());
                    finalPriceContractCollectVOList.add(priceContractCollectVO);
                });
            }
        }
        return priceContractCollectVOList;
    }

    @Override
    public List<PriceContractCollectVO> getGuidePriceAreaByDocIds(List<Long> docIds) {
        List<PriceGuideDetailEntity> list = priceGuideDetailService.queryPriceGuideDetailByDocIds(docIds);
        List<PriceContractCollectVO> PriceContractCollectVOList = new ArrayList<>();
        if (com.baomidou.mybatisplus.core.toolkit.CollectionUtils.isNotEmpty(list)){
            for (PriceGuideDetailEntity guide : list) {
                PriceContractCollectVO priceContractCollectVO = new PriceContractCollectVO();
                BigDecimal min = guide.getMinPrice() == null ? BigDecimal.ZERO : guide.getMinPrice();
                BigDecimal max = guide.getMaxPrice() == null ? BigDecimal.ZERO : guide.getMaxPrice();
                min = min.divide(BigDecimal.ONE, 4, BigDecimal.ROUND_HALF_UP);
                max = max.divide(BigDecimal.ONE, 4, BigDecimal.ROUND_HALF_UP);
                priceContractCollectVO.setGuidePriceArea(min + "-" + max);
                min = guide.getMinTaxPrice() == null ? BigDecimal.ZERO : guide.getMinTaxPrice();
                max = guide.getMaxTaxPrice() == null ? BigDecimal.ZERO : guide.getMaxTaxPrice();
                min = min.divide(BigDecimal.ONE, 4, BigDecimal.ROUND_HALF_UP);
                max = max.divide(BigDecimal.ONE, 4, BigDecimal.ROUND_HALF_UP);
                priceContractCollectVO.setGuideTaxPriceArea(min + "-" + max);
                priceContractCollectVO.setDocId(guide.getDocId());
                PriceContractCollectVOList.add(priceContractCollectVO);
            }
        }
        return PriceContractCollectVOList;
    }
}
