package com.ejianc.business.profinance.plan.service.impl;

import com.baomidou.mybatisplus.core.conditions.query.LambdaQueryWrapper;
import com.baomidou.mybatisplus.core.toolkit.Wrappers;
import com.ejianc.business.profinance.bean.RegisterDetailEntity;
import com.ejianc.business.profinance.bean.RegulateEntity;
import com.ejianc.business.profinance.enums.PaymentContractEnum;
import com.ejianc.business.profinance.plan.bean.*;
import com.ejianc.business.profinance.plan.service.*;
import com.ejianc.business.profinance.service.IRegisterDetailService;
import com.ejianc.business.profinance.service.IRegulateService;
import com.ejianc.foundation.support.api.IBillTypeApi;
import com.ejianc.framework.auth.session.SessionManager;
import com.ejianc.framework.auth.session.UserContext;
import com.ejianc.framework.core.exception.BusinessException;
import com.ejianc.framework.core.response.BillStateEnum;
import com.ejianc.framework.core.response.CommonResponse;
import com.ejianc.framework.skeleton.billState.service.ICommonBusinessService;
import org.apache.commons.collections.CollectionUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.stereotype.Service;

import java.math.BigDecimal;
import java.util.*;
import java.util.stream.Collectors;

@Service("plan")
public class PlanBpmServiceImpl implements ICommonBusinessService {
	private final Logger logger = LoggerFactory.getLogger(this.getClass());
	private final SessionManager sessionManager;
	private final IBillTypeApi billTypeApi;
	private final IPlanService service;
	private final IPlanContractService planContractService;
	private final IPlanMaterialService planMaterialService;
	private final IPlanTemporaryService planTemporaryService;
	private final IPlanFeeService planFeeService;
	private final IRegulateService regulateService;
	private final IRegisterDetailService registerDetailService;
	private static final String SOURCE_TYPE = "项目资金计划";
	private static final String LINK_URL = "/ejc-profinance-frontend/#/plan/card?id=";
	private static final String CHECK_CLOSE_STATE_ERROR = "操作失败！该计划存在已关闭的数据，不允许撤销审批。";

	public PlanBpmServiceImpl(SessionManager sessionManager,
	                          IBillTypeApi billTypeApi,
	                          IPlanService service,
	                          IPlanContractService planContractService,
	                          IPlanMaterialService planMaterialService,
	                          IPlanTemporaryService planTemporaryService,
	                          IPlanFeeService planFeeService,
	                          IRegulateService regulateService,
	                          IRegisterDetailService registerDetailService) {
		this.sessionManager = sessionManager;
		this.billTypeApi = billTypeApi;
		this.service = service;
		this.planContractService = planContractService;
		this.planMaterialService = planMaterialService;
		this.planTemporaryService = planTemporaryService;
		this.planFeeService = planFeeService;
		this.regulateService = regulateService;
		this.registerDetailService = registerDetailService;
	}

	/**
	 * 终审审核完回调
	 *
	 * @param billId 单据id
	 * @param state  单据状态
	 *
	 * @return 响应信息
	 */
	@Override
	public CommonResponse<String> afterApprovalProcessor(Long billId, Integer state, String billTypeCode) {
		logger.info("终审审核完回调--start，billId={},state={},billTypeCode={}", billId, state, billTypeCode);

		// 审批通过/已提交
		PlanEntity entity = service.selectById(billId);
		if (entity == null) {
			throw new BusinessException("查询不到单据信息");
		}

		UserContext userContext = sessionManager.getUserContext();

		// 区分提交和审批
		if (state.equals(BillStateEnum.COMMITED_STATE.getBillStateCode())) {
			// 直审更新提交相关字段
			entity.setCommitDate(new Date());
			entity.setCommitUserCode(userContext.getUserCode());
			entity.setCommitUserName(userContext.getUserName());
		}
		entity.setBillStateName(BillStateEnum.getEnumByStateCode(state).getDescription());
		// 生效时间
		entity.setEffectiveDate(new Date());

		// 执行更新
		service.saveOrUpdate(entity, false);

		// 更新子表单据状态
		updateSubBillState(billId, state);

		// 同步项目资金计划子表数据到付款管理的中间表
		insertMiddleTable(billId);

		logger.info("终审审核完回调--end");
		return CommonResponse.success("终审审核完回调成功");
	}

	/**
	 * 有审批流的撤回前回调
	 *
	 * @param billId 单据id
	 * @param state  单据状态
	 *
	 * @return 响应信息
	 */
	@Override
	public CommonResponse<String> beforeHasBpmBack(Long billId, Integer state, String billTypeCode) {
		return bpmBackCheck(billId, state, billTypeCode);
	}

	/**
	 * 弃审前事件回调
	 *
	 * @param billId 单据id
	 * @param state  单据状态
	 *
	 * @return 响应信息
	 */
	@Override
	public CommonResponse<String> beforeAbstainingProcessor(Long billId, Integer state, String billTypeCode) {
		return bpmBackCheck(billId, state, billTypeCode);
	}

	private CommonResponse<String> bpmBackCheck(Long billId, Integer state, String billTypeCode) {
		PlanEntity entity = service.selectById(billId);
		if (entity == null) {
			throw new BusinessException("查询不到单据信息");
		}

		// 是否被下游引用
		CommonResponse<String> res = billTypeApi.checkQuote(billTypeCode, billId);
		logger.info("平台返回查询被引用情况：{}，{}", res.isSuccess(), res.getMsg());

		if (!res.isSuccess()) {
			return CommonResponse.error("当前单据已被项目还款单引用，不能撤回/弃审！");
		}

		LambdaQueryWrapper<PlanEntity> lambdaQuery = Wrappers.lambdaQuery();
		lambdaQuery.eq(PlanEntity::getProjectId, entity.getProjectId());
		lambdaQuery.orderByDesc(PlanEntity::getPlanDate);
		List<PlanEntity> entityList = service.list(lambdaQuery);
		if (CollectionUtils.isNotEmpty(entityList) && !Objects.equals(entity.getId(), entityList.get(0).getId())) {
			throw new BusinessException("操作失败！该单据不是项目的最新计划，不允许撤销审批。");
		}

		// 校验是否存在对应的【付款登记】单据，无论单据状态是自由态还是审批通过、审批中等状态，如果是，则不允许撤销审批，并提示“操作失败！该计划已存在支付信息，不允许撤销审批。”
		LambdaQueryWrapper<RegisterDetailEntity> queryWrapper = Wrappers.lambdaQuery();
		queryWrapper.exists("select 1 from `ejc-profinance`.ejc_payment_register_detail where source_parent_id = " + billId);
		int count = registerDetailService.count(queryWrapper);
		if (count > 0) {
			return CommonResponse.error("操作失败！该计划已存在支付信息，不允许撤销审批。");
		}

		// 校验是否存在子表数据的“关闭状态”为“已关闭”，如果存在，则不允许撤销审批，并提示“操作失败！该计划存在已不关闭的数据，不允许撤销审批。”
		checkCloseState(entity);

		// 删除中间表待处理数据
		delMidData(billId);

		// 更新子表单据状态
		updateSubBillState(billId, state);

		return CommonResponse.success("单据撤回/弃审成功");
	}

	private void checkCloseState(PlanEntity planEntity) {
		List<PlanContractEntity> planContractList = planEntity.getPlanContractList().stream().filter(e -> e.getContractPlanPayment() != null).collect(Collectors.toList());
		List<PlanMaterialEntity> planMaterialList = planEntity.getPlanMaterialList();
		List<PlanTemporaryEntity> planTemporaryList = planEntity.getPlanTemporaryList();
		List<PlanFeeEntity> planFeeList = planEntity.getPlanFeeList();

		if (CollectionUtils.isNotEmpty(planContractList)) {
			for (PlanContractEntity entity : planContractList) {
				if ("已关闭".equals(entity.getCloseState())) {
					throw new BusinessException(CHECK_CLOSE_STATE_ERROR);
				}
			}
		}

		if (CollectionUtils.isNotEmpty(planMaterialList)) {
			for (PlanMaterialEntity entity : planMaterialList) {
				if ("已关闭".equals(entity.getCloseState())) {
					throw new BusinessException(CHECK_CLOSE_STATE_ERROR);
				}
			}
		}

		if (CollectionUtils.isNotEmpty(planTemporaryList)) {
			for (PlanTemporaryEntity entity : planTemporaryList) {
				if ("已关闭".equals(entity.getCloseState())) {
					throw new BusinessException(CHECK_CLOSE_STATE_ERROR);
				}
			}
		}

		if (CollectionUtils.isNotEmpty(planFeeList)) {
			for (PlanFeeEntity entity : planFeeList) {
				if ("已关闭".equals(entity.getCloseState())) {
					throw new BusinessException(CHECK_CLOSE_STATE_ERROR);
				}
			}
		}
	}

	/**
	 * 删除中间表待处理数据
	 *
	 * @param planId 项目资金计划id
	 */
	private void delMidData(Long planId) {
		LambdaQueryWrapper<RegulateEntity> queryWrapper = Wrappers.lambdaQuery();
		queryWrapper.eq(RegulateEntity::getSourceParentId, planId);
		regulateService.remove(queryWrapper);
	}

	/**
	 * 更新子表单据状态
	 *
	 * @param billId 项目资金计划id
	 * @param state  单据状态
	 */
	private void updateSubBillState(Long billId, Integer state) {
		planContractService.updateBillState(billId, state);
		planMaterialService.updateBillState(billId, state);
		planTemporaryService.updateBillState(billId, state);
		planFeeService.updateBillState(billId, state);
	}

	/**
	 * 插入中间表
	 *
	 * @param planId 项目资金计划id
	 */
	private void insertMiddleTable(Long planId) {
		PlanEntity planEntity = service.selectById(planId);
		if (planEntity == null) {
			throw new BusinessException("查询不到项目资金计划信息");
		}
		List<PlanContractEntity> planContractList = planEntity.getPlanContractList().stream().filter(e -> e.getContractId() != null).collect(Collectors.toList());
		List<PlanMaterialEntity> planMaterialList = planEntity.getPlanMaterialList();
		List<PlanTemporaryEntity> planTemporaryList = planEntity.getPlanTemporaryList();
		List<PlanFeeEntity> planFeeList = planEntity.getPlanFeeList();

		Collection<RegulateEntity> entities = new ArrayList<>();

		if (CollectionUtils.isNotEmpty(planContractList)) {
			insertPlanContractList(planEntity, planContractList, entities);
		}

		if (CollectionUtils.isNotEmpty(planMaterialList)) {
			insertPlanMaterialList(planEntity, planMaterialList, entities);
		}

		if (CollectionUtils.isNotEmpty(planTemporaryList)) {
			insertPlanTemporaryList(planEntity, planTemporaryList, entities);
		}

		if (CollectionUtils.isNotEmpty(planFeeList)) {
			insertPlanFeeList(planEntity, planFeeList, entities);
		}

		if (CollectionUtils.isNotEmpty(entities)) {
			regulateService.saveBatch(entities, 10);
		}
	}

	private static void insertPlanContractList(PlanEntity p, List<PlanContractEntity> planContractList, Collection<RegulateEntity> entities) {
		for (PlanContractEntity e : planContractList) {
			RegulateEntity regulate = new RegulateEntity();

			// 来源
			regulate.setSourceType(SOURCE_TYPE);
			// 来源id
			regulate.setSourceId(e.getId());
			// 来源父id
			regulate.setSourceParentId(e.getPlanId());
			// 穿透地址
			regulate.setRuteLinkUrl(LINK_URL + e.getPlanId());
			// 穿透名字
			regulate.setRuteLinkName(SOURCE_TYPE);
			// 项目部id
			regulate.setOrgId(p.getOrgId());
			// 项目部编码
			regulate.setOrgCode(p.getOrgCode());
			// 项目部名称
			regulate.setOrgName(p.getOrgName());


			// 计划单号
			regulate.setBillCode(p.getBillCode());
			// 本次计划日期
			regulate.setPlanDate(p.getPlanDate());
			// 项目所属组织
			regulate.setParentOrgId(p.getParentOrgId());
			regulate.setParentOrgCode(p.getParentOrgCode());
			regulate.setParentOrgName(p.getParentOrgName());
			// 项目编码（隐藏）
			regulate.setProjectId(p.getProjectId());
			regulate.setProjectCode(p.getProjectCode());
			// 项目名称
			regulate.setProjectName(p.getProjectName());
			// 支出类型
			regulate.setPaymentContractFlag(PaymentContractEnum.合同付款.getCode());
			// 合同编码
			regulate.setContractId(e.getContractId());
			regulate.setContractCode(e.getContractCode());
			// 合同名称
			regulate.setContractName(e.getContractName());
			// 收款单位
			regulate.setSupplierOrgId(e.getContractPartybId());
			regulate.setSupplierOrgCode(null);
			regulate.setSupplierOrgName(e.getContractPartybName());
			// 无合同费用类型
			regulate.setFeeTypeId(null);
			regulate.setFeeTypeCode(null);
			regulate.setFeeTypeName(null);
			// 结算金额（隐藏）
			regulate.setSettleMny(e.getContractTotalSettleMny());
			// 支付对象（隐藏）
			regulate.setPayTargetType(null);
			// 收款人
			regulate.setSupplierUserId(null);
			regulate.setSupplierUserCode(null);
			regulate.setSupplierUserName(null);
			// 收款单位/收款人
			regulate.setSupplierName(e.getContractPartybName());
			// 计划支付金额
			regulate.setPlanPayMny(e.getContractPlanPayment());
			// 已支付金额
			regulate.setPaidMny(e.getTotalActualPaymentMny());
			// 未支付金额（隐藏）
			regulate.setUnpaidMny(e.getUnpaidMny());
			// 关闭状态（隐藏）
			regulate.setCloseState(e.getCloseState());
			// 关闭时间（隐藏）
			regulate.setCloseDateTime(e.getCloseDateTime());
			// 关闭人（隐藏）
			regulate.setCloseUserId(e.getCloseUserId());
			regulate.setCloseUserCode(e.getCloseUserCode());
			regulate.setCloseUserName(e.getCloseUserName());

			// 支付状态
			regulate.setPaymentState("未支付");

			// 合同大类id
			regulate.setSourceCategoryType(e.getContractSourceType());
			// 合同大类名称
			regulate.setSourceCategoryTypeName(e.getContractSourceTypeName());
			// 合同类别id
			regulate.setContractCategoryId(e.getContractCategoryId());
			// 合同类别名称
			regulate.setContractCategoryName(e.getContractCategoryName());
			// 合同金额
			regulate.setContractTaxMny(e.getContractTaxMny());
			// 未支付金额(元)
			regulate.setUnpaidMny(e.getContractPlanPayment());
			// 已支付金额(元)
			regulate.setPaidMny(BigDecimal.ZERO);
			// 来源单据编号
			regulate.setSourceBillCode(e.getContractCode());
			// 来源穿透地址
			regulate.setSourceLinkUrl(e.getContractRouteUrl());
			// 来源穿透名字
			regulate.setSourceLinkName(PaymentContractEnum.合同付款.getName());

			entities.add(regulate);
		}
	}


	private static void insertPlanMaterialList(PlanEntity p, List<PlanMaterialEntity> planMaterialList, Collection<RegulateEntity> entities) {
		for (PlanMaterialEntity e : planMaterialList) {
			RegulateEntity regulate = new RegulateEntity();

			// 来源
			regulate.setSourceType(SOURCE_TYPE);
			// 来源id
			regulate.setSourceId(e.getId());
			// 来源父id
			regulate.setSourceParentId(e.getPlanId());
			// 穿透地址
			regulate.setRuteLinkUrl(LINK_URL + e.getPlanId());
			// 穿透名字
			regulate.setRuteLinkName(SOURCE_TYPE);
			// 项目部id
			regulate.setOrgId(p.getOrgId());
			// 项目部编码
			regulate.setOrgCode(p.getOrgCode());
			// 项目部名称
			regulate.setOrgName(p.getOrgName());


			// 计划单号
			regulate.setBillCode(p.getBillCode());
			// 本次计划日期
			regulate.setPlanDate(p.getPlanDate());
			// 项目所属组织
			regulate.setParentOrgId(p.getParentOrgId());
			regulate.setParentOrgCode(p.getParentOrgCode());
			regulate.setParentOrgName(p.getParentOrgName());
			// 项目编码（隐藏）
			regulate.setProjectId(p.getProjectId());
			regulate.setProjectCode(p.getProjectCode());
			// 项目名称
			regulate.setProjectName(p.getProjectName());
			// 支出类型
			regulate.setPaymentContractFlag(PaymentContractEnum.零星材料.getCode());
			// 合同编码
			regulate.setContractId(null);
			regulate.setContractCode(null);
			// 合同名称
			regulate.setContractName(null);
			// 收款单位
			regulate.setSupplierOrgId(e.getMaterialPartybId());
			regulate.setSupplierOrgCode(null);
			regulate.setSupplierOrgName(e.getMaterialPartybName());
			// 无合同费用类型
			regulate.setFeeTypeId(null);
			regulate.setFeeTypeCode(null);
			regulate.setFeeTypeName(null);
			// 结算金额（隐藏）
			regulate.setSettleMny(e.getMaterialSettleMny());
			// 支付对象（隐藏）
			regulate.setPayTargetType(null);
			// 收款人
			regulate.setSupplierUserId(null);
			regulate.setSupplierUserCode(null);
			regulate.setSupplierUserName(null);
			// 收款单位/收款人
			regulate.setSupplierName(e.getMaterialPartybName());
			// 计划支付金额
			regulate.setPlanPayMny(e.getMaterialPlanPaymentMny());
			// 已支付金额
			regulate.setPaidMny(e.getTotalActualPaymentMny());
			// 未支付金额（隐藏）
			regulate.setUnpaidMny(e.getUnpaidMny());
			// 关闭状态（隐藏）
			regulate.setCloseState(e.getCloseState());
			// 关闭时间（隐藏）
			regulate.setCloseDateTime(e.getCloseDateTime());
			// 关闭人（隐藏）
			regulate.setCloseUserId(e.getCloseUserId());
			regulate.setCloseUserCode(e.getCloseUserCode());
			regulate.setCloseUserName(e.getCloseUserName());

			// 支付状态
			regulate.setPaymentState("未支付");

			// 未支付金额(元)
			regulate.setUnpaidMny(e.getMaterialPlanPaymentMny());
			// 已支付金额(元)
			regulate.setPaidMny(BigDecimal.ZERO);
			// 来源单据编号
			regulate.setSourceBillCode(e.getMaterialSettleCode());
			// 来源穿透地址
			regulate.setSourceLinkUrl(e.getMaterialSettleRouteUrl());
			// 来源穿透名字
			regulate.setSourceLinkName(PaymentContractEnum.零星材料.getName());
			regulate.setSettleDate(e.getMaterialSettleDate());
			entities.add(regulate);
		}
	}


	private static void insertPlanTemporaryList(PlanEntity p, List<PlanTemporaryEntity> planTemporaryList, Collection<RegulateEntity> entities) {
		for (PlanTemporaryEntity e : planTemporaryList) {
			RegulateEntity regulate = new RegulateEntity();

			// 来源
			regulate.setSourceType(SOURCE_TYPE);
			// 来源id
			regulate.setSourceId(e.getId());
			// 来源父id
			regulate.setSourceParentId(e.getPlanId());
			// 穿透地址
			regulate.setRuteLinkUrl(LINK_URL + e.getPlanId());
			// 穿透名字
			regulate.setRuteLinkName(SOURCE_TYPE);
			// 项目部id
			regulate.setOrgId(p.getOrgId());
			// 项目部编码
			regulate.setOrgCode(p.getOrgCode());
			// 项目部名称
			regulate.setOrgName(p.getOrgName());


			// 计划单号
			regulate.setBillCode(p.getBillCode());
			// 本次计划日期
			regulate.setPlanDate(p.getPlanDate());
			// 项目所属组织
			regulate.setParentOrgId(p.getParentOrgId());
			regulate.setParentOrgCode(p.getParentOrgCode());
			regulate.setParentOrgName(p.getParentOrgName());
			// 项目编码（隐藏）
			regulate.setProjectId(p.getProjectId());
			regulate.setProjectCode(p.getProjectCode());
			// 项目名称
			regulate.setProjectName(p.getProjectName());
			// 支出类型
			regulate.setPaymentContractFlag(PaymentContractEnum.临时机械.getCode());
			// 合同编码
			regulate.setContractId(null);
			regulate.setContractCode(null);
			// 合同名称
			regulate.setContractName(null);
			// 收款单位
			regulate.setSupplierOrgId(e.getTemporaryPartybId());
			regulate.setSupplierOrgCode(null);
			regulate.setSupplierOrgName(e.getTemporaryPartybName());
			// 无合同费用类型
			regulate.setFeeTypeId(null);
			regulate.setFeeTypeCode(null);
			regulate.setFeeTypeName(null);
			// 结算金额（隐藏）
			regulate.setSettleMny(e.getTemporarySettleMny());
			// 支付对象（隐藏）
			regulate.setPayTargetType(null);
			// 收款人
			regulate.setSupplierUserId(null);
			regulate.setSupplierUserCode(null);
			regulate.setSupplierUserName(null);
			// 收款单位/收款人
			regulate.setSupplierName(e.getTemporaryPartybName());
			// 计划支付金额
			regulate.setPlanPayMny(e.getTemporaryPlanPaymentMny());
			// 已支付金额
			regulate.setPaidMny(e.getTotalActualPaymentMny());
			// 未支付金额（隐藏）
			regulate.setUnpaidMny(e.getUnpaidMny());
			// 关闭状态（隐藏）
			regulate.setCloseState(e.getCloseState());
			// 关闭时间（隐藏）
			regulate.setCloseDateTime(e.getCloseDateTime());
			// 关闭人（隐藏）
			regulate.setCloseUserId(e.getCloseUserId());
			regulate.setCloseUserCode(e.getCloseUserCode());
			regulate.setCloseUserName(e.getCloseUserName());

			// 支付状态
			regulate.setPaymentState("未支付");

			// 未支付金额(元)
			regulate.setUnpaidMny(e.getTemporaryPlanPaymentMny());
			// 已支付金额(元)
			regulate.setPaidMny(BigDecimal.ZERO);
			// 来源单据编号
			regulate.setSourceBillCode(e.getTemporarySettleCode());
			// 来源穿透地址
			regulate.setSourceLinkUrl(e.getTemporarySettleRouteUrl());
			// 来源穿透名字
			regulate.setSourceLinkName(PaymentContractEnum.临时机械.getName());
			// 结算时间
			regulate.setSettleDate(e.getTemporarySettleDate());
			entities.add(regulate);
		}
	}


	private static void insertPlanFeeList(PlanEntity p, List<PlanFeeEntity> planFeeList, Collection<RegulateEntity> entities) {
		for (PlanFeeEntity e : planFeeList) {
			RegulateEntity regulate = new RegulateEntity();

			// 来源
			regulate.setSourceType(SOURCE_TYPE);
			// 来源id
			regulate.setSourceId(e.getId());
			// 来源父id
			regulate.setSourceParentId(e.getPlanId());
			// 穿透地址
			regulate.setRuteLinkUrl(LINK_URL + e.getPlanId());
			// 穿透名字
			regulate.setRuteLinkName(SOURCE_TYPE);
			// 项目部id
			regulate.setOrgId(p.getOrgId());
			// 项目部编码
			regulate.setOrgCode(p.getOrgCode());
			// 项目部名称
			regulate.setOrgName(p.getOrgName());


			// 计划单号
			regulate.setBillCode(p.getBillCode());
			// 本次计划日期
			regulate.setPlanDate(p.getPlanDate());
			// 项目所属组织
			regulate.setParentOrgId(p.getParentOrgId());
			regulate.setParentOrgCode(p.getParentOrgCode());
			regulate.setParentOrgName(p.getParentOrgName());
			// 项目编码（隐藏）
			regulate.setProjectId(p.getProjectId());
			regulate.setProjectCode(p.getProjectCode());
			// 项目名称
			regulate.setProjectName(p.getProjectName());
			// 支出类型
			regulate.setPaymentContractFlag(PaymentContractEnum.零星费用.getCode());
			// 合同编码
			regulate.setContractId(null);
			regulate.setContractCode(null);
			// 合同名称
			regulate.setContractName(null);
			// 收款单位
			regulate.setSupplierOrgId(e.getSupplierOrgId());
			regulate.setSupplierOrgCode(e.getSupplierOrgCode());
			regulate.setSupplierOrgName(e.getSupplierOrgName());
			// 无合同费用类型
			regulate.setFeeTypeId(e.getPayTypeNoContractId());
			regulate.setFeeTypeCode(e.getPayTypeNoContractCode());
			regulate.setFeeTypeName(e.getPayTypeNoContractName());
			// 结算金额（隐藏）
			regulate.setSettleMny(e.getFeeShouldPayTaxMny());
			// 支付对象（隐藏）
			regulate.setPayTargetType(e.getPaymentObject());
			// 收款人
			regulate.setSupplierUserId(e.getSupplierUserId());
			regulate.setSupplierUserCode(e.getSupplierUserCode());
			regulate.setSupplierUserName(e.getSupplierUserName());
			// 收款单位/收款人
			regulate.setSupplierName(e.getSupplierName());
			// 计划支付金额
			regulate.setPlanPayMny(e.getFeePlanPaymentMny());
			// 已支付金额
			regulate.setPaidMny(e.getTotalActualPaymentMny());
			// 未支付金额（隐藏）
			regulate.setUnpaidMny(e.getUnpaidMny());
			// 关闭状态（隐藏）
			regulate.setCloseState(e.getCloseState());
			// 关闭时间（隐藏）
			regulate.setCloseDateTime(e.getCloseDateTime());
			// 关闭人（隐藏）
			regulate.setCloseUserId(e.getCloseUserId());
			regulate.setCloseUserCode(e.getCloseUserCode());
			regulate.setCloseUserName(e.getCloseUserName());

			// 支付状态
			regulate.setPaymentState("未支付");

			// 未支付金额(元)
			regulate.setUnpaidMny(e.getFeePlanPaymentMny());
			// 已支付金额(元)
			regulate.setPaidMny(BigDecimal.ZERO);
			// 来源单据编号
			regulate.setSourceBillCode(e.getFeeCode());
			// 来源穿透地址
			regulate.setSourceLinkUrl(e.getFeeRouteUrl());
			// 来源穿透名字
			regulate.setSourceLinkName(PaymentContractEnum.零星费用.getName());
			//发生时间
			regulate.setHappenDate(e.getHappenDate());

			entities.add(regulate);
		}
	}

}
