package com.ejianc.business.profinance.plan.service.impl;

import com.alibaba.fastjson.JSONObject;
import com.baomidou.mybatisplus.core.conditions.query.QueryWrapper;
import com.baomidou.mybatisplus.core.conditions.update.LambdaUpdateWrapper;
import com.baomidou.mybatisplus.core.toolkit.Wrappers;
import com.ejianc.business.contractbase.pool.settlepool.api.ISettlePoolApi;
import com.ejianc.business.profinance.plan.bean.PlanContractEntity;
import com.ejianc.business.profinance.plan.mapper.PlanContractMapper;
import com.ejianc.business.profinance.plan.service.IPlanContractService;
import com.ejianc.framework.core.exception.BusinessException;
import com.ejianc.framework.core.response.BillStateEnum;
import com.ejianc.framework.core.response.CommonResponse;
import com.ejianc.framework.skeleton.template.BaseServiceImpl;
import org.springframework.stereotype.Service;

import java.math.BigDecimal;
import java.util.*;

/**
 * 项目资金计划-支出合同
 *
 * @author generator
 */
@Service("planContractService")
public class PlanContractServiceImpl extends BaseServiceImpl<PlanContractMapper, PlanContractEntity> implements IPlanContractService {

	private final ISettlePoolApi settlePoolApi;
	private static final String CONTRACT_ID = "contract_id";

	public PlanContractServiceImpl(ISettlePoolApi settlePoolApi) {
		this.settlePoolApi = settlePoolApi;
	}

	/**
	 * 根据项目资金计划id，修改项目资金计划-支出合同的单据状态
	 *
	 * @param planId        项目资金计划id
	 * @param billStateCode 单据状态
	 */
	@Override
	public void updateBillState(Long planId, Integer billStateCode) {
		LambdaUpdateWrapper<PlanContractEntity> updateWrapper = Wrappers.lambdaUpdate();
		updateWrapper.eq(PlanContractEntity::getPlanId, planId);
		updateWrapper.set(PlanContractEntity::getBillState, billStateCode);
		updateWrapper.set(PlanContractEntity::getBillStateName, BillStateEnum.getEnumByStateCode(billStateCode).getDescription());
		super.update(updateWrapper);
	}

	/**
	 * 根据合同ids查询累计结算金额、累计申请预付款(不含本次)、累计已扣减预付款(不含本次)、累计申请进度款(不含本次)、累计已申请付款(不含本次)
	 *
	 * @param contractIds 合同id集合
	 *
	 * @return 累计结算金额、累计申请预付款(不含本次)、累计已扣减预付款(不含本次)、累计申请进度款(不含本次)、累计已申请付款(不含本次)
	 */
	@Override
	public JSONObject fetchContractValue(Collection<Long> contractIds) {
		JSONObject jsonObject = new JSONObject();

		// 累计结算金额
		CommonResponse<Map<String, Object>> res = settlePoolApi.fetchTotalSettleTaxMny(contractIds);
		if (!res.isSuccess()) {
			throw new BusinessException("调用结算池接口异常，失败原因：" + res.getMsg());
		}
		jsonObject.put("totalSettleTaxMny", initContractValue(res.getData(), contractIds));

		// 累计申请预付款(不含本次)
		QueryWrapper<PlanContractEntity> queryWrapper = Wrappers.query();
		queryWrapper.in("bill_state", Arrays.asList(BillStateEnum.COMMITED_STATE.getBillStateCode(), BillStateEnum.PASSED_STATE.getBillStateCode()));
		queryWrapper.in(CONTRACT_ID, contractIds);
		queryWrapper.eq("contract_payment_type", "预付款");
		queryWrapper.groupBy(CONTRACT_ID);
		queryWrapper.select("contract_id as contractId,(ifnull(sum(contract_plan_payment), 0) - ifnull(sum(release_apply_mny), 0)) as total");
		List<Map<String, Object>> resultMap = super.listMaps(queryWrapper);
		jsonObject.put("lastTotalContractPlanPayment", initContractValue(converToMap(resultMap), contractIds));

		// 累计已扣减预付款(不含本次)
		QueryWrapper<PlanContractEntity> queryWrapper1 = Wrappers.query();
		queryWrapper1.in("bill_state", Arrays.asList(BillStateEnum.COMMITED_STATE.getBillStateCode(), BillStateEnum.PASSED_STATE.getBillStateCode()));
		queryWrapper1.in(CONTRACT_ID, contractIds);
		queryWrapper1.eq("contract_payment_type", "进度款");
		queryWrapper1.groupBy(CONTRACT_ID);
		queryWrapper1.select("contract_id as contractId,ifnull(sum(contract_deduction_advance_payment), 0) as total");
		List<Map<String, Object>> resultMap1 = super.listMaps(queryWrapper1);
		jsonObject.put("lastTotalDeductionAdvancePayment", initContractValue(converToMap(resultMap1), contractIds));

		// 累计申请进度款(不含本次)
		QueryWrapper<PlanContractEntity> queryWrapper2 = Wrappers.query();
		queryWrapper2.in("bill_state", Arrays.asList(BillStateEnum.COMMITED_STATE.getBillStateCode(), BillStateEnum.PASSED_STATE.getBillStateCode()));
		queryWrapper2.in(CONTRACT_ID, contractIds);
		queryWrapper2.eq("contract_payment_type", "进度款");
		queryWrapper2.groupBy(CONTRACT_ID);
		queryWrapper2.select("contract_id as contractId,(ifnull(sum(contract_plan_payment), 0) - ifnull(sum(release_apply_mny), 0)) as total");
		List<Map<String, Object>> resultMap2 = super.listMaps(queryWrapper2);
		jsonObject.put("lastTotalApplySchedulePayment", initContractValue(converToMap(resultMap2), contractIds));

		// 累计已申请付款(不含本次)
		QueryWrapper<PlanContractEntity> queryWrapper3 = Wrappers.query();
		queryWrapper3.in("bill_state", Arrays.asList(BillStateEnum.COMMITED_STATE.getBillStateCode(), BillStateEnum.PASSED_STATE.getBillStateCode()));
		queryWrapper3.in(CONTRACT_ID, contractIds);
		queryWrapper3.groupBy(CONTRACT_ID);
		queryWrapper3.select("contract_id as contractId,(ifnull(sum(contract_plan_payment), 0) - ifnull(sum(release_apply_mny), 0)) as total");
		List<Map<String, Object>> resultMap3 = super.listMaps(queryWrapper3);
		jsonObject.put("lastTotalAppliedPayment", initContractValue(converToMap(resultMap3), contractIds));

		return jsonObject;
	}

	private Map<String, Object> converToMap(List<Map<String, Object>> listMap) {
		Map<String, Object> resultMap = new HashMap<>();
		for (Map<String, Object> map : listMap) {
			if (!resultMap.containsKey(map.get("contractId").toString())) {
				resultMap.put(map.get("contractId").toString(), map.get("total").toString());
			}
		}
		return resultMap;
	}

	/**
	 * 初始化不存在的数据
	 *
	 * @param existMap    已存在的数据
	 * @param contractIds 合同ids
	 *
	 * @return Map<String, Object>
	 */
	private Map<String, Object> initContractValue(Map<String, Object> existMap, Collection<Long> contractIds) {
		Map<String, Object> map = new HashMap<>();
		if (!existMap.isEmpty()) {
			map.putAll(existMap);
			for (Long contractId : contractIds) {
				if (!map.containsKey(contractId.toString())) {
					map.put(contractId.toString(), BigDecimal.ZERO);
				}
			}
		} else {
			for (Long contractId : contractIds) {
				map.put(contractId.toString(), BigDecimal.ZERO);

			}
		}

		return map;
	}
}
