package com.ejianc.business.market.controller;

import cn.hutool.core.io.IoUtil;
import com.alibaba.fastjson.JSONObject;
import com.baomidou.mybatisplus.core.metadata.IPage;
import com.ejianc.business.market.bean.CloseCostEntity;
import com.ejianc.business.market.service.ICloseCostService;
import com.ejianc.business.market.vo.CloseCostVO;
import com.ejianc.foundation.orgcenter.api.IOrgApi;
import com.ejianc.foundation.orgcenter.vo.OrgVO;
import com.ejianc.foundation.support.api.IBillCodeApi;
import com.ejianc.foundation.support.api.IBillTypeApi;
import com.ejianc.foundation.support.vo.BillCodeParam;
import com.ejianc.framework.auth.session.SessionManager;
import com.ejianc.framework.core.context.InvocationInfoProxy;
import com.ejianc.framework.core.exception.BusinessException;
import com.ejianc.framework.core.kit.collection.ListUtil;
import com.ejianc.framework.core.kit.mapper.BeanMapper;
import com.ejianc.framework.core.response.BillStateEnum;
import com.ejianc.framework.core.response.CommonResponse;
import com.ejianc.framework.core.response.Parameter;
import com.ejianc.framework.core.response.QueryParam;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.core.io.ClassPathResource;
import org.springframework.web.bind.annotation.*;

import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import java.io.IOException;
import java.io.Serializable;
import java.util.List;
import java.util.stream.Collectors;

/**
 * 成本关门
 *
 * @author baipengyan
 * @version 1.0
 * @since JDK 1.8
 */
@RestController
@RequestMapping("closeCost")
public class CloseCostController implements Serializable {
	private static final long serialVersionUID = -6108432280361278283L;
	private static final String RULE_CODE = "CLOSE_COST";
	private static final String BILL_CODE = "EJCBT202209000016";
	private final Logger logger = LoggerFactory.getLogger(this.getClass());
	private final SessionManager sessionManager;
	private final IBillTypeApi billTypeApi;
	private final IBillCodeApi billCodeApi;
	private final IOrgApi iOrgApi;
	private final ICloseCostService service;


	public CloseCostController(SessionManager sessionManager, IBillTypeApi billTypeApi, IBillCodeApi billCodeApi, IOrgApi iOrgApi, ICloseCostService service) {
		this.sessionManager = sessionManager;
		this.billTypeApi = billTypeApi;
		this.billCodeApi = billCodeApi;
		this.iOrgApi = iOrgApi;
		this.service = service;
	}

	/**
	 * 保存修改
	 *
	 * @param saveOrUpdateVO vo
	 *
	 * @return CommonResponse<CloseCostVO>
	 */
	@PostMapping(value = "/saveOrUpdate")
	public CommonResponse<CloseCostVO> saveOrUpdate(@RequestBody CloseCostVO saveOrUpdateVO) {
		CloseCostEntity entity = BeanMapper.map(saveOrUpdateVO, CloseCostEntity.class);
		if (null == entity.getId() || 0 == entity.getId()) {
			BillCodeParam billCodeParam = BillCodeParam.build(RULE_CODE, InvocationInfoProxy.getTenantid(), saveOrUpdateVO);
			CommonResponse<String> billCode = billCodeApi.generateBillCode(billCodeParam);
			if (billCode.isSuccess()) {
				entity.setBillCode(billCode.getData());//此处需要根据实际修改 删除本行或者上一行
			} else {
				throw new BusinessException("网络异常， 编码生成失败， 请稍后再试");
			}
		}
		entity.setBillName("成本关门");
		entity.setBillStateName(BillStateEnum.UNCOMMITED_STATE.getDescription());
		service.saveOrUpdate(entity, false);
		return CommonResponse.success("保存或修改单据成功！", BeanMapper.map(service.selectById(entity.getId()), CloseCostVO.class));
	}

	/**
	 * 查询详情
	 *
	 * @param id 主键
	 *
	 * @return CommonResponse<CloseCostVO>
	 */
	@GetMapping(value = "/queryDetail")
	public CommonResponse<CloseCostVO> queryDetail(@RequestParam(value = "id", required = true) Long id) {
		CloseCostEntity entity = service.selectById(id);
		CloseCostVO vo = BeanMapper.map(entity, CloseCostVO.class);
		return CommonResponse.success("查询详情数据成功！", vo);
	}

	/**
	 * 删除
	 *
	 * @param vos 待删除的vos
	 *
	 * @return CommonResponse<String>
	 */
	@PostMapping(value = "/delete")
	public CommonResponse<String> delete(@RequestBody List<CloseCostVO> vos) {
		if (ListUtil.isNotEmpty(vos)) {
			for (CloseCostVO vo : vos) {
				CommonResponse<String> resp = billTypeApi.checkQuote(BILL_CODE, vo.getId());
				if (!resp.isSuccess()) {
					return CommonResponse.error("删除失败！" + resp.getMsg());
				}
			}
		}
		service.removeByIds(vos.stream().map(CloseCostVO::getId).collect(Collectors.toList()), true);
		return CommonResponse.success("删除成功！");
	}

	/**
	 * 分页查询
	 *
	 * @param param 请求参数
	 *
	 * @return CommonResponse<JSONObject>
	 */
	@PostMapping(value = "/queryList")
	public CommonResponse<JSONObject> queryList(@RequestBody QueryParam param) {
		JSONObject resp = new JSONObject();

		List<String> fuzzyFields = param.getFuzzyFields();
		fuzzyFields.add("projectName");

		param.getParams().put("tenantId", new Parameter(QueryParam.EQ, InvocationInfoProxy.getTenantid()));
		//若当前上下文为项目部，则根据项目部Id来进行查询
		if (OrgVO.ORG_TYPE_DEPARTMENT.equals(Integer.valueOf(InvocationInfoProxy.getOrgType()))) {
			param.getParams().put("orgId", new Parameter(QueryParam.EQ, InvocationInfoProxy.getOrgId()));
		} else {
			CommonResponse<List<OrgVO>> orgResp = iOrgApi.findChildrenByParentIdWithoutProjectDept(InvocationInfoProxy.getOrgId());
			if (!orgResp.isSuccess()) {
				logger.error("分页查询失败，获取当前本下组织信息失败, {}", orgResp.getMsg());
				return CommonResponse.error("查询失败，获取组织信息失败！");
			}
			param.getParams().put("parentOrgId", new Parameter(QueryParam.IN,
					orgResp.getData().stream().map(OrgVO::getId).collect(Collectors.toList())));
		}

		IPage<CloseCostEntity> page = service.queryPage(param, false);
		List<CloseCostVO> closeCost = BeanMapper.mapList(page.getRecords(), CloseCostVO.class);
		resp.put("current", page.getCurrent());
		resp.put("size", page.getSize());
		resp.put("pages", page.getPages());
		resp.put("total", page.getTotal());
		resp.put("records", closeCost);
		return CommonResponse.success("查询列表数据成功！", resp);
	}


	/**
	 * 导出
	 *
	 * @param param    查询参数
	 * @param response 响应头
	 */
	@PostMapping(value = "/excelExport")
	public void excelExport(@RequestBody QueryParam param, HttpServletResponse response) throws IOException {
		ClassPathResource resource = new ClassPathResource("excel/成本关门统计表模板.xlsx");
		IoUtil.copy(resource.getInputStream(), response.getOutputStream());
	}

	/**
	 * 自动关门
	 */
	@PostMapping(value = "/autoCloseCost")
	public CommonResponse<String> autoCloseCost(HttpServletRequest request) {
		logger.info("自动关门--------start");
		service.autoCloseCost();
		logger.info("自动关门--------end");
		return CommonResponse.success("请求成功");
	}


}
