package com.ejianc.business.market.controller;

import cn.hutool.core.io.IoUtil;
import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.JSONObject;
import com.baomidou.mybatisplus.core.metadata.IPage;
import com.baomidou.mybatisplus.core.toolkit.Wrappers;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.ejianc.business.market.bean.CloseCostEntity;
import com.ejianc.business.market.service.ICloseCostService;
import com.ejianc.business.market.vo.*;
import com.ejianc.foundation.orgcenter.api.IOrgApi;
import com.ejianc.foundation.orgcenter.vo.OrgVO;
import com.ejianc.foundation.support.api.IBillCodeApi;
import com.ejianc.foundation.support.api.IBillTypeApi;
import com.ejianc.foundation.support.vo.BillCodeParam;
import com.ejianc.framework.auth.session.SessionManager;
import com.ejianc.framework.core.context.InvocationInfoProxy;
import com.ejianc.framework.core.exception.BusinessException;
import com.ejianc.framework.core.kit.collection.ListUtil;
import com.ejianc.framework.core.kit.mapper.BeanMapper;
import com.ejianc.framework.core.response.BillStateEnum;
import com.ejianc.framework.core.response.CommonResponse;
import com.ejianc.framework.core.response.Parameter;
import com.ejianc.framework.core.response.QueryParam;
import com.ejianc.framework.core.util.ExcelExport;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.core.io.ClassPathResource;
import org.springframework.web.bind.annotation.*;

import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import java.io.IOException;
import java.io.Serializable;
import java.math.BigDecimal;
import java.sql.Wrapper;
import java.util.*;
import java.util.stream.Collectors;

/**
 * 成本关门
 *
 * @author baipengyan
 * @version 1.0
 * @since JDK 1.8
 */
@RestController
@RequestMapping("closeCost")
public class CloseCostController implements Serializable {
	private static final long serialVersionUID = -6108432280361278283L;
	private static final String RULE_CODE = "CLOSE_COST";
	private static final String BILL_CODE = "EJCBT202209000016";
	private final Logger logger = LoggerFactory.getLogger(this.getClass());
	private final SessionManager sessionManager;
	private final IBillTypeApi billTypeApi;
	private final IBillCodeApi billCodeApi;
	private final IOrgApi iOrgApi;
	private final ICloseCostService service;


	public CloseCostController(SessionManager sessionManager, IBillTypeApi billTypeApi, IBillCodeApi billCodeApi, IOrgApi iOrgApi, ICloseCostService service) {
		this.sessionManager = sessionManager;
		this.billTypeApi = billTypeApi;
		this.billCodeApi = billCodeApi;
		this.iOrgApi = iOrgApi;
		this.service = service;
	}

	/**
	 * 保存修改
	 *
	 * @param saveOrUpdateVO vo
	 *
	 * @return CommonResponse<CloseCostVO>
	 */
	@PostMapping(value = "/saveOrUpdate")
	public CommonResponse<CloseCostVO> saveOrUpdate(@RequestBody CloseCostVO saveOrUpdateVO) {
		CloseCostEntity entity = BeanMapper.map(saveOrUpdateVO, CloseCostEntity.class);
		if (null == entity.getId() || 0 == entity.getId()) {
			BillCodeParam billCodeParam = BillCodeParam.build(RULE_CODE, InvocationInfoProxy.getTenantid(), saveOrUpdateVO);
			CommonResponse<String> billCode = billCodeApi.generateBillCode(billCodeParam);
			if (billCode.isSuccess()) {
				entity.setBillCode(billCode.getData());//此处需要根据实际修改 删除本行或者上一行
			} else {
				throw new BusinessException("网络异常， 编码生成失败， 请稍后再试");
			}
		}
		entity.setBillName("成本关门");
		entity.setBillStateName(BillStateEnum.UNCOMMITED_STATE.getDescription());
		service.saveOrUpdate(entity, false);
		CloseCostVO vo = BeanMapper.map(entity, CloseCostVO.class);

		//排序
		List<CloseCostInfoVO> list = vo.getCloseCostInfoList().stream()
				.sorted(Comparator.comparing(CloseCostInfoVO::getSequence, Comparator.nullsFirst(Integer::compareTo)))
				.collect(Collectors.toList());
		vo.setCloseCostInfoList(list);
		return CommonResponse.success("保存或修改单据成功！", vo);
	}

	/**
	 * 查询详情
	 *
	 * @param id 主键
	 *
	 * @return CommonResponse<CloseCostVO>
	 */
	@GetMapping(value = "/queryDetail")
	public CommonResponse<CloseCostVO> queryDetail(@RequestParam(value = "id", required = true) Long id) {
		CloseCostEntity entity = service.selectById(id);
		CloseCostVO vo = BeanMapper.map(entity, CloseCostVO.class);

		//排序
		List<CloseCostInfoVO> list = vo.getCloseCostInfoList().stream()
				.sorted(Comparator.comparing(CloseCostInfoVO::getSequence, Comparator.nullsFirst(Integer::compareTo)))
				.collect(Collectors.toList());
		vo.setCloseCostInfoList(list);
		return CommonResponse.success("查询详情数据成功！", vo);
	}

	/**
	 * 删除
	 *
	 * @param vos 待删除的vos
	 *
	 * @return CommonResponse<String>
	 */
	@PostMapping(value = "/delete")
	public CommonResponse<String> delete(@RequestBody List<CloseCostVO> vos) {
		if (ListUtil.isNotEmpty(vos)) {
			for (CloseCostVO vo : vos) {
				CommonResponse<String> resp = billTypeApi.checkQuote(BILL_CODE, vo.getId());
				if (!resp.isSuccess()) {
					return CommonResponse.error("删除失败！" + resp.getMsg());
				}
			}
		}
		service.removeByIds(vos.stream().map(CloseCostVO::getId).collect(Collectors.toList()), true);
		return CommonResponse.success("删除成功！");
	}

	/**
	 * 分页查询
	 *
	 * @param param 请求参数
	 *
	 * @return CommonResponse<JSONObject>
	 */
	@PostMapping(value = "/queryList")
	public CommonResponse<JSONObject> queryList(@RequestBody QueryParam param) {
		JSONObject resp = new JSONObject();

		List<String> fuzzyFields = param.getFuzzyFields();
		fuzzyFields.add("projectName");

		param.getParams().put("tenantId", new Parameter(QueryParam.EQ, InvocationInfoProxy.getTenantid()));

		/**
		 * 组织隔离
		 */
		List<OrgVO> orgVOList = null;
		//普通组织 id
		List<Long> commonOrgIds = new ArrayList<>();
		if (!param.getParams().containsKey("orgId")) {
			//若当前上下文为项目部，则根据项目部Id来进行查询
			if (OrgVO.ORG_TYPE_DEPARTMENT.equals(Integer.valueOf(InvocationInfoProxy.getOrgType()))) {
				param.getParams().put("orgId", new Parameter(QueryParam.EQ, InvocationInfoProxy.getOrgId()));
			} else {
				CommonResponse<List<OrgVO>> orgResp = iOrgApi.findChildrenByParentIdWithoutProjectDept(InvocationInfoProxy.getOrgId());
				if (!orgResp.isSuccess()) {
					logger.error("分页查询失败，获取当前本下组织信息失败, {}", orgResp.getMsg());
					return CommonResponse.error("查询失败，获取组织信息失败！");
				}
				param.getParams().put("parentOrgId", new Parameter(QueryParam.IN,
						orgResp.getData().stream().map(OrgVO::getId).collect(Collectors.toList())));
			}

		} else {
			Long orgId = Long.valueOf(param.getParams().get("orgId").getValue().toString());
			param.getParams().remove("orgId");
			orgVOList = (List<OrgVO>) getRespData(iOrgApi.findChildrenByParentId(orgId), true, "查询失败，获取当前本下组织信息失败。");
			orgVOList.stream().forEach(org -> {
				//普通组织
				commonOrgIds.add(org.getId());
			});
			if (CollectionUtils.isNotEmpty(commonOrgIds)) {
				param.getParams().put("parentOrgId", new Parameter(QueryParam.IN, commonOrgIds));
			}
		}

		IPage<CloseCostEntity> page = service.queryPage(param, false);
		List<CloseCostVO> closeCost = BeanMapper.mapList(page.getRecords(), CloseCostVO.class);
		resp.put("current", page.getCurrent());
		resp.put("size", page.getSize());
		resp.put("pages", page.getPages());
		resp.put("total", page.getTotal());
		resp.put("records", closeCost);
		return CommonResponse.success("查询列表数据成功！", resp);
	}


	/**
	 * 导出
	 *
	 * @param param    查询参数
	 * @param response 响应头
	 */
	@PostMapping(value = "/excelExport")
	public void excelExport(@RequestBody QueryParam param, HttpServletResponse response) throws IOException {
		ClassPathResource resource = new ClassPathResource("excel/成本关门统计表模板.xlsx");
		IoUtil.copy(resource.getInputStream(), response.getOutputStream());
	}

	/**
	 * 自动关门
	 */
	@PostMapping(value = "/autoCloseCost")
	public CommonResponse<String> autoCloseCost(HttpServletRequest request) {
		logger.info("自动关门--------start");
		service.autoCloseCost();
		logger.info("自动关门--------end");
		return CommonResponse.success("请求成功");
	}
	/**
	 * 根据项目id获取子表金额统计
	 * @param
	 * @return
	 */
	@RequestMapping(value = "getDetailByProjectId", method= RequestMethod.GET)
	CommonResponse<JSONObject> getDetailByProjectId(@RequestParam(value = "projectId", required = true) Long projectId) {
		HashMap<String, BigDecimal> result = service.getDetailByProjectId(projectId);
		JSONObject jsonObject = new JSONObject();
		jsonObject.putAll(result);
		return CommonResponse.success("查询成功！",jsonObject);
	}
	/**
	 * 根据项目id获取主表信息
	 * @param
	 * @return
	 */
	@RequestMapping(value = "getByProjectId", method= RequestMethod.GET)
	CommonResponse<List<CloseCostVO>> getByProjectId(@RequestParam(value = "projectId", required = true) Long projectId) {
		List<CloseCostEntity> list = service.list(Wrappers.<CloseCostEntity>lambdaQuery()
				.eq(CloseCostEntity::getProjectId, projectId)
				.in(CloseCostEntity::getBillState, Arrays.asList(1, 3))
		);
		if(ListUtil.isNotEmpty(list)){
			List<CloseCostVO> closeCostVOS = BeanMapper.mapList(list, CloseCostVO.class);
			return CommonResponse.success("查询成功！",closeCostVOS);

		}
		return CommonResponse.success("查询成功,无相关数据！");
	}
	/**
	 * 根据项目id获取ES支出合同报表数据
	 * @param
	 * @return
	 */

    //项目支出合同台账es索引名
    private final static String INDEX_EX_CONTRACTT = "ex_contractt";

	@RequestMapping(value = "queryESContract", method= RequestMethod.GET)
	CommonResponse<List<JSONObject>> queryESContract(@RequestParam(value = "projectId", required = true) Long projectId) {
		List<JSONObject> result = service.queryESContract(projectId,INDEX_EX_CONTRACTT);
		if(ListUtil.isNotEmpty(result)){
			return CommonResponse.success("查询成功！",result);
		} else{
			return CommonResponse.error("根据项目id查询es数据失败，请查询项目支出合同台账是否存在该项目数据！");
		}

	}

	/**
	 * @param param
	 * @Description 子表金额信息导出
	 * @Return void
	 */
		@RequestMapping(value = "/excelCloseCostInfoExport", method = RequestMethod.POST)
	@ResponseBody
	public void excelCloseCostInfoExport(@RequestBody List<CloseCostInfoVO> vo, HttpServletResponse response) {
		//todo:字段翻译等等
		Map<String, Object> beans = new HashMap<>();
		beans.put("records", vo);
		ExcelExport.getInstance().export("CloseCostInfo-export.xlsx", beans, response);
	}


	/**
	 * @param param
	 * @Description 子表台账信息导出
	 * @Return void
	 */
	@RequestMapping(value = "/excelReportExport", method = RequestMethod.POST)
	@ResponseBody
	public void excelReportExport(@RequestBody List<CloseCostZiReportVO> vo, HttpServletResponse response) {
		//todo:字段翻译等等
		Map<String, Object> beans = new HashMap<>();
		beans.put("records", vo);
		ExcelExport.getInstance().export("CloseCostZiReport-export.xlsx", beans, response);
	}

	/**
	 * @Description 分页查询成本关门台账信息
	 */
	@PostMapping(value = "/queryReportList")
	public CommonResponse<Page<CloseCostReportVO>> queryReportList(@RequestBody QueryParam param) {
        /**
         * 组织隔离
         */
        List<OrgVO> orgVOList = null;
        //普通组织 id
        List<Long> commonOrgIds = new ArrayList<>();
        if (!param.getParams().containsKey("orgId")) {
            //若当前上下文为项目部，则根据项目部Id来进行查询
            if (OrgVO.ORG_TYPE_DEPARTMENT.equals(Integer.valueOf(InvocationInfoProxy.getOrgType()))) {
                param.getParams().put("orgId", new Parameter(QueryParam.EQ, InvocationInfoProxy.getOrgId()));
            } else {
                CommonResponse<List<OrgVO>> orgResp = iOrgApi.findChildrenByParentIdWithoutProjectDept(InvocationInfoProxy.getOrgId());
                if (!orgResp.isSuccess()) {
                    logger.error("分页查询失败，获取当前本下组织信息失败, {}", orgResp.getMsg());
                    return CommonResponse.error("查询失败，获取组织信息失败！");
                }
                param.getParams().put("parentOrgId", new Parameter(QueryParam.IN,
                        orgResp.getData().stream().map(OrgVO::getId).collect(Collectors.toList())));
            }

        } else {
            Long orgId = Long.valueOf(param.getParams().get("orgId").getValue().toString());
            param.getParams().remove("orgId");
            orgVOList = (List<OrgVO>) getRespData(iOrgApi.findChildrenByParentId(orgId), true, "查询失败，获取当前本下组织信息失败。");
            orgVOList.stream().forEach(org -> {
                //普通组织
                commonOrgIds.add(org.getId());
            });
            if (CollectionUtils.isNotEmpty(commonOrgIds)) {
                param.getParams().put("parentOrgId", new Parameter(QueryParam.IN, commonOrgIds));
            }
        }
		param.getFuzzyFields().add("projectName");
		Page<CloseCostReportVO> closeCostReportVOS = service.queryReportList(param);

		return CommonResponse.success("查询台账信息成功！",closeCostReportVOS);
	}

	/**
	 * @param param
	 * @Description 成本关门台账导出
	 * @Return void
	 */
	@RequestMapping(value = "/excelCloseCostReportExport", method = RequestMethod.POST)
	@ResponseBody
	public void excelCloseCostReportExport(@RequestBody QueryParam param, HttpServletResponse response) {
		/**
		 * 组织隔离
		 */
		List<OrgVO> orgVOList = null;
		//普通组织 id
		List<Long> commonOrgIds = new ArrayList<>();
		if (!param.getParams().containsKey("orgId")) {
			//若当前上下文为项目部，则根据项目部Id来进行查询
			if (OrgVO.ORG_TYPE_DEPARTMENT.equals(Integer.valueOf(InvocationInfoProxy.getOrgType()))) {
				param.getParams().put("orgId", new Parameter(QueryParam.EQ, InvocationInfoProxy.getOrgId()));
			} else {
				CommonResponse<List<OrgVO>> orgResp = iOrgApi.findChildrenByParentIdWithoutProjectDept(InvocationInfoProxy.getOrgId());
				if (!orgResp.isSuccess()) {
					logger.error("分页查询失败，获取当前本下组织信息失败, {}", orgResp.getMsg());
				}
				param.getParams().put("parentOrgId", new Parameter(QueryParam.IN,
						orgResp.getData().stream().map(OrgVO::getId).collect(Collectors.toList())));
			}

		} else {
			Long orgId = Long.valueOf(param.getParams().get("orgId").getValue().toString());
			param.getParams().remove("orgId");
			orgVOList = (List<OrgVO>) getRespData(iOrgApi.findChildrenByParentId(orgId), true, "查询失败，获取当前本下组织信息失败。");
			orgVOList.stream().forEach(org -> {
				//普通组织
				commonOrgIds.add(org.getId());
			});
			if (CollectionUtils.isNotEmpty(commonOrgIds)) {
				param.getParams().put("parentOrgId", new Parameter(QueryParam.IN, commonOrgIds));
			}
		}
		param.getFuzzyFields().add("projectName");
		Page<CloseCostReportVO> closeCostReportVOS = service.queryReportList(param);

		//todo:字段翻译等等
		Map<String, Object> beans = new HashMap<>();
		beans.put("records", closeCostReportVOS);
		ExcelExport.getInstance().export("closeCostReport-export.xlsx", beans, response);
	}

	/**
	 * 获取RPC数据
	 * resp 返回值
	 * isMustSuc 是否必须成功
	 * errMsg 失败提示
	 */
	private Object getRespData(CommonResponse<?> resp, boolean isMustSuc, String errMsg) {
		if (isMustSuc && !resp.isSuccess()) {
			throw new BusinessException(StringUtils.isNoneBlank(errMsg) ? errMsg : "调用Rpc服务失败");
		}
		return resp.getData();
	}


	/**
	 * 根据项目id获取目标、分析数据
	 * @param
	 * @return
	 */
	@RequestMapping(value = "getTargetReportFit", method= RequestMethod.GET)
	CommonResponse<TargetCostFitDTO> getTargetReportFit(@RequestParam(value = "projectId", required = true) Long projectId) {
		TargetCostFitDTO targetReportFit = service.getTargetReportFit(projectId);
		return CommonResponse.success("查询成功！",targetReportFit);
	}
}
