package com.ejianc.business.market.service.impl;

import com.baomidou.mybatisplus.core.conditions.query.LambdaQueryWrapper;
import com.baomidou.mybatisplus.core.toolkit.Wrappers;
import com.ejianc.business.market.bean.AdjustApplicationEntity;
import com.ejianc.business.market.bean.ProjectSetEntity;
import com.ejianc.business.enums.*;
import com.ejianc.business.market.service.IAdjustApplicationService;
import com.ejianc.business.market.service.IProjectSetService;
import com.ejianc.foundation.share.api.IProjectSetApi;
import com.ejianc.foundation.support.api.IBillTypeApi;
import com.ejianc.framework.auth.session.SessionManager;
import com.ejianc.framework.auth.session.UserContext;
import com.ejianc.framework.core.exception.BusinessException;
import com.ejianc.framework.core.response.BillStateEnum;
import com.ejianc.framework.core.response.CommonResponse;
import com.ejianc.framework.skeleton.billState.service.ICommonBusinessService;
import org.apache.commons.collections.CollectionUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import java.util.Date;
import java.util.List;
import java.util.Objects;

@Service("adjustApplication")
public class AdjustApplicationBpmServiceImpl implements ICommonBusinessService {
	private static final String BUSINESS_STATUS1= "在建";
	private static final String BUSINESS_STATUS2= "项目中止";
	private static final String BUSINESS_STATUS3= "竣工";
	private static final String BUSINESS_STATUS4= "合同状态";
	private static final String SETTLE_STATUS = "决算状态";
	private static final String ARCHIVE_STATUS = "资料状态";
	private static final String CAPITAL_STATUS = "资金管控状态";
	private final Logger logger = LoggerFactory.getLogger(this.getClass());
	private final SessionManager sessionManager;
	private final IBillTypeApi billTypeApi;
	private final IProjectSetApi projectSetApi;
	private final IAdjustApplicationService service;
	@Autowired
	private IProjectSetService projectSetService;

	public AdjustApplicationBpmServiceImpl(
			SessionManager sessionManager,
			IBillTypeApi billTypeApi,
			IProjectSetApi projectSetApi,
			IAdjustApplicationService service) {
		this.sessionManager = sessionManager;
		this.billTypeApi = billTypeApi;
		this.projectSetApi = projectSetApi;
		this.service = service;
	}

	/**
	 * 终审审核完回调
	 *
	 * @param billId 单据id
	 * @param state  单据状态
	 *
	 * @return 响应信息
	 */
	@Override
	public CommonResponse<String> afterApprovalProcessor(Long billId, Integer state, String billTypeCode) {
		logger.info("终审审核完回调--start，billId={},state={},billTypeCode={}", billId, state, billTypeCode);

		// 审批通过/已提交
		AdjustApplicationEntity entity = service.selectById(billId);
		if (entity == null) {
			throw new BusinessException("查询不到单据信息");
		}

		// 区分提交和审批
		UserContext userContext = sessionManager.getUserContext();
		if (state.equals(BillStateEnum.COMMITED_STATE.getBillStateCode())) {
			// 直审更新提交相关字段
			entity.setCommitDate(new Date());
			entity.setCommitUserCode(userContext.getUserCode());
			entity.setCommitUserName(userContext.getUserName());
		}
		entity.setBillStateName(BillStateEnum.getEnumByStateCode(state).getDescription());
		// 生效时间
		entity.setEffectiveDate(new Date());

		// 执行更新
		service.saveOrUpdate(entity, false);


		//修改项目立项的状态
		ProjectSetEntity setEntity = projectSetService.getById(entity.getProjectId());
//		setEntity.setProjectStatus(ProjectStatusNewEnum.getCodeByName(entity.getAdjustStateType()));//项目状态 在建
//		setEntity.setBusinessStatus(entity.getStateAfterAdjustment());//业务状态 复工
//		projectSetService.updateById(setEntity);
		syncProjectSet(entity,setEntity);
		// 更新项目池状态
		syncProjectPool(entity);

		logger.info("终审审核完回调--end");
		return CommonResponse.success("终审审核完回调成功");
	}

	/**
	 * 有审批流的撤回前回调
	 *
	 * @param billId 单据id
	 * @param state  单据状态
	 *
	 * @return 响应信息
	 */
	@Override
	public CommonResponse<String> beforeHasBpmBack(Long billId, Integer state, String billTypeCode) {
		return bpmBackCheck(billId, state, billTypeCode);
	}

	/**
	 * 弃审前事件回调
	 *
	 * @param billId 单据id
	 * @param state  单据状态
	 *
	 * @return 响应信息
	 */
	@Override
	public CommonResponse<String> beforeAbstainingProcessor(Long billId, Integer state, String billTypeCode) {
		return bpmBackCheck(billId, state, billTypeCode);
	}

	private CommonResponse<String> bpmBackCheck(Long billId, Integer state, String billTypeCode) {
		AdjustApplicationEntity entity = service.selectById(billId);
		if (entity == null) {
			throw new BusinessException("查询不到单据信息");
		}

		// 校验该申请单是否为该项目最新的申请单，含未生效的
		LambdaQueryWrapper<AdjustApplicationEntity> lambdaQuery = Wrappers.lambdaQuery();
		lambdaQuery.eq(AdjustApplicationEntity::getProjectId, entity.getProjectId());
		lambdaQuery.orderByDesc(AdjustApplicationEntity::getApplyDate, AdjustApplicationEntity::getCreateTime);
		List<AdjustApplicationEntity> entityList = service.list(lambdaQuery);
		if (CollectionUtils.isNotEmpty(entityList) && !Objects.equals(entity.getId(), entityList.get(0).getId())) {
			throw new BusinessException("操作失败！只能撤销最新申请的单据。");
		}
		//修改项目立项的状态
		ProjectSetEntity setEntity = projectSetService.getById(entity.getProjectId());
		setEntity.setProjectStatus(ProjectStatusEnum.getCodeByName(entity.getProjectStatus()));
		setEntity.setBusinessStatus(BusinessStatusEnum.getCodeByName(entity.getBusinessStatus()));
		setEntity.setSettleStatus(SettleStatusEnum.getCodeByName(entity.getSettleStatus()));
		setEntity.setArchiveStatus(ArchiveStatusEnum.getCodeByName(entity.getArchiveStatus()));
		setEntity.setCapitalStatus(CapitalStatusEnum.getCodeByName(entity.getCapitalStatus()));
		setEntity.setChangeStatusDate(entity.getApplyDate());
		projectSetService.updateById(setEntity);
		revertProjectPool(entity);
		return CommonResponse.success("单据撤回/弃审成功");
	}


	/**
	 * 更新项目池状态
	 *
	 * @param entity 实体
	 */
	private void syncProjectPool(AdjustApplicationEntity entity) {
		//项目状态和业务状态
		if (BUSINESS_STATUS1.equals(entity.getAdjustStateType()) || BUSINESS_STATUS2.equals(entity.getAdjustStateType())||BUSINESS_STATUS3.equals(entity.getAdjustStateType())||BUSINESS_STATUS4.equals(entity.getAdjustStateType())) {
			CommonResponse<String> res = projectSetApi.changeProjectStatus(entity.getProjectId(), ProjectStatusEnum.getCodeByName(entity.getAdjustStateType()), BusinessStatusEnum.getCodeByCode(entity.getStateAfterAdjustment()), null, null, null);
			if (!res.isSuccess()) {
				logger.info("工程状态调整申请【{}】更新【项目池】业务状态为【{}】失败，失败原因：{}", entity.getBillCode(), entity.getStateAfterAdjustment(), res.getMsg());
				throw new BusinessException("工程状态调整申请【" + entity.getBillCode() + "】更新【项目池】业务状态为【" + entity.getStateAfterAdjustment() + "】失败，失败原因：" + res.getMsg());
			}
		}
		//决算状态
		if (SETTLE_STATUS.equals(entity.getAdjustStateType())) {
			if ("18".equals(entity.getStateAfterAdjustment())){
				entity.setStateAfterAdjustment("1");
			}
			CommonResponse<String> res = projectSetApi.changeProjectStatus(entity.getProjectId(), null, null, entity.getStateAfterAdjustment(), null, null);
			if (!res.isSuccess()) {
				logger.info("工程状态调整申请【{}】更新【项目池】结算状态为【{}】失败，失败原因：{}", entity.getBillCode(), entity.getStateAfterAdjustment(), res.getMsg());
				throw new BusinessException("工程状态调整申请【" + entity.getBillCode() + "】更新【项目池】结算状态为【" + entity.getStateAfterAdjustment() + "】失败，失败原因：" + res.getMsg());
			}
		}
		//资料状态
		if (ARCHIVE_STATUS.equals(entity.getAdjustStateType())) {
			CommonResponse<String> res = projectSetApi.changeProjectStatus(entity.getProjectId(), null, null, null, entity.getStateAfterAdjustment(), null);
			if (!res.isSuccess()) {
				logger.info("工程状态调整申请【{}】更新【项目池】资料归档状态为【{}】失败，失败原因：{}", entity.getBillCode(), entity.getStateAfterAdjustment(), res.getMsg());
				throw new BusinessException("工程状态调整申请【" + entity.getBillCode() + "】更新【项目池】资料归档状态为【" + entity.getStateAfterAdjustment() + "】失败，失败原因：" + res.getMsg());
			}
		}
		//资金管控状态
		if (CAPITAL_STATUS.equals(entity.getAdjustStateType())) {
			CommonResponse<String> res = projectSetApi.changeProjectStatus(entity.getProjectId(),null, null, null, null, entity.getStateAfterAdjustment());
			if (!res.isSuccess()) {
				logger.info("工程状态调整申请【{}】更新【项目池】资金管控状态为【{}】失败，失败原因：{}", entity.getBillCode(), entity.getStateAfterAdjustment(), res.getMsg());
				throw new BusinessException("工程状态调整申请【" + entity.getBillCode() + "】更新【项目池】资金管控状态为【" + entity.getStateAfterAdjustment() + "】失败，失败原因：" + res.getMsg());
			}
		}
	}
	/**
	 * 修改项目立项的状态
	 *
	 * @param entity 实体
	 */
	private void syncProjectSet(AdjustApplicationEntity entity, ProjectSetEntity projectSetEntity) {
		//项目状态和业务状态
		if (BUSINESS_STATUS1.equals(entity.getAdjustStateType()) || BUSINESS_STATUS2.equals(entity.getAdjustStateType())||BUSINESS_STATUS3.equals(entity.getAdjustStateType())||BUSINESS_STATUS4.equals(entity.getAdjustStateType())) {
			projectSetEntity.setProjectStatus(ProjectStatusEnum.getCodeByName(entity.getAdjustStateType()));
			projectSetEntity.setBusinessStatus(BusinessStatusEnum.getCodeByCode(entity.getStateAfterAdjustment()));
		}
		//决算状态
		if (SETTLE_STATUS.equals(entity.getAdjustStateType())) {
			if ("18".equals(entity.getStateAfterAdjustment())){
				entity.setStateAfterAdjustment("1");
			}
			projectSetEntity.setSettleStatus(entity.getStateAfterAdjustment());
		}
		//资料状态
		if (ARCHIVE_STATUS.equals(entity.getAdjustStateType())) {
			projectSetEntity.setArchiveStatus(entity.getStateAfterAdjustment());

		}
		//资金管控状态
		if (CAPITAL_STATUS.equals(entity.getAdjustStateType())) {
			projectSetEntity.setCapitalStatus(entity.getStateAfterAdjustment());

		}
		projectSetEntity.setChangeStatusDate(entity.getApplyDate());//状态改变时间

		projectSetService.updateById(projectSetEntity);
	}
	/**
	 * 还原项目池状态
	 *
	 * @param entity 实体
	 */
	private void revertProjectPool(AdjustApplicationEntity entity) {
		CommonResponse<String> res = projectSetApi.changeProjectStatus(entity.getProjectId(),
				ProjectStatusEnum.getCodeByName(entity.getProjectStatus()),
				BusinessStatusEnum.getCodeByName(entity.getBusinessStatus()),
				SettleStatusEnum.getCodeByName(entity.getSettleStatus()),
				ArchiveStatusEnum.getCodeByName(entity.getArchiveStatus()),
				CapitalStatusEnum.getCodeByName(entity.getCapitalStatus()));
		if (!res.isSuccess()) {
			logger.info("工程状态调整申请【{}】更新【项目池】业务状态为【{}】，结算状态为【{}】，资料归档状态为【{}】，资金管控状态为【{}】失败，失败原因：{}", entity.getBillCode(), entity.getBusinessStatus(), entity.getSettleStatus(), entity.getArchiveStatus(), entity.getCapitalStatus(), res.getMsg());
			throw new BusinessException("工程状态调整申请【" + entity.getBillCode() + "】更新【项目池】业务状态为【" + entity.getBusinessStatus() + "】，结算状态为【" + entity.getSettleStatus() + "】，资料归档状态为【" + entity.getArchiveStatus() + "】，资金管控状态为【" + entity.getCapitalStatus() + "】失败，失败原因：" + res.getMsg());
		}
	}

}
