package com.ejianc.business.promaterial.settlement.service.impl;

import com.baomidou.mybatisplus.core.conditions.query.LambdaQueryWrapper;
import com.baomidou.mybatisplus.core.conditions.query.QueryWrapper;
import com.ejianc.business.promaterial.check.service.ICheckService;
import com.ejianc.business.promaterial.contract.bean.ContractEntity;
import com.ejianc.business.promaterial.contract.service.IContractService;
import com.ejianc.business.promaterial.pricelib.service.IPriceHistoryService;
import com.ejianc.business.promaterial.reconciliation.service.IReconciliationDetailService;
import com.ejianc.business.promaterial.settlement.bean.SettlementEntity;
import com.ejianc.business.promaterial.settlement.mapper.SettlementMapper;
import com.ejianc.business.promaterial.settlement.service.ISettlementCollectService;
import com.ejianc.business.promaterial.settlement.service.ISettlementDetailService;
import com.ejianc.business.promaterial.settlement.service.ISettlementFeeService;
import com.ejianc.business.promaterial.settlement.service.ISettlementService;
import com.ejianc.business.promaterial.settlement.vo.ContractSettlementRecordVO;
import com.ejianc.business.promaterial.settlement.vo.SettlementRecordVO;
import com.ejianc.business.promaterial.settlement.vo.SettlementVO;
import com.ejianc.foundation.file.api.IAttachmentApi;
import com.ejianc.foundation.orgcenter.api.IOrgApi;
import com.ejianc.foundation.support.api.IBillCodeApi;
import com.ejianc.foundation.support.api.IBillTypeApi;
import com.ejianc.foundation.support.api.IParamConfigApi;
import com.ejianc.foundation.support.vo.BillCodeParam;
import com.ejianc.framework.auth.session.SessionManager;
import com.ejianc.framework.core.context.InvocationInfoProxy;
import com.ejianc.framework.core.exception.BusinessException;
import com.ejianc.framework.core.kit.mapper.BeanMapper;
import com.ejianc.framework.core.kit.time.DateFormatUtil;
import com.ejianc.framework.core.response.BillStateEnum;
import com.ejianc.framework.core.response.CommonResponse;
import com.ejianc.framework.core.response.Parameter;
import com.ejianc.framework.core.response.QueryParam;
import com.ejianc.framework.skeleton.dataPush.ISystemDataPushService;
import com.ejianc.framework.skeleton.template.BaseServiceImpl;
import org.apache.commons.collections.CollectionUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.stereotype.Service;
import redis.clients.jedis.JedisPool;

import java.math.BigDecimal;
import java.math.RoundingMode;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;

/**
 * 结算单主表
 *
 * @author generator
 *
 */
@Service("settlementService")
public class SettlementServiceImpl extends BaseServiceImpl<SettlementMapper, SettlementEntity> implements ISettlementService{
    @Autowired
    private IBillCodeApi billCodeApi;

    private Logger logger = LoggerFactory.getLogger(this.getClass());

    private static final String BILL_CODE_C = "JS_CODE";//此处需要根据实际修改
    @Autowired
    private IContractService contractService;//合同


    /*根据合同id查询*/
    @Override
    public SettlementRecordVO querySettleRecord(Long contractId) {
        SettlementRecordVO vo = new SettlementRecordVO();
        ContractEntity contractEntity = contractService.selectById(contractId);
        vo.setContractId(contractId);
        vo.setContractTaxMny(contractEntity.getContractTaxMny());
        LambdaQueryWrapper<SettlementEntity> wrapper = new LambdaQueryWrapper<>();
        wrapper.eq(SettlementEntity::getContractId, contractId);
        wrapper.eq(SettlementEntity::getDr, 0);
        wrapper.in(SettlementEntity::getBillState,1,3);
        wrapper.orderByAsc(SettlementEntity::getCreateTime);
        List<SettlementEntity> list = super.list(wrapper);
        List<SettlementVO> vos = BeanMapper.mapList(list, SettlementVO.class);
        if(CollectionUtils.isNotEmpty(vos)){
            BigDecimal totalSettlementTaxMny = vos.stream().filter(e -> e.getSettlementTaxMny() != null).map(SettlementVO::getSettlementTaxMny).reduce(BigDecimal.ZERO, BigDecimal::add);
            vo.setTotalSettlementTaxMny(totalSettlementTaxMny);
        }else{
            vo.setTotalSettlementTaxMny(BigDecimal.ZERO);
        }

        if(vo.getContractTaxMny()==null || BigDecimal.ZERO.compareTo(vo.getContractTaxMny()) == 0) {
            vo.setSettleRatio(BigDecimal.ZERO);
        } else {
            vo.setSettleRatio(vo.getTotalSettlementTaxMny().divide(vo.getContractTaxMny(), 8, RoundingMode.HALF_UP));
        }
        vo.setSettlementList(vos);
        //查询是否有最终结算单
        LambdaQueryWrapper<SettlementEntity> wrapper2 = new LambdaQueryWrapper<>();
        wrapper2.eq(SettlementEntity::getContractId, contractId);
        wrapper2.eq(SettlementEntity::getDr, 0);
        wrapper2.eq(SettlementEntity::getSignatureType, 1);//最终
        List<SettlementEntity> list2 = super.list(wrapper2);
        vo.setFlag(!CollectionUtils.isNotEmpty(list2));
        //查询是否有未生效的过程结算单
        LambdaQueryWrapper<SettlementEntity> wrapper3 = new LambdaQueryWrapper<>();
        wrapper3.eq(SettlementEntity::getContractId, contractId);
        wrapper3.eq(SettlementEntity::getDr, 0);
        wrapper3.notIn(SettlementEntity::getBillState, 1,3);
        wrapper3.eq(SettlementEntity::getSignatureType, 0);//过程
        List<SettlementEntity> list3 = super.list(wrapper3);
        vo.setFlagTwo(!CollectionUtils.isNotEmpty(list3));
        return vo;
    }

    @Override
    public CommonResponse<SettlementVO> saveOrUpdate(SettlementVO saveOrUpdateVO) {
//        // 校验结算明细
//        if(CollectionUtils.isEmpty(saveOrUpdateVO.getSettlementDetailList())){
//            throw new BusinessException("检测到当前结算单的材料明细为空，无法保存！");
//        }
        SettlementEntity entity = BeanMapper.map(saveOrUpdateVO, SettlementEntity.class);
        if(entity.getId() == null || entity.getId() == 0){
            if (null!=entity.getContractId() && this.queryExist(entity.getContractId())){
                throw new BusinessException("该合同下有未生效的结算单！");
            }
            BillCodeParam billCodeParam = BillCodeParam.build(BILL_CODE_C,InvocationInfoProxy.getTenantid(),saveOrUpdateVO);
            CommonResponse<String> billCode = billCodeApi.generateBillCode(billCodeParam);
            if(billCode.isSuccess()) {
                entity.setBillCode(billCode.getData());//此处需要根据实际修改 删除本行或者上一行
            }else{
                throw new BusinessException("网络异常， 编码生成失败， 请稍后再试");
            }
        }
        //初始化签章，签字状态
        entity.setSignStatus(0);
        super.saveOrUpdate(entity, false);

        SettlementVO vo = BeanMapper.map(entity, SettlementVO.class);
        return CommonResponse.success("保存或修改单据成功！",vo);
    }

    /*是否有为生效的结算单*/
    public boolean queryExist(Long contractId){
        LambdaQueryWrapper<SettlementEntity> wrapper = new LambdaQueryWrapper<>();
        wrapper.eq(SettlementEntity::getContractId, contractId);
        wrapper.notIn(SettlementEntity::getBillState, 1, 3);//未生效条件
        List<SettlementEntity> list = super.list(wrapper);
        if (CollectionUtils.isNotEmpty(list)) {//有数据表示有未生效的结算单
            return true;
        }
        return false;
    }

    /*获取最近一次含本期结算金额和结算日期 ，结算次数*/
    @Override
    public CommonResponse<Map> getDateMny(Long contractId) {
        Map<String, Object> map = new HashMap<>();
        BigDecimal currentTaxMny = BigDecimal.ZERO; // 结算金额
        BigDecimal currentMny = BigDecimal.ZERO; // 结算金额(无税)
        String sTDate  = null;
        LambdaQueryWrapper<SettlementEntity> wrapper = new LambdaQueryWrapper<>();
        wrapper.orderByDesc(SettlementEntity::getCreateTime);
        wrapper.in(SettlementEntity::getBillState, 1,3);//防止修改时获取本单据的数据
        wrapper.eq(SettlementEntity::getDr, 0);
        wrapper.eq(SettlementEntity::getContractId, contractId);
        List<SettlementEntity> list = super.list(wrapper);
        if(CollectionUtils.isNotEmpty(list)){
            SettlementEntity settlementEntity = list.get(0);
            currentTaxMny = list.stream().filter(e -> e.getSettlementTaxMny() != null).map(SettlementEntity::getSettlementTaxMny).reduce(BigDecimal.ZERO, BigDecimal::add);
            currentMny  = list.stream().filter(e -> e.getSettlementMny() != null).map(SettlementEntity::getSettlementMny).reduce(BigDecimal.ZERO, BigDecimal::add);
            sTDate = DateFormatUtil.formatDate("yyyy-MM-dd", settlementEntity.getSettlementDate());
        }
        map.put("currentTaxMny",currentTaxMny);
        map.put("currentMny",currentMny);
        map.put("sTDate",sTDate);
        map.put("settlementNum",list.size());
        return CommonResponse.success("获取金额和时间、结算次数成功,！",map);
    }


    /*删除*/
    @Override
    public CommonResponse<String> delete(List<SettlementVO> vos) {
        super.removeByIds(vos.stream().map(SettlementVO::getId).collect(Collectors.toList()), true);
        return CommonResponse.success("删除成功！");
    }

    /**
     * 根据合同id查询当前合同下结算列表
     *
     * @param contractId 合同id
     * @return 查询结果
     */
    @Override
    public ContractSettlementRecordVO queryDetailRecord(Long contractId) {
        ContractSettlementRecordVO settleRecordVO = new ContractSettlementRecordVO();

        ContractEntity contract = contractService.selectById(contractId);
        settleRecordVO.setContractId(contract.getId());
        settleRecordVO.setContractTaxMny(contract.getContractTaxMny());
        settleRecordVO.setPerformanceStatus(contract.getPerformanceStatus());

        QueryWrapper<SettlementEntity> listQuery = new QueryWrapper<>();
        listQuery.eq("contract_id", contractId);
        listQuery.in("bill_state", BillStateEnum.PASSED_STATE.getBillStateCode(), BillStateEnum.COMMITED_STATE
                .getBillStateCode());
        listQuery.orderByDesc("settlement_date");
        List<SettlementEntity> list = super.list(listQuery);
        List<SettlementVO> rentSettlementVOS = BeanMapper.mapList(list, SettlementVO.class);
        settleRecordVO.setSettleList(rentSettlementVOS);

        //累计结算金额
        BigDecimal contractSettleMny = BigDecimal.ZERO;
        if (CollectionUtils.isNotEmpty(list)) {
            contractSettleMny = list.stream().map(SettlementEntity::getSettlementTaxMny)
                    .reduce(BigDecimal.ZERO, BigDecimal::add);
        }
        settleRecordVO.setContractSettleMny(contractSettleMny);

        if (BigDecimal.ZERO.compareTo(settleRecordVO.getContractTaxMny()) == 0) {
            settleRecordVO.setSettleRate(BigDecimal.ZERO);
        }
        else {
            settleRecordVO.setSettleRate((settleRecordVO.getContractSettleMny().multiply(BigDecimal.valueOf(100)))
                    .divide(settleRecordVO.getContractTaxMny(), 8, RoundingMode.HALF_UP));
        }
        return settleRecordVO;
    }

    /**
     * 根据合同id查询对应的结算列表
     *
     * @param settlementVO 合同id、结算类型
     * @return 查询结果
     */
    @Override
    public List<SettlementVO> querySettlementByContractId(SettlementVO settlementVO) {
        List<SettlementEntity> list = this.queryListByContractId(settlementVO);
        List<SettlementVO> rtnList = new ArrayList<>();
        if (CollectionUtils.isNotEmpty(list)) {
            rtnList = BeanMapper.mapList(list, SettlementVO.class);
            for (SettlementVO vo : rtnList) {
                vo.setSettlementDateStr(DateFormatUtil.formatDate("yyyy-MM-dd", vo.getSettlementDate()));
                vo.setBillCodeLink(vo.getBillCode());
                vo.setBillStateName(BillStateEnum.getEnumByStateCode(vo.getBillState()).getDescription());
            }
        }
        return rtnList;
    }

    /**
     * 根据合同id查询结算单列表
     *
     * @param settlementVO 合同id、结算单类型
     * @return 查询结果
     */
    private List<SettlementEntity> queryListByContractId(SettlementVO settlementVO) {
        QueryParam queryParam = new QueryParam();
        queryParam.getParams().put("contractId", new Parameter(QueryParam.EQ, settlementVO.getContractId()));
        queryParam.getOrderMap().put("settlementDate", QueryParam.DESC);
        return super.queryList(queryParam, false);
    }

}
