package com.ejianc.business.promaterial.plan.service.impl;

import com.alibaba.fastjson.JSONObject;
import com.baomidou.mybatisplus.core.conditions.query.LambdaQueryWrapper;
import com.baomidou.mybatisplus.core.metadata.IPage;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.ejianc.business.promaterial.contract.enums.ChangeTypeEnum;
import com.ejianc.business.promaterial.plan.bean.BatPlanDetailEntity;
import com.ejianc.business.promaterial.plan.bean.MasterPlanDetailEntity;
import com.ejianc.business.promaterial.plan.bean.MasterPlanEntity;
import com.ejianc.business.promaterial.plan.mapper.BatPlanDetailMapper;
import com.ejianc.business.promaterial.plan.service.IBatPlanDetailService;
import com.ejianc.business.promaterial.plan.service.IMasterPlanService;
import com.ejianc.business.promaterial.plan.vo.BatPlanDetailVO;
import com.ejianc.business.promaterial.plan.vo.BatPlanEnum;
import com.ejianc.business.promaterial.plan.vo.BatPlanVO;
import com.ejianc.business.promaterial.pricelib.vo.MaterialPriceCheckCodeEnum;
import com.ejianc.business.promaterial.pricelib.vo.MaterialPriceHistoryApiVO;
import com.ejianc.business.targetcost.vo.ParamsCheckDsVO;
import com.ejianc.business.targetcost.vo.ParamsCheckVO;
import com.ejianc.foundation.orgcenter.api.IOrgApi;
import com.ejianc.foundation.orgcenter.vo.OrgVO;
import com.ejianc.foundation.share.api.IPriceDepotParamSetApi;
import com.ejianc.foundation.share.api.IProjectPoolApi;
import com.ejianc.foundation.share.consts.PrinceDepotEnum;
import com.ejianc.foundation.share.vo.dto.PriceDepotParamSetDTO;
import com.ejianc.foundation.support.api.IBillCodeApi;
import com.ejianc.foundation.support.api.IDefdocApi;
import com.ejianc.foundation.support.api.IParamConfigApi;
import com.ejianc.foundation.support.vo.BillCodeParam;
import com.ejianc.foundation.support.vo.BillParamVO;
import com.ejianc.foundation.support.vo.DefdocDetailVO;
import com.ejianc.framework.core.context.InvocationInfoProxy;
import com.ejianc.framework.core.exception.BusinessException;
import com.ejianc.framework.core.kit.collection.CollectionUtil;
import com.ejianc.framework.core.kit.mapper.BeanMapper;
import com.ejianc.framework.core.response.BillStateEnum;
import com.ejianc.framework.core.response.CommonResponse;
import com.ejianc.framework.core.response.Parameter;
import com.ejianc.framework.core.response.QueryParam;
import com.ejianc.framework.core.util.ComputeUtil;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import com.ejianc.framework.skeleton.template.BaseServiceImpl;

import com.ejianc.business.promaterial.plan.mapper.BatPlanMapper;
import com.ejianc.business.promaterial.plan.bean.BatPlanEntity;
import com.ejianc.business.promaterial.plan.service.IBatPlanService;
import org.springframework.transaction.annotation.Propagation;
import org.springframework.transaction.annotation.Transactional;

import java.math.BigDecimal;
import java.util.*;
import java.util.stream.Collectors;

/**
 * 材料批次采购计划主表
 *
 * @author generator
 */
@Service("batPlanService")
public class BatPlanServiceImpl extends BaseServiceImpl<BatPlanMapper, BatPlanEntity> implements IBatPlanService {

    private Logger logger = LoggerFactory.getLogger(this.getClass());

    /** 参照查询：经办人字段名 */
    private static final String CONDITION_TRUSTEES_ID = "employeeId";
    /** 参照查询：项目ID */
    private static final String CONDITION_PROJECT_ID = "projectId";
    /** 参照查询：组织ID */
    private static final String CONDITION_ORG_ID = "orgId";

    /** BILL_CODE */
    private static final String BILL_CODE = "PARCEL_PLAN_CODE";//此处需要根据实际修改

    @Autowired
    private IBillCodeApi billCodeApi;

    @Autowired
    private IOrgApi iOrgApi;

    @Autowired
    private IDefdocApi defdocApi;

    @Autowired
    private IProjectPoolApi projectPoolApi;

    /** 子表services */
    @Autowired
    private IBatPlanDetailService batPlanDetailService;
    @Autowired
    private IPriceDepotParamSetApi priceDepotParamSetApi;
    @Autowired
    private IMasterPlanService masterPlanService;
    @Autowired
    private IParamConfigApi paramConfigApi;
    private static final String PLAN_MNY_PARAM_CODE = "P-3n03G60227";//【总计划金额】控【批次计划金额】
    private static final String PLAN_NUM_PARAM_CODE = "P-IT32wl0228";//【总计划清单量】控【批次计划清单量】

    /**
     * 保存
     *
     * @param batPlanVO 保存数据
     * @return 保存结果
     */
    @Override
    @Transactional(rollbackFor = Exception.class, propagation = Propagation.REQUIRED)
    public BatPlanVO saveOrUpdateByVo(BatPlanVO batPlanVO) {
        // parentOrgCode如果是空的，则需要查询赋值
        if (StringUtils.isEmpty(batPlanVO.getParentOrgCode()) && batPlanVO.getParentOrgId() != null) {
            CommonResponse<OrgVO> orgResponse = iOrgApi.getOneById(batPlanVO.getParentOrgId());
            if (orgResponse.isSuccess()) {
                OrgVO orgVO = orgResponse.getData();
                batPlanVO.setParentOrgCode(orgVO.getCode());
            }
        }
        BatPlanEntity entity = BeanMapper.map(batPlanVO, BatPlanEntity.class);
        if (entity.getId() == null || entity.getId() == 0) {
            BillCodeParam billCodeParam = BillCodeParam.build(BILL_CODE, InvocationInfoProxy.getTenantid(), batPlanVO);
            CommonResponse<String> billCode = billCodeApi.generateBillCode(billCodeParam);
            if (billCode.isSuccess()) {
                entity.setBillCode(billCode.getData());//此处需要根据实际修改 删除本行或者上一行
            }
            else {
                throw new BusinessException("网络异常， 编码生成失败， 请稍后再试");
            }
        }
        // 采购状态默认待采购
        if (entity.getPlanState() == null) {
            entity.setPlanState(BatPlanEnum.PLAN_STATE_WAIT.getPlanState());
        }
        super.saveOrUpdate(entity, false);
        return this.copyVoByEntity(entity);
    }

    /**
     * 根据id查询详情
     *
     * @param id 单据id
     * @return 详情
     */
    @Override
    public BatPlanVO queryBatPlanDetailById(Long id) {
        BatPlanEntity entity = super.selectById(id);
        return this.copyVoByEntity(entity);
    }

    /**
     * 查询采购计划列表
     *
     * @param param 查询条件
     * @return 查询结果
     */
    @Override
    public IPage<BatPlanVO> queryBatPlanPage(QueryParam param) {
        this.setDefaultParam(param);
        return this.queryBatPlanVO(param);
    }

    /**
     * 导出
     * @param param 查询参数
     * @return 查询结果
     */
    @Override
    public List<BatPlanVO> excelExportBatPlan(QueryParam param) {
        this.setDefaultParam(param);
        param.setPageIndex(1);
        // 默认10000条
        param.setPageSize(10000);
        List<BatPlanEntity> entityList = super.queryList(param);
        if (CollectionUtils.isNotEmpty(entityList)) {
            List<BatPlanVO> list = BeanMapper.mapList(entityList,BatPlanVO.class);
            for (BatPlanVO vo: list) {
                // 单据状态翻译
//                vo.setBillStateName(BillStateEnum.getEnumByStateCode(vo.getBillState()).getDescription());
                vo.setBillStateName(this.getBillStateName(vo.getBillState()));
                // 采购状态翻译
                vo.setPlanStateName(BatPlanEnum.getDescriptionByStateCode(vo.getPlanState()));
            }
            return list;
        }
        return new ArrayList<>();
    }

    /**
     * 参照查询
     *
     * @param param        查询参数
     * @param condition    参数
     * @param searchObject 模糊查询匹配字段
     * @return 查询结果
     */
    @Override
    public IPage<BatPlanVO> refBatPlanData(QueryParam param, String condition, String searchObject) {
        this.setDefaultParam(param);
//        param.getFuzzyFields().add("billCode");
        // 只查询待采购
        param.getParams().put("planState", new Parameter(QueryParam.EQ, BatPlanEnum.PLAN_STATE_WAIT.getPlanState()));
        // 单据状态
        param.getParams().put("billState", new Parameter(QueryParam.IN,
                Arrays.asList(BillStateEnum.COMMITED_STATE.getBillStateCode(),
                        BillStateEnum.PASSED_STATE.getBillStateCode())));
        // searchObject做模糊查询处理
        if (StringUtils.isNotEmpty(searchObject)) {
            // 系统自带的searchObject实际上是做EQ处理的，在此置为null，手动做like处理
            param.setSearchObject(null);
            JSONObject searchJson = JSONObject.parseObject(searchObject);
            logger.info("参照查询searchObject：【{}】", searchJson);
            if (searchJson != null && !searchJson.isEmpty()) {
                for (Map.Entry<String, Object> entry : searchJson.entrySet()) {
                    param.getParams().put(entry.getKey(), new Parameter(QueryParam.LIKE, entry.getValue()));
                }
            }
        }
        if (StringUtils.isNotEmpty(condition)) {
            // 处理condition */
            JSONObject _con = JSONObject.parseObject(condition);
            logger.info("参照查询condition：【{}】", _con);
//            for (Map.Entry<String, Object> entry : _con.entrySet()) {
//                // 经办人字段名称不一致，特殊处理
//                if (CONDITION_TRUSTEES_ID.equals(entry.getKey())) {
//                    param.getParams().put("trusteesId", new Parameter(QueryParam.EQ, _con.get
//                    (CONDITION_TRUSTEES_ID)));
//                }
//                else {
//                    param.getParams().put(entry.getKey(), new Parameter(QueryParam.EQ, entry.getValue()));
//                }
//            }
            // 经办人
            if (_con.containsKey(CONDITION_TRUSTEES_ID)) {
                param.getParams().put("trusteesId", new Parameter(QueryParam.EQ, _con.get(CONDITION_TRUSTEES_ID)));
            }
            // 项目id
            if (_con.containsKey(CONDITION_PROJECT_ID)) {
                Long projectId = Long.parseLong(_con.get(CONDITION_PROJECT_ID).toString());
                CommonResponse<List<Long>> projectResponse = projectPoolApi.queryProjectIdsByParentProjectId(projectId);
                if (!projectResponse.isSuccess()) {
                    throw new BusinessException("查询项目信息失败！");
                }
                param.getParams().put("projectId", new Parameter(QueryParam.IN, projectResponse.getData()));
            }
            // 组织id
            if (_con.containsKey(CONDITION_ORG_ID)) {
                Long orgId = Long.valueOf(_con.get(CONDITION_ORG_ID).toString());
                CommonResponse<OrgVO> orgResp = iOrgApi.getOneById(orgId);
                OrgVO orgVO = orgResp.getData();
                /** 数据隔离，如果查询组织为项目部，查询orgId，否则查询parentOrgId本下 */
                if (OrgVO.ORG_TYPE_DEPARTMENT.equals(orgVO.getOrgType())) {
                    param.getParams().put("orgId", new Parameter(QueryParam.EQ, orgId));
                }
                else {
                    param.getParams().put("parentOrgId", new Parameter(QueryParam.IN, iOrgApi
                            .findChildrenByParentIdWithoutProjectDept(orgId).getData().stream().map(OrgVO::getId)
                            .collect(Collectors.toList())));
                }
//                param.getParams().put("parentOrgId", new Parameter(QueryParam.EQ, ));
            }

        }
        IPage<BatPlanVO> pageData = this.queryBatPlanVO(param);
        // 查询采购清单
        if (CollectionUtils.isNotEmpty(pageData.getRecords())) {
            List<BatPlanVO> batPlanVOList = pageData.getRecords();
            List<String> idList = batPlanVOList.stream().map(BatPlanVO::getId).map(String::valueOf)
                    .collect(Collectors.toList());
            // 查询子表
            QueryParam detailParam = new QueryParam();
            detailParam.getParams().put("planId", new Parameter(QueryParam.IN, idList));
            List<BatPlanDetailEntity> detailEntityList = batPlanDetailService.queryList(detailParam);
            List<BatPlanDetailVO> detailVOList = BeanMapper.mapList(detailEntityList, BatPlanDetailVO.class);
            Map<Long, List<BatPlanDetailVO>> detailMap = new HashMap<>();
            List<BatPlanDetailVO> mapList;
            // 按照planID对子表结果分组
            for (BatPlanDetailVO vo : detailVOList) {
                if (detailMap.containsKey(vo.getPlanId())) {
                    detailMap.get(vo.getPlanId()).add(vo);
                }
                else {
                    mapList = new ArrayList<>();
                    mapList.add(vo);
                    detailMap.put(vo.getPlanId(), mapList);
                }
            }
            // 将采购计划清单添加到采购计划中
            batPlanVOList.forEach(batPlanVO -> {
                batPlanVO.setBatPlanDetailList(detailMap.get(batPlanVO.getId()));
            });
        }
        return pageData;
    }

    /**
     * 修改采购计划状态
     *
     * @param idList    id列表
     * @param planState 引用类型
     */
    @Override
    public void updateBatPlanByQuoteType(List<String> idList, Integer planState) {
        if (StringUtils.isEmpty(BatPlanEnum.getDescriptionByStateCode(planState))) {
            throw new BusinessException("计划状态错误");
        }
        // 批量更改
        if (CollectionUtils.isNotEmpty(idList)) {
            List<BatPlanEntity> entityList = (List<BatPlanEntity>) super.listByIds(idList);
            if (CollectionUtils.isNotEmpty(entityList)) {
                for (BatPlanEntity entity : entityList) {
                    // 采购完成不允许做其他操作
                    if (BatPlanEnum.PLAN_STATE_OVER.getPlanState().equals(entity.getPlanState())) {
                        logger.info("采购计划编码【{}】计划名称【{}】已采购完成！", entity.getBillCode(), entity.getPlanName());
                        throw new BusinessException("采购计划编码【" + entity.getBillCode() + "】计划名称【" +
                                entity.getPlanName() + "】已采购完成！");
                    }

                    entity.setPlanState(planState);
                }
//                entityList.forEach(entity -> {
//                    entity.setPlanState(planState);
//                });
                super.saveOrUpdateBatch(entityList);
            }
        }
    }

    /**
     * 查询采购类别
     *
     * @return 采购类别自定义档案
     */
    @Override
    public List<DefdocDetailVO> refPurchaseDate() {
        String wzDefCode = "pro-supply-material-supply-content";
        String zzcDefCode = "pro-supply-revolving-materials-supply-content";
        List<DefdocDetailVO> list = new ArrayList<>();
        CommonResponse<List<DefdocDetailVO>> wzRes = defdocApi.getDefDocByDefCode(wzDefCode);
        if (wzRes.isSuccess()) {
            list.addAll(wzRes.getData());
        }
        CommonResponse<List<DefdocDetailVO>> zzcRes = defdocApi.getDefDocByDefCode(zzcDefCode);
        if (zzcRes.isSuccess()) {
            list.addAll(zzcRes.getData());
        }
        return list;
    }

    /**
     * 参数控制
     * @param vo
     * @return
     */
    @Override
    public ParamsCheckVO checkParams(BatPlanVO vo) {
        List<ParamsCheckVO> paramsCheckVOS = new ArrayList<>();
        ParamsCheckVO paramsCheckVO = new ParamsCheckVO();
        paramsCheckVO.setWarnType("none");
        /*添加参数控制区域---*/
        paramsCheckVOS.addAll(this.checkParamsByMasterPlanMny(vo));//总计划控制批次计划金额
        paramsCheckVOS.addAll(this.checkParamsByMasterPlanNum(vo));//总计划清单量控制批次计划清单量

        ParamsCheckVO priceParam = this.priceCheckParams(vo);
        if(priceParam!=null){//价格库控制
            paramsCheckVOS.add(priceParam);
        }
        /*添加参数控制区域---*/
        Map<String, List<ParamsCheckDsVO>> map = new HashMap<>();
        String[] paramsArray = {"alert", "warn", "none"};
        if(CollectionUtils.isNotEmpty(paramsCheckVOS)){
            for (ParamsCheckVO checkVO : paramsCheckVOS) {
                String warnType = checkVO.getWarnType();
                if(map.containsKey(warnType)){
                    List<ParamsCheckDsVO> checkDsVOS = map.get(warnType);
                    checkDsVOS.addAll(checkVO.getDataSource());
                    map.put(warnType,checkDsVOS);
                }else {
                    map.put(warnType,checkVO.getDataSource());
                }
            }
        }
        for (String s : paramsArray) {
            if(map.containsKey(s)){
                paramsCheckVO.setWarnType(s);
                paramsCheckVO.setDataSource(map.get(s));
                if(CollectionUtils.isEmpty(paramsCheckVO.getDataSource())){
                    paramsCheckVO.setWarnType("none");
                }else {
                    return paramsCheckVO;
                }
            }
        }
        return paramsCheckVO;
    }

    @Override
    public ParamsCheckVO priceCheckParams(BatPlanVO batPlanVO) {
        MaterialPriceHistoryApiVO materialPriceHistoryApiVO = new MaterialPriceHistoryApiVO();
        ArrayList<MaterialPriceHistoryApiVO> materialPriceHistoryApiVOS = new ArrayList<>();
            //招标立项
            if(CollectionUtil.isNotEmpty(batPlanVO.getBatPlanDetailList())){
                for (BatPlanDetailVO detailVO : batPlanVO.getBatPlanDetailList()) {
                    if(!"del".equals(detailVO.getRowState()) && detailVO.getMaterialId()!=null){
                        MaterialPriceHistoryApiVO priceHistoryApiVO = new MaterialPriceHistoryApiVO();
                        priceHistoryApiVO.setMaterialId(detailVO.getMaterialId());
                        priceHistoryApiVO.setMaterialName(detailVO.getMaterialName());
                        priceHistoryApiVO.setSpec(detailVO.getSpec());
                        priceHistoryApiVO.setPrice(detailVO.getPrice());
//                        priceHistoryApiVO.setTaxPrice(detailVO.getTaxPrice());
                        priceHistoryApiVO.setHistoryTaxPriceArea(detailVO.getHistoryTaxPriceArea());
                        priceHistoryApiVO.setHistoryPriceArea(detailVO.getHistoryPriceArea());
                        priceHistoryApiVO.setGuideTaxPriceArea(detailVO.getGuideTaxPriceArea());
                        priceHistoryApiVO.setGuidePriceArea(detailVO.getGuidePriceArea());
                        materialPriceHistoryApiVOS.add(priceHistoryApiVO);
                    }
                }
            }
        materialPriceHistoryApiVO.setMaterialPriceHistoryApiVOList(materialPriceHistoryApiVOS);
        materialPriceHistoryApiVO.setOrgId(batPlanVO.getOrgId());
        ParamsCheckVO paramsCheckVO = priceCheckParams(materialPriceHistoryApiVO);
        return  paramsCheckVO;
    }

    public ParamsCheckVO priceCheckParams(MaterialPriceHistoryApiVO materialPriceHistoryApiVO) {
        // 三种控制方式：不控制，提醒，无法保存 (默认为提醒)
        ParamsCheckVO paramsCheckVO = new ParamsCheckVO();
        paramsCheckVO.setWarnType("none");
        String[] paramsArray = {"none", "warn", "alert"};
        List<ParamsCheckVO> paramsCheckVOS = new ArrayList<>();
        List<MaterialPriceHistoryApiVO> detailList = materialPriceHistoryApiVO.getMaterialPriceHistoryApiVOList();
        if(CollectionUtils.isEmpty(detailList)){//传入集合为空直接返回结果
            logger.info("detailList为空直接返回结果");
            return paramsCheckVO;
        }
        String MAX_CODE = MaterialPriceCheckCodeEnum.材料批次计划最高价.getCode();
        String MIN_CODE =MaterialPriceCheckCodeEnum.材料批次计划最低价.getCode();
        Long orgId = materialPriceHistoryApiVO.getOrgId();
        if(null==orgId){
            logger.info("orgId为空");
            return paramsCheckVO;
        }
        //查询高价比例
//        CommonResponse<BillParamVO> maxParamByCode = paramConfigApi.getBillParamByCode(MAX_CODE);
        CommonResponse<List<BillParamVO>> maxParamByCode = paramConfigApi.getBillParamByCodeAndOrgId(MAX_CODE,orgId);
        if(!maxParamByCode.isSuccess() && null == maxParamByCode.getData()){
            logger.info(maxParamByCode.getMsg());
            return paramsCheckVO;
//            throw new BusinessException("获取历史高价控制参数失败");
        }
        //查询高价比例
//        CommonResponse<BillParamVO> minParamByCode = paramConfigApi.getBillParamByCode(MIN_CODE);
        CommonResponse<List<BillParamVO>> minParamByCode = paramConfigApi.getBillParamByCodeAndOrgId(MIN_CODE,orgId);
        if(!minParamByCode.isSuccess() && null == minParamByCode.getData()){
            logger.info(minParamByCode.getMsg());
            return paramsCheckVO;
//            throw new BusinessException("获取历史低价控制参数失败");
        }
        List<BillParamVO> maxParamVOS = maxParamByCode.getData();//校验结果
        List<BillParamVO> minParamVOS = minParamByCode.getData();
        logger.info("材料高价控制信息返回："+JSONObject.toJSONString(maxParamVOS));
        logger.info("材料低价控制信息返回："+JSONObject.toJSONString(minParamVOS));
        //todo 取优先级
        boolean flag = true;
        logger.info("传入参数--{}",JSONObject.toJSONString(PrinceDepotEnum.材料价格库.getCode()));
        CommonResponse<List<PriceDepotParamSetDTO>> commonResponse = priceDepotParamSetApi.queryPriceDepotParamSetList(PrinceDepotEnum.材料价格库.getCode());
        if(!commonResponse.isSuccess()){
            throw new BusinessException(commonResponse.getMsg());
        }
        List<PriceDepotParamSetDTO> data = commonResponse.getData();
        Integer priorityFlag = data.get(0).getPriorityFlag();
        if(null!=priorityFlag&&2==priorityFlag){
            //指导价优先
            flag = false;
        }
        if(CollectionUtils.isNotEmpty(maxParamVOS)){//遍历高价
            for (BillParamVO maxParamVO : maxParamVOS) {
                ParamsCheckVO paramsCheckVOMax = new ParamsCheckVO();//高价
                List<ParamsCheckDsVO> checkDsVOSMax = new ArrayList<>();
                BigDecimal roleValueMax = maxParamVO.getRoleValue();//高价校验比例
                paramsCheckVOMax.setWarnType(paramsArray[maxParamVO.getControlType()]);//高价赋值控制类型
                for (MaterialPriceHistoryApiVO  detailVO: detailList) {//遍历明细
                    //根据优先级取价格
                    String getHistoryPriceArea = "";
                    if(flag){
                        //历史价格优先   先去历史价  为空再取  指导价
                        getHistoryPriceArea = detailVO.getHistoryPriceArea();
                        if(com.baomidou.mybatisplus.core.toolkit.StringUtils.isEmpty(getHistoryPriceArea)){
                            getHistoryPriceArea = detailVO.getGuidePriceArea();
                        }
                    }else{
                        //指导价优先   先去指导价  为空再取  历史价
                        getHistoryPriceArea = detailVO.getGuidePriceArea();
                        if(com.baomidou.mybatisplus.core.toolkit.StringUtils.isEmpty(getHistoryPriceArea)){
                            getHistoryPriceArea = detailVO.getHistoryPriceArea();
                        }
                    }
                    if(getHistoryPriceArea!=null && !getHistoryPriceArea.isEmpty()){//历史区间不为空"-"
                        String[] split = getHistoryPriceArea.split("-");//截取高低价
                        BigDecimal maxPrice = new BigDecimal(split[1]);
                        BigDecimal minPrice =new BigDecimal(split[0]);
                        logger.info("物资：{}--高价：{}---低价：{}",detailVO.getMaterialName()+detailVO.getSpec(),split[1],split[0]);
                        if(maxPrice.compareTo(minPrice)==0){//相等跳过循环
                            continue;
                        }
                        BigDecimal price = detailVO.getPrice()==null?BigDecimal.ZERO:detailVO.getPrice().setScale(4, BigDecimal.ROUND_HALF_UP);//物资单价
                        BigDecimal maxPriceParam = ComputeUtil.safeDiv(ComputeUtil.safeMultiply(maxPrice, roleValueMax), new BigDecimal("100")).setScale(4, BigDecimal.ROUND_HALF_UP);
                        if (price.compareTo(maxPriceParam) >0) {
                            ParamsCheckDsVO paramsCheckDsVO = new ParamsCheckDsVO();
                            paramsCheckDsVO.setOrgName(maxParamVO.getOrgName());
                            paramsCheckDsVO.setWarnItem(detailVO.getMaterialName()+(detailVO.getSpec()==null ? "" : "+"+detailVO.getSpec()));
                            paramsCheckDsVO.setWarnName("材料单价大于历史最高价");
                            StringBuffer stringBuffer = new StringBuffer();
                            stringBuffer.append("材料单价：").append(price)
                                    .append("，历史价格区间:(").append(getHistoryPriceArea)
                                    .append("),历史最高价*").append(roleValueMax).append("%:").append(maxPriceParam)
                                    .append("，超出最高价：").append(ComputeUtil.safeSub(price,maxPriceParam).setScale(4,BigDecimal.ROUND_HALF_UP));
                            paramsCheckDsVO.setContent(stringBuffer.toString());
                            checkDsVOSMax.add(paramsCheckDsVO);
                        }
                    }
                }
                paramsCheckVOMax.setDataSource(checkDsVOSMax);
                paramsCheckVOS.add(paramsCheckVOMax);
            }
        }
        if(CollectionUtils.isNotEmpty(minParamVOS)){//遍历低价
            for (BillParamVO minParamVO : minParamVOS) {
                ParamsCheckVO paramsCheckVOMin = new ParamsCheckVO();//高价
                List<ParamsCheckDsVO> checkDsVOSMin = new ArrayList<>();
                BigDecimal roleValueMin = minParamVO.getRoleValue();//低价校验比例
                paramsCheckVOMin.setWarnType(paramsArray[minParamVO.getControlType()]);//低价赋值控制类型
                for (MaterialPriceHistoryApiVO  detailVO: detailList) {//遍历明细
                    //根据优先级取价格
                    String getHistoryPriceArea = "";
                    if(flag){
                        //历史价格优先   先去历史价  为空再取  指导价
                        getHistoryPriceArea = detailVO.getHistoryPriceArea();
                        if(com.baomidou.mybatisplus.core.toolkit.StringUtils.isEmpty(getHistoryPriceArea)){
                            getHistoryPriceArea = detailVO.getGuidePriceArea();
                        }
                    }else{
                        //指导价优先   先去指导价  为空再取  历史价
                        getHistoryPriceArea = detailVO.getGuidePriceArea();
                        if(com.baomidou.mybatisplus.core.toolkit.StringUtils.isEmpty(getHistoryPriceArea)){
                            getHistoryPriceArea = detailVO.getHistoryPriceArea();
                        }
                    }
                    if(getHistoryPriceArea!=null && !getHistoryPriceArea.isEmpty()){//历史区间不为空"-"
                        String[] split = getHistoryPriceArea.split("-");//截取高低价
                        BigDecimal maxPrice = new BigDecimal(split[1]);
                        BigDecimal minPrice =new BigDecimal(split[0]);
                        logger.info("物资：{}--高价：{}---低价：{}",detailVO.getMaterialName()+detailVO.getSpec(),split[1],split[0]);
                        if(maxPrice.compareTo(minPrice)==0){//相等跳过循环
                            continue;
                        }
                        BigDecimal price = detailVO.getPrice()==null?BigDecimal.ZERO:detailVO.getPrice().setScale(4, BigDecimal.ROUND_HALF_UP);//物资单价
                        BigDecimal minPriceParam = ComputeUtil.safeDiv(ComputeUtil.safeMultiply(minPrice, roleValueMin), new BigDecimal("100")).setScale(4, BigDecimal.ROUND_HALF_UP);
                        if(price.compareTo(minPriceParam) < 0){
                            ParamsCheckDsVO paramsCheckDsVO = new ParamsCheckDsVO();
                            paramsCheckDsVO.setOrgName(minParamVO.getOrgName());
                            paramsCheckDsVO.setWarnItem(detailVO.getMaterialName()+(detailVO.getSpec()==null ? "" : "+"+detailVO.getSpec()));
                            paramsCheckDsVO.setWarnName("材料单价小于历史最低价");
                            StringBuffer stringBuffer = new StringBuffer();
                            stringBuffer.append("材料单价：").append(price)
                                    .append("，历史价格区间:(").append(getHistoryPriceArea)
                                    .append("),历史最低价*").append(roleValueMin).append("%:").append(minPriceParam)
                                    .append("，低于最低价：").append(ComputeUtil.safeSub(minPriceParam,price).setScale(4,BigDecimal.ROUND_HALF_UP));
                            paramsCheckDsVO.setContent(stringBuffer.toString());
                            checkDsVOSMin.add(paramsCheckDsVO);
                        }
                    }
                }
                paramsCheckVOMin.setDataSource(checkDsVOSMin);
                paramsCheckVOS.add(paramsCheckVOMin);
            }
        }
        /*添加参数控制区域---*/
        String[] paramsArr = {"alert", "warn", "none"};
        Map<String, List<ParamsCheckDsVO>> map = new HashMap<>();
        if(org.apache.commons.collections.CollectionUtils.isNotEmpty(paramsCheckVOS)){
            for (ParamsCheckVO checkVO : paramsCheckVOS) {
                String warnType = checkVO.getWarnType();
                if(map.containsKey(warnType)){
                    List<ParamsCheckDsVO> checkDsVOS = map.get(warnType);
                    checkDsVOS.addAll(checkVO.getDataSource());
                    map.put(warnType,checkDsVOS);
                }else {
                    map.put(warnType,checkVO.getDataSource());
                }
            }
        }
        for (String s : paramsArr) {
            if(map.containsKey(s)){
                paramsCheckVO.setWarnType(s);
                paramsCheckVO.setDataSource(map.get(s));
                if(org.apache.commons.collections.CollectionUtils.isEmpty(paramsCheckVO.getDataSource())){
                    paramsCheckVO.setWarnType("none");
                }else{
                    return paramsCheckVO;
                }
            }
        }
        return paramsCheckVO;
    }

    /**
     * 单据管控-批次计划金额大于总计划金额
     *
     * @return
     */
    public List<ParamsCheckVO> checkParamsByMasterPlanMny(BatPlanVO vo) {
        // 三种控制方式：不控制，提醒，无法保存 (默认为提醒)
        String[] paramsArray = {"none", "warn", "alert"};
        List<ParamsCheckVO> paramsCheckVOS = new ArrayList<>();

        BigDecimal temporaryMoney = vo.getTemporaryMoney() == null ? BigDecimal.ZERO : vo.getTemporaryMoney();//本次批次计划金额
        BigDecimal planTaxMny;//总计划金额
        BigDecimal totalMoney = temporaryMoney;//累计批次计划金额  默认赋值本次
        //查询总计划
        LambdaQueryWrapper<MasterPlanEntity> planLambda = new LambdaQueryWrapper<>();
        planLambda.eq(MasterPlanEntity::getProjectId, vo.getProjectId());
        planLambda.in(MasterPlanEntity::getBillState, 1, 3);
        planLambda.eq(MasterPlanEntity::getDr, 0);
        List<MasterPlanEntity> planList = masterPlanService.list(planLambda);
        if (CollectionUtils.isNotEmpty(planList)) {
            planTaxMny = planList.stream().filter(e -> e.getTotalPlanAmt() != null).map(MasterPlanEntity::getTotalPlanAmt).reduce(BigDecimal.ZERO, BigDecimal::add);
        } else {
            return paramsCheckVOS;
        }
        //查询批次计划
        LambdaQueryWrapper<BatPlanEntity> Lambda = new LambdaQueryWrapper<>();
        Lambda.eq(BatPlanEntity::getProjectId, vo.getProjectId());
        Lambda.in(BatPlanEntity::getBillState, 1, 3);
        List<BatPlanEntity> list = super.list(Lambda);//查询项目下所有批次计划
        if (CollectionUtils.isNotEmpty(list)) {
            totalMoney = list.stream().filter(e -> e.getTemporaryMoney() != null).map(BatPlanEntity::getTemporaryMoney).reduce(totalMoney, BigDecimal::add);
        }

        CommonResponse<List<BillParamVO>> billParamByCode = paramConfigApi.getBillParamByCodeAndOrgId(PLAN_MNY_PARAM_CODE, vo.getOrgId());

        if (billParamByCode.isSuccess() && null != billParamByCode.getData()) {
            List<BillParamVO> data = billParamByCode.getData();
            logger.info("总计划金额控制批次计划金额信息返回："+JSONObject.toJSONString(data));
            if(CollectionUtils.isNotEmpty(data)){
                for (BillParamVO datum : data) {
                    ParamsCheckVO paramsCheckVO = new ParamsCheckVO();
                    List<ParamsCheckDsVO> checkDsVOS = new ArrayList<>();
                    BigDecimal roleValue = datum.getRoleValue();
                    BigDecimal comMny = ComputeUtil.safeDiv(ComputeUtil.safeMultiply(planTaxMny, roleValue), new BigDecimal("100")).setScale(2, BigDecimal.ROUND_HALF_UP);
                    paramsCheckVO.setWarnType(paramsArray[datum.getControlType()]);
                    if (totalMoney.compareTo(comMny) > 0) {
                        ParamsCheckDsVO paramsCheckDsVO = new ParamsCheckDsVO();
                        paramsCheckDsVO.setOrgName(datum.getOrgName());
                        paramsCheckDsVO.setWarnItem("批次计划金额超总计划金额");
                        paramsCheckDsVO.setWarnName("累计批次计划金额大于总计划金额");
                        StringBuffer stringBuffer = new StringBuffer();
                        stringBuffer.append("本次批次计划金额：").append(temporaryMoney.setScale(2, BigDecimal.ROUND_HALF_UP))
                                .append("元，含本次累计批次计划金额：").append(totalMoney.setScale(2, BigDecimal.ROUND_HALF_UP))
                                .append("元，总计划金额*").append(roleValue).append("%:").append(comMny.setScale(2, BigDecimal.ROUND_HALF_UP))
                                .append("元。超出金额：").append(ComputeUtil.safeSub(totalMoney, comMny).setScale(2, BigDecimal.ROUND_HALF_UP)).append("元");
                        paramsCheckDsVO.setContent(stringBuffer.toString());
                        checkDsVOS.add(paramsCheckDsVO);
                    }
                    paramsCheckVO.setDataSource(checkDsVOS);
                    paramsCheckVOS.add(paramsCheckVO);
                }
            }
        }
        else {
            logger.info(billParamByCode.getMsg());
            throw new BusinessException("获取控制参数失败");
        }
        return paramsCheckVOS;
    }

    /**
     * 单据管控-批次计划清单量大于批次计划清单量
     *
     * @return
     */
    public List<ParamsCheckVO> checkParamsByMasterPlanNum(BatPlanVO vo) {
        // 三种控制方式：不控制，提醒，无法保存 (默认为提醒)
        String[] paramsArray = {"none", "warn", "alert"};
        List<ParamsCheckVO> paramsCheckVOS = new ArrayList<>();

        //查询总计划
        LambdaQueryWrapper<MasterPlanEntity> planLambda = new LambdaQueryWrapper<>();
        planLambda.eq(MasterPlanEntity::getProjectId, vo.getProjectId());
        planLambda.in(MasterPlanEntity::getBillState, 1, 3);
        planLambda.eq(MasterPlanEntity::getDr, 0);
        List<MasterPlanEntity> planList = masterPlanService.list(planLambda);
        if (CollectionUtils.isEmpty(planList)) {
            return paramsCheckVOS;
        }
        List<MasterPlanDetailEntity> masterPlanDetailList = masterPlanService.selectById(planList.get(0).getId()).getMasterPlanDetailList();

        //查批次计划累计子表
        List<BatPlanDetailEntity> detailEntityList = baseMapper.getBatPlanDetailNumByProjectId(vo.getProjectId());
        //批次计划分类累加数量
        Map<Long, BigDecimal> batPlanTypeMap = new HashMap<>();
        //批次计划档案累加数量
        Map<Long, BigDecimal> batPlanDocMap = new HashMap<>();
        if(CollectionUtils.isNotEmpty(detailEntityList)){
            batPlanDocMap = detailEntityList.stream().filter(e -> null != e.getMaterialId()).collect(Collectors.toMap(BatPlanDetailEntity::getMaterialId, BatPlanDetailEntity::getNum, (key1, key2) -> key2));
            batPlanTypeMap = detailEntityList.stream().filter(e -> null == e.getMaterialId()).collect(Collectors.toMap(BatPlanDetailEntity::getMaterialTypeId, BatPlanDetailEntity::getNum, (key1, key2) -> key2));
        }

        //总计划分类数量
        Map<Long, BigDecimal> masterPlanTypeMap = masterPlanDetailList.stream().filter(t -> null == t.getMaterialId() && !ChangeTypeEnum.中止项.toString().equals(t.getChangeType())).collect(Collectors.toMap(MasterPlanDetailEntity::getMaterialCategoryId, MasterPlanDetailEntity::getNum, (key1, key2) -> key2));
        //总计划档案数量
        Map<Long, BigDecimal> masterPlanDocMny = masterPlanDetailList.stream().filter(t -> null != t.getMaterialId() && !ChangeTypeEnum.中止项.toString().equals(t.getChangeType())).collect(Collectors.toMap(MasterPlanDetailEntity::getMaterialId, MasterPlanDetailEntity::getNum, (key1, key2) -> key2));

        //当前单据吗明细
        List<BatPlanDetailVO> detailList = vo.getBatPlanDetailList();

        CommonResponse<List<BillParamVO>> billParamByCode = paramConfigApi.getBillParamByCodeAndOrgId(PLAN_NUM_PARAM_CODE, vo.getOrgId());
        if (billParamByCode.isSuccess() && null != billParamByCode.getData()) {
            List<BillParamVO> data = billParamByCode.getData();
            logger.info("总计划清单量控制批次计划清单量信息返回："+JSONObject.toJSONString(data));
            if(CollectionUtils.isNotEmpty(data)){
                for (BillParamVO datum : data) {
                    ParamsCheckVO paramsCheckVO = new ParamsCheckVO();
                    List<ParamsCheckDsVO> checkDsVOS = new ArrayList<>();
                    BigDecimal roleValue = datum.getRoleValue();
                    paramsCheckVO.setWarnType(paramsArray[datum.getControlType()]);
                    if(CollectionUtils.isNotEmpty(detailList)){
                        //遍历明细
                        for (BatPlanDetailVO detailVO : detailList) {
                            //总计划清单数量
                            BigDecimal comNum = BigDecimal.ZERO;
                            //含本次批次计划 数量  明细
                            BigDecimal totalNum = BigDecimal.ZERO;

                            if(masterPlanDocMny.containsKey(detailVO.getMaterialId())){ //总计划是否有该清单
                                comNum = masterPlanDocMny.get(detailVO.getMaterialId());
                                totalNum = ComputeUtil.safeAdd(batPlanDocMap.containsKey(detailVO.getMaterialId()) ? batPlanDocMap.get(detailVO.getMaterialId()) : BigDecimal.ZERO, detailVO.getNum());
                            }else if(masterPlanTypeMap.containsKey(detailVO.getMaterialTypeId())) { //总计划是否有该明细
                                comNum = masterPlanDocMny.get(detailVO.getMaterialTypeId());
                                totalNum = ComputeUtil.safeAdd(batPlanTypeMap.containsKey(detailVO.getMaterialTypeId()) ? batPlanTypeMap.get(detailVO.getMaterialTypeId()) : BigDecimal.ZERO, detailVO.getNum());
                            }else {//都不属于
                                continue;
                            }
                            comNum = ComputeUtil.safeDiv(ComputeUtil.safeMultiply(comNum, roleValue), new BigDecimal("100")).setScale(4, BigDecimal.ROUND_HALF_UP);
                            logger.info("materialName-{}，materialTypeName-{}，comNum-{},totalNum-{}", detailVO.getMaterialName(), detailVO.getMaterialTypeName(), comNum, totalNum);
                            if (totalNum.compareTo(comNum) > 0) {
                                ParamsCheckDsVO paramsCheckDsVO = new ParamsCheckDsVO();
                                paramsCheckDsVO.setWarnItem(StringUtils.isNotBlank(detailVO.getMaterialName()) ? detailVO.getMaterialName() : detailVO.getMaterialTypeName() + (detailVO.getSpec()==null ? "" : "+"+detailVO.getSpec()));
                                paramsCheckDsVO.setOrgName(datum.getOrgName());
                                paramsCheckDsVO.setWarnName("批次计划清单数量大于总计划清单数量");
                                StringBuffer stringBuffer = new StringBuffer();
                                stringBuffer.append(StringUtils.isNotBlank(detailVO.getMaterialName()) ? detailVO.getMaterialName() : detailVO.getMaterialTypeName()).append(detailVO.getSpec()==null ? "" : " "+detailVO.getSpec()).append(" ").append("本次批次计划数量：").append(detailVO.getNum().setScale(4, BigDecimal.ROUND_HALF_UP))
                                        .append("，含本次累计批次计划数量：").append(totalNum.setScale(4, BigDecimal.ROUND_HALF_UP))
                                        .append("，总计划数量*").append(roleValue).append("%:").append(comNum.setScale(4, BigDecimal.ROUND_HALF_UP))
                                        .append("。超出数量：").append(ComputeUtil.safeSub(totalNum, comNum).setScale(4, BigDecimal.ROUND_HALF_UP));
                                paramsCheckDsVO.setContent(stringBuffer.toString());
                                checkDsVOS.add(paramsCheckDsVO);
                            }
                            logger.info("checkDsVOS-{}", JSONObject.toJSONString(checkDsVOS));
                        }
                    }
                    paramsCheckVO.setDataSource(checkDsVOS);
                    paramsCheckVOS.add(paramsCheckVO);
                }
            }
        } else {
            logger.info(billParamByCode.getMsg());
            throw new BusinessException("获取控制参数失败");
        }

        return paramsCheckVOS;
    }

    /**
     * 查询分页列表并转换成VO
     *
     * @param param 查询参数
     * @return 查询结果
     */
    private IPage<BatPlanVO> queryBatPlanVO(QueryParam param) {
        IPage<BatPlanEntity> page = super.queryPage(param, false);
        IPage<BatPlanVO> pageData = new Page<>(page.getCurrent(), page.getSize(), page.getTotal());
        List<BatPlanVO> batPlanVOList = BeanMapper.mapList(page.getRecords(), BatPlanVO.class);
        // 采购阶段转义
        for (BatPlanVO vo : batPlanVOList) {
            vo.setPlanStateName(BatPlanEnum.getDescriptionByStateCode(vo.getPlanState()));
        }
        pageData.setRecords(batPlanVOList);
        return pageData;
    }

    /**
     * 预制模糊查询字段
     *
     * @param param 查询参数
     */
    private void setDefaultParam(QueryParam param) {
        List<String> fuzzyFields = param.getFuzzyFields();
        fuzzyFields.add("billCode");
        fuzzyFields.add("planName");
        fuzzyFields.add("projectName");
        fuzzyFields.add("unitName");
        fuzzyFields.add("trusteesName");
        fuzzyFields.add("purchaseName");
        // 租户隔离 */
        param.getParams().put("tenantId", new Parameter(QueryParam.EQ, InvocationInfoProxy.getTenantid()));
        // 数据隔离 本下 没有组织orgId的删除下面代码 */

        CommonResponse<List<OrgVO>> orgResponse = iOrgApi.findChildrenByParentId(InvocationInfoProxy.getOrgId());
        if (!orgResponse.isSuccess()) {
            throw new BusinessException("查询失败，获取组织信息失败！");
        }
        List<OrgVO> orgVOList = orgResponse.getData();
        List<Long> commonOrgIds = new ArrayList<>();
        //项目部 id
        List<Long> departmentIds = new ArrayList<>();
        orgVOList.forEach(org -> {
            if (5 == org.getOrgType()) {
                //项目部
                departmentIds.add(org.getId());
            }
            else {
                //普通组织
                commonOrgIds.add(org.getId());
            }
        });
        // 查询本下
        if (CollectionUtils.isNotEmpty(commonOrgIds)) {
            // 要求主表有orgId字段，保存单据所属组织
            param.getParams().put("parentOrgId", new Parameter(QueryParam.IN, commonOrgIds));
        }
        // 查询项目部
        else if (CollectionUtils.isNotEmpty(departmentIds)) {
            // 要求主表有parentOrgId字段，保存单据所属项目部
            param.getParams().put("orgId", new Parameter(QueryParam.IN, departmentIds));
        }
    }

    /**
     * 根据实体copyVO
     *
     * @param entity 实体
     * @return vo
     */
    private BatPlanVO copyVoByEntity(BatPlanEntity entity) {
        BatPlanVO vo = BeanMapper.map(entity, BatPlanVO.class);
        vo.setPlanStateName(BatPlanEnum.getDescriptionByStateCode(vo.getPlanState()));
        return vo;
    }

    /**
     * 根据单据状态获取单据名称
     * @param billState 单据状态
     * @return 单据名称
     */
    private String getBillStateName(Integer billState){
        String billStateName = null;
        if (BillStateEnum.UNCOMMITED_STATE.getBillStateCode().equals(billState)) {
            billStateName = "自由态";
//            billStateName = BillStateEnum.getEnumByStateCode(billState).getDescription();
        }
        else if(BillStateEnum.COMMITED_STATE.getBillStateCode().equals(billState)){
            billStateName = "已提交";
//            billStateName = BillStateEnum.getEnumByStateCode(billState).getDescription();
        }
        else if(BillStateEnum.APPROVING_HAS_STATE.getBillStateCode().equals(billState)){
            billStateName = "审核中";
        }
        else if(BillStateEnum.PASSED_STATE.getBillStateCode().equals(billState)){
            billStateName = "审批通过";
        }
        else if(BillStateEnum.UNAPPROVED.getBillStateCode().equals(billState)){
            billStateName = "审批驳回";
//            billStateName = BillStateEnum.getEnumByStateCode(billState).getDescription();
        }
        else if(BillStateEnum.APPROVING_UNEXAM_STATE.getBillStateCode().equals(billState)){
            billStateName = "审核中";
        }
        return billStateName;
    }
}
