package com.ejianc.business.process.service.impl;

import com.ejianc.business.process.bean.PicKingEntity;
import com.ejianc.business.process.enums.BillPushStatusEnum;
import com.ejianc.business.process.enums.SupplierSignStatusEnum;
import com.ejianc.business.process.service.IPickingService;
import com.ejianc.foundation.share.api.IShareCooperateApi;
import com.ejianc.foundation.share.vo.CooperateVO;
import com.ejianc.foundation.support.api.IBillTypeApi;
import com.ejianc.framework.auth.session.SessionManager;
import com.ejianc.framework.auth.session.UserContext;
import com.ejianc.framework.core.kit.mapper.BeanMapper;
import com.ejianc.framework.core.response.BillStateEnum;
import com.ejianc.framework.core.response.CommonResponse;
import com.ejianc.framework.skeleton.billState.service.ICommonBusinessService;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import java.util.Date;

@Service("picKing")
public class PickingBpmServiceImpl implements ICommonBusinessService {

    private Logger logger = LoggerFactory.getLogger(this.getClass());
    @Autowired
    private SessionManager sessionManager;
    @Autowired
    private IPickingService service;
    @Autowired
    private IBillTypeApi billTypeApi;
    @Autowired
    private IShareCooperateApi shareCooperateApi;


    /**
     * 提交完回调
     *
     * @param billId 单据id
     * @param state  单据当前状态
     * @return 返回当前成功信息
     */
    @Override
    public CommonResponse<String> afterSubmitProcessor(Long billId, Integer state, String billTypeCode) {
        PicKingEntity e = service.selectById(billId);
        UserContext userContext = sessionManager.getUserContext();
        e.setCommitDate(new Date());
        e.setCommitUserCode(userContext.getUserCode());
        e.setCommitUserName(userContext.getUserName());
        service.saveOrUpdate(e, false);
        return CommonResponse.success("领料结算审批回调处理成功");
    }

    /**
     * 有审批流的撤回前回调
     *
     * @param billId 单据id
     * @param state  单据当前状态
     * @return 返回当前成功信息
     */
    @Override
    public CommonResponse<String> beforeHasBpmBack(Long billId, Integer state, String billTypeCode) {
        return bpmBackCheck(billId, state, billTypeCode);
    }


    /**
     * 终审审核完回调
     *
     * @param billId 单据id
     * @param state  单据当前状态
     * @return 返回当前成功信息
     */
    @Override
    public CommonResponse<String> afterApprovalProcessor(Long billId, Integer state, String billTypeCode) {
        PicKingEntity entity = service.selectById(billId);
        if (entity != null) {
            entity.setEffectiveDate(new Date());
        }
        if (state.equals(BillStateEnum.COMMITED_STATE.getBillStateCode())) {
            //直审时添加单据提交人等信息
            UserContext userContext = sessionManager.getUserContext();
            entity.setCommitDate(new Date());
            entity.setCommitUserCode(userContext.getUserCode());
            entity.setCommitUserName(userContext.getUserName());
        }

        //查询该单据是否支持协同分享，则向供方协同服务推送该单据
        CommonResponse<CooperateVO> cooperateResp = shareCooperateApi.queryCooperateBybillTypeCode(billTypeCode);
        if (!cooperateResp.isSuccess()) {
            logger.error("根据单据类型-{}查询其协同配置信息失败, 不进行单据推送操作，{}", billTypeCode, cooperateResp.getMsg());
            entity.setBillPushFlag(BillPushStatusEnum.未成功推送.getStatus());
        }

        service.saveOrUpdate(entity, false);

        // 协同推送
        if (cooperateResp.isSuccess()) {
            service.pushBillToSupCenter(BeanMapper.map(entity, PicKingEntity.class), billTypeCode, cooperateResp.getData());
        }

        return CommonResponse.success("领料结算审批回调成功！");
    }


    /**
     * 弃审前事件回调
     *
     * @param billId
     * @param state
     * @return
     */
    @Override
    public CommonResponse<String> beforeAbstainingProcessor(Long billId, Integer state, String billTypeCode) {
        return bpmBackCheck(billId, state, billTypeCode);
    }

    private CommonResponse<String> bpmBackCheck(Long billId, Integer state, String billTypeCode) {
        PicKingEntity entity = service.selectById(billId);
        //检测单据是否供应商已签字，
        if (SupplierSignStatusEnum.乙方已签字.getCode().equals(entity.getSupplierSignStatus())) {
            return CommonResponse.error("供应商已签字，无法执行此操作！");
        }

        //合同是否被其他单据引用
        CommonResponse<String> res = billTypeApi.checkQuote(billTypeCode, billId);
        logger.info("平台返回查询被引用情况" + res.isSuccess() + "----" + res.getMsg());

        if (!res.isSuccess()) {
            return CommonResponse.error("当前单据已被下游业务引用，不能弃审！");
        }

        if (BillPushStatusEnum.推送成功.getStatus().equals(entity.getBillPushFlag())) {
            //将推送至供方的单据作废
            boolean pushResult = service.delPushBill(entity, billTypeCode);

            if (!pushResult) {
                return CommonResponse.error("领料结算审批回调失败，删除协同服务单据失败！");
            } else {
                entity.setBillPushFlag(BillPushStatusEnum.未成功推送.getStatus());
                service.saveOrUpdate(entity, false);
            }
        }

        return CommonResponse.success("弃审/撤回前校验回调成功！");
    }

}
