package com.ejianc.business.rmat.api;


import com.alibaba.fastjson.JSONObject;
import com.baomidou.mybatisplus.core.conditions.query.LambdaQueryWrapper;
import com.baomidou.mybatisplus.core.conditions.query.QueryWrapper;
import com.baomidou.mybatisplus.core.conditions.update.LambdaUpdateWrapper;
import com.baomidou.mybatisplus.core.metadata.IPage;
import com.baomidou.mybatisplus.core.toolkit.Wrappers;
import com.ejianc.business.finance.pub.vo.PubContractSettleVO;
import com.ejianc.business.finance.util.MathUtil;
import com.ejianc.business.rmat.bean.RentContractEntity;
import com.ejianc.business.rmat.bean.RentSettlementEntity;
import com.ejianc.business.rmat.service.IRentContractService;
import com.ejianc.business.rmat.service.IRentSettlementService;
import com.ejianc.business.rmat.vo.RentContractVO;
import com.ejianc.business.rmat.vo.RmatProjectReportVo;
import com.ejianc.business.rmat.vo.RmatSettlementVO;
import com.ejianc.business.tax.vo.ContractResVO;
import com.ejianc.foundation.orgcenter.api.IOrgApi;
import com.ejianc.foundation.orgcenter.vo.OrgVO;
import com.ejianc.foundation.share.vo.CustomerVO;
import com.ejianc.foundation.share.vo.SupplierVO;
import com.ejianc.foundation.support.api.ICustomerApi;
import com.ejianc.foundation.support.api.ISupplierApi;
import com.ejianc.framework.auth.session.SessionManager;
import com.ejianc.framework.auth.session.UserContext;
import com.ejianc.framework.core.context.InvocationInfoProxy;
import com.ejianc.framework.core.kit.mapper.BeanMapper;
import com.ejianc.framework.core.response.*;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.web.bind.annotation.*;

import java.math.BigDecimal;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;
import java.util.Objects;
import java.util.stream.Collectors;

@RestController
@RequestMapping("/api/rmat/")
public class RmatContractApi {

    private Logger logger = LoggerFactory.getLogger(this.getClass());


    @Autowired
    private IRentSettlementService rentSettleService;

    @Autowired
    private IRentContractService rentContractService;

    @Autowired
    private ISupplierApi supplierApi;

    @Autowired
    private ICustomerApi customerApi;
    @Autowired
    private SessionManager sessionManager;
    @Autowired
    private IOrgApi orgApi;

    /**
     * @Author sunyj
     * @Date 2020/7/4
     * @Description 收票审批通过后更新周转材租赁合同合同开票金额等字段
     * @Param contractId  合同id
     * @Param invoiceMny、invoiceTaxMny  收票金额
     * @Return com.ejianc.framework.core.response.CommonResponse<java.lang.String>
     */
    @PostMapping(value = "rentAfterApproveInvoiceReceive")
    public CommonResponse<JSONObject> rentAfterApproveInvoiceReceive(@RequestParam("contractId") Long contractId, @RequestParam("invoiceMny") BigDecimal invoiceMny, @RequestParam("invoiceTaxMny") BigDecimal invoiceTaxMny) {
        RentContractEntity entity = rentContractService.selectById(contractId);
        if (null != entity) {
            BigDecimal sumInvoiceMny = entity.getSumInvoiceMny() == null ? BigDecimal.ZERO : entity.getSumInvoiceMny();
            BigDecimal sumInvoiceTaxMny = entity.getSumInvoiceTaxMny() == null ? BigDecimal.ZERO : entity.getSumInvoiceTaxMny();
            LambdaUpdateWrapper<RentContractEntity> updateWrapper = new LambdaUpdateWrapper<>();
            updateWrapper.eq(RentContractEntity::getId, contractId);
            updateWrapper.set(RentContractEntity::getSumInvoiceMny, sumInvoiceMny.add(invoiceMny));
            updateWrapper.set(RentContractEntity::getSumInvoiceTaxMny, sumInvoiceTaxMny.add(invoiceTaxMny));
            boolean flag = rentContractService.update(updateWrapper);
            if (flag) {
                return CommonResponse.success("更新成功");
            } else {
                return CommonResponse.error("更新异常");
            }
        } else {
            return CommonResponse.error("更新异常!未获取到单据信息");
        }

    }

    /**
     * @Author sunyj
     * @Date 2020/7/4
     * @Description 收票弃审通过后更新周转材租赁合同开票金额等字段
     * @Param contractId  合同id
     * @Param invoiceMny、invoiceTaxMny  收票金额
     * @Return com.ejianc.framework.core.response.CommonResponse<java.lang.String>
     */
    @PostMapping(value = "rentAfterRevocationInvoiceReceive")
    public CommonResponse<JSONObject> rentAfterRevocationInvoiceReceive(@RequestParam("contractId") Long contractId, @RequestParam("invoiceMny") BigDecimal invoiceMny, @RequestParam("invoiceTaxMny") BigDecimal invoiceTaxMny) {
        RentContractEntity entity = rentContractService.selectById(contractId);
        if (null != entity) {
            BigDecimal sumInvoiceMny = entity.getSumInvoiceMny() == null ? BigDecimal.ZERO : entity.getSumInvoiceMny();
            BigDecimal sumInvoiceTaxMny = entity.getSumInvoiceTaxMny() == null ? BigDecimal.ZERO : entity.getSumInvoiceTaxMny();

            LambdaUpdateWrapper<RentContractEntity> updateWrapper = new LambdaUpdateWrapper<>();
            updateWrapper.eq(RentContractEntity::getId, contractId);
            updateWrapper.set(RentContractEntity::getSumInvoiceMny, sumInvoiceMny.subtract(invoiceMny));
            updateWrapper.set(RentContractEntity::getSumInvoiceTaxMny, sumInvoiceTaxMny.subtract(invoiceTaxMny));
            boolean flag = rentContractService.update(updateWrapper);
            if (flag) {
                return CommonResponse.success("更新成功");
            } else {
                return CommonResponse.error("更新异常");
            }
        } else {
            return CommonResponse.error("更新异常!未获取到单据信息");
        }
    }

    /**
     * @Author sunyj
     * @Date 2020/7/4
     * @Description 租赁合同跳转收票, 根据主键查询合同相关信心
     * @Param contractId  合同id
     */
    @GetMapping(value = "getRentContractById")
    public CommonResponse<ContractResVO> getRentContractById(@RequestParam("contractId") Long contractId) {
        RentContractEntity rentContractEntity = rentContractService.selectById(contractId);
        ContractResVO rentContractVO = BeanMapper.map(rentContractEntity, ContractResVO.class);
        if (null != rentContractEntity) {
            CommonResponse<SupplierVO> supplierVOCommonResponse = supplierApi.queryById(rentContractEntity.getSupplierId());
            if (supplierVOCommonResponse.isSuccess() && supplierVOCommonResponse.getData() != null) {
                rentContractVO.setSupplierCreditCode(supplierVOCommonResponse.getData().getSocialCreditCode());
            }
            CommonResponse<CustomerVO> customerVOCommonResponse = customerApi.detailById(rentContractEntity.getCustomerId());
            if (supplierVOCommonResponse.isSuccess() && customerVOCommonResponse.getData() != null) {
                rentContractVO.setCustomerCreditCode(customerVOCommonResponse.getData().getSocialCreditCode());

            }
        } else {
            return CommonResponse.error("获取合同信息失败!");
        }

        return CommonResponse.success("获取合同信息成功", rentContractVO);
    }

    @PostMapping(value = "getRmatSettle")
    public CommonResponse<JSONObject> getRmatSettle(@RequestBody QueryParam param) {
        try {
            /** 模糊搜索配置字段示例 */
            List<String> fuzzyFields = param.getFuzzyFields();
            fuzzyFields.add("billCode");
            fuzzyFields.add("orgName");
            fuzzyFields.add("employeeName");
            fuzzyFields.add("customerName");
            fuzzyFields.add("supplierName");
            param.getParams().put("tenantId", new Parameter("eq", InvocationInfoProxy.getTenantid()));
            param.getOrderMap().put("createTime", QueryParam.DESC);
            //已生效状态的单据
            param.getComplexParams().add(ComplexParam.getApprovedComplexParam(ComplexParam.AND));

            IPage<RentSettlementEntity> pageData = rentSettleService.queryPage(param, false);
            List<RentSettlementEntity> entityList = pageData.getRecords();
            List<PubContractSettleVO> voList = new ArrayList<>();
            for (RentSettlementEntity entity : entityList) {
                PubContractSettleVO vo = new PubContractSettleVO();
                vo.setId(entity.getId());
                vo.setBillCode(entity.getBillCode());
                vo.setSettleMny(MathUtil.safeSub(entity.getSettlementTaxMny(), entity.getOffsetMny()));
                vo.setCreateUserCode(entity.getCreateUserCode());
                vo.setCreateTime(entity.getCreateTime());
                voList.add(vo);
            }
            JSONObject page = new JSONObject();
            page.put("records", voList);
            page.put("total", pageData.getTotal());
            page.put("current", pageData.getCurrent());
            page.put("size", pageData.getSize());
            page.put("pages", pageData.getPages());
            return CommonResponse.success("查询列表数据成功！", page);
        } catch (Exception e) {
            logger.error("系统异常：" + e.getMessage());
            e.printStackTrace();
        }
        return CommonResponse.error("查询失败");
    }

    /**
     * @Author yqls
     * @Date 2020/7/4
     * @Description 回写周转材租赁结算单累计付款金额
     * @Param settleId  结算单id
     * @Param payMny  本期已付款金额，正为回写，负为逆回写
     * @Return com.ejianc.framework.core.response.CommonResponse<java.lang.String>
     */
    @GetMapping(value = "updateRentSettleSumPayMny")
    public CommonResponse<String> updateRentSettleSumPayMny(@RequestParam(value = "settleId", required = true) Long settleId,
                                                            @RequestParam("payMny") BigDecimal payMny) {
        RentSettlementEntity entity = rentSettleService.selectById(settleId);
        if (entity == null) {
            return CommonResponse.error("没有找到相应周转材租赁结算单！");
        }
        //累计付款金额
        BigDecimal sumPayMny = entity.getSumPayMny();
        sumPayMny = MathUtil.safeAdd(sumPayMny, payMny);
        entity.setSumPayMny(sumPayMny);
        rentSettleService.saveOrUpdate(entity, false);
        return CommonResponse.success("回写周转材租赁结算单累计付款金额成功！");
    }


    /**
     * @Author sunyj
     * @Date 2020/7/4
     * @Description 根据条件查询结算单累计冲抵金额
     * @Param contractId  合同id
     */
    @GetMapping(value = "getRentSettlementById")
    public CommonResponse<RmatSettlementVO> getRentSettlementById(@RequestParam(value = "contractId", required = true) Long contractId, @RequestParam(value = "orgId", required = true) Long orgId) {
        RmatSettlementVO vo = new RmatSettlementVO();
        vo.setSumOffsetMny(new BigDecimal("0.00").setScale(2, BigDecimal.ROUND_HALF_UP));
        LambdaQueryWrapper<RentSettlementEntity> lambdachange = Wrappers.<RentSettlementEntity>lambdaQuery();
        lambdachange.eq(RentSettlementEntity::getTenantId, InvocationInfoProxy.getTenantid());
        lambdachange.eq(RentSettlementEntity::getContractId, contractId);
        lambdachange.eq(RentSettlementEntity::getOrgId, orgId);
        lambdachange.in(RentSettlementEntity::getBillState, BillStateEnum.PASSED_STATE.getBillStateCode(), BillStateEnum.COMMITED_STATE.getBillStateCode());
        List<RentSettlementEntity> list = rentSettleService.list(lambdachange);
        if (CollectionUtils.isNotEmpty(list)) {
            final BigDecimal[] sumOffsetMny = {new BigDecimal("0.00").setScale(2, BigDecimal.ROUND_HALF_UP)};
            list.forEach(e -> {
                sumOffsetMny[0] = sumOffsetMny[0].add(e.getOffsetMny() == null ? BigDecimal.ZERO : e.getOffsetMny());
            });
            //累计冲抵金额
            vo.setSumOffsetMny(sumOffsetMny[0]);
        }
        return CommonResponse.success("获取信息成功", vo);
    }


    /**
     * @Author yqls
     * @Date 2020/7/4
     * @Description 回写设备租赁结算单累计申请金额、剩余可申请金额
     * @Param settleId  结算单id
     * @Param payMny  本期已付款金额，正为回写，负为逆回写
     * @Return com.ejianc.framework.core.response.CommonResponse<java.lang.String>
     */
    @GetMapping(value = "updateRentSettleSumApplyMny")
    public CommonResponse<String> updateRentSettleSumApplyMny(@RequestParam(value = "settleId", required = true) Long settleId,
                                                              @RequestParam("applyMny") BigDecimal applyMny) {
        RentSettlementEntity entity = rentSettleService.selectById(settleId);
        if (entity == null) {
            return CommonResponse.error("没有找到相应周转材租赁结算单！");
        }
        //累计申请金额
        BigDecimal sumApplyMny = entity.getSumApplyMny();
        sumApplyMny = MathUtil.safeAdd(sumApplyMny, applyMny);
        entity.setSumApplyMny(sumApplyMny);
        //剩余可申请金额
        BigDecimal surplusApplyMny = MathUtil.safeSub(MathUtil.safeSub(entity.getSettlementTaxMny(), entity.getOffsetMny()), applyMny);
        entity.setSurplusApplyMny(surplusApplyMny);
        rentSettleService.saveOrUpdate(entity, false);
        return CommonResponse.success("回写周转材租赁结算单累计申请金额、剩余可申请金额成功！");
    }


    /**
     * @param prepayMny 本期预付款金额，正为回写，负为逆回写
     * @Author yqls
     * @Date 2020/7/4
     * @Description 回写周转材租赁合同累计付款金额、累计预付款金额
     * @Param contractId  合同id
     * @Param payMny  本期已付款金额，正为回写，负为逆回写
     * @Return com.ejianc.framework.core.response.CommonResponse<java.lang.String>
     */
    @GetMapping(value = "updateRentContractSumPayMny")
    public CommonResponse<String> updateRentContractSumPayMny(@RequestParam(value = "contractId", required = true) Long contractId,
                                                              @RequestParam("payMny") BigDecimal payMny, @RequestParam("prepayMny") BigDecimal prepayMny) {
        RentContractEntity entity = rentContractService.selectById(contractId);
        if (entity == null) {
            return CommonResponse.error("没有找到相应周转材租赁合同！");
        }
        //累计付款金额
        BigDecimal sumPayMny = entity.getSumPayMny();
        sumPayMny = MathUtil.safeAdd(sumPayMny, payMny);
        entity.setSumPayMny(sumPayMny);
        //累计预付款金额
        BigDecimal sumPrepayMny = entity.getSumPrepayMny();
        sumPrepayMny = MathUtil.safeAdd(sumPrepayMny, prepayMny);
        entity.setSumPrepayMny(sumPrepayMny);
        rentContractService.saveOrUpdate(entity, false);
        return CommonResponse.success("回写周转材租赁合同累计付款金额、累计预付款金额成功！");
    }

    /**
     * 查周转材合同结算
     *
     * @param projectId
     * @param lastDay
     * @return
     */
    @RequestMapping(value = "getMonthRmatMny", method = RequestMethod.GET)
    @ResponseBody
    public CommonResponse<List<RmatProjectReportVo>> getMonthRmatMny(@RequestParam(value = "projectId") Long projectId,
                                                                     @RequestParam(value = "lastDay") Integer lastDay) {

        return CommonResponse.success("查询列表数据成功", rentContractService.getMonthRmatMny(projectId, lastDay));
    }

    /**
     * 查周转材合同数量，金额，结算金额
     *
     * @param projectId
     * @return
     */
    @GetMapping(value = "getRmatContract")
    @ResponseBody
    public CommonResponse<RmatProjectReportVo> getRmatContract(@RequestParam(value = "projectId") Long projectId) {

        return CommonResponse.success("查询列表数据成功", rentContractService.getRmatContract(projectId));
    }

    /**
     * 按项目维度，获取合同金额(含税)
     *
     * @param projectId 项目id
     *
     * @return {@link CommonResponse}<{@link BigDecimal}>
     */
    @GetMapping(value = "fetchContractTaxMny")
    public CommonResponse<BigDecimal> fetchContractTaxMny(@RequestParam(value = "projectId") Long projectId) {
        return CommonResponse.success("查询列表数据成功", rentContractService.fetchContractTaxMny(projectId));
    }
    /**
     * 查周转材合同数量，金额，结算金额
     *
     * @param projectIds
     *
     * @return
     */
    @PostMapping(value = "/getContractMnyByProjectIds")
    @ResponseBody
    public CommonResponse<RmatProjectReportVo> getContractMnyByProjectIds(@RequestBody List<Long> projectIds){
        List<Long> orgIds = new ArrayList<>();
        UserContext userContext = sessionManager.getUserContext();
        String authOrgIds = userContext.getAuthOrgIds();
        if (StringUtils.isNotEmpty(authOrgIds)) {
            CommonResponse<List<OrgVO>> authResponse =
                    orgApi.findChildrenByParentIds(Arrays.stream(authOrgIds.split(",")).map(Long::parseLong).
                            collect(Collectors.toList()));
            orgIds = authResponse.getData().stream().map(OrgVO::getId).collect(Collectors.toList());
        }  else {
            orgIds = orgApi.findChildrenByParentId(InvocationInfoProxy.getOrgId()).getData().stream().map(OrgVO::getId).collect(Collectors.toList());
        }
        RmatProjectReportVo rmatProjectReportVo = new RmatProjectReportVo();
        QueryWrapper<RentContractEntity> objectQueryWrapper = new QueryWrapper<>();
        objectQueryWrapper.select("IFNUll(sum(contract_tax_mny),0) as contractTaxMny");
        objectQueryWrapper.in(CollectionUtils.isNotEmpty(projectIds),"project_id",projectIds);
        objectQueryWrapper.in(CollectionUtils.isNotEmpty(orgIds),"org_id",orgIds);

        objectQueryWrapper.in("bill_state", Arrays.asList(1,3));
        RentContractEntity one = rentContractService.getOne(objectQueryWrapper);
        if (!Objects.isNull(one)){
            rmatProjectReportVo.setMny(one.getContractTaxMny());
        }
        QueryWrapper<RentSettlementEntity> settleQueryWrapper = new QueryWrapper<>();
        settleQueryWrapper.select("IFNUll(sum(settlement_tax_mny),0) as settlementTaxMny");
        settleQueryWrapper.in(CollectionUtils.isNotEmpty(projectIds),"project_id",projectIds);
        settleQueryWrapper.in("bill_state", Arrays.asList(1,3));
        settleQueryWrapper.in(CollectionUtils.isNotEmpty(orgIds),"org_id",orgIds);
        RentSettlementEntity settle = rentSettleService.getOne(settleQueryWrapper);
        if (!Objects.isNull(settle)){
            rmatProjectReportVo.setSettleMny(settle.getSettlementTaxMny());
        }
        return CommonResponse.success("查询金额数据成功", rmatProjectReportVo);
    }


    @PostMapping(value = "getContractByCondition")
    @ResponseBody
    public CommonResponse<List<RentContractVO>> getContractByCondition(@RequestBody List<Long> enquiryPriceBillIds) {
        LambdaUpdateWrapper<RentContractEntity> lambdaUpdateWrapper = new LambdaUpdateWrapper<>();
        lambdaUpdateWrapper.in(RentContractEntity::getEnquiryPriceBillId, enquiryPriceBillIds);
        lambdaUpdateWrapper.orderByDesc(RentContractEntity::getSignDate);
        List<RentContractEntity> list = rentContractService.list(lambdaUpdateWrapper);

        return CommonResponse.success(BeanMapper.mapList(list, RentContractVO.class));
    }


    @GetMapping(value = "fetchSjzcje")
    public CommonResponse<BigDecimal> fetchSjzcje(@RequestParam(value = "projectId") Long projectId) {
        return CommonResponse.success(rentContractService.fetchSjzcje(projectId));
    }

}


