package com.ejianc.business.scheme.service.impl;

import com.baomidou.mybatisplus.core.conditions.query.LambdaQueryWrapper;
import com.baomidou.mybatisplus.core.conditions.update.LambdaUpdateWrapper;
import com.ejianc.business.scheme.bean.SchemeApprovalEntity;
import com.ejianc.business.scheme.bean.SchemeApprovalMarkEntity;
import com.ejianc.business.scheme.cons.BillCons;
import com.ejianc.business.scheme.service.ISchemeApprovalMarkService;
import com.ejianc.business.scheme.service.ISchemeApprovalService;
import com.ejianc.business.scheme.utils.MathUtil;
import com.ejianc.foundation.support.api.IBillTypeApi;
import com.ejianc.framework.core.context.InvocationInfoProxy;
import com.ejianc.framework.core.response.CommonResponse;
import com.ejianc.framework.skeleton.billState.service.ICommonBusinessService;
import org.apache.commons.collections.CollectionUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import java.math.BigDecimal;
import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.util.Date;
import java.util.List;
import java.util.Map;

@Service("schemeApproval") 
public class SchemeApprovalBpmServiceImpl implements ICommonBusinessService {

	private Logger logger = LoggerFactory.getLogger(this.getClass());

	private static final String SIGN = "mark";

	@Autowired
	private IBillTypeApi billTypeApi;

	@Autowired
	private ISchemeApprovalService approvalService;

	@Autowired
	private ISchemeApprovalMarkService markService;
	
	/**
	 * 提交前回调
	 * 
	 * @param billId
	 * @param state
	 * @return
	 */
	public CommonResponse<String> beforeSubmitProcessor(Long billId, Integer state, String billTypeCode) {
		//TODO
		return CommonResponse.success();
	};

	/**
	 * 提交完回调
	 * 
	 * @param
	 * @return
	 */
	@Override
	public CommonResponse<String> afterSubmitProcessor(Long billId, Integer state, String billTypeCode){
		//TODO
		return CommonResponse.success();
	}

	/**
	 * 审批节点审批中时节点审批前回调
	 *
	 * @param billId //单据id
	 * @param state  //单据状态
	 * @param billTypeCode  //单据类型code
	 * @param sign  //该流程标识
	 * @return
	 */
	@Override
	public CommonResponse<String> beforeInApprovalBack(Long billId, Integer state, String billTypeCode, String sign){
		logger.info("-----------审批节点审批中时节点审批前回调-----------");
		logger.info("sign-----------"+sign);
		if(SIGN.equals(sign)){
			LambdaQueryWrapper<SchemeApprovalMarkEntity> lambda = new LambdaQueryWrapper<>();
			lambda.eq(SchemeApprovalMarkEntity::getApprovalId, billId);
			lambda.eq(SchemeApprovalMarkEntity::getExpertId, InvocationInfoProxy.getUserid());
			lambda.ne(SchemeApprovalMarkEntity::getIsApprive, BillCons.NO);
			List<SchemeApprovalMarkEntity> markList = markService.list();
			if(CollectionUtils.isEmpty(markList)){
				return CommonResponse.error("未执行方案评分！");
			}
		}
		return CommonResponse.success();
	};

	/**
	 * 审批节点审批中时节点审批后回调
	 *
	 * @param billId //单据id
	 * @param state  //单据状态
	 * @param billTypeCode  //单据类型code
	 * @param taskId  //该流程id
	 * @param isEnd  //该节点是否结束
	 * @param sign  //该流程标识
	 * @return
	 */
	@Override
	public CommonResponse<String> afterInApprovalBack(Long billId, Integer state, String billTypeCode, String taskId, Boolean isEnd, String sign, Map<String,Object> other){
		logger.info("-----------审批节点审批中时节点审批后回调-----------");
		logger.info("开始时间-----------"+other.get("taskStartTime"));
		logger.info("审批意见-----------"+other.get("comment"));
		logger.info("other-----------"+other);
		if(SIGN.equals(sign)){
			// 1.查询人员未审批的打分记录
			LambdaQueryWrapper<SchemeApprovalMarkEntity> lambda = new LambdaQueryWrapper<>();
			lambda.eq(SchemeApprovalMarkEntity::getApprovalId, billId);
			lambda.eq(SchemeApprovalMarkEntity::getExpertId, InvocationInfoProxy.getEmployeeId());
			lambda.ne(SchemeApprovalMarkEntity::getIsApprive, BillCons.YES);
			List<SchemeApprovalMarkEntity> markList = markService.list(lambda);
			if(CollectionUtils.isNotEmpty(markList)){
				// 2.修改打分记录
				SchemeApprovalMarkEntity mark = markList.get(0);
				SimpleDateFormat format = new SimpleDateFormat("yyyy-MM-dd HH:mm:ss");
				Date appriveStartTime = new Date();
				try {
					appriveStartTime = format.parse(String.valueOf(other.get("taskStartTime")));
				} catch (ParseException e) {
					e.printStackTrace();
				}
				mark.setAppriveStartTime(appriveStartTime);
				mark.setAppriveEndTime(new Date());
				mark.setExpertOpion(String.valueOf(other.get("comment")));
				mark.setProcessInstanceId(Long.valueOf(taskId));
				mark.setIsApprive(BillCons.YES);
				markService.saveOrUpdate(mark);

				// 3.重新计算平均分
				this.calculateMark(billId);
			}
		}
		return CommonResponse.success();
	}

	/**
	 * 重新计算平均分
	 * @param billId
	 */
	private void calculateMark(Long billId) {
		BigDecimal averageMark = null;
		// 查询已审批的打分记录
		LambdaQueryWrapper<SchemeApprovalMarkEntity> lambda = new LambdaQueryWrapper<>();
		lambda.eq(SchemeApprovalMarkEntity::getApprovalId, billId);
		lambda.eq(SchemeApprovalMarkEntity::getIsApprive, BillCons.YES);
		List<SchemeApprovalMarkEntity> markList = markService.list(lambda);
		if(CollectionUtils.isNotEmpty(markList)){
			for(SchemeApprovalMarkEntity vo : markList){
				averageMark = MathUtil.safeAdd(averageMark, vo.getMark());
			}
			averageMark = MathUtil.safeDiv(averageMark, new BigDecimal(markList.size()));
		}
		// 修改报批已打分及平均分
		LambdaUpdateWrapper<SchemeApprovalEntity> updateWrapper = new LambdaUpdateWrapper<>();
		updateWrapper.set(SchemeApprovalEntity::getAverageMark, averageMark);
		updateWrapper.set(SchemeApprovalEntity::getIsMark, BillCons.YES);
		updateWrapper.eq(SchemeApprovalEntity::getId, billId);
		approvalService.update(updateWrapper);
	}

	/**
	 * 撤回
	 *
	 * @param billId
	 * @param state
	 * @param billTypeCode
	 * @return
	 */
	@Override
	public CommonResponse<String> afterHasBpmBack(Long billId, Integer state, String billTypeCode) {
		logger.info("撤回回调处理：billId: {}, state: {}", billId, state);
		// 查询最新一条已审批通过的打分记录
		LambdaQueryWrapper<SchemeApprovalMarkEntity> lambda = new LambdaQueryWrapper<>();
		lambda.eq(SchemeApprovalMarkEntity::getApprovalId, billId);
//		lambda.eq(SchemeApprovalMarkEntity::getExpertId, InvocationInfoProxy.getEmployeeId());
		lambda.eq(SchemeApprovalMarkEntity::getIsApprive, BillCons.YES);
		lambda.orderByDesc(SchemeApprovalMarkEntity::getCreateTime);
		List<SchemeApprovalMarkEntity> markList = markService.list(lambda);
		if(CollectionUtils.isNotEmpty(markList)){
			SchemeApprovalMarkEntity mark = markList.get(0);
			mark.setIsApprive(BillCons.NO);
			markService.saveOrUpdate(mark, false);
		}

		// 重新计算平均分
		this.calculateMark(billId);
		return CommonResponse.success("撤回回调处理成功！");
	}

	/**
	 * 终审审核前回调
	 * 
	 * @param billId
	 * @param state
	 * @return
	 */
	@Override
	public CommonResponse<String> beforeApprovalProcessor(Long billId, Integer state, String billTypeCode) {
		//TODO
		return CommonResponse.success();
	}

	/**
	 * 终审审核完回调
	 * 
	 * @param
	 * @return
	 */
	@Override
	public CommonResponse<String> afterApprovalProcessor(Long billId, Integer state, String billTypeCode) {
		// 生效时间
		SchemeApprovalEntity entity = approvalService.selectById(billId);
		entity.setPassTime(new Date());
		approvalService.saveOrUpdate(entity);
		return CommonResponse.success();
	}

	/**
	 * 弃审前事件回调
	 * 
	 * @param billId
	 * @param state
	 * @return
	 */
	@Override
	public CommonResponse<String> beforeAbstainingProcessor(Long billId, Integer state, String billTypeCode) {
		//TODO
		SchemeApprovalEntity entity = approvalService.selectById(billId);
		if(entity.getExpertArgumentState() == 1){
			return CommonResponse.error("外部专家已论证，不能弃审!");
		}
		if(entity.getChangeStatus()!=1){
			return CommonResponse.error("单据已变更，无法弃审！");
		}
		CommonResponse<String>  res = billTypeApi.checkQuote(billTypeCode,billId);
		if(!res.isSuccess()) {//单据未被下游单据引用
			return CommonResponse.error(res.getMsg());
		}
		return CommonResponse.success();
	}

	/**
	 * 弃审后事件回调
	 * 
	 * @param billId
	 * @param state
	 * @return
	 */
	@Override
	public CommonResponse<String> afterAbstainingProcessor(Long billId, Integer state, String billTypeCode) {
		logger.info("弃审后事件回调处理：billId: {}, state: {}", billId, state);
		// 修改所有打分记录未审批
		LambdaUpdateWrapper<SchemeApprovalMarkEntity> updateWrapper = new LambdaUpdateWrapper<>();
		updateWrapper.set(SchemeApprovalMarkEntity::getIsApprive, BillCons.NO);
		updateWrapper.eq(SchemeApprovalMarkEntity::getApprovalId, billId);
		markService.update(updateWrapper);
		// 修改报批未打分及平均分
		LambdaUpdateWrapper<SchemeApprovalEntity> updateWrapper2 = new LambdaUpdateWrapper<>();
		updateWrapper2.set(SchemeApprovalEntity::getAverageMark, null);
		updateWrapper2.set(SchemeApprovalEntity::getIsMark, BillCons.NO);
		updateWrapper2.eq(SchemeApprovalEntity::getId, billId);
		approvalService.update(updateWrapper2);
		return CommonResponse.success("弃审后事件回调成功！");
	}

}
