package com.ejianc.business.scientific.sci.utils;

import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.JSONArray;
import com.alibaba.fastjson.JSONObject;
import com.alibaba.fastjson.TypeReference;
import com.ejianc.framework.core.util.CamelAndUnderLineConverter;
import com.ejianc.framework.core.util.JsonUtils;
import com.fasterxml.jackson.databind.DeserializationFeature;
import com.fasterxml.jackson.databind.ObjectMapper;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.util.ArrayList;
import java.util.List;
import java.util.Set;

/**
 * @author yqls
 * @date 2024-11-13 20:32:04
 */
public class JSONUtils {
    private static Logger logger = LoggerFactory.getLogger(JSONUtils.class);

    // 可以全局共享，所以设计成static类型
    private static final ObjectMapper MAPPER;

    static {
        MAPPER = new ObjectMapper();
    }

    /**
     * 说明：json转对象
     *
     * @param json
     * @param cls
     * @return
     * @throws Exception
     */
    public static <T> T map(String json, Class<T> cls) {
        if (StringUtils.isEmpty(json)) {
            return null;
        }

        T obj = null;
        try {
            MAPPER.configure(DeserializationFeature.FAIL_ON_UNKNOWN_PROPERTIES, false);
            obj = MAPPER.readValue(json, cls);
        } catch (Exception e) {
            if (!JsonUtils.isJson(json)) {
                logger.error("content is not json: " + json + " --> " + e.getMessage());
            } else {
                logger.error("Convert to object failure: " + json, e);
            }
        }

        return obj;
    }

    /**
     * json转List
     * @param json
     * @param cls
     * @param <T>
     * @return
     */
    public static <T> List<T> json2ListMapper(String json, Class<T> cls){
        JSONArray array = JSONArray.parseArray(json);
        List<T> list = new ArrayList<>();
        for(Object obj : array){
            T vo = JSONUtils.map(JSONObject.toJSONString(obj), cls);
            if(null != vo){
                list.add(vo);
            }
        }
        return list;
    }

    /**
     * json转List
     * @param array
     * @param cls
     * @param <T>
     * @return
     */
    public static <T> List<T> mapList(JSONArray array, Class<T> cls){
        List<T> list = new ArrayList<>();
        for(Object obj : array){
            T vo = JSONUtils.map(JSONObject.toJSONString(obj), cls);
            if(null != vo){
                list.add(vo);
            }
        }
        return list;
    }

    /**
     * json转List
     * @param sourceList
     * @param cls
     * @param <T>
     * @return
     */
    public static <S,T> List<T> mapList(Iterable<S> sourceList, Class<T> cls){
        List<T> list = new ArrayList<>();
        for (S source : sourceList) {
            if (source != null) {
                T vo = JSONUtils.map(JSONObject.toJSONString(source), cls);
                list.add(vo);
            }
        }
        return list;
    }

    /**
     * json转List
     * @param json
     * @param type
     * @param <T>
     * @return
     */
    public static <T> List<T> json2List(String json, TypeReference<T> type){
        JSONArray array = JSONArray.parseArray(json);
        List<T> list = new ArrayList<>();
        for(Object obj : array){
            JSONUtils.lineToHump(obj);
            T vo = JSON.parseObject(JSONObject.toJSONString(obj), type);
            if(null != vo){
                list.add(vo);
            }
        }
        return list;
    }

    /**
     * json转List
     * @param jsonArray
     * @param cls
     * @param <T>
     * @return
     */
    public static <T> List<T> json2List(String jsonArray, Class<T> cls){
        if(StringUtils.isEmpty(jsonArray)){
            return new ArrayList<>();
        }
        JSONArray array = JSONArray.parseArray(jsonArray);
        array.stream().forEach(x->JSONUtils.lineToHump(x));
        List<T> list = array.toJavaList(cls);
        return list;
    }

    /**
     * 下划线转驼峰
     * @param obj
     */
    public static void lineToHump(Object obj) {
        if(obj instanceof JSONArray){
            JSONArray jsonArray = (JSONArray) obj;
            for(Object json : jsonArray){
                lineToHump(json);
            }
        } else if(obj instanceof JSONObject){
            JSONObject json = (JSONObject) obj;
            Set<String> keySet = json.keySet();
            String[] keyArray = keySet.toArray(new String[keySet.size()]);
            for(String key : keyArray){
                Object value = json.get(key);
                String[] keyStrs = key.split("_");
                if(keyStrs.length > 1){
                    String newKey = CamelAndUnderLineConverter.lineToHump(key);
                    json.remove(key);
                    json.put(newKey, value);
                }

            }
        }
    }
}
