package com.ejianc.foundation.share.controller;

import java.io.Serializable;
import java.util.ArrayList;
import java.util.Date;
import java.util.List;
import java.util.Map;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import com.ejianc.foundation.share.vo.CustomerLinkerVO;
import com.ejianc.foundation.support.api.IBillCodeApi;
import com.ejianc.foundation.support.vo.DefdocDetailVO;
import com.ejianc.foundation.util.ExcelReader;
import com.ejianc.framework.core.kit.collection.ListUtil;
import com.ejianc.framework.core.util.FileUtils;
import com.ejianc.framework.core.util.ImportTemplate;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.stereotype.Controller;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestMethod;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.ResponseBody;

import com.alibaba.fastjson.JSONObject;
import com.baomidou.mybatisplus.core.toolkit.IdWorker;
import com.ejianc.foundation.share.bean.CustomerCategoryEntity;
import com.ejianc.foundation.share.service.ICustomerCategoryService;
import com.ejianc.foundation.share.service.ICustomerService;
import com.ejianc.foundation.share.vo.CustomerCategoryVO;
import com.ejianc.foundation.share.vo.CustomerVO;
import com.ejianc.framework.core.context.InvocationInfoProxy;
import com.ejianc.framework.core.kit.mapper.BeanMapper;
import com.ejianc.framework.core.response.CommonResponse;
import com.ejianc.framework.core.response.Parameter;
import com.ejianc.framework.core.response.QueryParam;
import com.ejianc.framework.core.util.ResultAsTree;
import org.springframework.web.multipart.MultipartFile;
import org.springframework.web.multipart.MultipartHttpServletRequest;

import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;

/**
 * 客户分类
 *
 * @author guoming
 */
@SuppressWarnings({"unchecked", "rawtypes"})
@Controller
@RequestMapping("/customer/category/")
public class CustomerCategoryController implements Serializable {

    private static final long serialVersionUID = 1L;
    private static final String CUSTOMER_CATEGORY_BILL_CODE = "SUPPORT_CUSTOMER_CATEGORY";

    @Autowired
    private ICustomerCategoryService customerCategoryService;
    @Autowired
    private ICustomerService customerService;
    @Autowired
    private IBillCodeApi billCodeApi;

    @Value("${insideCustomerCategoryCode:CC-inside}")
    private String insideCustomerCategoryCode;//内部客户分类编码
    /**
     * 新增或者修改
     *
     * @param customerCategoryVo
     * @return
     */
    @RequestMapping(value = "/saveOrUpdate", method = RequestMethod.POST)
    @ResponseBody
    public CommonResponse<String> saveOrUpdate(@RequestBody CustomerCategoryVO customerCategoryVo) {
        Long tenantId = InvocationInfoProxy.getTenantid();
        if (customerCategoryVo.getId() != null && customerCategoryVo.getId() > 0) {
            CustomerCategoryEntity uniqueBean = customerCategoryService.queryDetail(customerCategoryVo.getId());
            uniqueBean.setCode(customerCategoryVo.getCode());
            uniqueBean.setName(customerCategoryVo.getName());
            uniqueBean.setSequence(customerCategoryVo.getSequence());
            uniqueBean.setDescription(customerCategoryVo.getDescription());
            customerCategoryService.saveOrUpdate(uniqueBean, false);

            return CommonResponse.success("修改成功");
        } else {
            CommonResponse<String> codeBatchByRuleCode = billCodeApi.getCodeBatchByRuleCode(CUSTOMER_CATEGORY_BILL_CODE, tenantId);
            if (codeBatchByRuleCode.isSuccess()) {
                customerCategoryVo.setCode(codeBatchByRuleCode.getData());
            }
            CustomerCategoryEntity customerCategoryEntity = BeanMapper.map(customerCategoryVo, CustomerCategoryEntity.class);
            customerCategoryEntity.setId(IdWorker.getId());
            if (customerCategoryEntity.getParentId() != null && customerCategoryEntity.getParentId() > 0) {
                CustomerCategoryEntity uniqueBean = customerCategoryService.queryDetail(customerCategoryEntity.getParentId());
                customerCategoryEntity.setInnerCode(uniqueBean.getInnerCode() + "|" + customerCategoryEntity.getId());
            } else {
                customerCategoryEntity.setInnerCode(customerCategoryEntity.getId().toString());
            }
            customerCategoryService.saveOrUpdate(customerCategoryEntity, false);

            return CommonResponse.success("保存成功");
        }
    }

    /**
     * 根据主键ID查询客户分类详情
     *
     * @param id
     * @return
     */
    @RequestMapping(value = "/queryDetail", method = RequestMethod.GET)
    @ResponseBody
    public CommonResponse<CustomerCategoryVO> queryDetail(@RequestParam Long id) {
        CustomerCategoryEntity entity = customerCategoryService.queryDetail(id);
        if (entity != null) {
            return CommonResponse.success(BeanMapper.map(entity, CustomerCategoryVO.class));
        } else {
            return CommonResponse.error("查询失败，参数错误！");
        }
    }

    /**
     * 删除
     *
     * @param id
     * @return
     */
    @RequestMapping(value = "/delete", method = RequestMethod.GET)
    @ResponseBody
    public CommonResponse<String> delete(@RequestParam Long id) {
        //先查询该分类下有没有子节点
        List<CustomerCategoryVO> children = customerCategoryService.queryListByPid(id);
        if (children != null && children.size() > 0) {
            return CommonResponse.error("该分类有子节点不允许删除");
        } else {
            //查询该分类下有没有客户信息
            List<CustomerVO> customerList = customerService.queryListByCategoryId(id);
            if (customerList != null && customerList.size() > 0) {
                return CommonResponse.error("该分类下已有客户信息，不允许删除");
            } else {
                customerCategoryService.delete(id);
                return CommonResponse.success("删除成功");
            }
        }
    }

    /**
     * 查询客户分类分页列表
     *
     * @param queryParam
     * @return
     */
    @RequestMapping(value = "/queryList", method = RequestMethod.POST)
    @ResponseBody
    public CommonResponse<JSONObject> queryList(@RequestBody QueryParam queryParam) {
        queryParam.getParams().put("tenantId", new Parameter(QueryParam.EQ, InvocationInfoProxy.getTenantid()));
        queryParam.getOrderMap().put("sequence", "asc");
        List<CustomerCategoryEntity> dataList = customerCategoryService.queryList(queryParam, false);
        List<Map> resultMapList = BeanMapper.mapList(dataList, Map.class);
        JSONObject jsonObject = new JSONObject();
        jsonObject.put("data", ResultAsTree.createTreeData(resultMapList));
//    	Map map = null;
//    	for(CustomerCategoryEntity entity: dataList) {
//    		map = new HashMap<>();
//    		map.put("id", entity.getId());
//    		map.put("key", entity.getId());
//    		map.put("title", entity.getName());
//    		map.put("parentId", entity.getParentId());
//    		resultMapList.add(map);
//    	}
//    	List<Map<String, Object>> result =  ResultAsTree.createTreeData(resultMapList);
//    	
//		JSONObject jsonObject = new JSONObject();
//		jsonObject.put("data", result);
        return CommonResponse.success(jsonObject);
    }

    /**
     * @Author mrsir_wxp
     * @Date 2021/4/30 客户分类树
     * @Description queryRefTree
     * @Param [searchText]
     * @Return java.util.List<java.util.Map < java.lang.String, java.lang.Object>>
     */
    @RequestMapping(value = "/queryRefTree", method = RequestMethod.GET)
    @ResponseBody
    public List<Map<String, Object>> queryRefTree(@RequestParam(value = "searchText", required = false) String searchText) {
        QueryParam queryParam = new QueryParam();
        queryParam.getParams().put("tenantId", new Parameter(QueryParam.EQ, InvocationInfoProxy.getTenantid()));
        if (StringUtils.isNotBlank(searchText)) {
            queryParam.getParams().put("name", new Parameter(QueryParam.LIKE, searchText));
        }
        queryParam.getParams().put("code", new Parameter(QueryParam.NE, insideCustomerCategoryCode));
        queryParam.getOrderMap().put("sequence", "asc");
        List<CustomerCategoryEntity> dataList = customerCategoryService.queryList(queryParam, false);
        if (ListUtil.isEmpty(dataList)) {
            return new ArrayList<>();
        }
        List<Map> resultMapList = BeanMapper.mapList(dataList, Map.class);
        return ResultAsTree.createTreeData(resultMapList);
    }

    /**
     * 导入模板 下载
     *
     * @param request
     * @param response
     */
    @RequestMapping(value = "/download")
    @ResponseBody
    public void download(HttpServletRequest request, HttpServletResponse response) {
        ImportTemplate.initialize(response);
        ImportTemplate.templetdownload(request, "customer-category-import.xlsx", "客户分类导入模板");
    }

    /**
     * excel导入
     *
     * @param request
     * @return
     */
    @RequestMapping(value = "/excelImport", method = RequestMethod.POST)
    @ResponseBody
    public CommonResponse<Object> excelImport(HttpServletRequest request, HttpServletResponse response) {
        Long tenantId = InvocationInfoProxy.getTenantid();
        MultipartHttpServletRequest multipartRequest = (MultipartHttpServletRequest) request;
        Map<String, MultipartFile> fileMap = multipartRequest.getFileMap();
        boolean isFailed = false;
        MultipartFile mf = null;
        for (Map.Entry<String, MultipartFile> entity : fileMap.entrySet()) {
            mf = entity.getValue();
            String originalFileName = mf.getOriginalFilename();
            String extName = null;
            originalFileName = originalFileName.replaceAll("\\/|\\/|\\||:|\\?|\\*|\"|<|>|\\p{Cntrl}", "_");
            originalFileName.replaceAll("00.", "");
            extName = FileUtils.getFileExt(originalFileName, false);
            if (!"xls".equals(extName) && !"xlsx".equals(extName)) {
                isFailed = true;
                break;
            }
        }

        if (isFailed) {
            return CommonResponse.error("文件格式不合法");
        } else {
            List<List<String>> result = ExcelReader.readExcel(mf);
            List<String> errorList = new ArrayList<String>();
            List<CustomerCategoryEntity> list = new ArrayList<>();
            if (result != null && result.size() > 0) {
                for (int i = 0; i < result.size(); i++) {
                    List<String> datas = result.get(i);
                    CustomerCategoryVO customerCategoryVO = new CustomerCategoryVO();
                    CommonResponse<String> codeBatchByRuleCode = billCodeApi.getCodeBatchByRuleCode(CUSTOMER_CATEGORY_BILL_CODE, tenantId);
                    if (codeBatchByRuleCode.isSuccess()) {
                        customerCategoryVO.setCode(codeBatchByRuleCode.getData());
                    } else {
                        customerCategoryVO.setCode(String.valueOf(new Date().getTime()));
                    }
                    if (StringUtils.isNotEmpty(datas.get(0))) {
                        String name = datas.get(0).toString();
                        CustomerCategoryVO vo = customerCategoryService.queryByName(name);
                        customerCategoryVO.setParentId(vo.getId());
                    }
                    if (StringUtils.isEmpty(datas.get(1))) {
                        errorList.add("第" + i + "行数据：" + datas.toString() + "导入失败，原因：客户分类名称不能为空！");
                        continue;
                    }
                    customerCategoryVO.setName(datas.get(1).toString());
                    customerCategoryVO.setDescription(datas.get(2).toString());
                    if (StringUtils.isNotEmpty(datas.get(3).toString())){
                        Pattern pattern = Pattern.compile("^[-+]?(([0-9]+)([.]([0-9]+))?|([.]([0-9]+))?)$");
                        Matcher isNum = pattern.matcher(datas.get(3).toString());
                        if (!isNum.matches()){
                            errorList.add("第" + i + "行数据：" + datas.toString() + "导入失败，原因：排序号必须为数字！");
                            continue;
                        }else {
                            customerCategoryVO.setDescription(datas.get(3).toString());
                        }
                    }
                    CustomerCategoryEntity customerCategoryEntity = BeanMapper.map(customerCategoryVO, CustomerCategoryEntity.class);
                    customerCategoryEntity.setId(IdWorker.getId());
                    if (customerCategoryEntity.getParentId() != null && customerCategoryEntity.getParentId() > 0) {
                        CustomerCategoryEntity uniqueBean = customerCategoryService.queryDetail(customerCategoryEntity.getParentId());
                        customerCategoryEntity.setInnerCode(uniqueBean.getInnerCode() + "|" + customerCategoryEntity.getId());
                    } else {
                        customerCategoryEntity.setInnerCode(customerCategoryEntity.getId().toString());
                    }
                    list.add(customerCategoryEntity);
                }
            }
            if (errorList.size() > 0) {
                errorList.add("导入成功：" + (result.size() - errorList.size() - 1) + "条, 失败：" + errorList.size() + "条");
                return CommonResponse.error("导入失败", errorList);
            } else {
                customerCategoryService.saveOrUpdateBatch(list,list.size());
                return CommonResponse.success("导入成功");
            }
        }
    }

}
