package com.ejianc.foundation.support.service.impl;

import com.alibaba.fastjson.JSONObject;
import com.baomidou.mybatisplus.core.conditions.query.QueryWrapper;
import com.ejianc.foundation.support.bean.CommonSNEntity;
import com.ejianc.foundation.support.mapper.CommonSNMapper;
import com.ejianc.foundation.support.service.ICommonSNService;
import com.ejianc.framework.cache.utils.RedisTool;
import com.ejianc.framework.core.context.InvocationInfoProxy;
import com.ejianc.framework.core.exception.BusinessException;
import com.ejianc.framework.skeleton.template.BaseServiceImpl;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import redis.clients.jedis.Jedis;
import redis.clients.jedis.JedisPool;

import java.util.ArrayList;
import java.util.List;

/**
 * @author CJ
 * @Description:
 * @date 2022/7/7 15:57
 */
@Service(value = "commonNSService")
public class CommonSNServiceImpl extends BaseServiceImpl<CommonSNMapper, CommonSNEntity> implements ICommonSNService {

    private Logger logger = LoggerFactory.getLogger(this.getClass());

    @Autowired
    private JedisPool jedisPool;

    private final String LOCK_KEY_PREFIX = "redis_lock_key::material";
    private final String OPERATE = "GENERATE_ITEM_CODE";
    private long retryInterval = 5000;
    private final int maxRetryTime = 10;
    private final String billType = "EJCBT202311000003";
    private final Integer defaultSnLength = 9;
    @Override
    public List<String> generateSnList(String code, int count, Integer snLength) {
        String lockKey = LOCK_KEY_PREFIX + code;


        Jedis jedis = jedisPool.getResource();
        logger.info("从jedisPool获取jedis对象，jedis对象-{}", jedis);
        boolean lock = getLock(lockKey, jedis);
        if(!lock) {
            logger.info("键-{}尝试获取锁失败,", lockKey);
            throw new BusinessException("编码生成失败，获取锁失败");
        }

        try {
            QueryWrapper<CommonSNEntity> query = new QueryWrapper<>();
            query.eq("source_type", billType);
            query.eq("dimension", code);
            query.eq("tenant_id", InvocationInfoProxy.getTenantid());

            CommonSNEntity curSn = super.getOne(query);
            Long startSn = 1L;
            Integer snLen = null != snLength ? snLength : defaultSnLength;
            if(null == curSn) {
                curSn = new CommonSNEntity();
                curSn.setDimension(code);
                curSn.setTenantId(InvocationInfoProxy.getTenantid());
                curSn.setSourceType(billType);
                curSn.setSnLength(snLen);

                //新的流水号从1开始
                curSn.setCurSn(Long.valueOf(count));
                curSn.setVersion(0);
            } else {
                startSn = curSn.getCurSn()+1;
                curSn.setCurSn(curSn.getCurSn()+count);
            }

            boolean rs = super.saveOrUpdate(curSn, false);
            if(!rs) {
                throw new BusinessException("生成流水号失败，获取最新流水号失败，请重试！");
            }

            List<String> codes = new ArrayList<>();
            for(long start = startSn; start <= curSn.getCurSn(); start++) {
                if(snLen > 0) {
                    codes.add(code + String.format("%0" + snLen + "d", start));
                } else {
                    codes.add(code + (start < 10 ? "0"+start : start));
                }
            }

            return codes;
        } finally {
            unLock(jedis, true, lockKey, OPERATE);
        }
    }

    public boolean getLock(String key, Jedis jedis) {
        int retryTime = 0;
        boolean lock = false;
        while(!lock && retryTime <= maxRetryTime) {
            try {
                if(retryTime >0 ){
                    logger.info("键-{}第{}次尝试获取redis锁, 第{}毫秒后开始尝试", key, retryTime, retryTime * retryInterval);
                    Thread.sleep(retryTime * retryInterval);
                }
                retryTime++;
                // 在数据维度层面进行加锁
                lock = RedisTool.tryLock(jedis, key, OPERATE, 600);
                if(lock) {
                    return true;
                }
            } catch (Exception e) {
                logger.error("根据键-{}获取reids锁异常", JSONObject.toJSONString(key));
                return false;
            }
        }
        return false;
    }

    // 解锁
    public void unLock(Jedis jedis, boolean locked, String key, String operate) {
        try {
            if(locked) {
                RedisTool.releaseLock(jedis, key, operate);
            }
        } finally {
            if(null != jedis) {
                jedis.close();
            }
        }
    }
}
