package com.ejianc.foundation.util;

import com.alibaba.fastjson.JSONObject;
import com.ejianc.foundation.share.bean.EjcCloudSystemEntity;
import com.ejianc.foundation.share.service.ICooperateService;
import com.ejianc.foundation.share.service.IEjcCloudSystemService;
import com.ejianc.foundation.share.service.ISupplierService;
import com.ejianc.foundation.share.vo.CooperateVO;
import com.ejianc.foundation.share.vo.SupplierVO;
import com.ejianc.framework.core.kit.collection.ListUtil;
import com.ejianc.framework.core.response.CommonResponse;
import com.ejianc.framework.core.response.QueryParam;
import com.ejianc.framework.core.util.DESUtils;
import com.ejianc.framework.skeleton.dataPush.ISystemDataPushService;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.stereotype.Component;

import java.math.BigDecimal;
import java.net.URLEncoder;
import java.util.Calendar;
import java.util.Date;
import java.util.List;

/**
 * @author CJ
 * @Description:
 * @date 2022/1/19 20:46
 */
@Component
public class BillShareHelper {

    private Logger logger = LoggerFactory.getLogger(this.getClass());

    @Autowired
    private ICooperateService cooperateService;

    @Autowired
    private ISupplierService supplierService;

    @Autowired
    private ISystemDataPushService systemDataPushService;

    @Value("${common.env.base-host}")
    private String BaseHost;

    @Autowired
    private IEjcCloudSystemService ejcCloudSystemService;

    /**
     * 微信后端信息链接
     * 思路：
     * 1、该地址指向供方后台地址，参数出h5地址外，还携带前端地址
     * 2、微信访问地址，供方后端根据微信提供的weixincode，调用idmAPI 获取对应用户信息
     * 3、后端根据参数判断当前用户是否对当前单据有操作权限，并将校验结果作为参数生成最终前端地址
     * 4、前端页面根据指定参数判断当前用户是否有操作权限，不再进行短信验证。
     *
     */
    private final String WEI_XIN_BACK_URL = "/ejc-supbusiness-web/commomsupbusiness/getBillShareLink";
    private final String WX_CODE_URL = "https://open.weixin.qq.com/connect/qrconnect?appid=wx270b19a26de3997e&scope=snsapi_login&response_type=code&redirect_uri=";

    public CommonResponse<JSONObject> getBillShareLink(Long billId, String billTypeCode, String supplierId, String h5FrontUrl, String wxFrontUrl) {
        JSONObject resp = new JSONObject();

        //设置单据当前系统信息
        QueryParam param = new QueryParam();
        List<EjcCloudSystemEntity> list =  ejcCloudSystemService.queryList(param,false);
        if(ListUtil.isEmpty(list)){
            return CommonResponse.error("益企联公有云平台内的系统标识编码 尚未配置！");
        }
        String ejcCloudSystemId = list.get(0).getCode();

        try {
            //查询该单据是否支持协同分享，则向供方协同服务推送该单据
            CooperateVO cooperateVO = cooperateService.queryCooperateByBillTypeCode(billTypeCode);
            if(null == cooperateVO) {
                logger.error("单据类型-{}未进行协同分享配置，{}", billTypeCode);
                return CommonResponse.error("单据类型未进行协同分享配置, 请设置协同信息后在进行此操作！");
            }

            if(1 != cooperateVO.getShareFlag()) {
                logger.error("单据类型-{}协同配不支持协同分享，{}", billTypeCode, JSONObject.toJSONString(cooperateVO));
                return CommonResponse.error("获取分享连接失败，该单据类型不支持协同分享！");
            }

            //获取供应商服务器地址
            CommonResponse<JSONObject> cacheData2 = systemDataPushService.getAuthInfoBySupplierId(supplierId);
            logger.info("获取到供应商id-{}企业配置信息：{}", supplierId, cacheData2.isSuccess());
            if(!cacheData2.isSuccess()) {
                logger.error("获取到供应商id-{}企业配置信息为空", supplierId);
                return CommonResponse.error("获取分享连接失败, 查找供应商企业配置信息失败！");
            }

            //查询供应商信息
            SupplierVO supplierVO = supplierService.queryDetail(Long.valueOf(supplierId));
            if(null == supplierVO) {
                logger.error("系统中未找查询到匹配的供应商id-{}信息", supplierId);
                return CommonResponse.error("获取分享连接失败, 查询供应商信息失败！");
            }

            //获取企业配置信息中的服务器地址
            String serverAddr = cacheData2.getData().get("appHost").toString();
            //获取企业配置信息中的服务器地址
            logger.info("供应商id-{}企业服务url地址：{}", supplierId, serverAddr);

            //连接失效时间
            Long timeInLong = getAeadTime(cooperateVO.getEffectiveHour());
            String weakFlag = cooperateVO.getWeakFlag().toString();

            //组装连接参数部分
            StringBuffer sp = new StringBuffer("id=");
            sp.append(billId)
                    .append("&tl=")
                    .append(timeInLong)
                    .append("&supplierTenant=")
                    .append(supplierVO.getTenant())
                    .append("&weakFlag=")
                    .append(weakFlag);

            StringBuffer wxParam = new StringBuffer(sp);
            wxParam.append("&frontUrl=").append(URLEncoder.encode(h5FrontUrl, "UTF-8"));

            //数据加密字段
            String secret = DESUtils.encrypt(sp.toString() + "&supplierId=" + supplierId);
            sp.append("&secret=").append(URLEncoder.encode(secret, "utf-8"));

            String wxUrlSecret = DESUtils.encrypt(wxParam.toString() + "&supplierId=" + supplierId);
            wxParam.append("&secret=").append(URLEncoder.encode(wxUrlSecret, "utf-8"));

            String wxShareLink = StringUtils.isNotBlank(wxFrontUrl) ? wxFrontUrl : WEI_XIN_BACK_URL;
            String wxRedirectUrl = URLEncoder.encode(serverAddr + wxShareLink + "?" + wxParam.append("&ejcCloudSystemId=").append(ejcCloudSystemId).toString(), "UTF-8");

            resp.put("mobileH5Url", serverAddr + h5FrontUrl + "?" + sp.toString());
            resp.put("mobileWxUrl", serverAddr + h5FrontUrl + "?" + sp.toString());
//            resp.put("mobileWxUrl", WX_CODE_URL + wxRedirectUrl + "&state=" + timeInLong);
            resp.put("timeout", timeInLong);
            logger.info("供应商id-{}单据分享url地址：{}", resp.toString());
        } catch (Exception e) {
            logger.error("根据参数billId-{},billTypeCode-{},supplierId-{},h5FrontUrl-{}, wxFrontUrl-{}生成分享链接异常",
                    billId, billTypeCode, supplierId, h5FrontUrl, wxFrontUrl, e);
            return CommonResponse.error("生成分享链接失败！");
        }

        return CommonResponse.success("获取分享连接成功！", resp);
    }

    /**
     * 获取当前连接失效时间
     *
     * @param efectiveHours
     * @return
     */
    private Long getAeadTime(BigDecimal efectiveHours) {
        Calendar cal= Calendar.getInstance();
        cal.setTime(new Date());
        // 24小时制
        cal.add(Calendar.HOUR, efectiveHours.intValue());
        return cal.getTimeInMillis();
    }

}
