package com.ejianc.business.signaturemanage.controller.api;


import com.alibaba.fastjson.JSONObject;
import com.alibaba.fastjson.serializer.SerializerFeature;
import com.baomidou.mybatisplus.core.conditions.query.QueryWrapper;
import com.ejianc.business.signaturemanage.bean.SignMgrEntity;
import com.ejianc.business.signaturemanage.bean.SignMgrLogEntity;
import com.ejianc.business.signaturemanage.bean.SignMgrPreviewEntity;
import com.ejianc.business.signaturemanage.bean.SignMgrSignatoryEntity;
import com.ejianc.business.signaturemanage.enums.SignMgrSignatoryEnum;
import com.ejianc.business.signaturemanage.service.ISignMgrLogService;
import com.ejianc.business.signaturemanage.service.ISignMgrService;
import com.ejianc.business.signaturemanage.service.ISignMgrSignatoryService;
import com.ejianc.business.signaturemanage.vo.*;
import com.ejianc.foundation.file.api.IAttachmentApi;
import com.ejianc.foundation.file.vo.AttachmentVO;
import com.ejianc.foundation.orgcenter.api.IEmployeeApi;
import com.ejianc.foundation.orgcenter.vo.EmployeeVO;
import com.ejianc.framework.core.context.InvocationInfoProxy;
import com.ejianc.framework.core.exception.BusinessException;
import com.ejianc.framework.core.kit.mapper.BeanMapper;
import com.ejianc.framework.core.response.CommonResponse;
import net.qiyuesuo.sdk.bean.user.UserDetail;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.web.bind.annotation.*;
import org.springframework.web.context.request.RequestAttributes;
import org.springframework.web.context.request.RequestContextHolder;

import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import java.io.*;
import java.util.*;
import java.util.stream.Collectors;

/**
 * 签章管理API控制器
 *
 * @author baipengyan
 * @Description: 签章管理API控制器
 */
@RestController
@RequestMapping(value = "/api/signaturemanage")
public class SignatureManageApi {
    private final Logger logger = LoggerFactory.getLogger(this.getClass());

    @Autowired
    private ISignMgrService signMgrService;

    @Autowired
    private ISignMgrSignatoryService signMgrSignatoryService;

    @Autowired
    private ISignMgrLogService signMgrLogService;

    @Autowired
    private IEmployeeApi employeeApi;// 获取用户信息

    @Autowired
    private IAttachmentApi attachmentApi; // 文件信息

    @Value("${ejc.fileServerUrl}")
    private String FILE_HOST; // 文件中心域名


    /**
     * 发起签章
     * 1.创建合同文档
     * 2.创建合同
     * 3.获取下一个待签章的人员列表
     * 4.推送消息通知
     *
     * @param initSignatureVO 发起签章VO
     * @return contractId
     */
    @PostMapping(value = "/sendSignature")
    public CommonResponse<Long> initiateSignature(@RequestBody InitSignatureVO initSignatureVO, HttpServletRequest httpServletRequest) {
        logger.info("发起签章初始数据initSignatureVO:{}", JSONObject.toJSONString(initSignatureVO, SerializerFeature.PrettyFormat, SerializerFeature.WriteMapNullValue));
        signMgrService.validation(initSignatureVO);
        // 获取上下文
        String authority = httpServletRequest.getHeader("authority");

        // 校验是否重发合同

        // 先获取发起人名称和手机号
        Long userId = InvocationInfoProxy.getUserid();
        CommonResponse<EmployeeVO> employeeResponse = employeeApi.queryEmployeByUserId(userId);
        logger.info("发起人信息：{}", JSONObject.toJSONString(employeeResponse));
        if (employeeResponse.isSuccess()) {
            String userName = employeeResponse.getData().getUserName();
            if (StringUtils.isEmpty(userName)) {
                throw new BusinessException("请维护发起人名称！");
            }
            String mobilePhone = employeeResponse.getData().getMobilePhone();
            if (StringUtils.isEmpty(mobilePhone)) {
                throw new BusinessException("请维护发起人手机号！");
            }
            initSignatureVO.setCreatorName(employeeResponse.getData().getUserName());
            initSignatureVO.setCreatorContact(employeeResponse.getData().getMobilePhone());
        } else {
            throw new BusinessException("未获取到发起人信息！");
        }

        // 获取文件地址
        String fileName;
        String billDocumentType = null;
        String url;
        CommonResponse<AttachmentVO> attachmentResp = attachmentApi.queryDetail(String.valueOf(initSignatureVO.getBillDocId()));
        logger.info("调用第三方接口返回的文件信息：{}", JSONObject.toJSONString(attachmentResp));
        if (attachmentResp.isSuccess()) {
            // url = attachmentResp.getData().getTruePath();
            url = FILE_HOST + attachmentResp.getData().getFilePath();
            if (StringUtils.isEmpty(url)) {
                throw new BusinessException("文件地址为空！");
            }
            fileName = attachmentResp.getData().getFileName();
            if (StringUtils.isEmpty(fileName)) {
                throw new BusinessException("文件名称为空！");
            }
            billDocumentType = fileName.substring(fileName.lastIndexOf(".") + 1, fileName.length());
        } else {
            throw new BusinessException("获取文件信息异常！");
        }

        Long documentId = signMgrService.createDocument(url, initSignatureVO.getBillDocumentName(), billDocumentType);
        Long contractId = signMgrService.createContract(documentId, initSignatureVO);

        QueryWrapper<SignMgrEntity> queryWrapper = new QueryWrapper<>();
        queryWrapper.eq("source_bill_id", contractId);
        SignMgrEntity signMgrEntity = signMgrService.getOne(queryWrapper);

        String billType = signMgrEntity.getBillType();
        Long billId = signMgrEntity.getBillId();
        String billRefCode = signMgrEntity.getBillRefCode();

        Integer maxSignOrder = -1;
        Integer maxSealSignOrder = -1;

        // 找到下一个签章人
        List<SignMgrSignatoryEntity> operators = signMgrService.nextSignatory(contractId, maxSignOrder, maxSealSignOrder);

        int status;
        if (operators.isEmpty()) {
            throw new BusinessException("找不到下一个签章人！");
        } else {
            // 发送消息通知
            signMgrService.sendMessage(contractId, operators, initSignatureVO.getBillCode(), initSignatureVO.getContractName());
            if (Objects.equals(operators.get(0).getSignatureType(), SignMgrSignatoryEnum.INTERNAL_UNIT.getValue())) {
                status = 3;
                // 待甲方签章
                signMgrService.sendStatus(billType, billId, billRefCode, status, authority);
            } else {
                status = 2;
                // 待乙方签章
                signMgrService.sendStatus(billType, billId, billRefCode, status, authority);
            }
        }
        return CommonResponse.success("发起签章，操作成功！", contractId);
    }


    /**
     * 契约锁，全局状态回调
     *
     * @param map
     * @return
     */
    @PostMapping(value = "/qysCallBack/global")
    public CommonResponse<Long> global(@RequestParam Map<String, String> map, HttpServletRequest httpServletRequest) {

        logger.info("回调入参map为：{}", JSONObject.toJSONString(map, SerializerFeature.PrettyFormat, SerializerFeature.WriteMapNullValue));

        switch (map.get("type")) {
            case "CONTRACT_COMPLETE":
                //     文件全部签署完成回调成功--{
                //     "callbackBusinessType":"CONTRACT",
                //             "tenantName":"云南工程建设总承包股份有限公司",
                //             "signatoriesInfo":"[{\"tenantId\":\"2940149858777051390\",\"tenantName\":\"\\u5317\\u4EAC\\u76CA\\u4F01\\u8054\\u79D1\\u6280\\u6709\\u9650\\u516C\\u53F8\",\"tenantType\":\"COMPANY\",\"receiverId\":\"2940148477009735713\",\"receiverName\":\"\\u90B9\\u5BCC\\u96C4\",\"receiverContact\":\"17600548718\",\"receiverType\":\"PERSONAL\",\"status\":\"COMPLETE\",\"serialNo\":1,\"sponsor\":false,\"operatorInfos\":[]},{\"tenantId\":\"2905355236628385838\",\"tenantName\":\"\\u4E91\\u5357\\u5DE5\\u7A0B\\u5EFA\\u8BBE\\u603B\\u627F\\u5305\\u80A1\\u4EFD\\u6709\\u9650\\u516C\\u53F8\",\"tenantType\":\"INNER_COMPANY\",\"receiverId\":\"2940148474610593798\",\"receiverName\":\"\\u7CFB\\u7EDF\\u7BA1\\u7406\\u5458\",\"receiverContact\":\"13120319591\",\"receiverType\":\"PERSONAL\",\"status\":\"COMPLETE\",\"serialNo\":2,\"sponsor\":true,\"operatorInfos\":[]}]",
                //             "contractId":"2940234542370402615",
                //             "tenantId":"2905355236628385838",
                //             "completeTime":"Sat Mar 12 20:07:01 GMT+08:00 2022",
                //             "callbackHeader":"{}",
                //             "documentId":"2940234541841920308",
                //             "callbackEventType":"ELECTRONIC",
                //             "sn":"null",
                //             "type":"CONTRACT_COMPLETE",
                //             "status":"COMPLETE"
                // }
                // 文件全部签署完成
                logger.info("文件全部签署完成回调成功--{}", JSONObject.toJSONString(map, SerializerFeature.PrettyFormat, SerializerFeature.WriteMapNullValue));
                break;
            case "CONTRACT_TERMINATE":
                // 文件作废

                break;
            case "CONTRACT_EXPIRED":
                // 文件过期

                break;
            default:
                break;
        }

        return CommonResponse.success("全局，回调成功！");
    }


    /**
     * 契约锁，发起方状态回调
     *
     * @param map
     * @return
     */
    @PostMapping(value = "/qysCallBack/organizer")
    public CommonResponse<String> organizer(@RequestParam Map<String, String> map, HttpServletRequest httpServletRequest) {
        logger.info("回调入参map为：{}", JSONObject.toJSONString(map));

        Long contractId = Long.valueOf(map.get("contractId"));
        int status;

        switch (map.get("type")) {
            case "CONTRACT_SIGN":
                // 文件签署-签署方签署

                break;
            case "INTERNAL_FLOW":
                // 每个节点签署完成
                logger.info("================================>单个节点回调-START<================================");
                QueryWrapper<SignMgrEntity> queryWrapper = new QueryWrapper<>();
                queryWrapper.eq("source_bill_id", contractId);
                SignMgrEntity signMgrEntity = signMgrService.getOne(queryWrapper);

                // 获取上下文
                String authority = signMgrService.login(httpServletRequest, signMgrEntity.getTenantId());

                InvocationInfoProxy.setTenantid(signMgrEntity.getTenantId());
                String key = "userCode";
                int idx = authority.indexOf(";" + key + "=");
                if (idx >= 0) {
                    String tmp = authority.substring(idx + 1);
                    InvocationInfoProxy.setUsercode(tmp.substring(0, tmp.indexOf(";")).replace(key + "=", ""));
                }

                logger.info("获取的上下文信息为：{}", authority);
                RequestContextHolder.getRequestAttributes().setAttribute("authority", authority, RequestAttributes.SCOPE_REQUEST);

                String billType = signMgrEntity.getBillType();
                Long billId = signMgrEntity.getBillId();
                String billRefCode = signMgrEntity.getBillRefCode();

                QueryWrapper<SignMgrSignatoryEntity> queryWrapper1 = new QueryWrapper<>();
                queryWrapper1.eq("pid", signMgrEntity.getId());
                List<SignMgrSignatoryEntity> actionList = signMgrSignatoryService.list(queryWrapper1);


                Map<String, Object> minMax = signMgrService.internalFlow(map, signMgrEntity, actionList);
                Integer maxSignOrder = (Integer) minMax.get("maxSignOrder");
                Integer maxSealSignOrder = (Integer) minMax.get("maxSealSignOrder");

                List<SignMgrLogEntity> slog = (List<SignMgrLogEntity>) minMax.get("SignMgrLogEntity");
                List<SignMgrSignatoryEntity> operators = signMgrService.nextSignatory(contractId, maxSignOrder, maxSealSignOrder);

                // 通知业务系统结果
                String res;
                if (operators.isEmpty()) {
                    // 全都签章完成
                    // 无待激活状态，判定全部完成,（不做操作的话等全局回调合同完成）这里可以把合同状态通知给业务系统
                    status = 4;
                    res = signMgrService.sendStatus(billType, billId, billRefCode, status, authority);
                } else {
                    // 发送消息通知
                    logger.info("================================>发送消息通知-START<================================");
                    signMgrService.sendMessage(contractId, operators, signMgrEntity.getBillCode(), signMgrEntity.getContractName());
                    logger.info("================================>发送消息通知-END<================================");
                    if (Objects.equals(operators.get(0).getSignatureType(), SignMgrSignatoryEnum.INTERNAL_UNIT.getValue())) {
                        status = 3;
                        // 待甲方签章
                        logger.info("================================>业务系统更新状态-待甲方签章-START<================================");
                        res = signMgrService.sendStatus(billType, billId, billRefCode, status, authority);
                        logger.info("================================>业务系统更新状态-待甲方签章-END<================================");
                    } else {
                        status = 2;
                        // 待乙方签章
                        logger.info("================================>业务系统更新状态-待乙方签章-START<================================");
                        res = signMgrService.sendStatus(billType, billId, billRefCode, status, authority);
                        logger.info("================================>业务系统更新状态-待乙方签章-END<================================");
                    }
                }

                slog.stream().peek(e -> e.setBillSysResp(res)).collect(Collectors.toList());
                signMgrLogService.updateBatchById(slog);

                // 异步任务-文件下载和上传
                signMgrService.fileTask(contractId, status, signMgrEntity, authority, billType, billId);

                logger.info("================================>单个节点回调-END<================================");
                break;
            case "CONTRACT_RECALL":
                // 文件撤回-发起方撤回

                break;
            default:
                break;
        }

        return CommonResponse.success("发起方状态，回调成功！");
    }


    /**
     * 契约锁，接收方状态回调
     *
     * @param map
     * @return
     */
    @PostMapping(value = "/qysCallBack/receiver")
    public CommonResponse<String> receiver(@RequestParam Map<String, String> map, HttpServletRequest httpServletRequest) {

        logger.info("回调入参map为：{}", JSONObject.toJSONString(map, SerializerFeature.PrettyFormat, SerializerFeature.WriteMapNullValue));
        Long contractId = Long.valueOf(map.get("contractId"));

        switch (map.get("type")) {
            case "ALLRECEIVER_SIGNATORY_COMPLATE":
                //     所有接收方签署完成回调成功--{
                //     "callbackBusinessType":"CONTRACT",
                //             "tenantName":"北京益企联科技有限公司",
                //             "contact":"17600548718",
                //             "contractId":"2940234542370402615",
                //             "tenantId":"2940149858777051390",
                //             "callbackHeader":"{}",
                //             "callbackEventType":"ELECTRONIC",
                //             "sn":"null",
                //             "type":"ALLRECEIVER_SIGNATORY_COMPLATE",
                //             "signatorySerialNo":"S_999999_proSubContract_557654601943093280_BT220216000000003_1502615982810415106",
                //             "signatoryType":"COMPANY",
                //             "status":"SIGNING"
                // }
                // 所有接收方签署完成
                logger.info("所有接收方签署完成回调成功--{}", JSONObject.toJSONString(map, SerializerFeature.PrettyFormat, SerializerFeature.WriteMapNullValue));
                // 修改所有接收方签署状态

                break;
            case "CONTRACT_SIGN":
                // 文件签署-签署方签署

                break;
            case "INTERNAL_FLOW":
                // 每个节点签署完成

                break;
            case "CONTRACT_REJECTED":
                // 文件退回-签署方拒绝

                break;
            case "CONTRACT_TERMINATE_REJECTED":
                // 文件作废拒绝

                break;
            default:
                break;
        }

        return CommonResponse.success("接收方状态，回调成功！");
    }


    /**
     * 获取私有云合同签署短链接
     *
     * @param privateSignUrlVO
     * @return privateSignUrl
     */
    @PostMapping(value = "/getPrivateSignUrl")
    public CommonResponse<String> privateSignUrl(@RequestBody PrivateSignUrlVO privateSignUrlVO) {
        String privateSignUrl = signMgrService.createPrivateSignUrl(privateSignUrlVO.getContractId(), privateSignUrlVO.getTenantType(), privateSignUrlVO.getTenantName(), privateSignUrlVO.getContact());


        return CommonResponse.success("获取私有云合同签署短链接，成功！", privateSignUrl);
    }


    @GetMapping(value = "/downloadHasSignedFile")
    public void downloadHasSignedFile(@RequestParam(value = "billId") Long billId, HttpServletResponse response) {

        if (billId == null) {
            throw new BusinessException("请检查业务id是否为空！");
        }

        QueryWrapper<SignMgrEntity> queryWrapper = new QueryWrapper<>();
        queryWrapper.eq("bill_id", billId);
        SignMgrEntity signMgrEntity = signMgrService.getOne(queryWrapper);

        if (signMgrEntity == null) {
            throw new BusinessException("查询不到对应数据，请检查重试！");
        }

        Long contractId = signMgrEntity.getSourceBillId();

        OutputStream out = null;
        String fileName = String.format("%s_%d", signMgrEntity.getBillDocumentName(), billId);
        File file = null;
        try {
            file = File.createTempFile(fileName, ".pdf");
            out = new FileOutputStream(file);
        } catch (Exception e) {
            logger.error("下载签章文件，失败！原因是：", e);
            throw new BusinessException("下载签章文件，失败！");
        }

        if (!file.isFile() || !file.exists()) {
            throw new BusinessException("初始化临时文件不存在！");
        }

        signMgrService.downloadHasSignedFile(contractId, out);
        logger.info("下载合同完成,业务ID为：{},契约锁合同ID为：{}", billId, contractId);

        InputStream inputStream = null;
        try {
            inputStream = new FileInputStream(file);
            try (BufferedInputStream bufferedInputStream = new BufferedInputStream(inputStream)) {
                BufferedOutputStream bufferedOutputStream = new BufferedOutputStream(response.getOutputStream());
                int length = 0;
                byte[] temp = new byte[1024 * 10];
                while ((length = bufferedInputStream.read(temp)) != -1) {
                    bufferedOutputStream.write(temp, 0, length);
                }
                bufferedOutputStream.flush();
                bufferedOutputStream.close();
                logger.info("下载签章文件成功，文件名称：{}", file.getName());
            }
            inputStream.close();
        } catch (IOException e) {
            logger.error("下载签章文件，失败！原因是：", e);
            throw new BusinessException("下载签章文件，失败！");
        } finally {
            // 成功后才删除签章文件
            file.delete();
            logger.info("删除签章文件：{},成功！", file.getName());
        }
    }

    @GetMapping(value = "/previewContract")
    public CommonResponse<List<SignMgrPreviewVO>> previewContract(@RequestParam(value = "billId") Long billId, @RequestParam(value = "status") List<String> status) {
        if (billId == null || status.isEmpty()) {
            throw new BusinessException("请检查参数重试！");
        }
        logger.info("合同ID为--{},状态--{}", billId, JSONObject.toJSONString(status));

        List<String> sourceTypes = new ArrayList<>();
        for (String s : status) {
            sourceTypes.add("signature_" + s);
        }
        List<SignMgrPreviewEntity> signMgrPreviewEntityList = signMgrService.previewContract(billId, sourceTypes);
        if (signMgrPreviewEntityList == null) {
            throw new BusinessException("查询不到相关记录，请检查参数后重试！");
        }
        List<SignMgrPreviewVO> signMgrPreviewVOS = BeanMapper.mapList(signMgrPreviewEntityList, SignMgrPreviewVO.class);
        return CommonResponse.success("请求成功", signMgrPreviewVOS);
    }


    /**
     * 签章详情
     *
     * @param billId
     * @return
     */
    @GetMapping(value = "/signDetail")
    public CommonResponse<SignDetailVO> signDetail(@RequestParam(value = "billId") Long billId) {
        SignDetailVO signDetailVO = new SignDetailVO();

        QueryWrapper<SignMgrEntity> queryWrapper = new QueryWrapper<>();
        queryWrapper.eq("bill_id", billId);
        SignMgrEntity signMgrEntity = signMgrService.getOne(queryWrapper);

        if (signMgrEntity != null) {
            QueryWrapper<SignMgrSignatoryEntity> queryWrapper1 = new QueryWrapper<>();
            queryWrapper1.eq("pid", signMgrEntity.getId());
            List<SignMgrSignatoryEntity> list = signMgrSignatoryService.list(queryWrapper1);
            if (list.isEmpty()) {
                throw new BusinessException("查询签署详情失败，请稍后尝试！");
            }

            signDetailVO.setTenantName(signMgrEntity.getTenantName());
            signDetailVO.setCreatorName(signMgrEntity.getCreatorName());
            signDetailVO.setCreateTime(signMgrEntity.getCreateTime());

            List<SignatoryDetailVO> sds = new ArrayList<>();
            list.sort(Comparator.comparing(SignMgrSignatoryEntity::getSignOrder));
            Map<Integer, List<SignMgrSignatoryEntity>> listMap = list.stream().collect(Collectors.groupingBy(SignMgrSignatoryEntity::getSignOrder));

            int number = 1;
            for (Integer key : listMap.keySet()) {
                List<SignMgrSignatoryEntity> values = listMap.get(key);

                SignatoryDetailVO signatoryDetailVO = new SignatoryDetailVO();
                List<ActionDetailVO> actionDetails = new ArrayList<>();

                for (SignMgrSignatoryEntity value : values) {
                    ActionDetailVO actionDetailVO = new ActionDetailVO();
                    actionDetailVO.setActionName(value.getName());
                    actionDetailVO.setSignatoryName(value.getSignatureName());
                    actionDetailVO.setActualSignatoryName(value.getActualSignatoryName());
                    actionDetailVO.setSigningDate(value.getEndTime());
                    actionDetailVO.setSignatureStatus(value.getJobStatus());
                    actionDetails.add(actionDetailVO);
                }

                Set<Integer> s = actionDetails.stream().map(ActionDetailVO::getSignatureStatus).collect(Collectors.toSet());
                if (!s.isEmpty()) {
                    if (s.contains(1)) {
                        signatoryDetailVO.setStatus(1);
                    } else if (s.size() == 1 && s.iterator().next().equals(2)) {
                        signatoryDetailVO.setStatus(2);
                    } else {
                        signatoryDetailVO.setStatus(0);
                    }
                }
                if (signMgrEntity.getTenantName().equals(values.get(0).getTenantName())) {
                    signatoryDetailVO.setSignatoryName("发起主体");
                } else {
                    signatoryDetailVO.setSignatoryName("签约方" + number);
                    number++;
                }
                signatoryDetailVO.setTenantName(values.get(0).getTenantName());
                // 契约锁展示逻辑：甲方接收人和手机号都传，以手机号为准，看存不存在契约锁公司人员里，存在则签署详情展示传的手机号对应真实的人，不存在则创建新的接收人和手机号/只传接收人，不传手机号，展示接受人名和创建人手机号/只传手机号，不传接收人，以手机号为准，如果手机号已在契约锁平台公司人员下，展示真实的接收人名，不在的话
                if (!Objects.equals(values.get(0).getSignatureType(), SignMgrSignatoryEnum.INTERNAL_UNIT.getValue())) {
                    signatoryDetailVO.setOperatorName(values.get(0).getReceiverName());
                    signatoryDetailVO.setContact(values.get(0).getContact());
                } else {
                    if (!StringUtils.isEmpty(values.get(0).getContact())) {
                        if (StringUtils.isEmpty(values.get(0).getReceiverName())) {
                            signatoryDetailVO.setOperatorName(null);
                        } else {
                            // 取真实的
                            UserDetail userDetail = signMgrService.userDetail(values.get(0).getContact());
                            logger.info("根据手机号获取的用户信息和公司信息----{}", JSONObject.toJSONString(userDetail));
                            if ("AUTH_FAILURE".equals(userDetail.getStatus())) {
                                signatoryDetailVO.setOperatorName(values.get(0).getReceiverName());
                            } else {
                                signatoryDetailVO.setOperatorName(userDetail.getName());
                            }
                        }
                        signatoryDetailVO.setContact(values.get(0).getContact());
                    } else {
                        if (StringUtils.isEmpty(values.get(0).getReceiverName())) {
                            signatoryDetailVO.setOperatorName(signMgrEntity.getCreatorName());
                            signatoryDetailVO.setContact(signMgrEntity.getCreatorContact());
                        } else {
                            signatoryDetailVO.setOperatorName(values.get(0).getReceiverName());
                            signatoryDetailVO.setContact(signMgrEntity.getCreatorContact());
                        }
                    }
                }
                signatoryDetailVO.setAcceptTime(values.get(0).getAcceptTime());
                signatoryDetailVO.setFinishTime(values.get(values.size() - 1).getEndTime());

                signatoryDetailVO.setActionDetails(actionDetails);
                sds.add(signatoryDetailVO);
            }
            signDetailVO.setSignatorys(sds);
        } else {
            throw new BusinessException("未查询到签章记录，请联系管理员处理！");
        }
        return CommonResponse.success("获取签署详情成功！", signDetailVO);
    }

}
