package com.ejianc.business.signaturemanage.controller;

import com.alibaba.fastjson.JSONObject;
import com.baomidou.mybatisplus.core.metadata.IPage;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.ejianc.business.signaturemanage.bean.ManagementEntity;
import com.ejianc.business.signaturemanage.bean.ManagementUserEntity;
import com.ejianc.business.signaturemanage.service.IManagementService;
import com.ejianc.business.signaturemanage.service.IManagementUserService;
import com.ejianc.business.signaturemanage.service.ISealManageService;
import com.ejianc.business.signaturemanage.vo.ManagementVO;
import com.ejianc.foundation.orgcenter.api.IOrgApi;
import com.ejianc.foundation.orgcenter.vo.OrgVO;
import com.ejianc.foundation.support.api.IBillCodeApi;
import com.ejianc.foundation.tenant.api.ITenantApi;
import com.ejianc.framework.auth.session.UserContext;
import com.ejianc.framework.core.context.InvocationInfoProxy;
import com.ejianc.framework.core.exception.BusinessException;
import com.ejianc.framework.core.kit.mapper.BeanMapper;
import com.ejianc.framework.core.response.CommonResponse;
import com.ejianc.framework.core.response.ComplexParam;
import com.ejianc.framework.core.response.Parameter;
import com.ejianc.framework.core.response.QueryParam;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Controller;
import org.springframework.web.bind.annotation.*;

import javax.servlet.http.HttpServletRequest;
import java.io.Serializable;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;

/**
 * @author:刘家乐
 * @version: 1
 * @document:
 */
@Controller
@RequestMapping("management")
public class ManagementController implements Serializable {
    private static final long serialVersionUID = 1L;
    /**
     * 此处需要根据实际修改
     */
    private static final String BILL_CODE = "LJL_MANAGEMENT";
    private Logger logger = LoggerFactory.getLogger(this.getClass());
    @Autowired
    private IBillCodeApi billCodeApi;
    @Autowired
    private IOrgApi orgApi;

    @Autowired
    private ITenantApi tenantApi;
    @Autowired
    private IManagementService service;

    @Autowired
    private ISealManageService sealService; // 同步签章
    @Autowired
    private IManagementUserService userService;

    /**
     * 列表查询
     *
     * @param queryParam 查询参数
     * @return 返回列表
     */
    @ResponseBody
    @RequestMapping(value = "/queryList", method = RequestMethod.POST)
    public CommonResponse<IPage<ManagementVO>> queryList(@RequestBody QueryParam queryParam) {
        //封装参数
        String checkSearch = "orgName";
        List<Long> orgIdList = new ArrayList<>();

        if (null != queryParam.getParams().get(checkSearch)) {
            //如果前端传入 orgName 就 只查询当前组织的印章
            String requestOrgId = String.valueOf(queryParam.getParams().get(checkSearch).getValue());
            queryParam.getParams().remove(checkSearch);
            queryParam.getParams().put("orgId", new Parameter(QueryParam.EQ, requestOrgId));
        } else {
            //如果前端未传入 orgName 查询本下和未匹配的印章
            CommonResponse<List<OrgVO>> orgResp = orgApi.findChildrenByParentId(InvocationInfoProxy.getOrgId());
            if (!orgResp.isSuccess()) {
                //返回错误信息
                IPage<ManagementVO> pageData = new Page<>(queryParam.getPageIndex(), queryParam.getPageSize(), 0);
                return CommonResponse.success("查询数表参照失败，项目组织信息查询失败！", pageData);
            }
            List<OrgVO> orgList = orgResp.getData();
            ComplexParam c1 = new ComplexParam();
            c1.setLogic(ComplexParam.AND);
            ComplexParam c2 = new ComplexParam();
            c2.setLogic(ComplexParam.OR);
            if (CollectionUtils.isNotEmpty(orgList)) {
                orgList.forEach(orgVO -> orgIdList.add(orgVO.getId()));
                if (CollectionUtils.isNotEmpty(orgIdList)) {
                    c2.getParams().put("orgId", new Parameter(QueryParam.IN, orgIdList));
                }
            }
            ComplexParam c3 = new ComplexParam();
            c3.setLogic(ComplexParam.OR);
            c3.getParams().put("isMatch", new Parameter(QueryParam.EQ, "0"));
            c1.getComplexParams().add(c2);
            c1.getComplexParams().add(c3);
            queryParam.getComplexParams().add(c1);
        }
        queryParam.getOrderMap().put("id", QueryParam.ASC);
        //模糊查询字段
        List<String> fuzzyFields = queryParam.getFuzzyFields();
        fuzzyFields.add("sourceSealName");
        fuzzyFields.add("orgName");
        fuzzyFields.add("sealSubType");
        fuzzyFields.add("sealType");
        fuzzyFields.add("sealUseName");
        //印章 同步印章会将其他印章也同步过来 我们这边只需要展示  公章,合同专用章,法定代表人章,个人
        // queryParam.getParams().put("sealSubTypeName", new Parameter(QueryParam.IN, "公章,合同专用章,法定代表人章,个人"));
        IPage<ManagementEntity> page = service.queryPage(queryParam, false);
        IPage<ManagementVO> pageData = new Page<>(page.getCurrent(), page.getSize(), page.getTotal());
        pageData.setRecords(BeanMapper.mapList(page.getRecords(), ManagementVO.class));
        return CommonResponse.success("查询列表数据成功！", pageData);
    }


    /**
     * 查询详情
     *
     * @param id 签章id
     * @return 返回签章设置 与 默认用户
     */
    @ResponseBody
    @RequestMapping(value = "/queryDetail", method = RequestMethod.GET)
    public CommonResponse<ManagementVO> queryDetail(@RequestParam String id) {
        ManagementEntity managementEntity = service.selectById(id);
        QueryParam queryParam = new QueryParam();
        queryParam.getParams().put("pid", new Parameter(QueryParam.EQ, managementEntity.getId()));
        queryParam.getParams().put("user_type", new Parameter(QueryParam.EQ, "user"));
        queryParam.getParams().put("tenant_id", new Parameter(QueryParam.EQ, InvocationInfoProxy.getTenantid()));
        List<ManagementUserEntity> managementUserEntities = userService.queryList(queryParam);
        managementEntity.setUserEntityList(managementUserEntities);
        ManagementVO map = BeanMapper.map(managementEntity, ManagementVO.class);
        return CommonResponse.success(map);
    }

    /**
     * 根据当前组织id 查询当前组织下签章
     * @param orgId 组织id
     * @return 返回当前组织下的印章
     */
    @ResponseBody
    @RequestMapping(value = "/querySealByOrgId", method = RequestMethod.GET)
    public CommonResponse<List<ManagementVO>> querySealByOrgId(@RequestParam(value = "orgId") Long orgId) {
        if (null == orgId) {
            throw new BusinessException("当前接口单位类型为空");
        } else {
            CommonResponse<List<OrgVO>> parentsByOrgId = orgApi.findParentsByOrgId(orgId);
            List<Long> OrgIdList = new ArrayList();
            if (parentsByOrgId.isSuccess()){
                OrgIdList= parentsByOrgId.getData().stream().map(OrgVO::getId).collect(Collectors.toList());
            }else {
                throw new BusinessException("网络错误 查询组织错误");
            }
            QueryParam queryParam = new QueryParam();
            queryParam.getParams().put("org_id", new Parameter(QueryParam.IN, OrgIdList));
            queryParam.getParams().put("sealSubTypeName", new Parameter(QueryParam.IN, "公章,合同专用章,法定代表人章,个人"));
            queryParam.getParams().put("sourceStatus", new Parameter(QueryParam.EQ, "NORMAL"));
            List<ManagementEntity> managementEntities = service.queryList(queryParam, false);
            List<ManagementVO> mapList = BeanMapper.mapList(managementEntities, ManagementVO.class);
            if (mapList.isEmpty()) {
                return CommonResponse.success("当前单位下没有印章");
            } else {
                return CommonResponse.success("查询印章成功", mapList);
            }

        }
    }


    /**
     * 查询当前 组织所有匹配的印章
     */
    @ResponseBody
    @RequestMapping(value = "/querySeals", method = RequestMethod.GET)
    public CommonResponse<IPage<ManagementVO>> querySeals(
            @RequestParam(defaultValue = "1") Integer pageNumber,
            @RequestParam(defaultValue = "10") Integer pageSize,
            @RequestParam(value = "condition", required = false) String condition
    ) {
        //封装参数 org Id
        QueryParam queryParam = new QueryParam();
        queryParam.setPageIndex(pageNumber);
        queryParam.setPageSize(pageSize);
        queryParam.getParams().put("sealSubTypeName", new Parameter(QueryParam.IN, "公章,合同专用章,法定代表人章,个人"));
        queryParam.getParams().put("sourceStatus", new Parameter(QueryParam.EQ, "NORMAL"));
        //判断是否传入condition 传入 查询 id未当前分类的和全分类可用的
        if (StringUtils.isNotBlank(condition)) {
            Map<String, Object> conditionMap = JSONObject.parseObject(condition, Map.class);
            if (null != conditionMap.get("orgId")) {
                //类型转换 拿到 orgId
                Long id = Long.parseLong(conditionMap.get("orgId").toString());
                CommonResponse<List<OrgVO>> parentsByOrgId = orgApi.findParentsByOrgId(id);
                List<Long> OrgIdList = new ArrayList();
                if (parentsByOrgId.isSuccess()){
                    OrgIdList= parentsByOrgId.getData().stream().map(OrgVO::getId).collect(Collectors.toList());
                }else {
                    throw new BusinessException("网络错误 查询组织错误");
                }
                queryParam.getParams().put("orgId", new Parameter(QueryParam.IN, OrgIdList));
            }
        } else {
            return CommonResponse.success("数据为空", new Page<>(0, 1, 0));
        }
        queryParam.getOrderMap().put("create_time", QueryParam.DESC);
        IPage<ManagementEntity> pageData = service.queryPage(queryParam, false);
        IPage<ManagementVO> result = new Page<>(pageData.getCurrent(), pageData.getSize(), pageData.getTotal());
        result.setRecords(BeanMapper.mapList(pageData.getRecords(), ManagementVO.class));
        if (!result.getRecords().isEmpty()) {
            for (ManagementVO managementVO : result.getRecords()) {
                String sealType = managementVO.getSealType();
                switch (sealType) {
                    case "COMPANY":
                        managementVO.setSealTypeStr("企业公章");
                        break;
                    case "LP":
                        managementVO.setSealTypeStr("法定代表人章");
                        break;
                    case "PERSONAL":
                        managementVO.setSealTypeStr("个人签章");
                        break;
                    default:
                        managementVO.setSealTypeStr("未定义");
                }
            }
        }
        return CommonResponse.success(result);
    }


    /**
     * 根据当前组织id 查询当前组织下签章
     *
     * @param queryParam 组织id 分页信息
     * @return 返回当前组织下的印章
     */
    @ResponseBody
    @RequestMapping(value = "/querySeal", method = RequestMethod.POST)
    public CommonResponse<IPage<ManagementVO>> querySeal(@RequestBody QueryParam queryParam) {
        queryParam.getParams().put("sealSubTypeName", new Parameter(QueryParam.IN, "公章,合同专用章,法定代表人章,个人"));
        queryParam.getParams().put("sourceStatus", new Parameter(QueryParam.EQ, "NORMAL"));
        IPage<ManagementEntity> page = service.queryPage(queryParam);
        List<ManagementVO> categoryVoList =
                BeanMapper.mapList(page.getRecords(), ManagementVO.class);
        IPage<ManagementVO> management = new Page<>();
        management.setCurrent(page.getCurrent());
        management.setRecords(categoryVoList);
        management.setSize(page.getSize());
        management.setTotal(page.getTotal());
        management.setPages(page.getPages());
        return CommonResponse.success("查询成功！", management);
    }


    /**
     * 定时任务同步
     */
    @ResponseBody
    @RequestMapping(value = "/synchronizationAuto", method = RequestMethod.POST)
    public CommonResponse<String> synchronizationAuto(HttpServletRequest request) {

        logger.info("定时任务开始执行");
        List<ManagementVO> data = sealService.queryAllSeal();
        if (null == data) {
            logger.info("定时任务 无对应印章");
            return CommonResponse.success("定时任务 查询印章失败定时任务 无对应印章");
        }
        List<ManagementVO> managementVOList = BeanMapper.mapList(data, ManagementVO.class);
        List<ManagementEntity> managementEntityList = BeanMapper.mapList(managementVOList, ManagementEntity.class);
        String authority = request.getHeader("authority");
        // for (ManagementEntity managementEntity : managementEntityList) {
        //     service.syncUpdate(managementEntity, authority);
        // }
        // 批量保存或修改
        service.batchSaveOrUpdate(managementEntityList);
        logger.info(String.format("同步成功 已同步%d个", managementEntityList.size()));
        // 异步更新图片地址
        service.syncUpdateImgUrl(managementEntityList.parallelStream().map(ManagementEntity::getSourceSealId).collect(Collectors.toList()), authority);
        return CommonResponse.success("同步成功");
    }

    /**
     * 同步用户当前部门
     */
    @ResponseBody
    @RequestMapping(value = "/synchronizationMan", method = RequestMethod.POST)
    public CommonResponse<String> synchronizationMan(HttpServletRequest request) {
        String userContext = UserContext.getUserContext().getUserName();
        Long orgId = InvocationInfoProxy.getOrgId();
        OrgVO data = orgApi.getOneById(orgId).getData();
        String companyName = data.getName();
        logger.info(companyName + ":主动请求同步开始执行");
        List<ManagementVO> response = sealService.querySealsByCompanyName(companyName);
        if (response.isEmpty()) {
            return CommonResponse.success(UserContext.getUserContext().getOrgName() + "无对应签章，请提醒管理员");
        }
        logger.info("判断返回信息:" + JSONObject.toJSONString(response));
        List<ManagementVO> managementVOList = BeanMapper.mapList(response, ManagementVO.class);
        List<ManagementEntity> managementEntityList = BeanMapper.mapList(managementVOList, ManagementEntity.class);
        String authority = request.getHeader("authority");
        // //循环保存
        // for (ManagementEntity managementEntity : managementEntityList) {
        //     service.syncUpdate(managementEntity, authority);
        // }
        // 批量保存或修改
        service.batchSaveOrUpdate(managementEntityList);
        logger.info(String.format("同步成功 已同步%d个", managementEntityList.size()));
        // 异步更新图片地址
        service.syncUpdateImgUrl(managementEntityList.parallelStream().map(ManagementEntity::getSourceSealId).collect(Collectors.toList()), authority);
        return CommonResponse.success("同步成功");
    }

    /**
     * 修改接口
     */
    @ResponseBody
    @RequestMapping(value = "/updateManagement", method = RequestMethod.POST)
    public CommonResponse<ManagementVO> updateManagement(@RequestBody ManagementVO vo) {
        return service.updateManagement(vo);
    }

    /**
     * @param orgId 组织id
     * @return 当前组织下所有启用的印章
     */
    @ResponseBody
    @RequestMapping(value = "/getManagementByOrgId", method = RequestMethod.GET)
    public CommonResponse<List<ManagementVO>> getManagementByOrgId(@RequestParam(value = "orgId") Long orgId) {
        QueryParam queryParam = new QueryParam();
        CommonResponse<List<OrgVO>> parentsByOrgId = orgApi.findParentsByOrgId(orgId);
        List<Long> OrgIdList = new ArrayList();
        if (parentsByOrgId.isSuccess()){
            OrgIdList= parentsByOrgId.getData().stream().map(OrgVO::getId).collect(Collectors.toList());
        }else {
            throw new BusinessException("网络错误 查询组织错误");
        }
        queryParam.getParams().put("orgId", new Parameter(QueryParam.IN, OrgIdList));
        queryParam.getParams().put("sealSubTypeName", new Parameter(QueryParam.IN, "公章,合同专用章,法定代表人章,个人"));
        List<ManagementEntity> managementEntities = service.queryList(queryParam);
        if (managementEntities.isEmpty()) {
            return CommonResponse.success("当前组织无印章");
        } else {
            for (ManagementEntity entity : managementEntities) {
                QueryParam query = new QueryParam();
                query.getParams().put("pid", new Parameter(QueryParam.EQ, entity.getId()));
                List<ManagementUserEntity> managementUserEntities = userService.queryList(query);
                entity.setUserEntityList(managementUserEntities);
            }
            List<ManagementVO> management = BeanMapper.mapList(managementEntities, ManagementVO.class);
            return CommonResponse.success(management);
        }
    }

}
