package com.ejianc.business.signaturemanage.service.impl;

import com.alibaba.fastjson.JSONObject;
import com.alibaba.fastjson.serializer.SerializerFeature;
import com.ejianc.business.signaturemanage.bean.ManagementEntity;
import com.ejianc.business.signaturemanage.enums.UserTypeEnum;
import com.ejianc.business.signaturemanage.service.IManagementService;
import com.ejianc.business.signaturemanage.service.ISealManageService;
import com.ejianc.business.signaturemanage.utils.uploadFileCenter;
import com.ejianc.business.signaturemanage.vo.ManagementUserVO;
import com.ejianc.business.signaturemanage.vo.ManagementVO;
import com.ejianc.foundation.file.vo.AttachmentVO;
import com.ejianc.framework.core.exception.BusinessException;
import com.ejianc.framework.core.response.CommonResponse;
import net.qiyuesuo.sdk.SDKClient;
import net.qiyuesuo.sdk.api.SealService;
import net.qiyuesuo.sdk.bean.company.Company;
import net.qiyuesuo.sdk.bean.seal.Seal;
import net.qiyuesuo.sdk.bean.seal.SealCategory;
import net.qiyuesuo.sdk.bean.seal.SealCondition;
import net.qiyuesuo.sdk.bean.seal.SealPermissionGroupBean;
import net.qiyuesuo.sdk.bean.user.User;
import net.qiyuesuo.sdk.common.exception.PrivateAppException;
import net.qiyuesuo.sdk.impl.SealServiceImpl;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.scheduling.annotation.Async;
import org.springframework.stereotype.Service;

import java.io.File;
import java.io.FileOutputStream;
import java.io.OutputStream;
import java.util.ArrayList;
import java.util.List;

@Service("sealManageService")
public class SealManageServiceImpl implements ISealManageService {
    private final Logger logger = LoggerFactory.getLogger(this.getClass());

    @Value("${qiyuesuo.client.url}")
    private String url;// 私有化开放平台请求地址
    @Value("${qiyuesuo.client.accessKey}")
    private String accessKey;// 私有化开放平台申请的token
    @Value("${qiyuesuo.client.accessSecret}")
    private String accessSecret;// 私有化开放平台申请的secret
    @Value("${qiyuesuo.contract.categoryId}")
    private Long categoryId;// 用印流程ID

    @Value("${common.env.base-host}")
    private String BASE_HOST;

    @Autowired
    private uploadFileCenter uploadFileCenter;

    @Autowired
    private IManagementService service;


    /**
     * 初始化sdk的client
     *
     * @return client
     */
    private SDKClient getSdkClient() {
        // 初始化client
        SDKClient client = new SDKClient(url, accessKey, accessSecret);
        // 开启防止重放攻击
        client.enableNonce();
        return client;
    }


    /**
     * 公司印章列表
     *
     * @param companyId
     * @param companyName
     * @param category
     * @return
     */
    public List<Seal> sealList(Long companyId, String companyName, String category) {
        if (StringUtils.isBlank(companyName)) {
            throw new BusinessException("公司名称不能为空！");
        }

        SDKClient client = getSdkClient();
        SealService sealService = new SealServiceImpl(client);

        List<Seal> list = null;
        logger.info("根据公司获取印章列表入参：{}", companyName);
        try {
            list = sealService.sealList(companyId, companyName, category);
        } catch (PrivateAppException e) {
            logger.error("请求契约锁异常，请检查请求参数！", e);
            throw new BusinessException("请求契约锁异常，请检查请求参数！");
        }

        return list;
    }

    public Seal detail(Long sealId) {
        if (sealId == null) {
            throw new BusinessException("印章ID不能为空！");
        }

        SDKClient client = getSdkClient();
        SealService sealService = new SealServiceImpl(client);

        Seal seal = null;
        try {
            seal = sealService.detail(sealId);
        } catch (PrivateAppException e) {
            e.printStackTrace();
            throw new BusinessException("请求契约锁异常，请检查请求参数！");
        }

        return seal;
    }

    @Async(value = "downloadFileTask")
    public void replaceImgUrl(Long sealId, String authority, ManagementEntity entity) {
        if (sealId == null) {
            throw new BusinessException("获取印章图片，请求的印章id不能为空！");
        }
        logger.info("获取印章图片，请求的印章ID-{}", sealId);
        SDKClient client = getSdkClient();
        SealService sealService = new SealServiceImpl(client);

        OutputStream outputStream;
        File file;
        try {

            file = File.createTempFile(String.valueOf(sealId), ".png");
            outputStream = new FileOutputStream(file);
            Seal request = new Seal();
            request.setId(sealId);
            try {
                sealService.image(request, outputStream);
            } catch (PrivateAppException e) {
                logger.error("获取印章图片失败！", e);
                throw new BusinessException("请求契约锁异常，请检查请求参数！");
            }
        } catch (Exception e) {
            logger.error("获取印章图片失败！", e);
            throw new BusinessException("获取印章图片失败！");
        }

        // 上传文件中心，获取在线url地址
        CommonResponse<List<AttachmentVO>> uploadResp = uploadFileCenter.uploadFile(file, authority, "seal", sealId, "BT211123000000005", "true", BASE_HOST);
        if (!uploadResp.isSuccess()) {
            throw new BusinessException("上传文件中心失败！");
        }
        logger.info("文件上传结果：{}", JSONObject.toJSONString(uploadResp.getData()));
        AttachmentVO attachmentVO = JSONObject.parseObject(JSONObject.toJSONString(uploadResp.getData().get(0)), AttachmentVO.class);

        String imgUrl = attachmentVO.getOnlinePath();
        entity.setImgUrl(imgUrl);
        service.saveOrUpdate(entity, false);
        file.delete();
    }


    /**
     * 解析印章管理员和使用者信息
     *
     * @param sealGroup
     * @return
     */
    public List<ManagementUserVO> getUsers(List<SealPermissionGroupBean> sealGroup, Boolean isAdmin) {
        List<ManagementUserVO> userList = new ArrayList<>();
        if (!sealGroup.isEmpty()) {
            for (SealPermissionGroupBean s : sealGroup) {
                ManagementUserVO m = new ManagementUserVO();
                m.setSourceUserId(s.getId());
                m.setSourceUserName(s.getName());
                List<User> users = s.getUsers();
                // 这个users判null，不是判空
                // 未实名认证用户user信息为空,不处理
                if (users != null) {
                    for (User user : users) {
                        m.setSourcePhone(user.getMobile());
                        m.setSourceEmail(user.getEmail());
                    }
                }
                if (isAdmin) {
                    m.setUserType(UserTypeEnum.MANGER.getTypeName());
                } else {
                    m.setUserType(UserTypeEnum.USER.getTypeName());
                }
                userList.add(m);
            }
        }
        return userList;
    }

    /**
     * 查询所有内部企业的印章列表
     *
     * @return
     */
    @Override
    public List<Company> companySealList() {
        SDKClient client = getSdkClient();
        SealService sealService = new SealServiceImpl(client);

        List<Company> companyList = null;
        SealCondition sealCondition = new SealCondition();
        // 印章类型：PHYSICS("物理签章"),ELECTRONIC("电子签章"),不传默认查询电子章
        sealCondition.setCategory(SealCategory.ELECTRONIC);
        // 需要查询的印章状态：ALL("查询全部")，NORMAL("正常")，FREEZE("冻结")，INVALID("失效")，默认查询NORMAL状态
        sealCondition.setSealQueryStatus("NORMAL");

        try {
            companyList = sealService.innercompanySealList(sealCondition);
        } catch (PrivateAppException e) {
            e.printStackTrace();
            throw new BusinessException("请求契约锁异常，请检查请求参数！");
        }
        return companyList;
    }


    /**
     * 查询公司下的印章列表
     *
     * @param
     * @return
     */
    public List<ManagementVO> querySealsByCompanyName(String companyName) {
        List<ManagementVO> list = new ArrayList<>();
        List<Seal> sealList = this.sealList(null, companyName, null);

        logger.info("请求的印章列表:{}", JSONObject.toJSONString(sealList, SerializerFeature.PrettyFormat, SerializerFeature.WriteMapNullValue));

        /* List<CompletableFuture<String>> futures = new ArrayList<>(); */

        if (!sealList.isEmpty()) {
            for (Seal seal : sealList) {
                Seal s = this.detail(seal.getId());
                logger.info("当前印章id-{},印章详情:{}", seal.getId(), JSONObject.toJSONString(s, SerializerFeature.PrettyFormat, SerializerFeature.WriteMapNullValue));

                ManagementVO managementVO = new ManagementVO();
                // 印章ID（第三方返回）@
                managementVO.setSourceSealId(s.getId());
                // 印章名称（第三方返回）@
                managementVO.setSourceSealName(s.getName());
                // 印章分类，PHYSICS(物理章),ELECTRONIC(电子章)@
                managementVO.setSealCategory(String.valueOf(seal.getCategory()));
                // 印章图片KEY@
                managementVO.setSourceKey(s.getSealKey());
                // 图片地址@
                /* CompletableFuture<String> future = CompletableFuture.supplyAsync(() -> sealManageService.getImgUrl(s.getId(), authority)).thenApplyAsync(img -> {
                    managementVO.setImgUrl(img);
                    return null;
                });
                futures.add(future); */

                // 第三方返回的印章状态名称@
                JSONObject jsonObject = JSONObject.parseObject(s.getStatus());

                switch (jsonObject.getString("key")) {
                    case "NORMAL":
                        managementVO.setSourceStatusDescription(jsonObject.getString("description"));
                    case "FREEZE":
                        managementVO.setSourceStatusDescription(jsonObject.getString("description"));
                    case "DELETE":
                        managementVO.setSourceStatusDescription(jsonObject.getString("description"));
                    case "INVALID":
                        managementVO.setSourceStatusDescription(jsonObject.getString("description"));
                        break;
                    default:
                        throw new IllegalStateException("Unexpected value: " + jsonObject.getString("key"));
                }
                // 第三方返回的印章状态，NORMAL("正 常"),FREEZE("冻结"),DELETE("删 除"),INVALID("失效")@
                managementVO.setSourceStatus(jsonObject.getString("key"));
                // 签章类型，COMPANY("企业公章"),LP("法定代表人章")@
                managementVO.setSealType(String.valueOf(s.getType()));
                // 公司ID,第三方@
                managementVO.setSourceCompanyId(seal.getOwner());
                // 所属公司名称 第三方@
                managementVO.setSourceCompanyName(seal.getOwnerName());
                // 印章类型（类别）子类别@
                // managementVO.setSealSubType("");
                // 印章类型名称子类别@
                managementVO.setSealSubTypeName(s.getSealCategoryName());


                // 用户id（第三方系统返回的）@
                // 用户名称 （第三方系统返回的）@
                // 手机号
                // 邮箱
                // 用户类型 user- 使用者 manger-管理员
                List<SealPermissionGroupBean> sealAdminGroup = s.getSealAdminGroup();
                List<SealPermissionGroupBean> sealUseGroup = s.getSealUseGroup();
                List<ManagementUserVO> managementUsers = new ArrayList<>();
                List<ManagementUserVO> sa = this.getUsers(sealAdminGroup, true);
                if (sa != null) {
                    managementUsers.addAll(sa);
                }
                List<ManagementUserVO> su = this.getUsers(sealUseGroup, false);
                if (su != null) {
                    managementUsers.addAll(su);
                }
                managementVO.setUserEntityList(managementUsers);
                list.add(managementVO);
            }


            /* CompletableFuture<Void> allOf = CompletableFuture.allOf(futures.toArray(new CompletableFuture[futures.size()]));

            try {
                allOf.get();
            } catch (InterruptedException e) {
                e.printStackTrace();
            } catch (ExecutionException e) {
                e.printStackTrace();
            } */
        }
        logger.info("返回的印章数据：{}", JSONObject.toJSONString(list));
        return list;
    }


    /**
     * 查询所有内部企业的印章列表
     *
     * @return
     */
    public List<ManagementVO> queryAllSeal() {
        List<ManagementVO> data = new ArrayList<>();
        List<Company> companyList = this.companySealList();

        logger.info("请求的公司列表:{}", JSONObject.toJSONString(companyList, SerializerFeature.PrettyFormat, SerializerFeature.WriteMapNullValue));

        /* List<Future<String>> futures = new ArrayList<>(); */

        if (!companyList.isEmpty()) {
            for (Company company : companyList) {
                for (Seal seal : company.getSeals()) {
                    Seal s = this.detail(seal.getId());
                    logger.info("当前公司-{},当前印章id-{},印章详情:{}", company.getName(), seal.getId(), JSONObject.toJSONString(s, SerializerFeature.PrettyFormat, SerializerFeature.WriteMapNullValue));

                    ManagementVO managementVO = new ManagementVO();
                    // 印章ID（第三方返回）@
                    managementVO.setSourceSealId(s.getId());
                    // 印章名称（第三方返回）@
                    managementVO.setSourceSealName(s.getName());
                    // 签章类型，COMPANY("企业公章"),LP("法定代表人章")@
                    managementVO.setSealType(String.valueOf(s.getType()));
                    // 印章图片KEY@
                    managementVO.setSourceKey(s.getSealKey());
                    // 图片地址@
                    /* CompletableFuture<String> future = CompletableFuture.supplyAsync(() -> sealManageService.getImgUrl(s.getId(), authority)).thenApplyAsync(img -> {
                        managementVO.setImgUrl(img);
                        return null;
                    });
                    futures.add(future); */

                    // 第三方返回的印章状态名称@
                    JSONObject jsonObject = JSONObject.parseObject(s.getStatus());

                    switch (jsonObject.getString("key")) {
                        case "NORMAL":
                            managementVO.setSourceStatusDescription(jsonObject.getString("description"));
                        case "FREEZE":
                            managementVO.setSourceStatusDescription(jsonObject.getString("description"));
                        case "DELETE":
                            managementVO.setSourceStatusDescription(jsonObject.getString("description"));
                        case "INVALID":
                            managementVO.setSourceStatusDescription(jsonObject.getString("description"));
                            break;
                        default:
                            throw new IllegalStateException("Unexpected value: " + jsonObject.getString("key"));
                    }
                    // 第三方返回的印章状态，NORMAL("正 常"),FREEZE("冻结"),DELETE("删 除"),INVALID("失效")@
                    managementVO.setSourceStatus(jsonObject.getString("key"));
                    // 印章分类，PHYSICS(物理章),ELECTRONIC(电子章)@
                    managementVO.setSealCategory(String.valueOf(seal.getCategory()));
                    // 公司ID,第三方@
                    managementVO.setSourceCompanyId(seal.getOwner());
                    // 所属公司名称 第三方@
                    managementVO.setSourceCompanyName(seal.getOwnerName());
                    // 印章类型（类别）子类别@
                    // managementVO.setSealSubType("");
                    // 印章类型名称子类别@
                    managementVO.setSealSubTypeName(s.getSealCategoryName());


                    // 用户id（第三方系统返回的）@
                    // 用户名称 （第三方系统返回的）@
                    // 手机号
                    // 邮箱
                    // 用户类型 user- 使用者 manger-管理员
                    List<SealPermissionGroupBean> sealAdminGroup = s.getSealAdminGroup();
                    List<SealPermissionGroupBean> sealUseGroup = s.getSealUseGroup();
                    List<ManagementUserVO> managementUsers = new ArrayList<>();
                    List<ManagementUserVO> sa = this.getUsers(sealAdminGroup, true);
                    if (sa != null) {
                        managementUsers.addAll(sa);
                    }
                    List<ManagementUserVO> su = this.getUsers(sealUseGroup, false);
                    if (su != null) {
                        managementUsers.addAll(su);
                    }
                    managementVO.setUserEntityList(managementUsers);
                    data.add(managementVO);
                }

            }

            /* CompletableFuture<Void> allOf = CompletableFuture.allOf(futures.toArray(new CompletableFuture[futures.size()]));

            try {
                allOf.get();
            } catch (InterruptedException e) {
                e.printStackTrace();
            } catch (ExecutionException e) {
                e.printStackTrace();
            } */
        }

        return data;
    }

}
