package com.ejianc.business.signaturemanage.service.impl;

import cn.hutool.core.io.FileUtil;
import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.serializer.SerializerFeature;
import com.ejianc.business.signaturemanage.enums.WatermarkSchemeEnum;
import com.ejianc.business.signaturemanage.service.ISignatureCommonService;
import com.ejianc.business.signaturemanage.vo.WatermarkConfigVO;
import com.ejianc.business.signaturemanage.vo.WatermarkVO;
import com.ejianc.foundation.file.api.IAttachmentApi;
import com.ejianc.foundation.file.vo.AttachmentVO;
import com.ejianc.framework.core.exception.BusinessException;
import com.ejianc.framework.core.response.CommonResponse;
import net.qiyuesuo.sdk.SDKClient;
import net.qiyuesuo.sdk.api.ContractService;
import net.qiyuesuo.sdk.bean.contract.WaterMarkContent;
import net.qiyuesuo.sdk.bean.contract.WaterMarkLocation;
import net.qiyuesuo.sdk.bean.contract.WaterMarkType;
import net.qiyuesuo.sdk.bean.document.DocumentCreateByUrl;
import net.qiyuesuo.sdk.bean.document.DownloadDocRequest;
import net.qiyuesuo.sdk.common.exception.PrivateAppException;
import net.qiyuesuo.sdk.impl.ContractServiceImpl;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.scheduling.annotation.Async;
import org.springframework.stereotype.Service;
import org.springframework.util.Assert;

import javax.annotation.PostConstruct;
import java.io.File;
import java.io.IOException;
import java.io.OutputStream;
import java.nio.file.Files;
import java.nio.file.Paths;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.function.Function;

/**
 * @author baipengyan
 * @date 2022/5/18
 * @description
 */
@Service("signatureCommonService")
public class SignatureCommonServiceImpl implements ISignatureCommonService {
    private final Logger logger = LoggerFactory.getLogger(this.getClass());
    private final Map<String, Function<WatermarkVO, File>> watermarkMap = new HashMap<>();
    @Value("${ejc.fileServerUrl}")
    private String FILE_HOST; // 文件中心
    @Autowired
    private IAttachmentApi attachmentApi; // 文件信息
    @Value("${qiyuesuo.client.url}")
    private String url;// 私有化开放平台请求地址
    @Value("${qiyuesuo.client.accessKey}")
    private String accessKey;// 私有化开放平台申请的token
    @Value("${qiyuesuo.client.accessSecret}")
    private String accessSecret;// 私有化开放平台申请的secret

    @PostConstruct
    public void watermarkDispatcher() {
        // 加水印方案：契约锁
        watermarkMap.put(WatermarkSchemeEnum.QI_YUE_SUO.getCode(), watermarkVO -> {
            try {
                return addWatermarkIfQiyuesuo(watermarkVO);
            } catch (IOException ignored) {
            }
            return null;
        });

        // 其他方案
    }

    /**
     * 合同添加水印
     *
     * @param watermarkVO 合同文件添加水印参数
     */
    @Override
    public File addWatermark(WatermarkVO watermarkVO) {
        Function<WatermarkVO, File> res = watermarkMap.get(watermarkVO.getScheme());
        Assert.notNull(res, "合同添加水印的方案不存在，请联系管理员!");
        return res.apply(watermarkVO);
    }

    /**
     * 删除临时文件
     *
     * @param file 要删除的文件
     */
    @Override
    @Async(value = "commonTask")
    public void cleanFile(File file) throws IOException {
        Files.delete(Paths.get(file.getPath()));
    }

    private SDKClient getSdkClient() {
        // 初始化client
        SDKClient client = new SDKClient(url, accessKey, accessSecret);
        // 开启防止重放攻击
        client.enableNonce();
        return client;
    }

    private File addWatermarkIfQiyuesuo(WatermarkVO watermarkVO) throws IOException {
        logger.info("合同文件添加水印，入参：{}", JSON.toJSONString(watermarkVO, SerializerFeature.WriteMapNullValue, SerializerFeature.PrettyFormat));

        CommonResponse<AttachmentVO> res = attachmentApi.queryDetail(String.valueOf(watermarkVO.getBillId()));
        if (!res.isSuccess()) {
            logger.error(res.getMsg());
            throw new BusinessException("获取合同文件失败！");
        }

        AttachmentVO attachmentVO = res.getData();
        logger.info("合同文件详情：{}", JSON.toJSONString(attachmentVO, SerializerFeature.WriteMapNullValue, SerializerFeature.PrettyFormat));
        String contractUrl = FILE_HOST + attachmentVO.getFilePath();
        Assert.hasText(contractUrl, "合同文件网络路径不能为空！");
        String fullName = attachmentVO.getFileName();
        String contractTitle = fullName.substring(0, fullName.lastIndexOf("."));
        Assert.hasText(contractTitle, "合同文件名称不能为空！");
        String contractFileType = fullName.substring(fullName.lastIndexOf(".") + 1);
        Assert.hasText(contractFileType, "合同文件类型不能为空！");

        SDKClient sdkClient = getSdkClient();
        ContractService contractService = new ContractServiceImpl(sdkClient);

        // 水印参数组装
        List<WaterMarkContent> waterMarks = new ArrayList<>();
        for (WatermarkConfigVO vo : watermarkVO.getWatermarkConfigs()) {
            WaterMarkContent waterMarkContent = new WaterMarkContent();
            waterMarkContent.setType(WaterMarkType.valueOf(vo.getType()));
            waterMarkContent.setContent(vo.getContent());
            waterMarkContent.setImageBase64(vo.getImageBase64());
            waterMarkContent.setFontSize(vo.getFontSize());
            waterMarkContent.setColor(vo.getColor());
            waterMarkContent.setRotateAngle(vo.getRotateAngle());
            waterMarkContent.setTransparency(vo.getTransparency());
            waterMarkContent.setLocation(WaterMarkLocation.valueOf(vo.getLocation()));
            waterMarks.add(waterMarkContent);
        }

        DocumentCreateByUrl documentCreateByUrl = new DocumentCreateByUrl(contractUrl, contractTitle, contractFileType, waterMarks);
        // 创建合同文档
        Long documentId = null;
        try {
            documentId = contractService.createDocumentByUrl(documentCreateByUrl);
        } catch (PrivateAppException e) {
            throw new BusinessException(e.getMessage());
        }
        logger.info("合同文档id--{}", documentId);
        // 下载合同文档
        DownloadDocRequest request = new DownloadDocRequest();

        File file = File.createTempFile(contractTitle, ".pdf");
        OutputStream out = FileUtil.getOutputStream(file);
        request.setDocumentId(documentId);
        request.setContact(contractTitle);
        request.setOutputStream(out);
        try {
            contractService.downloadDoc(request);
        } catch (PrivateAppException e) {
            throw new BusinessException(e.getMessage());
        }

        return file;
    }
}
