package com.ejianc.business.signaturemanage.controller.api;

import cn.hutool.core.io.FileUtil;
import cn.hutool.core.io.IoUtil;
import com.ejianc.business.signaturemanage.service.ISignatureCommonService;
import com.ejianc.business.signaturemanage.vo.WatermarkVO;
import com.ejianc.foundation.file.vo.AttachmentVO;
import com.google.common.base.Stopwatch;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.http.MediaType;
import org.springframework.util.Assert;
import org.springframework.validation.annotation.Validated;
import org.springframework.web.bind.annotation.PostMapping;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RestController;

import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import java.io.BufferedInputStream;
import java.io.File;
import java.io.IOException;
import java.io.OutputStream;
import java.util.concurrent.TimeUnit;

/**
 * @author baipengyan
 * @date 2022/5/17
 * @description 签章通用接口实现控制器
 */
@RestController
@RequestMapping(value = "/common")
public class SignatureCommonApi {

    private final Logger logger = LoggerFactory.getLogger(this.getClass());

    @Autowired
    private ISignatureCommonService service; // 签章通用服务


    /**
     * 合同文件添加水印
     *
     * @param watermarkVO 合同文件添加水印参数
     * @param response    响应
     */
    @PostMapping(value = "/addWatermark", consumes = MediaType.APPLICATION_JSON_UTF8_VALUE)
    public void addWatermark(@Validated @RequestBody WatermarkVO watermarkVO, HttpServletResponse response) throws IOException {
        File file = service.addWatermark(watermarkVO);
        Assert.notNull(file, "合同文件添加水印失败！");
        BufferedInputStream in = FileUtil.getInputStream(file);
        response.setCharacterEncoding("UTF-8");
        response.setHeader("Content-Disposition", "attachment;fileName=" + file.getName());
        response.setContentType(MediaType.APPLICATION_PDF_VALUE);
        OutputStream out = response.getOutputStream();
        IoUtil.copy(in, out);
        out.flush();
        in.close();
        out.close();
        service.cleanFile(file);
    }

    /**
     * 获取加水印的合同附件
     *
     * @param watermarkVO 合同文件添加水印参数
     * @return AttachmentVO 文件中心的水印附件
     */
    @PostMapping(value = "/fetchWatermarkAttachment")
    public AttachmentVO fetchWatermarkAttachment(@RequestBody WatermarkVO watermarkVO, HttpServletRequest request) throws IOException {
        logger.info("获取加水印的合同附件，开始");
        Stopwatch start = Stopwatch.createStarted();
        String authority = request.getHeader("authority");
        Assert.hasText(authority, "获取加水印的合同附件，上下文不能为空！");
        File file = service.addWatermark(watermarkVO);
        Assert.notNull(file, "合同文件添加水印失败！");
        Stopwatch end = start.stop();
        logger.info("添加水印，结束。共计耗时：「{}」秒",end.elapsed(TimeUnit.SECONDS));
        Stopwatch start1 = Stopwatch.createStarted();
        Assert.notNull(watermarkVO.getBillId(), "业务id不能为空！");
        Assert.hasText(watermarkVO.getBillType(), "业务类型不能为空！");
        Assert.hasText(watermarkVO.getSourceType(), "单据类型不能为空！");
        AttachmentVO attachmentVO = service.fetchAttachment(file, authority, watermarkVO);
        service.cleanFile(file);
        Stopwatch end1 = start1.stop();
        logger.info("获取加水印的合同附件，结束。共计耗时：「{}」秒",end1.elapsed(TimeUnit.SECONDS));
        return attachmentVO;
    }
}
