package com.ejianc.business.signaturemanage.service.impl;

import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.serializer.SerializerFeature;
import com.baomidou.mybatisplus.core.conditions.query.LambdaQueryWrapper;
import com.baomidou.mybatisplus.core.toolkit.IdWorker;
import com.ejianc.business.signaturemanage.bean.SignMgrEntity;
import com.ejianc.business.signaturemanage.bean.SignMgrSignatoryEntity;
import com.ejianc.business.signaturemanage.enums.DelFlagEnum;
import com.ejianc.business.signaturemanage.enums.SignMgrSignatoryEnum;
import com.ejianc.business.signaturemanage.mapper.SignatureRectMapper;
import com.ejianc.business.signaturemanage.service.IAsyncInformService;
import com.ejianc.business.signaturemanage.service.ISignMgrService;
import com.ejianc.business.signaturemanage.service.ISignMgrSignatoryService;
import com.ejianc.business.signaturemanage.service.ISignatureRectService;
import com.ejianc.business.signaturemanage.vo.*;
import com.ejianc.foundation.file.api.IAttachmentApi;
import com.ejianc.foundation.file.vo.AttachmentVO;
import com.ejianc.foundation.orgcenter.api.IUserApi;
import com.ejianc.foundation.usercenter.vo.UserVO;
import com.ejianc.framework.auth.session.UserContext;
import com.ejianc.framework.core.context.InvocationInfoProxy;
import com.ejianc.framework.core.exception.BusinessException;
import com.ejianc.framework.core.response.CommonResponse;
import net.qiyuesuo.sdk.SDKClient;
import net.qiyuesuo.sdk.api.ContractService;
import net.qiyuesuo.sdk.bean.company.TenantType;
import net.qiyuesuo.sdk.bean.contract.*;
import net.qiyuesuo.sdk.bean.document.DocumentCreateByUrl;
import net.qiyuesuo.sdk.bean.sign.PreSignUrlRequest;
import net.qiyuesuo.sdk.bean.sign.Signatory;
import net.qiyuesuo.sdk.bean.sign.SignatoryRect;
import net.qiyuesuo.sdk.common.exception.PrivateAppException;
import net.qiyuesuo.sdk.impl.ContractServiceImpl;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.stereotype.Service;
import org.springframework.util.Assert;
import org.springframework.web.context.request.RequestAttributes;
import org.springframework.web.context.request.RequestContextHolder;
import org.springframework.web.context.request.ServletRequestAttributes;

import javax.servlet.http.HttpServletRequest;
import java.util.*;
import java.util.stream.Collectors;
import java.util.stream.Stream;

/**
 * @author baipengyan
 * @date 2022/6/2
 * @description 签章定位业务接口实现类
 */
@Service("signatureRectService")
public class SignatureRectServiceImpl implements ISignatureRectService {

	private final Logger logger = LoggerFactory.getLogger(this.getClass());
	private final ISignMgrService signMgrService;
	private final ISignMgrSignatoryService signMgrSignatoryService;
	private final IAsyncInformService asyncInformService;
	private final SignatureRectMapper signatureRectMapper;
	private final IAttachmentApi attachmentApi; // 文件中心附件信息接口
	private final IUserApi userApi; // 用户信息接口
	@Value("${qiyuesuo.client.url}")
	private String url; // 私有化开放平台请求地址
	@Value("${qiyuesuo.client.accessKey}")
	private String accessKey; // 私有化开放平台申请的token
	@Value("${qiyuesuo.client.accessSecret}")
	private String accessSecret; // 私有化开放平台申请的secret
	@Value("${common.env.base-host}")
	private String baseHost; // 域名
	@Value("${ejc.fileServerUrl}")
	private String fileHost; // 文件中心
	@Value("${qiyuesuo.contract.categoryId}")
	private Long categoryId;// 用印流程id


	public SignatureRectServiceImpl(ISignMgrService signMgrService, IAttachmentApi attachmentApi, IUserApi userApi, SignatureRectMapper signatureRectMapper, ISignMgrSignatoryService signMgrSignatoryService, IAsyncInformService asyncInformService) {
		this.signMgrService = signMgrService;
		this.attachmentApi = attachmentApi;
		this.userApi = userApi;
		this.signatureRectMapper = signatureRectMapper;
		this.signMgrSignatoryService = signMgrSignatoryService;
		this.asyncInformService = asyncInformService;
	}


	/**
	 * 契约锁：初始化sdk
	 *
	 * @return client
	 */
	private SDKClient getSdkClient() {
		// 初始化client
		SDKClient client = new SDKClient(url, accessKey, accessSecret);
		// 开启防止重放攻击
		client.enableNonce();
		return client;
	}


	/**
	 * 删除签章数据：双删除（本地逻辑删除，第三方合同删除）
	 *
	 * @param contractId PM系统合同id
	 */
	public void deleteSignatureRect(Long contractId) {
		LambdaQueryWrapper<SignMgrEntity> queryWrapper = new LambdaQueryWrapper<>();
		queryWrapper.eq(SignMgrEntity::getDelFlag, DelFlagEnum.NORMAL.getDelFlag());
		queryWrapper.eq(SignMgrEntity::getBillId, contractId);
		SignMgrEntity signMgrEntity = signMgrService.getOne(queryWrapper);

		if (signMgrEntity != null) {
			Assert.notNull(signMgrEntity.getSourceBillId(), "第三方电子签章合同id不存在！");

			// 1.校验合同状态
			ContractDetail contractDetail = this.thirdContractDetail(signMgrEntity.getSourceBillId(), false);
			Assert.notNull(contractDetail, "第三方电子签章合同详情查询不到！");
			if (contractDetail.getStatus() != ContractStatus.DRAFT) {
				throw new BusinessException("第三方电子签章合同详情不是草稿状态，不能删除PM系统签章数据第三方电子签章数据！");
			}

			// 2.删除签章数据
			this.deleteContractSignatureData(contractId);

			// 3.删除第三方电子签章数据
			this.deleteThirdContract(signMgrEntity.getSourceBillId());
		}
	}


	/**
	 * 获取预签署链接
	 *
	 * @param signatureVO 发起签章VO
	 *
	 * @return String 预签署链接
	 */
	public String fetchPreSignUrl(SignatureVO signatureVO) {
		// 0.手动校验参数
		signatureVO.getSignatoryVOS().forEach(signatoryVO -> signatoryVO.getSignatoryActionVOS().forEach(signatoryActionVO -> signatoryActionVO.getOperatorVOS().forEach(operatorVO -> {
			if (signatoryVO.getSignatureType() == 0) {
				Assert.notNull(operatorVO.getOperatorId(), "操作人id（内部单位平台用户id）「operatorId」不能为空！");
				Assert.hasLength(operatorVO.getOperatorName(), "操作人姓名「operatorName」不能为空！");
				Assert.hasLength(operatorVO.getOperatorContact(), "操作人联系方式「operatorContact」不能为空！");
			}
		})));

		// 1.删除签章数据：双删除（本地逻辑删除，第三方合同删除）
		this.deleteSignatureRect(signatureVO.getBillId());

		// 2.解析合同文件信息
		Map<String, String> map = this.parseAttachment(signatureVO.getBillDocId());

		// 3.根据合同文件创建合同文档
		Long documentId = this.createContractDocument(map.get("url"), map.get("title"), map.get("fileType"), null);
		Assert.notNull(documentId, "获取预签署链接：根据PM系统合同文件id：" + signatureVO.getBillDocId() + "，创建合同文档失败！");

		// 4.根据合同文档创建合同
		Long contractId = this.createContract(documentId, signatureVO, false);
		Assert.notNull(contractId, "获取预签署链接：根据第三方电子签章合同文档id：" + documentId + "，创建合同失败！");

		// 5.根据合同获取预签署链接
		return this.preSignUrl(contractId, true, true, false, true, documentId, baseHost);
	}


	/**
	 * 根据文件id获取附件信息
	 *
	 * @param fileId 文件id
	 *
	 * @return Map<String, String> 附件信息
	 */
	public Map<String, String> parseAttachment(Long fileId) {
		// 校验参数
		Assert.notNull(fileId, "fileId不能为空");

		Map<String, String> map = new HashMap<>();
		CommonResponse<AttachmentVO> res = attachmentApi.queryDetail(String.valueOf(fileId));
		if (!res.isSuccess() || res.getData() == null) {
			logger.error("调用文件中心接口，获取附件失败！原因：{}", res.getMsg());
			throw new BusinessException("调用文件中心接口，获取附件失败！原因：" + res.getMsg());
		}
		logger.info("合同文件详情：{}", JSON.toJSONString(res.getData(), SerializerFeature.WriteMapNullValue, SerializerFeature.PrettyFormat));
		String fileUrl = fileHost + res.getData().getFilePath();
		Assert.hasText(fileUrl, "合同文件网络路径不能为空！");
		String fullName = res.getData().getFileName();
		String title = fullName.substring(0, fullName.lastIndexOf("."));
		Assert.hasText(title, "合同文件名称不能为空！");
		String fileType = fullName.substring(fullName.lastIndexOf(".") + 1);
		Assert.hasText(fileType, "合同文件类型不能为空！");

		map.put("url", fileUrl);
		map.put("title", title);
		map.put("fileType", fileType);
		return map;
	}


	/**
	 * 发起签章
	 *
	 * @param contractId PM系统合同id
	 */
	public void sendSignature(Long contractId) {
		// 校验参数
		Assert.notNull(contractId, "发起签章，PM系统合同id不能为空！");

		// 0.获取合同签章数据
		LambdaQueryWrapper<SignMgrEntity> queryWrapper = new LambdaQueryWrapper<>();
		queryWrapper.eq(SignMgrEntity::getDelFlag, DelFlagEnum.NORMAL.getDelFlag());
		queryWrapper.eq(SignMgrEntity::getBillId, contractId);
		SignMgrEntity signMgrEntity = signMgrService.getOne(queryWrapper);
		Assert.notNull(signMgrEntity, "发起签章，根据PM系统合同id：" + contractId + "，未找到签章数据！");
		Assert.notNull(signMgrEntity.getSourceBillId(), "发起签章，根据PM系统合同id：" + contractId + "，未找到第三方电子签章合同id！");

		LambdaQueryWrapper<SignMgrSignatoryEntity> wrapper = new LambdaQueryWrapper<>();
		wrapper.eq(SignMgrSignatoryEntity::getDelFlag, DelFlagEnum.NORMAL.getDelFlag());
		wrapper.eq(SignMgrSignatoryEntity::getPid, signMgrEntity.getId());
		List<SignMgrSignatoryEntity> signatoryEntities = signMgrSignatoryService.list(wrapper);
		Assert.notEmpty(signatoryEntities, "发起签章，根据PM系统合同id：" + contractId + "，未找到签章数据！");

		// 1.调用第三方电子签章接口，发起合同
		this.sendContract(signMgrEntity.getSourceBillId());

		// 2.获取下一批签署动作
		List<SignMgrSignatoryEntity> actions = this.nextAction(signatoryEntities, true);

		// 3.获取上下文
		String authority = this.getAuthority();
		// RequestAttributes attributes = this.getAttributes();

		// 4.业务系统签章状态通知
		Integer status;
		if (CollectionUtils.isNotEmpty(actions)) {
			if (Objects.equals(SignMgrSignatoryEnum.INTERNAL_UNIT.getValue(), actions.get(0).getSignatureType())) {
				status = SignMgrSignatoryEnum.WAITE_PARTY_A.getValue();
			} else {
				status = SignMgrSignatoryEnum.WAITE_PARTY_B.getValue();
			}
		} else {
			status = SignMgrSignatoryEnum.WAITE_PARTY_A.getValue();
		}
		asyncInformService.informBusinessSystem(signMgrEntity.getBillType(), contractId, signMgrEntity.getBillRefCode(), status, authority);


		// 5.发送消息通知通知
		asyncInformService.informMessage(Collections.singletonList("sys"), actions, contractId, signMgrEntity.getBillCode(), signMgrEntity.getContractName());
	}


	/**
	 * 根据PM系统合同id获取第三方电子签章合同id
	 *
	 * @param contractId PM系统合同id
	 *
	 * @return 第三方电子签章合同id
	 */
	public Long getThirdContractId(Long contractId) {
		LambdaQueryWrapper<SignMgrEntity> queryWrapper = new LambdaQueryWrapper<>();
		queryWrapper.eq(SignMgrEntity::getDelFlag, DelFlagEnum.NORMAL.getDelFlag());
		queryWrapper.eq(SignMgrEntity::getBillId, contractId);
		SignMgrEntity signMgrEntity = signMgrService.getOne(queryWrapper);

		// 校验参数
		Assert.notNull(signMgrEntity, "PM系统合同id：" + contractId + "，电子签章记录不存在！");
		Assert.notNull(signMgrEntity.getSourceBillId(), "第三方电子签章，合同id不能为空！");
		return signMgrEntity.getSourceBillId();
	}


	/**
	 * 根据路径创建合同文档
	 *
	 * @param url        文件路径
	 * @param title      合同文档名称
	 * @param fileType   文件类型：pdf, doc, docx, wps, rtf, png, gif, jpg, jpeg, tiff, html, htm, xls, xlsx, txt
	 * @param waterMarks 水印
	 *
	 * @return Long 文档ID
	 */
	public Long createContractDocument(String url, String title, String fileType, List<WaterMarkContent> waterMarks) {
		// 校验参数
		Assert.hasText(url, "根据路径创建合同文档，文件路径不能为空！");
		Assert.hasText(title, "根据路径创建合同文档，合同文档名称不能为空！");
		Assert.hasText(fileType, "根据路径创建合同文档，文件类型不能为空！");

		ContractService contractService = new ContractServiceImpl(this.getSdkClient());
		Long documentId = null;
		try {
			documentId = contractService.createDocumentByUrl(new DocumentCreateByUrl(url, title, fileType, waterMarks));
		} catch (PrivateAppException e) {
			logger.error("第三方电子签章，创建合同文档失败！入参：PM系统-文件路径url：{}，文件名称：{}，文件类型：{}，错误信息：{}", url, title, fileType, e.getMessage());
		}
		return documentId;
	}


	/**
	 * 创建合同
	 *
	 * @param documentId  第三方电子签章合同文档id
	 * @param signatureVO 发起签章VO
	 * @param send        是否立即发起合同
	 *
	 * @return Long 第三方电子签章合同id
	 */
	public Long createContract(Long documentId, SignatureVO signatureVO, Boolean send) {
		// 校验参数
		Assert.notNull(documentId, "创建合同，第三方电子签章合同文档id不能为空！");
		Assert.notNull(signatureVO, "创建合同，发起签章VO不能为空！");
		Assert.notNull(send, "创建合同，是否立即发起合同不能为空！");

		SignMgrEntity signMgrEntity = new SignMgrEntity();
		List<SignMgrSignatoryEntity> signatoryEntities = signMgrEntity.getSignMgrSignatoryEntities();

		ContractService contractService = new ContractServiceImpl(this.getSdkClient());
		CreateContractRequest request = new CreateContractRequest();
		List<Signatory> signatories = new ArrayList<>();
		request.setCategoryId(categoryId);
		// todo : 郑州一建要求 乙方名称+文件名称+单据编号
		request.setSubject(signatureVO.getContractName());
		request.setSn(signatureVO.getBillCode());
		request.setSend(send);
		request.setDocuments(Collections.singletonList(documentId));
		request.setDescription(signatureVO.getSignRequirements());
		// 当前登录用户信息
		UserVO userVO = this.getUserVO(UserContext.getUserContext().getUserId());
		request.setCreatorName(userVO.getUserName());
		Assert.hasText(userVO.getUserMobile(), "当前用户手机号不能为空，请在系统内维护手机号后重试！");
		request.setCreatorContact(userVO.getUserMobile());
		request.setTenantName(signatureVO.getTenantName());
		// 必须签署可增加
		request.setExtraSign(true);
		request.setMustSign(true);

		signMgrEntity.setDelFlag(DelFlagEnum.NORMAL.getDelFlag());
		signMgrEntity.setBillId(signatureVO.getBillId());
		signMgrEntity.setBillType(signatureVO.getBillType());
		signMgrEntity.setBillCode(signatureVO.getBillCode());
		signMgrEntity.setContractName(signatureVO.getContractName());
		signMgrEntity.setContractTaxMny(signatureVO.getContractTaxMny());
		signMgrEntity.setProjectId(signatureVO.getProjectId());
		signMgrEntity.setProjectCode(signatureVO.getProjectCode());
		signMgrEntity.setProjectName(signatureVO.getProjectName());
		signMgrEntity.setBillRefCode(signatureVO.getBillRefCode());

		signMgrEntity.setSourceDocId(documentId);
		signMgrEntity.setBillDocId(signatureVO.getBillDocId());
		signMgrEntity.setBillDocumentName(signatureVO.getBillDocumentName());
		signMgrEntity.setBillDocumentType(signatureVO.getBillDocumentType());
		signMgrEntity.setSignRequirements(signatureVO.getSignRequirements());
		signMgrEntity.setCreatorName(signatureVO.getCreatorName());
		signMgrEntity.setCreatorContact(signatureVO.getCreatorContact());
		signMgrEntity.setTenantName(signatureVO.getTenantName());

		List<SignatoryVO> signatoryVOS = signatureVO.getSignatoryVOS();
		signatoryVOS.sort(Comparator.comparing(SignatoryVO::getSignOrder));
		// 签署方
		StringBuilder stringBuilder = new StringBuilder();
		for (SignatoryVO signatoryVO : signatoryVOS) {
			Signatory signatory = new Signatory();
			signatory.setTenantType(TenantType.valueOf(signatoryVO.getTenantType()));
			signatory.setTenantName(signatoryVO.getTenantName());

			String signatoryHead;
			boolean flag = Objects.equals(SignMgrSignatoryEnum.INTERNAL_UNIT.getValue(), signatoryVO.getSignatureType());
			if (flag) {
				signatoryHead = "T_";
			} else {
				signatoryHead = "S_";
			}

			String signatoryNo = String.valueOf(
					stringBuilder.append(signatoryHead)
							.append(InvocationInfoProxy.getTenantid()).append("_")
							.append(signatureVO.getBillRefCode()).append("_")
							.append(signatureVO.getBillId()).append("_")
							.append(signatureVO.getBillType()).append("_")
							.append(IdWorker.getId())
			);
			signatory.setSignatoryNo(signatoryNo);
			signatory.setReceiverName(signatoryVO.getReceiverName());
			signatory.setContact(signatoryVO.getContact());
			signatory.setSerialNo(signatoryVO.getSignOrder());

			List<Action> actions = new ArrayList<>();
			List<SignatoryActionVO> signatoryActionVOS = signatoryVO.getSignatoryActionVOS();
			for (SignatoryActionVO signatoryActionVO : signatoryActionVOS) {
				SignMgrSignatoryEntity sms = new SignMgrSignatoryEntity();
				Action action = new Action();
				action.setType(ActionType.valueOf(signatoryActionVO.getSignActionType()));

				// 字符串重置
				stringBuilder.delete(0, stringBuilder.length());
				String actionNo = String.valueOf(
						stringBuilder.append(signatoryHead)
								.append(InvocationInfoProxy.getTenantid()).append("_")
								.append(signatureVO.getBillRefCode()).append("_")
								.append(signatureVO.getBillId()).append("_")
								.append(signatureVO.getBillType()).append("_")
								.append(signatoryActionVO.getSignActionType()).append("_")
								.append(signatoryActionVO.getSealSignOrder()).append("_")
								.append(IdWorker.getId())
				);
				action.setActionNo(actionNo);
				action.setName(signatoryActionVO.getName());
				action.setSerialNo(signatoryActionVO.getSealSignOrder());

				// 内部单位指定默认印章
				if (flag && StringUtils.isNotBlank(signatoryActionVO.getSourceSealId()) && StringUtils.isNotBlank(signatoryActionVO.getSourceSealName())) {
					Set<Long> ids = Stream.of(signatoryActionVO.getSourceSealId().split(",")).map(e -> Long.parseLong(e.trim())).collect(Collectors.toSet());
					Set<String> names = Stream.of(signatoryActionVO.getSourceSealName().split(",")).collect(Collectors.toSet());
					action.setSealIds(ids);
					action.setSealNames(names);
				}

				List<Operator> operators = new ArrayList<>();
				List<OperatorVO> ops = signatoryActionVO.getOperatorVOS();
				if (CollectionUtils.isNotEmpty(ops)) {
					for (OperatorVO op : ops) {
						// 内部人员校验
						if (op.getOperatorId() != null && Objects.equals(SignMgrSignatoryEnum.INTERNAL_UNIT.getValue(), signatoryVO.getSignatureType())) {
							CommonResponse<List<UserVO>> user = userApi.queryListByIdsTenantId(new String[] {String.valueOf(op.getOperatorId())}, InvocationInfoProxy.getTenantid());
							if (!user.isSuccess() || CollectionUtils.isEmpty(user.getData())) {
								throw new BusinessException("内部人员校验失败！");
							}
						}

						Operator operator = new Operator();
						operator.setOperatorName(op.getOperatorName());
						operator.setOperatorContact(op.getOperatorContact());
						operators.add(operator);

						sms.setSignatureId(op.getOperatorId());
						sms.setSignatureName(op.getOperatorName());
						sms.setSignatureContact(op.getOperatorContact());

						sms.setSignActionType(signatoryActionVO.getSignActionType());
						sms.setName(signatoryActionVO.getName());
						sms.setSealSignOrder(signatoryActionVO.getSealSignOrder());
						sms.setSourceSealId(signatoryActionVO.getSourceSealId());
						sms.setSourceSealName(signatoryActionVO.getSourceSealName());

						sms.setSealId(signatoryActionVO.getSealId());
						sms.setSealName(signatoryActionVO.getSealName());
						sms.setSignatureUserIds(signatoryActionVO.getSignatureUserIds());
						sms.setActionNo(actionNo);

						sms.setTenantName(signatoryVO.getTenantName());
						sms.setTenantType(signatoryVO.getTenantType());
						sms.setReceiverName(signatoryVO.getReceiverName());
						sms.setContact(signatoryVO.getContact());
						sms.setSignOrder(signatoryVO.getSignOrder());

						sms.setSignatureType(signatoryVO.getSignatureType());
						sms.setSignatoryNo(signatoryNo);
						sms.setJobStatus(SignMgrSignatoryEnum.TO_BE_ACTIVATED.getValue());
						sms.setDelFlag(DelFlagEnum.NORMAL.getDelFlag());

						signatoryEntities.add(sms);
					}
					action.setActionOperators(operators);
				}

				// 签署位置
				List<SignatoryRectVO> signatoryRectVOS = signatoryActionVO.getSignatoryRectVOS();
				if (CollectionUtils.isNotEmpty(signatoryRectVOS)) {
					List<SignatoryRect> signatoryRects = new ArrayList<>();
					for (SignatoryRectVO rect : signatoryRectVOS) {
						// 个人签字或法人签字
						if ("PERSONAL".equals(signatoryActionVO.getSignActionType()) || "LP".equals(signatoryActionVO.getSignActionType())) {
							// 是否开启关键字定位：默认开启
							if (StringUtils.isNotBlank(rect.getKeywords())) {
								List<String> keywords = Arrays.asList(rect.getKeywords().split(","));
								if (CollectionUtils.isNotEmpty(keywords)) {
									SignatoryRect sr = new SignatoryRect();
									sr.setDocumentId(documentId);
									sr.setRectType(StamperType.SEAL_CORPORATE);
									for (String keyword : keywords) {
										if (StringUtils.isNotBlank(keyword)) {
											sr.setKeyword(keyword);
											signatoryRects.add(sr);
										}
									}
								}
							}

							// 是否开启日期关键字定位
							if (Boolean.TRUE.equals(rect.getTimeStampFlag()) && StringUtils.isNotBlank(rect.getTimeStampKeyWords())) {
								List<String> timeStampKeyWords = Arrays.asList(rect.getTimeStampKeyWords().split(","));
								if (CollectionUtils.isNotEmpty(timeStampKeyWords)) {
									SignatoryRect sr = new SignatoryRect();
									sr.setDocumentId(documentId);
									sr.setRectType(StamperType.TIMESTAMP);
									for (String timeStampKeyWord : timeStampKeyWords) {
										if (StringUtils.isNotBlank(timeStampKeyWord)) {
											sr.setKeyword(timeStampKeyWord);
											signatoryRects.add(sr);
										}
									}
								}
							}
						}

						// 公章
						if ("CORPORATE".equals(signatoryActionVO.getSignActionType())) {
							// 是否开启关键字定位：默认开启
							if (StringUtils.isNotBlank(rect.getKeywords())) {
								List<String> keywords = Arrays.asList(rect.getKeywords().split(","));
								if (CollectionUtils.isNotEmpty(keywords)) {
									SignatoryRect sr = new SignatoryRect();
									sr.setDocumentId(documentId);
									sr.setRectType(StamperType.SEAL_CORPORATE);
									for (String keyword : keywords) {
										if (StringUtils.isNotBlank(keyword)) {
											sr.setKeyword(keyword);
											sr.setKeywordIndex(rect.getKeywordIndex());
											signatoryRects.add(sr);
										}
									}
								}
							}

							// 是否开启日期关键字定位
							if (Boolean.TRUE.equals(rect.getTimeStampFlag()) && StringUtils.isNotBlank(rect.getTimeStampKeyWords())) {
								List<String> timeStampKeyWords = Arrays.asList(rect.getTimeStampKeyWords().split(","));
								if (CollectionUtils.isNotEmpty(timeStampKeyWords)) {
									SignatoryRect sr = new SignatoryRect();
									sr.setDocumentId(documentId);
									sr.setRectType(StamperType.TIMESTAMP);
									for (String timeStampKeyWord : timeStampKeyWords) {
										if (StringUtils.isNotBlank(timeStampKeyWord)) {
											sr.setKeyword(timeStampKeyWord);
											sr.setKeywordIndex(rect.getKeywordIndex());
											signatoryRects.add(sr);
										}
									}
								}
							}

							//  是否开启骑缝章
							if (Boolean.TRUE.equals(rect.getAcrossFlag()) && StringUtils.isNotBlank(rect.getAcrossSetting())) {
								if ("ACROSS_PAGE_ODD".equals(rect.getAcrossSetting())) {
									SignatoryRect sr = new SignatoryRect();
									sr.setDocumentId(documentId);
									sr.setRectType(StamperType.ACROSS_PAGE_ODD);
									sr.setOffsetY(rect.getOffsetY());
									signatoryRects.add(sr);
								}
								if ("ACROSS_PAGE".equals(rect.getAcrossSetting())) {
									SignatoryRect sr = new SignatoryRect();
									sr.setDocumentId(documentId);
									sr.setRectType(StamperType.ACROSS_PAGE);
									sr.setOffsetY(rect.getOffsetY());
									signatoryRects.add(sr);
								}
							}
						}
					}
					if (CollectionUtils.isNotEmpty(signatoryRects)) {
						action.setLocations(signatoryRects);
					}
				}
				actions.add(action);
			}
			signatory.setActions(actions);
			signatories.add(signatory);
		}
		request.setSignatories(signatories);

		Long contractId;
		try {
			contractId = contractService.createContractByCategory(request);
		} catch (PrivateAppException e) {
			throw new BusinessException("第三方电子签章创建合同失败，原因是：" + e.getMessage());
		}
		signMgrEntity.setSourceBillId(contractId);

		logger.info("================================>请求第三方电子签章发起合同，并写入签章业务数据-START<================================");
		logger.info("即将写入的签章数据signMgrEntity为：{}", JSON.toJSONString(signMgrEntity, SerializerFeature.PrettyFormat, SerializerFeature.WriteMapNullValue));
		signMgrService.saveOrUpdate(signMgrEntity, false);
		logger.info("================================>请求第三方电子签章发起合同，写入签章管理数据成功-END<================================");

		return contractId;
	}


	/**
	 * 发起合同
	 *
	 * @param contractId 第三方电子签章合同id
	 */
	public void sendContract(Long contractId) {
		// 校验参数
		Assert.notNull(contractId, "发起合同，第三方电子签章合同id不能为空！");

		ContractServiceImpl contractService = new ContractServiceImpl(getSdkClient());
		SendContractRequest request = new SendContractRequest();
		request.setContractId(contractId);
		try {
			contractService.send(request);
		} catch (PrivateAppException e) {
			logger.error("发起合同失败，PM系统合同id：{}，第三方电子签章错误信息：{}", contractId, e.getMessage());
		}
	}


	/**
	 * 预签署页面
	 *
	 * @param contractId     第三方电子签章合同id
	 * @param canSaveOrClose 是否展示保存并关闭按钮
	 * @param canSetParam    是否可以填参
	 * @param canSend        是否可以发起合同
	 * @param canSetStamper  是否可以指定签署位置
	 * @param documentId     页面需展示文档的ID
	 * @param invalidToPage  链接失效后跳转地址
	 *
	 * @return 预签署页面链接
	 */
	public String preSignUrl(Long contractId, Boolean canSaveOrClose, Boolean canSetParam, Boolean canSend, Boolean canSetStamper, Long documentId, String invalidToPage) {
		// 校验参数
		Assert.notNull(contractId, "预签署页面，第三方电子签章合同id不能为空！");
		Assert.notNull(canSaveOrClose, "预签署页面，第三方电子签章是否展示保存并关闭按钮不能为空！");
		Assert.notNull(canSetParam, "预签署页面，第三方电子签章是否可以填参不能为空！");
		Assert.notNull(canSend, "预签署页面，第三方电子签章是否可以发起合同不能为空！");
		Assert.notNull(canSetStamper, "预签署页面，第三方电子签章是否可以指定签署位置不能为空！");
		Assert.notNull(documentId, "预签署页面，第三方电子签章页面需展示文档的ID不能为空！");
		Assert.hasText(invalidToPage, "预签署页面，第三方电子签章链接失效后跳转地址不能为空！");

		ContractService contractService = new ContractServiceImpl(this.getSdkClient());
		PreSignUrlRequest request = new PreSignUrlRequest();
		request.setContractId(contractId);
		request.setCanSaveOrClose(canSaveOrClose);
		request.setCanSetParam(canSetParam);
		request.setCanSend(canSend);
		request.setCanSetStamper(canSetStamper);
		request.setDocumentId(documentId);
		request.setInvalidToPage(invalidToPage);
		String preSignUrl = null;
		try {
			preSignUrl = contractService.preSignUrl(request);
		} catch (PrivateAppException e) {
			logger.error("获取预签署页面链接失败，第三方电子签章合同id：{}，是否可以发起合同：{}，链接失效后跳转地址：{}，第三方电子签章错误信息：{}", contractId, canSend, invalidToPage, e.getMessage());
		}
		logger.info("第三方电子签章合同id：{}，预签署链接:{}", contractId, preSignUrl);
		return preSignUrl;
	}


	/**
	 * 根据第三方电子签章合同id查询第三方电子签章合同详情
	 *
	 * @param contractId 第三方电子签章合同id
	 * @param countSeal  是否统计印章
	 *
	 * @return ContractDetail 第三方电子签章合同详情
	 */
	public ContractDetail thirdContractDetail(Long contractId, Boolean countSeal) {
		// 校验参数
		Assert.notNull(contractId, "根据第三方电子签章合同id查询第三方电子签章合同详情，第三方电子签章合同id不能为空！");
		Assert.notNull(countSeal, "根据第三方电子签章合同id查询第三方电子签章合同详情，第三方电子签章是否统计印章不能为空！");

		ContractService contractService = new ContractServiceImpl(this.getSdkClient());
		ContractDetail detail;
		try {
			detail = contractService.detail(contractId, true);
		} catch (PrivateAppException e) {
			logger.error("第三方电子签章合同id：{}，查询合同详情失败！失败原因：{}", contractId, e.getMessage());
			throw new BusinessException("第三方电子签章根据合同id查询合同详情失败！失败原因：" + e.getMessage());
		}
		return detail;
	}


	/**
	 * 根据第三方电子签章合同id删除第三方电子签章合同
	 *
	 * @param contractId 第三方电子签章合同id
	 */
	public void deleteThirdContract(Long contractId) {
		// 校验参数
		Assert.notNull(contractId, "根据第三方电子签章合同id删除合同，第三方电子签章合同id不能为空！");

		ContractService contractService = new ContractServiceImpl(this.getSdkClient());
		try {
			contractService.delete(contractId);
		} catch (PrivateAppException e) {
			logger.error("第三方电子签章合同id：{}，删除合同失败！失败原因：{}", contractId, e.getMessage());
			throw new BusinessException("第三方电子签章根据合同id删除合同失败！失败原因：" + e.getMessage());
		}
	}


	/**
	 * 根据PM系统合同id逻辑删除合同签章数据
	 *
	 * @param contractId PM系统合同id
	 */
	public void deleteContractSignatureData(Long contractId) {
		// 删除主表、子表、预览表、日志表数据
		signatureRectMapper.deleteContractSignatureData(contractId);
	}


	/**
	 * 根据PM系统合同id查询合同签章数据吗，查询下一个签署动作
	 *
	 * @param entities 签署动作数据
	 * @param flag     是否是初始发起状态
	 *
	 * @return List<SignMgrSignatoryEntity> 签署动作
	 */
	public List<SignMgrSignatoryEntity> nextAction(List<SignMgrSignatoryEntity> entities, boolean flag) {
		// 校验参数
		Assert.notEmpty(entities, "查询下一个签署动作，签署动作数据不能为空！");
		List<SignMgrSignatoryEntity> operators = new ArrayList<>();

		// 0.根据是否是初始发起状态分治
		if (flag) {
			entities.sort(
					Comparator.comparing(SignMgrSignatoryEntity::getSignOrder)
							.thenComparing(SignMgrSignatoryEntity::getSealSignOrder)
			);

			List<SignMgrSignatoryEntity> entityList = entities.stream()
					.filter(e -> Objects.equals(e.getSignOrder(), entities.get(0).getSignOrder()) && Objects.equals(e.getSealSignOrder(), entities.get(0).getSealSignOrder()))
					.collect(Collectors.toList());
			operators.addAll(entityList);

			this.updateBatchToBeSigned(entityList);
		} else {
			List<SignMgrSignatoryEntity> entityList = entities.stream().filter(e -> Objects.equals(e.getJobStatus(), SignMgrSignatoryEnum.TO_BE_ACTIVATED.getValue()))
					.collect(Collectors.toList());

			if (CollectionUtils.isNotEmpty(entityList)) {
				entityList.sort(Comparator.comparing(SignMgrSignatoryEntity::getSignOrder).thenComparing(SignMgrSignatoryEntity::getSealSignOrder));

				List<SignMgrSignatoryEntity> entityList1 = entityList.stream().filter(e -> Objects.equals(e.getSignOrder(), entityList.get(0).getSignOrder()) && Objects.equals(e.getSealSignOrder(), entityList.get(0).getSealSignOrder()))
						.collect(Collectors.toList());
				operators.addAll(entityList1);

				this.updateBatchToBeSigned(entityList1);
			}
		}
		return operators;
	}


	/**
	 * 批量更新任务状态为待签章
	 *
	 * @param entityList 签署动作
	 */
	private void updateBatchToBeSigned(List<SignMgrSignatoryEntity> entityList) {
		entityList.forEach(e -> e.setJobStatus(SignMgrSignatoryEnum.TO_BE_SIGNED.getValue()));
		signMgrSignatoryService.saveOrUpdateBatch(entityList, 5);
	}


	/**
	 * 获取请求头authority（有上下文）
	 *
	 * @return authority
	 */
	private String getAuthority() {
		HttpServletRequest request = ((ServletRequestAttributes) Objects.requireNonNull(RequestContextHolder.getRequestAttributes())).getRequest();
		return request.getHeader("authority");
	}


	/**
	 * 获取上下文
	 *
	 * @return RequestAttributes
	 */
	public RequestAttributes getAttributes() {
		return RequestContextHolder.getRequestAttributes();
	}


	/**
	 * 根据用户id获取用户信息
	 *
	 * @param userId 用户id
	 *
	 * @return 用户信息
	 */
	public UserVO getUserVO(Long userId) {
		Assert.notNull(userId, "获取用户信息，用户id不能为空！");
		CommonResponse<UserVO> response = userApi.findUserByUserId(userId);
		if (!response.isSuccess() || response.getData() == null) {
			throw new BusinessException("根据用户id，获取用户信息失败，失败原因：" + response.getMsg());
		}
		return response.getData();
	}

}
