package com.ejianc.business.signaturemanage.service.impl;

import cn.hutool.http.HttpUtil;
import cn.hutool.http.Method;
import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.serializer.SerializerFeature;
import com.ejianc.business.signaturemanage.bean.SignMgrSignatoryEntity;
import com.ejianc.business.signaturemanage.enums.SignMgrSignatoryEnum;
import com.ejianc.business.signaturemanage.service.IAsyncInformService;
import com.ejianc.business.signaturemanage.service.ISignMgrSignatoryService;
import com.ejianc.foundation.message.api.IPushMessageApi;
import com.ejianc.foundation.message.vo.PushMsgParameter;
import com.ejianc.foundation.metadata.vo.MdReferVO;
import com.ejianc.foundation.orgcenter.api.IUserApi;
import com.ejianc.foundation.support.api.IBillTypeApi;
import com.ejianc.foundation.usercenter.vo.UserVO;
import com.ejianc.framework.core.context.InvocationInfoProxy;
import com.ejianc.framework.core.exception.BusinessException;
import com.ejianc.framework.core.response.CommonResponse;
import com.google.common.base.Stopwatch;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.retry.annotation.Backoff;
import org.springframework.retry.annotation.Recover;
import org.springframework.retry.annotation.Retryable;
import org.springframework.scheduling.annotation.Async;
import org.springframework.stereotype.Service;
import org.springframework.util.Assert;
import org.springframework.web.context.request.RequestContextHolder;
import org.springframework.web.context.request.ServletRequestAttributes;

import javax.annotation.PostConstruct;
import javax.servlet.http.HttpServletRequest;
import java.util.*;
import java.util.concurrent.TimeUnit;
import java.util.function.Function;

/**
 * @author baipengyan
 * @date 2022/6/9
 * @description
 */
@Service("asyncInformService")
public class IAsyncInformServiceImpl implements IAsyncInformService {
	private final Logger logger = LoggerFactory.getLogger(this.getClass());
	private final IBillTypeApi billTypeApi;
	private final IUserApi userApi;
	private final IPushMessageApi pushMessageApi;
	private final ISignMgrSignatoryService signMgrSignatoryService;
	private final Map<String, Function<Map<String, Object>, Void>> channelMap = new HashMap<>();
	@Value("${common.env.base-host}")
	private String BASE_HOST;

	public IAsyncInformServiceImpl(IBillTypeApi billTypeApi, IUserApi userApi, IPushMessageApi pushMessageApi, ISignMgrSignatoryService signMgrSignatoryService) {
		this.billTypeApi = billTypeApi;
		this.userApi = userApi;
		this.pushMessageApi = pushMessageApi;
		this.signMgrSignatoryService = signMgrSignatoryService;
	}

	@PostConstruct
	public void channelDispatcher() {
		// 消息渠道：系统通知
		channelMap.put("sys", this::systemInformMessage);
		// 消息渠道：短信通知
		// 消息渠道：微信通知
		// 消息渠道：钉钉通知
	}


	/**
	 * 异步通知业务系统签章状态
	 *
	 * @param billType  单据类型
	 * @param billId    单据id
	 * @param referCode 唯一标识
	 * @param status    签章状态
	 * @param authority 请求头上下文
	 */
	@Override
	// @Async(value = "attributesTask")
	@Retryable(value = {Exception.class}, maxAttempts = 3, backoff = @Backoff(delay = 2000, multiplier = 1.5), stateful = true)
	public void informBusinessSystem(String billType, Long billId, String referCode, int status, String authority) {
		// 校验参数
		Assert.hasText(billType, "单据类型不能为空！");
		Assert.notNull(billId, "单据id不能为空！");
		Assert.hasLength(referCode, "唯一标识不能为空！");
		Assert.hasText(authority, "请求头上下文不能为空！");

		logger.info("异步通知业务系统签章状态，单据类型：{}，单据id：{}，唯一标识：{}，签章状态：{}，请求头上下文：{}", billType, billId, referCode, status, authority);

		// ServletRequestAttributes requestAttributes = (ServletRequestAttributes) RequestContextHolder.getRequestAttributes();
		// HttpServletRequest request = Objects.requireNonNull(requestAttributes).getRequest();
		// logger.info("异步通知业务系统签章状态的authority：{}", request.getHeader("authority"));
		//
		// if (StringUtils.isBlank(request.getHeader("authority"))) {
		// 	requestAttributes.setAttribute("authority", authority, RequestAttributes.SCOPE_REQUEST);
		// 	InvocationInfoProxy.setExtendAttribute("authority", authority);
		// 	logger.info("informBusinessSystem-异步通知业务系统签章状态的authority：{}", authority);
		// }


		// 0.根据单据类型查询元数据信息
		CommonResponse<MdReferVO> mdRefResp = billTypeApi.queryMetadataByBillType(billType);
		if (!mdRefResp.isSuccess() || mdRefResp.getData() == null) {
			logger.error("异步通知业务系统签章状态，根据单据类型billType：{}，查询元数据信息失败,原因：{}！", billType, mdRefResp.getMsg());
			throw new BusinessException("异步通知业务系统签章状态，根据单据类型billType：" + billType + "，查询元数据信息失败，失败原因：" + mdRefResp.getMsg());
		}

		// 1.拼接对应业务系统的异步通知接口地址
		String entityName = mdRefResp.getData().getEntityName().replace("Entity", "") + "Signature";
		String apiUrl = entityName.substring(0, 1).toLowerCase() + entityName.substring(1);
		String url = BASE_HOST + mdRefResp.getData().getProjectName() + "/" + apiUrl + "/changeStatus";

		Map<String, Object> param = new HashMap<>();
		param.put("billId", billId);
		param.put("refCode", referCode);
		param.put("status", status);

		logger.info("异步通知业务系统签章状态，请求开始，请求地址：{}，请求参数：{}！", url, JSON.toJSONString(param, SerializerFeature.WriteMapNullValue));
		Stopwatch start = Stopwatch.createStarted();
		String response = HttpUtil.createRequest(Method.POST, url)
				.header("authority", authority)
				.body(JSON.toJSONString(param))
				.timeout(10000)
				.execute().body();
		Assert.hasLength(response, "异步通知业务系统签章状态，请求失败，返回结果为空！");
		logger.info("调用业务系统url-{}，param-{}，签章状态结果：{}", url, JSON.toJSONString(param, SerializerFeature.PrettyFormat, SerializerFeature.WriteMapNullValue), response);

		CommonResponse<String> commonResponse = JSON.parseObject(response, CommonResponse.class);
		if (!commonResponse.isSuccess()) {
			logger.error("异步通知业务系统签章状态，请求「成功」，返回结果「失败」：{}！", commonResponse.getMsg());
			throw new BusinessException("异步通知业务系统签章状态，请求「成功」，返回结果「失败」：" + commonResponse.getMsg());
		}
		logger.info("异步通知业务系统签章状态，请求「成功」，返回结果「成功」，耗时「{}」秒！", start.elapsed(TimeUnit.SECONDS));
	}


	/**
	 * 异步通知业务系统签章状态，重试失败
	 *
	 * @param e         异常
	 * @param billType  单据类型
	 * @param billId    单据id
	 * @param referCode 唯一标识
	 * @param status    签章状态
	 * @param authority 请求头上下文
	 */
	@Recover
	public void informBusinessSystemRecover(Exception e, String billType, Long billId, String referCode, int status, String authority) {
		logger.error("异步通知业务系统签章状态，重试失败，请求参数--billType：{}，billId：{}，referCode：{}，status：{}，authority：{}，失败原因：{}！", billType, billId, referCode, status, authority, e.getMessage());
	}


	/**
	 * 异步消息通知
	 *
	 * @param channels     消息渠道
	 * @param operators    签署动作
	 * @param contractId   第三方电子签章合同id
	 * @param billCode     单据编号
	 * @param contractName 合同名称
	 */
	@Override
	@Async(value = "attributesTask")
	@Retryable(value = {Exception.class}, maxAttempts = 3, backoff = @Backoff(delay = 3000, multiplier = 1.5), stateful = true)
	public void informMessage(List<String> channels, List<SignMgrSignatoryEntity> operators, Long contractId, String billCode, String contractName, String authority) {
		logger.info("异步消息通知，请求开始，请求参数--channels：{}，operators：{}，contractId：{}，billCode：{}，contractName：{}", JSON.toJSONString(channels, SerializerFeature.PrettyFormat, SerializerFeature.WriteMapNullValue), JSON.toJSONString(operators, SerializerFeature.PrettyFormat, SerializerFeature.WriteMapNullValue), contractId, billCode, contractName);

		ServletRequestAttributes requestAttributes = (ServletRequestAttributes) RequestContextHolder.getRequestAttributes();
		HttpServletRequest request = Objects.requireNonNull(requestAttributes).getRequest();
		logger.info("异步消息通知的authority：{}", request.getHeader("authority"));

		if (StringUtils.isBlank(request.getHeader("authority"))) {
			InvocationInfoProxy.setExtendAttribute("authority", authority);
			logger.info("informMessage-异步消息通知的authority：{}", authority);
		} else {
			InvocationInfoProxy.setExtendAttribute("authority", request.getHeader("authority"));
			logger.info("informMessage-异步消息通知的authority：{}", request.getHeader("authority"));
		}

		channels.forEach(channel -> {
			Map<String, Object> map = new HashMap<>();
			map.put("channel", channel);
			map.put("operators", operators);
			map.put("contractId", contractId);
			map.put("billCode", billCode);
			map.put("contractName", contractName);

			Function<Map<String, Object>, Void> result = channelMap.get(channel);
			Assert.notNull(result, "异步消息通知，请求失败，消息通道不存在！");
			result.apply(map);
		});
	}


	/**
	 * 系统通知
	 *
	 * @param map 消息通知参数
	 *
	 * @return null
	 */
	public Void systemInformMessage(Map<String, Object> map) {

		// 0.获取参数
		String channel = (String) map.get("channel");
		List<SignMgrSignatoryEntity> operators = (List<SignMgrSignatoryEntity>) map.get("operators");
		Long contractId = (Long) map.get("contractId");
		String billCode = (String) map.get("billCode");
		String contractName = (String) map.get("contractName");

		// 1.封装消息
		PushMsgParameter parameter = new PushMsgParameter();
		// 消息渠道
		parameter.setChannel(new String[] {channel});
		// 消息类型
		parameter.setMsgType("task");
		// 消息主题
		String subject = "你有一份新的文件【" + billCode + "_" + contractName + "】需要签署!";
		parameter.setSubject(subject);
		// 接收人
		String[] receivers;
		// 消息内容
		String content;

		if (Objects.equals(operators.get(0).getSignatureType(), SignMgrSignatoryEnum.INTERNAL_UNIT.getValue())) {

			// 接收人
			receivers = operators.stream().map(e -> String.valueOf(e.getSignatureId())).toArray(String[]::new);
			parameter.setReceivers(receivers);

			// 消息内容
			if ("PERSONAL".equals(operators.get(0).getSignActionType())) {
				content = "您有一个私有云待签署合同:<a href=\"" + BASE_HOST + "ejc-signaturemanage-frontend/#/privateSign?" + "contractId=" + contractId + "&tenantType="
						+ operators.get(0).getTenantType() + "&tenantName=" + operators.get(0).getTenantName() + "&contact=" + operators.get(0).getSignatureContact() + "\">点我签署</a>";
			} else {
				content = "您有一个私有云待签署合同:<a href=\"" + BASE_HOST + "ejc-signaturemanage-frontend/#/privateSign?" + "contractId=" + contractId + "&tenantType="
						+ operators.get(0).getTenantType() + "&tenantName=" + operators.get(0).getTenantName() + "&contact=" + "" + "\">点我签署</a>";
			}
			parameter.setContent(content);

			// 租户ID
			CommonResponse<UserVO> user = userApi.findUserByUserId(Long.valueOf(receivers[0]));
			if (!user.isSuccess() || user.getData() == null) {
				throw new BusinessException("获取用户信息异常！");
			}
			logger.info("查询到的租户列表：{}", JSON.toJSONString(user.getData(), SerializerFeature.PrettyFormat, SerializerFeature.WriteMapNullValue));
			String tenantId = String.valueOf(user.getData().getTenantId());
			parameter.setTenantId(tenantId);

			logger.info("发送通知入参：消息发送渠道类型：{}，接收人：{}，消息类型：{}，消息主题：{}，消息内容：{}，租户ID：{}", channel, receivers, "task", subject, content, tenantId);
			CommonResponse<String> response = pushMessageApi.pushMessage(parameter);
			if (!response.isSuccess()) {
				logger.error("发送系统消息通知失败，响应：{}", JSON.toJSONString(response, SerializerFeature.PrettyFormat, SerializerFeature.WriteMapNullValue));
				throw new BusinessException("发送系统消息通知失败，失败原因：" + response.getMsg());
			}
		} else {
			logger.info("外部单位发送消息通知");
		}

		// 接收时间
		operators.forEach(o -> o.setAcceptTime(new Date()));
		signMgrSignatoryService.saveOrUpdateBatch(operators, 5, false);
		logger.info("消息通知成功，已设置接收时间，当前通知人信息为：{}", JSON.toJSONString(operators, SerializerFeature.PrettyFormat, SerializerFeature.WriteMapNullValue));

		return null;
	}


	/**
	 * 异步消息通知，重试失败
	 *
	 * @param e   异常
	 * @param map 消息通知参数
	 */
	@Recover
	public Void systemInformMessage(Exception e, Map<String, Object> map) {
		String channel = (String) map.get("channel");
		List<SignMgrSignatoryEntity> operators = (List<SignMgrSignatoryEntity>) map.get("operators");
		Long contractId = (Long) map.get("contractId");
		String billCode = (String) map.get("billCode");
		String contractName = (String) map.get("contractName");

		logger.error("异步消息通知，重试失败，请求参数--channel：{}，operators：{}，contractId：{}，billCode：{}，contractName：{}，失败原因：{}！", channel, JSON.toJSONString(operators, SerializerFeature.PrettyFormat, SerializerFeature.WriteMapNullValue), contractId, billCode, contractName, e.getMessage());
		return null;
	}
}
