package com.ejianc.business.signaturemanage.controller;

import com.alibaba.fastjson.JSONObject;
import com.ejianc.business.signaturemanage.service.ISignatureRectService;
import com.ejianc.business.signaturemanage.vo.SignatureVO;
import com.ejianc.framework.core.context.InvocationInfoProxy;
import com.ejianc.framework.core.response.CommonResponse;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.validation.annotation.Validated;
import org.springframework.web.bind.annotation.*;

import javax.validation.Valid;
import javax.validation.constraints.NotNull;

/**
 * @author baipengyan
 * @date 2022/6/1
 * @description 签章定位
 */
@Validated
@RestController
@RequestMapping(value = "/signatureRect")
public class SignatureRectController {

	private final Logger logger = LoggerFactory.getLogger(this.getClass());

	private final ISignatureRectService service;

	public SignatureRectController(ISignatureRectService service) {
		this.service = service;
	}


	/**
	 * 签章定位：逻辑删除签章合同数据
	 */
	@GetMapping(value = "/deleteSignatureRect")
	public CommonResponse<String> deleteSignatureRect(@NotNull(message = "PM系统合同id不能为空！") @RequestParam(value = "contractId") Long contractId) {
		service.deleteSignatureRect(contractId);
		return CommonResponse.success("删除签章合同数据成功");
	}

	/**
	 * 签章定位：获取预签署链接
	 */
	@PostMapping(value = "/fetchPreSignUrl")
	public CommonResponse<String> fetchPreSignUrl(@Valid @RequestBody SignatureVO signatureVO) {
		return CommonResponse.success("获取预签署链接，请求成功！", service.fetchPreSignUrl(signatureVO));
	}

	/**
	 * 签章定位：发起签章
	 */
	@GetMapping(value = "/sendSignature")
	public CommonResponse<String> sendSignature(@NotNull(message = "PM系统合同id不能为空！") @RequestParam(value = "contractId") Long contractId) {
		service.sendSignature(contractId);
		return CommonResponse.success("当前合同id：" + contractId + "，发起签章成功！");
	}

	/**
	 * 撤回合同
	 * 撤回状态是“签署中”或“填参中”的合同
	 *
	 * @param contractRecallParamStr {contractId: Long 撤回签章单据主键Id, recallReason: String 签章撤回原因}
	 * @return
	 */
	@PostMapping(value = "/recallContract")
	public CommonResponse<String> recallContract(@RequestBody String contractRecallParamStr) {
		logger.info("【签章撤回】用户-{}执行单据签章撤回操作，参数-{}", InvocationInfoProxy.getUserid(), contractRecallParamStr);
		if(StringUtils.isBlank(contractRecallParamStr)) {
			return CommonResponse.error("合同签章撤回失败，参数为空！");
		}
		JSONObject jsonParam = JSONObject.parseObject(contractRecallParamStr);
		Long billId = jsonParam.getLong("contractId");
		String recallReason = jsonParam.getString("recallReason");
		if(StringUtils.isBlank(recallReason)) {
			//签章撤回原因为空是，使用默认签章撤回原因
			recallReason = "PM平台用户撤回签章";
		}

		String rs = service.recallContract(billId, recallReason);
		if(StringUtils.isNotBlank(rs)) {
			return CommonResponse.error(rs);
		}

		return CommonResponse.success("撤回合同成功！");
	}

}
