package com.ejianc.business.signaturemanage.controller.api;

import cn.hutool.core.io.FileUtil;
import cn.hutool.core.io.IoUtil;
import com.baomidou.mybatisplus.core.metadata.IPage;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.ejianc.business.signaturemanage.bean.ManagementEntity;
import com.ejianc.business.signaturemanage.enums.WatermarkSchemeEnum;
import com.ejianc.business.signaturemanage.service.IManagementService;
import com.ejianc.business.signaturemanage.service.ISignatureCommonService;
import com.ejianc.business.signaturemanage.vo.ContractVO;
import com.ejianc.business.signaturemanage.vo.ManagementVO;
import com.ejianc.business.signaturemanage.vo.SignMgrPreviewVO;
import com.ejianc.business.signaturemanage.vo.WatermarkVO;
import com.ejianc.foundation.file.vo.AttachmentVO;
import com.ejianc.foundation.orgcenter.api.IOrgApi;
import com.ejianc.framework.core.context.InvocationInfoProxy;
import com.ejianc.framework.core.kit.mapper.BeanMapper;
import com.ejianc.framework.core.response.CommonResponse;
import com.ejianc.framework.core.response.Parameter;
import com.ejianc.framework.core.response.QueryParam;
import com.google.common.base.Stopwatch;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.http.MediaType;
import org.springframework.util.Assert;
import org.springframework.validation.annotation.Validated;
import org.springframework.web.bind.annotation.*;

import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import java.io.BufferedInputStream;
import java.io.File;
import java.io.IOException;
import java.io.OutputStream;
import java.util.List;
import java.util.concurrent.TimeUnit;

/**
 * @author baipengyan
 * @date 2022/5/17
 * @description 签章通用接口实现控制器
 */
@RestController
@RequestMapping(value = "/common")
public class SignatureCommonApi {

	private final Logger logger = LoggerFactory.getLogger(this.getClass());

	@Autowired
	private ISignatureCommonService service; // 签章通用服务

	@Autowired
	private IManagementService managementService; // 签章管理服务

	@Autowired
	private IOrgApi iOrgApi;


	/**
	 * 合同文件添加水印
	 *
	 * @param watermarkVO 合同文件添加水印参数
	 * @param response    响应
	 */
	@PostMapping(value = "/addWatermark", consumes = MediaType.APPLICATION_JSON_UTF8_VALUE)
	public void addWatermark(@Validated @RequestBody WatermarkVO watermarkVO, HttpServletResponse response) throws IOException {
		watermarkVO = service.addWatermark(watermarkVO);
		File file = watermarkVO.getResultFile();
		Assert.notNull(file, "合同文件添加水印失败！");
		BufferedInputStream in = FileUtil.getInputStream(watermarkVO.getResultFile());
		response.setCharacterEncoding("UTF-8");
		response.setHeader("Content-Disposition", "attachment;fileName=" + file.getName());
		response.setContentType(MediaType.APPLICATION_PDF_VALUE);
		OutputStream out = response.getOutputStream();
		IoUtil.copy(in, out);
		out.flush();
		in.close();
		out.close();
		service.cleanFile(file);
	}

	/**
	 * 获取加水印的合同附件
	 *
	 * @param watermarkVO 合同文件添加水印参数
	 *
	 * @return AttachmentVO 文件中心的水印附件
	 */
	@PostMapping(value = "/fetchWatermarkAttachment")
	public AttachmentVO fetchWatermarkAttachment(@Validated @RequestBody WatermarkVO watermarkVO, HttpServletRequest request) throws IOException {
		logger.info("获取加水印的合同附件，开始");
		Stopwatch start = Stopwatch.createStarted();
		String authority = request.getHeader("authority");
		watermarkVO.setReqAuth(authority);
		Assert.hasText(authority, "获取加水印的合同附件，上下文不能为空！");
		watermarkVO = service.addWatermark(watermarkVO);
		Stopwatch end = start.stop();
		logger.info("添加水印，结束。共计耗时：「{}」秒", end.elapsed(TimeUnit.SECONDS));
		if(!WatermarkSchemeEnum.PM_FILE_SYSTEM.getCode().equals(watermarkVO.getScheme())) {
			Assert.notNull(watermarkVO.getResultFile(), "合同文件添加水印失败！");
			Stopwatch start1 = Stopwatch.createStarted();
			Assert.notNull(watermarkVO.getBillId(), "业务id不能为空！");
			Assert.hasText(watermarkVO.getBillType(), "业务类型不能为空！");
			Assert.hasText(watermarkVO.getSourceType(), "单据类型不能为空！");
			AttachmentVO attachmentVO = service.fetchAttachment(watermarkVO.getResultFile(), authority, watermarkVO);
			watermarkVO.setResultAttach(attachmentVO);
			service.cleanFile(watermarkVO.getResultFile());
			Stopwatch end1 = start1.stop();
			logger.info("获取加水印的合同附件，结束。共计耗时：「{}」秒", end1.elapsed(TimeUnit.SECONDS));
		} else {
			Assert.notNull(watermarkVO.getResultAttach(), "合同文件添加水印失败！");
		}

		return watermarkVO.getResultAttach();
	}


	/**
	 * 根据参数生成水印配置
	 *
	 * @param fileId     要加水印的文件id
	 * @param billId     业务id
	 * @param billCode   合同编码
	 * @param billType   单据类型编码
	 * @param sourceType 业务类型编码
	 *
	 * @return WatermarkVO 水印配置
	 */
	@GetMapping(value = "/fetchWatermarkConfig")
	public WatermarkVO fetchWatermarkConfig(@RequestParam(value = "fileId", required = true) Long fileId,
	                                        @RequestParam(value = "billId", required = true) Long billId,
	                                        @RequestParam(value = "billCode", required = true) String billCode,
	                                        @RequestParam(value = "billType", required = true) String billType,
	                                        @RequestParam(value = "sourceType", required = true) String sourceType) {
		return service.watermarkConfigConvert(fileId, billId, billCode, billType, sourceType);
	}


	/**
	 * 根据合同获取文件中心已签章的合同信息
	 *
	 * @param contracts 合同
	 *
	 * @return List<SignMgrPreviewVO> 已签章的合同信息
	 */
	@PostMapping(value = "/fetchSignedContract")
	public CommonResponse<List<SignMgrPreviewVO>> fetchSignedContract(@Validated @RequestBody List<ContractVO> contracts) {
		return CommonResponse.success(service.fetchSignedContract(contracts));
	}


	@GetMapping(value = "/signatureManagementRef")
	public CommonResponse<IPage<ManagementVO>> signatureManagementRef(@RequestParam(value = "pageNumber", required = true) Integer pageNumber,
	                                   @RequestParam(value = "pageSize", required = true) Integer pageSize,
	                                   @RequestParam(value = "condition", required = false) String condition,
	                                   @RequestParam(value = "searchObject", required = false) String searchObject,
	                                   @RequestParam(value = "searchText", required = false) String searchText) {
		QueryParam param = new QueryParam();
		param.setPageSize(pageSize);
		param.setPageIndex(pageNumber);
		param.setSearchText(searchText);
		param.setSearchObject(searchObject);

		List<String> fuzzyFields = param.getFuzzyFields();
		fuzzyFields.add("sourceSealName");

		// 租户隔离
		param.getParams().put("tenantId", new Parameter(QueryParam.EQ, InvocationInfoProxy.getTenantid()));
		// 组织隔离：查询本上
		param.getParams().put("orgId", new Parameter(QueryParam.IN, iOrgApi.findParentsByOrgId(InvocationInfoProxy.getOrgId())));

		IPage<ManagementEntity> page = managementService.queryPage(param, false);
		IPage<ManagementVO> pageData = new Page<>(page.getCurrent(), page.getSize(), page.getTotal());
		pageData.setRecords(BeanMapper.mapList(page.getRecords(), ManagementVO.class));

		return CommonResponse.success("印章管理参照查询成功！", pageData);
	}
}
