package com.ejianc.business.sub.service.impl;

import com.baomidou.mybatisplus.core.conditions.query.LambdaQueryWrapper;
import com.baomidou.mybatisplus.core.conditions.update.LambdaUpdateWrapper;
import com.baomidou.mybatisplus.core.metadata.IPage;
import com.baomidou.mybatisplus.core.toolkit.Wrappers;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.ejianc.business.budget.vo.cons.CostTypeEnum;
import com.ejianc.business.cost.api.ICostDetailApi;
import com.ejianc.business.cost.vo.CostDetailVO;
import com.ejianc.business.finance.util.BillTypeCodeEnum;
import com.ejianc.business.finance.util.ValidateUtil;
import com.ejianc.business.finance.vo.ParamsCheckDsVO;
import com.ejianc.business.finance.vo.ParamsCheckVO;
import com.ejianc.business.sub.bean.*;
import com.ejianc.business.sub.mapper.FinishMapper;
import com.ejianc.business.sub.service.*;
import com.ejianc.business.sub.utils.BigDecimalUtils;
import com.ejianc.business.sub.utils.TreeNodeBUtil;
import com.ejianc.business.sub.vo.*;
import com.ejianc.business.tax.api.IInvoiceApi;
import com.ejianc.business.utils.ComputeUtil;
import com.ejianc.foundation.orgcenter.api.IOrgApi;
import com.ejianc.foundation.orgcenter.vo.OrgVO;
import com.ejianc.foundation.support.api.IBillCodeApi;
import com.ejianc.foundation.support.api.IParamConfigApi;
import com.ejianc.foundation.support.vo.BillParamVO;
import com.ejianc.framework.auth.session.SessionManager;
import com.ejianc.framework.auth.session.UserContext;
import com.ejianc.framework.core.context.InvocationInfoProxy;
import com.ejianc.framework.core.exception.BusinessException;
import com.ejianc.framework.core.kit.mapper.BeanMapper;
import com.ejianc.framework.core.response.BillStateEnum;
import com.ejianc.framework.core.response.CommonResponse;
import com.ejianc.framework.core.response.Parameter;
import com.ejianc.framework.core.response.QueryParam;
import com.ejianc.framework.skeleton.template.BaseServiceImpl;
import com.fasterxml.jackson.databind.ObjectMapper;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import java.math.BigDecimal;
import java.text.SimpleDateFormat;
import java.util.*;
import java.util.stream.Collectors;

/**
 * <p>
 * 分包完工结算表 服务实现类
 * </p>
 *
 * @author zhangwx
 * @since 2020-06-05
 */
@Service("FinishServiceImpl")
public class FinishServiceImpl extends BaseServiceImpl<FinishMapper, FinishEntity> implements IFinishService {

    private static final String SUB_FINISH_BILL_CODE = "SUB_FINISH";
    private static final String SUB_FINISH = "SUB_FINISH";
    private static final String SUB_FINISH_COST = "SUB_FINISH_COST";
    private static final String CHECK_PARAM_CODE = "P-2452lK17";

    private final Logger logger = LoggerFactory.getLogger(getClass());

    @Autowired
    private IContractService contractService;

    @Autowired
    private IBillCodeApi billCodeApi;

    @Autowired
    private IParamConfigApi paramConfigApi;

    @Autowired
    private IFinishDetailService finishDetailService;

    @Autowired
    private IFinishCostService finishCostService;

    @Autowired
    private ISettleService settleService;

    @Autowired
    private IOddjobService oddJobService;

    @Autowired
    private IOrgApi orgApi;

    @Autowired
    private ICostDetailApi costDetailApi;

    @Autowired
    private SessionManager sessionManager;

    @Autowired
    private IInvoiceApi invoiceApi;


    @Override
    public FinishVO insertOrUpdate(FinishVO finishVO) {
        if (checkSameContract(finishVO)) {
            throw new BusinessException("该合同在相同组织下已经做过完工结算！");
        }
        if (checkUnEffectSettle(finishVO)) {
            throw new BusinessException("该合同与组织下存在未生效过程结算单，不允许新增！");
        }
        if (checkUnEffectOddjob(finishVO)) {
            throw new BusinessException("该合同与组织下存在未生效零星用工单，不允许新增！");
        }
        if (!(finishVO.getId() != null && finishVO.getId() > 0)) { //新增
            if (checkFinishContract(finishVO)) {
                throw new BusinessException("该合同在相同组织下已经完工，不允许新增！");
            }
        }

        // 保存时校验合同version是否一致
        if (!ValidateUtil.validateUpstreamVersion(String.valueOf(finishVO.getContractId()),
                BillTypeCodeEnum.分包合同.getBillTypeCode(), finishVO.getContractVersion())) {
            throw new BusinessException("该合同已被更新，请刷新后重做！");
        }

        if (StringUtils.isEmpty(finishVO.getBillCode())) {
            CommonResponse<String> billCode = billCodeApi.getCodeBatchByRuleCode(SUB_FINISH_BILL_CODE, InvocationInfoProxy.getTenantid());
            if (billCode.isSuccess()) {
                finishVO.setBillCode(billCode.getData());
            } else {
                throw new BusinessException("网络异常， 编码生成失败， 请稍后再试");
            }
        }

        if (checkSameBillCode(finishVO)) {
            throw new BusinessException("存在相同编码，不允许保存!");
        }
        finishVO.setRelationFlag("0");
        finishVO.setProportionFlag("0");
        FinishEntity finishEntity = BeanMapper.map(finishVO, FinishEntity.class);
        //如果该结算单有结算成本明细，并且金额不等于主表结算差额，不允许保存
        List<FinishCostEntity> finishCostList = finishEntity.getFinishCostList();
        if(CollectionUtils.isNotEmpty(finishCostList)){
            BigDecimal money = finishCostList.stream().map(FinishCostEntity::getMoney).reduce(BigDecimal.ZERO, BigDecimal::add);
            BigDecimal shouldPayTaxMny = finishEntity.getShouldPayTaxMny();
            if(money.compareTo(shouldPayTaxMny) != 0){
                throw new BusinessException("结算差额和成本明细金额不相等，不允许保存!");
            }
        }

        //保存前清空主键和父主键，重新生成
        List<FinishDetailEntity> beforeDetails = finishEntity.getFinishDetailList();
        if (CollectionUtils.isNotEmpty(beforeDetails) && null == finishEntity.getId()) {
            for (FinishDetailEntity cdEntity : beforeDetails) {
                cdEntity.setId(null);
                cdEntity.setParentId(null);
            }
        }

        CommonResponse<BigDecimal> invoiceReceiceMnyRes = invoiceApi.getInvoiceReceiceMny(finishEntity.getContractId(), finishEntity.getOrgId());
        if (invoiceReceiceMnyRes.isSuccess()) {
            finishEntity.setSumInvoiceTaxMny(ComputeUtil.nullToZero(invoiceReceiceMnyRes.getData()));
        }

        //保存修改合同，原合同完工状态要恢复过来
        if (finishEntity.getId() != null) {
            FinishEntity oldFinish = this.getById(finishEntity.getId());
            Long oldContractId = oldFinish.getContractId();
            if (oldContractId != null && !oldContractId.equals(finishEntity.getContractId())) {
                this.writeBackContractFinsishFlag(oldFinish, Boolean.FALSE);
            }
        }

        super.saveOrUpdate(finishEntity, false);

        List<FinishDetailEntity> finishDetailEntities = finishEntity.getFinishDetailList();
        if (CollectionUtils.isNotEmpty(finishDetailEntities)) {
            Map<String, Long> idMap = new HashMap<>();
            for (FinishDetailEntity cdEntity : finishDetailEntities) {
                idMap.put(cdEntity.getTid(), cdEntity.getId());
            }
            for (FinishDetailEntity cdEntity : finishDetailEntities) {
                if (StringUtils.isNotEmpty(cdEntity.getTpid())) {
                    cdEntity.setParentId(idMap.get(cdEntity.getTpid()));
                }
            }
            finishDetailService.saveOrUpdateBatch(finishDetailEntities, finishDetailEntities.size(), false);
        }

        // 回写合同完工结算状态
        this.writeBackContractFinsishFlag(finishEntity, Boolean.TRUE);
        return queryDetail(finishEntity.getId());
    }

    private Boolean checkSameBillCode(FinishVO finishVO) {
        Long tenantId = InvocationInfoProxy.getTenantid();
        LambdaQueryWrapper<FinishEntity> lambda = new LambdaQueryWrapper<>();
        lambda.eq(FinishEntity::getBillCode, finishVO.getBillCode());
        lambda.eq(FinishEntity::getTenantId, tenantId);
        if (null != finishVO.getId() && finishVO.getId() > 0) {
            lambda.ne(FinishEntity::getId, finishVO.getId());
        }
        return super.list(lambda).size() > 0;
    }

    private Boolean checkSameContract(FinishVO finishVO) {
        Long tenantId = InvocationInfoProxy.getTenantid();
        LambdaQueryWrapper<FinishEntity> lambda = new LambdaQueryWrapper<>();
        lambda.eq(FinishEntity::getContractId, finishVO.getContractId());
        lambda.eq(FinishEntity::getOrgId, finishVO.getOrgId());
        lambda.eq(FinishEntity::getTenantId, tenantId);
        if (null != finishVO.getId() && finishVO.getId() > 0) {
            lambda.ne(FinishEntity::getId, finishVO.getId());
        }
        return super.list(lambda).size() > 0;
    }

    private Boolean checkUnEffectSettle(FinishVO finishVO) {
        Long tenantId = InvocationInfoProxy.getTenantid();
        LambdaQueryWrapper<SettleEntity> lambda = new LambdaQueryWrapper<>();
        lambda.eq(SettleEntity::getContractId, finishVO.getContractId());
        lambda.eq(SettleEntity::getOrgId, finishVO.getOrgId());
        lambda.notIn(SettleEntity::getBillState, BillStateEnum.PASSED_STATE.getBillStateCode(), BillStateEnum.COMMITED_STATE.getBillStateCode());
        lambda.eq(SettleEntity::getTenantId, tenantId);
        return settleService.list(lambda).size() > 0;
    }

    private Boolean checkUnEffectOddjob(FinishVO finishVO) {
        Long tenantId = InvocationInfoProxy.getTenantid();
        LambdaQueryWrapper<OddjobEntity> lambda = new LambdaQueryWrapper<>();
        lambda.eq(OddjobEntity::getContractId, finishVO.getContractId());
        lambda.eq(OddjobEntity::getOrgId, finishVO.getOrgId());
        lambda.notIn(OddjobEntity::getBillState, BillStateEnum.PASSED_STATE.getBillStateCode(), BillStateEnum.COMMITED_STATE.getBillStateCode());
        lambda.eq(OddjobEntity::getTenantId, tenantId);
        return oddJobService.list(lambda).size() > 0;
    }

    private Boolean checkFinishContract(FinishVO finishVO) {
        Long tenantId = InvocationInfoProxy.getTenantid();
        LambdaQueryWrapper<ContractEntity> lambda = new LambdaQueryWrapper<>();
        lambda.eq(ContractEntity::getId, finishVO.getContractId());
        lambda.eq(ContractEntity::getTenantId, tenantId);
        lambda.eq(ContractEntity::getFinishFlag, true);
        return contractService.list(lambda).size() > 0;
    }

    @Override
    public FinishVO queryDetail(Long id) {
        FinishEntity finishEntity = super.selectById(id);
        FinishVO finishVO = BeanMapper.map(finishEntity, FinishVO.class);
        QueryParam queryParam = new QueryParam();
        queryParam.getParams().put("finishId", new Parameter(QueryParam.EQ, id));
        queryParam.getOrderMap().put("treeIndex", "asc");
        List<FinishDetailEntity> detailEntityList = finishDetailService.queryList(queryParam, false);
        if (CollectionUtils.isNotEmpty(detailEntityList)) {

            for (FinishDetailEntity cdEntity : detailEntityList) {
                cdEntity.setTid(cdEntity.getId().toString());
                cdEntity.setTpid(cdEntity.getParentId() != null && cdEntity.getParentId() > 0 ? cdEntity.getParentId().toString() : "");
                cdEntity.setRowState("edit");
            }
            List<FinishDetailVO> resultMapList = BeanMapper.mapList(detailEntityList, FinishDetailVO.class);
            sortIntMethod(resultMapList);
            List<FinishDetailVO> finishDetailVOS = TreeNodeBUtil.buildTree(resultMapList);
            finishVO.setFinishDetailList(finishDetailVOS);
        }
        // 【未付金额】=【完工结算金额】-【已付金额】
        finishVO.setUnPaidMoney(ComputeUtil.safeSub(finishEntity.getSettleTaxMny(), finishEntity.getPaidMoney()));
        return finishVO;
    }
    private static void sortIntMethod(List<FinishDetailVO> list){
        Collections.sort(list, new Comparator(){
            @Override
            public int compare(Object o1, Object o2) {
                ObjectMapper objectMapper = new ObjectMapper();
                FinishDetailVO o1Detail = objectMapper.convertValue(o1, FinishDetailVO.class);
                FinishDetailVO o2Detail = objectMapper.convertValue(o2, FinishDetailVO.class);
                String[] strs1 = o1Detail.getTreeIndex().split("\\.");
                String[] strs2 = o2Detail.getTreeIndex().split("\\.");
                int length = strs1.length > strs2.length ? strs1.length : strs2.length;
                for (int i = 0; i < length; i++) {
                    int num1 = 0;
                    int num2 = 0;
                    try {
                        num1 = Integer.parseInt(strs1[i]);
                        num2 = Integer.parseInt(strs2[i]);
                    } catch (Exception e) {

                    }
                    if (num1 < num2)
                        return -1;
                    if (num1 > num2)
                        return 1;
                }
                return 0;
            }
        });
    }
    @Override
    public CommonResponse<IPage<FinishVO>> queryListVOs(QueryParam param) {
        /** 模糊搜索配置字段示例 */
        List<String> fuzzyFields = param.getFuzzyFields();
        fuzzyFields.add("orgName");
        fuzzyFields.add("billCode");
        fuzzyFields.add("contractName");
//        fuzzyFields.add("customerName");
        fuzzyFields.add("supplierName");
        fuzzyFields.add("employeeName");
        fuzzyFields.add("memo");
        param.getParams().put("tenantId", new Parameter(QueryParam.EQ, InvocationInfoProxy.getTenantid()));
        // 组织本下
        UserContext userContext = sessionManager.getUserContext();
        String authOrgIds = userContext.getAuthOrgIds();
        if (StringUtils.isNotEmpty(authOrgIds)) {
            CommonResponse<List<OrgVO>> authResponse =
                    orgApi.findChildrenByParentIds(Arrays.stream(authOrgIds.split(",")).map(Long::parseLong).
                            collect(Collectors.toList()));
            param.getParams().put("orgId", new Parameter(QueryParam.IN, authResponse.getData().stream().map(OrgVO::getId).collect(Collectors.toList())));
        } else {
            param.getParams().put("orgId", new Parameter(QueryParam.IN, orgApi.findChildrenByParentId(InvocationInfoProxy.getOrgId()).getData().stream().map(OrgVO::getId).collect(Collectors.toList())));
        }
        param.getOrderMap().put("createTime", QueryParam.DESC);
        IPage<FinishEntity> page = queryPage(param, false);
        List<FinishVO> finishVOList = BeanMapper.mapList(page.getRecords(), FinishVO.class);
        IPage<FinishVO> finishVOIPage = new Page<>();
        finishVOIPage.setCurrent(page.getCurrent());
        finishVOIPage.setRecords(finishVOList);
        finishVOIPage.setSize(page.getSize());
        finishVOIPage.setTotal(page.getTotal());
        finishVOIPage.setPages(page.getPages());
        return CommonResponse.success("查询成功！", finishVOIPage);
    }

    @Override
    public CommonResponse<String> deleteByIds(List<FinishVO> vos) {
        List<Long> ids = vos.stream().map(FinishVO::getId).collect(Collectors.toList());
        // 回写合同完工结算状态
        List<FinishEntity> entityList = BeanMapper.mapList(super.listByIds(ids), FinishEntity.class);
        for (FinishEntity finishEntity : entityList) {
            this.writeBackContractFinsishFlag(finishEntity, Boolean.FALSE);
        }
        if (CollectionUtils.isNotEmpty(vos)) {
            this.removeByIds(ids, false);
        }
        return CommonResponse.success("删除成功！");
    }

    @Override
    public FinishRecordVO queryDetailRecord(Long id) {
        ContractEntity contractEntity = contractService.selectById(id);
        FinishRecordVO finishRecordVO = new FinishRecordVO();
        BigDecimal contractTaxMny = contractEntity.getContractTaxMny() == null ? BigDecimal.ZERO : contractEntity.getContractTaxMny();
        BigDecimal sumSettleTaxMny = (contractEntity.getSumSettleTaxMny() == null || !contractEntity.getFinishFlag()) ? BigDecimal.ZERO : contractEntity.getSumSettleTaxMny();
        BigDecimal sumScale = BigDecimalUtils.safeDiv(sumSettleTaxMny, contractTaxMny).multiply(new BigDecimal(100));
        finishRecordVO.setContractId(id);
        finishRecordVO.setFinishFlag(contractEntity.getFinishFlag());
        finishRecordVO.setContractStatus(Integer.valueOf(contractEntity.getContractStatus()));
        finishRecordVO.setPurchaseType(Integer.valueOf(contractEntity.getPurchaseType()));
        finishRecordVO.setContractName(contractEntity.getContractName());
        finishRecordVO.setSupplierName(contractEntity.getSupplierName());
        finishRecordVO.setContractTaxMny(contractTaxMny);
        finishRecordVO.setSumSettleTaxMny(sumSettleTaxMny);
        finishRecordVO.setSumScale(sumScale);

        LambdaQueryWrapper<FinishEntity> lambda = Wrappers.<FinishEntity>lambdaQuery();
        lambda.eq(FinishEntity::getContractId, id);
        List<FinishEntity> entitiesAllState = super.list(lambda);
        lambda.in(FinishEntity::getBillState, BillStateEnum.PASSED_STATE.getBillStateCode(), BillStateEnum.COMMITED_STATE.getBillStateCode());
        List<FinishEntity> entities = super.list(lambda);
        List<FinishRecordDetailVO> finishRecordDetailVOList = BeanMapper.mapList(entities, FinishRecordDetailVO.class);
        if (CollectionUtils.isNotEmpty(finishRecordDetailVOList)) {
            finishRecordDetailVOList.forEach(entity -> {
                BigDecimal settleTaxMny = entity.getSettleTaxMny() == null ? BigDecimal.ZERO : entity.getSettleTaxMny();
                BigDecimal scale = contractTaxMny == BigDecimal.ZERO ? BigDecimal.ZERO : (ComputeUtil.safeDiv(settleTaxMny,contractTaxMny)).multiply(new BigDecimal(100));
                entity.setSumScale(scale);
            });
        }
        finishRecordVO.setDetailList(BeanMapper.mapList(entitiesAllState, FinishRecordDetailVO.class));

        LambdaQueryWrapper<SettleEntity> lambda2 = Wrappers.<SettleEntity>lambdaQuery();
        lambda2.eq(SettleEntity::getContractId, id);
        lambda2.in(SettleEntity::getBillState, BillStateEnum.PASSED_STATE.getBillStateCode(), BillStateEnum.COMMITED_STATE.getBillStateCode());
        List<SettleEntity> settleList = settleService.list(lambda2);
        // 过程结算金额
        BigDecimal settleTaxMny = new BigDecimal(0);
        for (SettleEntity settle : settleList) {
            settleTaxMny = settleTaxMny.add(settle.getSettleTaxMny() == null ? BigDecimal.ZERO : settle.getSettleTaxMny());
        }
        finishRecordVO.setSettleTaxMny(settleTaxMny);
        finishRecordVO.setSettleBalanceMny(sumSettleTaxMny.subtract(settleTaxMny));
        return finishRecordVO;
    }

    @Override
    public FinishVO queryDetailAdd(Long contractId) {
        ContractEntity contractEntity = contractService.selectById(contractId);
        FinishVO finishVO = BeanMapper.map(contractEntity, FinishVO.class);
        finishVO.setBillCode(null);
        finishVO.setEmployeeId(Long.valueOf(InvocationInfoProxy.getEmployeeId()));
        finishVO.setEmployeeName(null);
        finishVO.setContractId(contractEntity.getId());
        finishVO.setBillState(null);
        finishVO.setCreateUserCode(null);
        finishVO.setCreateTime(null);
        finishVO.setUpdateUserCode(null);
        finishVO.setUpdateTime(null);
        finishVO.setId(null);
        finishVO.setVersion(null);
        finishVO.setTaxMny(null);
        finishVO.setSumInvoiceTaxMny(null);
        finishVO.setContractVersion(contractEntity.getVersion());
        return finishVO;
    }

    @Override
    public List<FinishVO> queryExportList(QueryParam param) {
        param.setPageIndex(0);
        param.setPageSize(-1);
        List<FinishVO> resVos = queryListVOs(param).getData().getRecords();
        if (!resVos.isEmpty()) {
            for (int i = 0; i < resVos.size(); i++) {
                FinishVO vo = resVos.get(i);
                if (vo.getContractType() != null) {
                    if (vo.getContractType().equals(Long.valueOf("1270328729526124545"))) {
                        vo.setContractTypeName("专业分包");
                    } else if (vo.getContractType().equals(Long.valueOf("1270328674299723778"))) {
                        vo.setContractTypeName("劳务分包");
                    }
                }
                vo.setBillStateName(BillStateEnum.getEnumByStateCode(vo.getBillState()).getDescription());
            }
            ;
        }
        return resVos;
    }

    @Override
    public CommonResponse<FinishVO> pushCost(FinishVO finishVO) {
        FinishEntity finishEntity = baseMapper.selectById(finishVO.getId());
        //如果该结算单有结算成本明细，并且金额不等于主表结算差额，不允许保存
        if(CollectionUtils.isNotEmpty(finishVO.getFinishCostList())){
            List<FinishCostVO> finishCostList = finishVO.getFinishCostList();
            BigDecimal money = finishCostList.stream().map(FinishCostVO::getMoney).reduce(BigDecimal.ZERO, BigDecimal::add);
            BigDecimal shouldPayTaxMny = finishEntity.getShouldPayTaxMny();
            if(money.compareTo(shouldPayTaxMny) != 0){
                throw new BusinessException("结算差额和成本明细金额不相等，不允许保存!");
            }
        }
        if (CollectionUtils.isNotEmpty(finishVO.getFinishCostList())) {
            List<FinishCostEntity> costEntities = BeanMapper.mapList(finishVO.getFinishCostList(), FinishCostEntity.class);
            finishEntity.setFinishCostList(costEntities);
        }
        super.saveOrUpdate(finishEntity, false);
        //推送数据
        costPush(finishEntity);
        return CommonResponse.success(queryDetail(finishVO.getId()));
    }

    @Override
    public void costPush(FinishEntity finishEntity) {
        Boolean isPush = true;
        LambdaQueryWrapper<FinishCostEntity> costWrapper = new LambdaQueryWrapper<>();
        costWrapper.eq(FinishCostEntity::getFinishId, finishEntity.getId());
        List<FinishCostEntity> costEntities = finishCostService.list(costWrapper);
        if (CollectionUtils.isEmpty(costEntities)) {
            isPush = false;
        } else if (isPush) {
            BigDecimal totalMny = BigDecimal.ZERO;
            for (FinishCostEntity costEntity : costEntities) {
                if (null == costEntity.getSubjectId()) {
                    isPush = false;
                    break;
                } else {
                    BigDecimal money = null != costEntity.getMoney() ? costEntity.getMoney() : BigDecimal.ZERO;
                    totalMny = totalMny.add(money);
                }
            }
            if (null == finishEntity.getShouldPayTaxMny() || totalMny.compareTo(finishEntity.getShouldPayTaxMny()) != 0) {
                isPush = false;
            }
        }

        finishEntity.setFinishCostList(costEntities);

        //更新是否关联
        LambdaUpdateWrapper<FinishEntity> updateWrapper = new LambdaUpdateWrapper<>();
        updateWrapper.eq(FinishEntity::getId, finishEntity.getId());
        updateWrapper.set(FinishEntity::getRelationFlag, isPush ? "1" : "0");//(1:是，0：否)
        super.update(updateWrapper);

        //判断之前的单据是否关联
        String oldRelationFlag = finishEntity.getRelationFlag();
        //之前已关联
        if (oldRelationFlag.equals("1")) {
            if (isPush) {
                saveCost(finishEntity);
            }
            if (!isPush) {
                //删除成本中心之前的数据
                costDetailApi.deleteSubject(finishEntity.getId());
            }
        }
        //之前未关联
        if (oldRelationFlag.equals("0")) {
            //if (isPush) {
                saveCost(finishEntity);
            //}
        }
    }

    @Override
    public void pullCost(Long id) {
        //更新是否关联
        LambdaUpdateWrapper<FinishEntity> updateWrapper = new LambdaUpdateWrapper<>();
        updateWrapper.eq(FinishEntity::getId, id);
        updateWrapper.set(FinishEntity::getRelationFlag, "0");//(1:是，0：否)
        super.update(updateWrapper);

        costDetailApi.deleteSubject(id);
    }

    @Override
    public ParamsCheckVO checkParams(Integer purchaseType, BigDecimal settleTaxMny, BigDecimal contractTaxMny,Long orgId) {
        Long curOrgId = Optional.ofNullable(orgId).orElse(InvocationInfoProxy.getOrgId());
        // 存放预警结果
        Map<String, List<ParamsCheckDsVO>> paramsCheckVOMap = new HashMap<>();
        paramsCheckVOMap.put("alert",new ArrayList<>());
        paramsCheckVOMap.put("warn",new ArrayList<>());
        // 三种控制方式：不控制，提醒，无法保存 (默认为提醒)
        String[] paramsArray = {"none", "warn", "alert"};
        ParamsCheckVO paramsCheckVO = new ParamsCheckVO();
        List<ParamsCheckDsVO> checkDsVOS = new ArrayList<>();
        CommonResponse<List<BillParamVO>> billParamByCode = paramConfigApi.getBillParamByCodeAndOrgId(CHECK_PARAM_CODE, curOrgId);
        if (billParamByCode.isSuccess() && null != billParamByCode.getData()) {
            List<BillParamVO> data = billParamByCode.getData();
            for (BillParamVO datum : data) {
                BigDecimal roleValue = datum.getRoleValue();
                BigDecimal comMny = contractTaxMny.multiply(roleValue.divide(BigDecimal.valueOf(100)
                        , 2, BigDecimal.ROUND_HALF_UP));

                if (2 == purchaseType) {
                    // 集采合同直接赋值为不控制
                    paramsCheckVO.setWarnType(paramsArray[0]);
                    return paramsCheckVO;
                } else {
                    paramsCheckVO.setWarnType(paramsArray[datum.getControlType()]);
                }

                if (settleTaxMny.compareTo(comMny) > 0) {
                    ParamsCheckDsVO paramsCheckDsVO = new ParamsCheckDsVO();
                    paramsCheckDsVO.setOrgName(datum.getOrgName());
                    paramsCheckDsVO.setWarnItem("合同超结");
                    paramsCheckDsVO.setWarnName("完工结算金额大于合同金额");
                    StringBuffer stringBuffer = new StringBuffer();
                    stringBuffer.append("本次结算金额：").append(settleTaxMny.toString())
                            .append("元，合同金额*").append(roleValue).append("%:")
                            .append(comMny).append("元。超出金额：")
                            .append(settleTaxMny.subtract(comMny)).append("元");
                    paramsCheckDsVO.setContent(stringBuffer.toString());
                    checkDsVOS.add(paramsCheckDsVO);
                    updateParamsCheckVOMap(paramsArray, paramsCheckVOMap, datum, paramsCheckDsVO);
                }
            }
        } else {
            logger.info(billParamByCode.getMsg());
            throw new BusinessException("获取控制参数失败");
        }

        //处理刚性、柔性预警
        ParamsCheckVO pc = new ParamsCheckVO();
        if (CollectionUtils.isNotEmpty(paramsCheckVOMap.get("alert"))) {
            pc.setWarnType("alert");
            pc.setDataSource(paramsCheckVOMap.get("alert"));
        } else if (CollectionUtils.isNotEmpty(paramsCheckVOMap.get("warn"))) {
            pc.setWarnType("warn");
            pc.setDataSource(paramsCheckVOMap.get("warn"));
        } else {
            pc.setWarnType("none");
            pc.setDataSource(null);
        }
        return pc;
    }

    /**
     * 更新参数控制结果
     *
     * @param paramsArray      参数数组
     * @param paramsCheckVOMap 预警结果map
     * @param billParamVO      控制参数
     * @param paramsCheckDsVO  预警信息
     */
    private static void updateParamsCheckVOMap(String[] paramsArray, Map<String, List<ParamsCheckDsVO>> paramsCheckVOMap, BillParamVO billParamVO, ParamsCheckDsVO paramsCheckDsVO) {
        if ("alert".equals(paramsArray[billParamVO.getControlType()])) {
            List<ParamsCheckDsVO> alert = paramsCheckVOMap.get("alert");
            alert.add(paramsCheckDsVO);
        }
        if ("warn".equals(paramsArray[billParamVO.getControlType()])) {
            List<ParamsCheckDsVO> warn = paramsCheckVOMap.get("warn");
            warn.add(paramsCheckDsVO);
        }
    }

    private void saveCost(FinishEntity finishEntity) {
        //税率
        BigDecimal taxRate = finishEntity.getTaxRate().divide(new BigDecimal(100));
        BigDecimal number = taxRate.add(new BigDecimal(1));
        //明细
        List<CostDetailVO> costDetailVOList = new ArrayList<CostDetailVO>();

        //费用
        List<FinishCostEntity> finishCostList = finishEntity.getFinishCostList();
        ContractEntity contractEntity = contractService.selectById(finishEntity.getContractId());
        Long contractType = contractEntity.getContractType();
        String contractTypeName = "";
        Integer costType = null;
        String sourceType = SUB_FINISH + "_PRO";
        //劳务分包
        if(contractType.equals(1270328729526124545l)){
            sourceType = SUB_FINISH + "_LAB";
            contractTypeName = CostTypeEnum.LABOR_COST_TYPE.getName();
            costType = CostTypeEnum.LABOR_COST_TYPE.getType();
        }
        //专业分包
        else {
            contractTypeName = CostTypeEnum.MAJOR_COST_TYPE.getName();
            costType = CostTypeEnum.MAJOR_COST_TYPE.getType();
        }
        String period = "";
        if(finishEntity.getSettleDate() != null){
            SimpleDateFormat ft = new SimpleDateFormat("yyyy-MM");
            period = ft.format(finishEntity.getSettleDate());
        }
        logger.info("----------------------------finishCostList:"+finishCostList.size());
        if (CollectionUtils.isNotEmpty(finishCostList) && finishCostList.size()>0) {
            for (FinishCostEntity costEntity : finishCostList) {
                CostDetailVO costDetailVO = new CostDetailVO();
                costDetailVO.setSubjectId(costEntity.getSubjectId());
                costDetailVO.setSourceId(costEntity.getFinishId());
                costDetailVO.setSourceDetailId(costEntity.getId());
                costDetailVO.setHappenTaxMny(costEntity.getMoney());
                //无税值计算
                BigDecimal pushSettlementMny = costEntity.getMoney();
                BigDecimal happenMny = pushSettlementMny.divide(number, 2, BigDecimal.ROUND_HALF_UP);
                costDetailVO.setHappenMny(happenMny);
                costDetailVO.setHappenDate(finishEntity.getSettleDate());
                costDetailVO.setCreateUserName(sessionManager.getUserContext().getUserName());
                costDetailVO.setSourceType(sourceType);
                costDetailVO.setSourceTabType(SUB_FINISH_COST);
                costDetailVO.setProjectId(finishEntity.getProjectId());

                costDetailVO.setCostType(costType);
                costDetailVO.setCostTypeName(contractTypeName);
                costDetailVO.setPeriod(period);
                costDetailVO.setShareFlag(0);
                costDetailVO.setSourceBillCode(finishEntity.getBillCode());
                costDetailVO.setSourceBillName(finishEntity.getContractName());
                costDetailVO.setSourceBillUrl("/ejc-sub-frontend/#/finish/card?id="+finishEntity.getId());
                //costDetailVO.setNum(0);
                costDetailVOList.add(costDetailVO);
            }
        }else{
            logger.info("---------------------------------------------子表成本分摊页签数据为空，推送主表-----------------------");
            //TODO  推送主表数据  组装一下推一条就行
            CostDetailVO costDetailVO = new CostDetailVO();
            costDetailVO.setSourceId(finishEntity.getId());
            costDetailVO.setHappenTaxMny(finishEntity.getShouldPayTaxMny());
            costDetailVO.setHappenMny(finishEntity.getShouldPayMny());
            costDetailVO.setCreateUserName(sessionManager.getUserContext().getUserName());
            costDetailVO.setSourceType(sourceType);
            //costDetailVO.setSourceTabType(SUB_FINISH); //推送主表数据不需要传子表类型
            costDetailVO.setProjectId(finishEntity.getProjectId());
            costDetailVO.setCostType(costType);
            costDetailVO.setCostTypeName(contractTypeName);
            costDetailVO.setPeriod(period);
            costDetailVO.setShareFlag(0);
            costDetailVO.setSourceBillCode(finishEntity.getBillCode());
            costDetailVO.setSourceBillName(finishEntity.getContractName());
            costDetailVO.setSourceBillUrl("/ejc-sub-frontend/#/finish/card?id="+finishEntity.getId());
            costDetailVOList.add(costDetailVO);
        }

        //成本中心
        if (CollectionUtils.isNotEmpty(costDetailVOList)) {
            CommonResponse<String> stringCommonResponse = costDetailApi.saveSubject(costDetailVOList);
            logger.info("----------------------完工结算推送结果："+stringCommonResponse.isSuccess());
            if (stringCommonResponse.isSuccess()) {
            } else {
                throw new BusinessException(stringCommonResponse.getMsg());
            }
        }

    }

    /**
     * 回写合同完工结算状态
     *
     * @param finishEntity
     * @param finishFlag
     */
    private void writeBackContractFinsishFlag(FinishEntity finishEntity, Boolean finishFlag) {
        LambdaUpdateWrapper<ContractEntity> updateWrapper = new LambdaUpdateWrapper<>();
        //根据完工状态回写合同状态，合同状态3已封账，2履约中
        updateWrapper.set(ContractEntity::getContractStatus, finishFlag ? "3" : "2");
        updateWrapper.set(ContractEntity::getFinishFlag, finishFlag);
        updateWrapper.eq(ContractEntity::getId, finishEntity.getContractId());
        ContractEntity contractEntity = contractService.getById(finishEntity.getContractId());
        contractService.update(contractEntity, updateWrapper, false);
    }
}
