package com.ejianc.business.sub.controller.api;

import com.baomidou.mybatisplus.core.conditions.query.LambdaQueryWrapper;
import com.baomidou.mybatisplus.core.toolkit.Wrappers;
import com.ejianc.business.finance.util.MathUtil;
import com.ejianc.business.sub.bean.ContractEntity;
import com.ejianc.business.sub.bean.FinishEntity;
import com.ejianc.business.sub.bean.SettleEntity;
import com.ejianc.business.sub.service.IContractService;
import com.ejianc.business.sub.service.IFinishService;
import com.ejianc.business.sub.service.ISettleService;
import com.ejianc.business.sub.vo.SettleVO;
import com.ejianc.business.sub.vo.SubSettleVO;
import com.ejianc.framework.core.context.InvocationInfoProxy;
import com.ejianc.framework.core.response.BillStateEnum;
import com.ejianc.framework.core.response.CommonResponse;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.web.bind.annotation.GetMapping;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.RestController;

import java.math.BigDecimal;
import java.util.List;

@RestController
@RequestMapping("/api/subForPay/")
public class SubContractForPayApi {
	private Logger logger = LoggerFactory.getLogger(this.getClass());

	@Autowired
	IContractService service;

	@Autowired
	private ISettleService settleService;

	@Autowired
	private IFinishService finishService;

	/**
	 * @Author yqls
	 * @Date 2020/7/4
	 * @Description 回写分包合同累计付款金额、累计预付款金额
	 * @Param contractId  合同id
	 * @Param payMny  本期已付款金额，正为回写，负为逆回写
	 * @param prepayMny 本期预付款金额，正为回写，负为逆回写
	 * @Return com.ejianc.framework.core.response.CommonResponse<java.lang.String>
	 */
	@GetMapping(value = "updateSubContractSumPayMny")
	public CommonResponse<String> updateSubContractSumPayMny(
			@RequestParam(value = "contractId", required = true) Long contractId,
			@RequestParam("payMny") BigDecimal payMny,
			@RequestParam("prepayMny") BigDecimal prepayMny) {
		ContractEntity entity = service.getById(contractId);
		if (entity == null) {
			return CommonResponse.error("没有找到相应分包合同！");
		}
		//累计付款金额
		BigDecimal sumPayMny = entity.getSumPayMny();
		sumPayMny = MathUtil.safeAdd(sumPayMny, payMny);
		entity.setSumPayMny(sumPayMny);
		//累计预付款金额
		BigDecimal sumPrepayMny = entity.getSumPrepayMny();
		sumPrepayMny = MathUtil.safeAdd(sumPrepayMny, prepayMny);
		entity.setSumPrepayMny(sumPrepayMny);
		service.saveOrUpdate(entity, false);
		return CommonResponse.success("回写分包合同累计付款金额、累计预付款金额成功！");
	}

	/**
	 * @Author yqls
	 * @Date 2020/7/4
	 * @Description 回写分包结算单累计付款金额
	 * @Param settleId  结算单id
	 * @Param payMny  本期已付款金额，正为回写，负为逆回写
	 * @Return com.ejianc.framework.core.response.CommonResponse<java.lang.String>
	 */
	@GetMapping(value = "updateSubSettleSumPayMny")
	public CommonResponse<String> updateSubSettleSumPayMny(
			@RequestParam(value = "settleId", required = true) Long settleId,
			@RequestParam("payMny") BigDecimal payMny) {
		SettleEntity settle = settleService.getById(settleId);//过程结算
		FinishEntity finish = finishService.getById(settleId);//完工结算
		//累计付款金额
		if(settle != null){
			BigDecimal sumPayMny = settle.getSumPayMny();
			sumPayMny = MathUtil.safeAdd(sumPayMny, payMny);
			settle.setSumPayMny(sumPayMny);
			settleService.saveOrUpdate(settle, false);
			return CommonResponse.success("回写分包结算单累计付款金额成功！");
		} else {
			if(finish != null){
				return CommonResponse.success("该结算单为分包完工结算单！");
			} else {
				return CommonResponse.error("没有找到相应分包结算单！");
			}
		}
	}

	/**
	 * @Author yqls
	 * @Date 2020/7/4
	 * @Description 回写分包结算单累计付款金额、剩余可申请金额
	 * @Param settleId  结算单id
	 * @Param applyMny  本期申请金额，正为回写，负为逆回写
	 * @Return com.ejianc.framework.core.response.CommonResponse<java.lang.String>
	 */
	@GetMapping(value = "updateSubSettleSumApplyMny")
	public CommonResponse<String> updateSubSettleSumApplyMny(
			@RequestParam(value = "settleId", required = true) Long settleId,
			@RequestParam("applyMny") BigDecimal applyMny) {
		SettleEntity settle = settleService.getById(settleId);//过程结算
		FinishEntity finish = finishService.getById(settleId);//完工结算
		if(settle != null){
			//累计申请金额
			BigDecimal sumApplyMny = settle.getSumApplyMny();
			sumApplyMny = MathUtil.safeAdd(sumApplyMny, applyMny);
			settle.setSumApplyMny(sumApplyMny);
			//剩余可申请金额
			BigDecimal surplusApplyMny = MathUtil.safeSub(
					MathUtil.safeSub(settle.getSettleTaxMny(), settle.getOffsetMny()), applyMny);
			settle.setSurplusApplyMny(surplusApplyMny);
			settleService.saveOrUpdate(settle, false);
			return CommonResponse.success("回写分包结算单累计申请金额、剩余可申请金额成功！");
		} else {
			if(finish != null){
				//累计申请金额
				BigDecimal sumApplyMny = finish.getSumApplyMny();
				sumApplyMny = MathUtil.safeAdd(sumApplyMny, applyMny);
				finish.setSumApplyMny(sumApplyMny);
				//剩余可申请金额
				BigDecimal surplusApplyMny = MathUtil.safeSub(finish.getSettleTaxMny(), applyMny);
				finish.setSurplusApplyMny(surplusApplyMny);
				finishService.saveOrUpdate(finish, false);
				return CommonResponse.success("回写完工结算单累计申请金额、剩余可申请金额成功！");
			} else {
				return CommonResponse.error("没有找到相应分包结算单！");
			}
		}
	}

	/**
	 * @Author yqls
	 * @Date 2020/7/16
	 * @Description 根据条件查询结算单累计冲抵金额
	 * @Param contractId  合同id
	 * @Param orgId  组织id
	 */
	@GetMapping(value = "getSubSettleById")
	public CommonResponse<SubSettleVO> getSubSettleById(
			@RequestParam(value = "contractId", required = true) Long contractId,
			@RequestParam(value = "orgId", required = true) Long orgId) {
		LambdaQueryWrapper<SettleEntity> wrapper = Wrappers.<SettleEntity>lambdaQuery();
		wrapper.eq(SettleEntity::getTenantId, InvocationInfoProxy.getTenantid());
		wrapper.eq(SettleEntity::getContractId, contractId);
		wrapper.eq(SettleEntity::getOrgId, orgId);
		wrapper.in(SettleEntity::getBillState, BillStateEnum.PASSED_STATE.getBillStateCode(),
				BillStateEnum.COMMITED_STATE.getBillStateCode());
		List<SettleEntity> list = settleService.list(wrapper);
		//累计冲抵金额
		BigDecimal sumOffsetMny = null;
		for(SettleEntity entity : list){
			sumOffsetMny = MathUtil.safeAdd(sumOffsetMny, entity.getOffsetMny());
		}
		SubSettleVO vo = new SubSettleVO();
		vo.setContractId(contractId);
		vo.setOrgId(orgId);
		vo.setSumOffsetMny(sumOffsetMny);
		return CommonResponse.success("获取信息成功",vo);
	}

	/**
	 * 根据合同、项目查询结算金额
	 * @param contractId
	 * @param projectId
	 * @return
	 */
	@GetMapping(value = "getSumSettleTaxMny")
	public CommonResponse<SubSettleVO> getSumSettleTaxMny(
			@RequestParam(value = "contractId", required = true) Long contractId,
			@RequestParam(value = "projectId", required = true) Long projectId){
		LambdaQueryWrapper<SettleEntity> wrapper = Wrappers.<SettleEntity>lambdaQuery();
		wrapper.eq(SettleEntity::getTenantId, InvocationInfoProxy.getTenantid());
		wrapper.eq(SettleEntity::getContractId, contractId);
		wrapper.eq(SettleEntity::getProjectId, projectId);
		wrapper.in(SettleEntity::getBillState, BillStateEnum.PASSED_STATE.getBillStateCode(),
				BillStateEnum.COMMITED_STATE.getBillStateCode());
		List<SettleEntity> settleList = settleService.list(wrapper);

		LambdaQueryWrapper<FinishEntity> wrapper1 = Wrappers.<FinishEntity>lambdaQuery();
		wrapper1.eq(FinishEntity::getTenantId,InvocationInfoProxy.getTenantid());
		wrapper1.eq(FinishEntity::getContractId,contractId);
		wrapper1.eq(FinishEntity::getProjectId,projectId);
		wrapper1.in(FinishEntity::getBillState, BillStateEnum.PASSED_STATE.getBillStateCode(),
				BillStateEnum.COMMITED_STATE.getBillStateCode());
		List<FinishEntity> finishList = finishService.list(wrapper1);


		BigDecimal settleTaxMny = null;
		for(SettleEntity entity : settleList){
			settleTaxMny = MathUtil.safeAdd(settleTaxMny, entity.getSettleTaxMny());
		}
		BigDecimal shouldPayTaxMny = null;
		for (FinishEntity entity : finishList) {
			shouldPayTaxMny = MathUtil.safeAdd(shouldPayTaxMny, entity.getShouldPayTaxMny());
		}
		SubSettleVO vo = new SubSettleVO();
		vo.setContractId(contractId);
		vo.setProjectId(projectId);
		vo.setSumSettleTaxMny(MathUtil.safeAdd(settleTaxMny , shouldPayTaxMny));
		return CommonResponse.success("获取信息成功",vo);
	}
}


