package com.ejianc.business.sub.service.impl;

import com.alibaba.fastjson.JSONObject;
import com.baomidou.mybatisplus.core.conditions.query.LambdaQueryWrapper;
import com.baomidou.mybatisplus.core.conditions.update.LambdaUpdateWrapper;
import com.baomidou.mybatisplus.core.metadata.IPage;
import com.baomidou.mybatisplus.core.toolkit.Wrappers;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.ejianc.business.accplat.consts.SystemCodeEnum;
import com.ejianc.business.cost.api.ICostDetailApi;
import com.ejianc.business.cost.vo.CostDetailVO;
import com.ejianc.business.finance.util.BillTypeCodeEnum;
import com.ejianc.business.finance.util.ValidateUtil;
import com.ejianc.business.finance.vo.ParamsCheckDsVO;
import com.ejianc.business.finance.vo.ParamsCheckVO;
import com.ejianc.business.sub.bean.*;
import com.ejianc.business.sub.mapper.FinishMapper;
import com.ejianc.business.sub.service.*;
import com.ejianc.business.sub.utils.BigDecimalUtils;
import com.ejianc.business.sub.utils.TreeNodeBUtil;
import com.ejianc.business.sub.vo.FinishDetailVO;
import com.ejianc.business.sub.vo.FinishRecordDetailVO;
import com.ejianc.business.sub.vo.FinishRecordVO;
import com.ejianc.business.sub.vo.FinishVO;
import com.ejianc.business.voucher.api.IVoucherApi;
import com.ejianc.business.voucher.consts.VoucherFlag;
import com.ejianc.business.voucher.consts.VoucherOptFlag;
import com.ejianc.business.voucher.utils.DataConvertUtil;
import com.ejianc.business.voucher.vo.VoucherInfo;
import com.ejianc.business.voucher.vo.VoucherParams;
import com.ejianc.foundation.orgcenter.api.IOrgApi;
import com.ejianc.foundation.orgcenter.vo.OrgVO;
import com.ejianc.foundation.support.api.IBillCodeApi;
import com.ejianc.foundation.support.api.IParamConfigApi;
import com.ejianc.foundation.support.vo.BillParamVO;
import com.ejianc.framework.auth.session.SessionManager;
import com.ejianc.framework.core.context.InvocationInfoProxy;
import com.ejianc.framework.core.exception.BusinessException;
import com.ejianc.framework.core.kit.mapper.BeanMapper;
import com.ejianc.framework.core.response.BillStateEnum;
import com.ejianc.framework.core.response.CommonResponse;
import com.ejianc.framework.core.response.Parameter;
import com.ejianc.framework.core.response.QueryParam;
import com.ejianc.framework.skeleton.template.BaseServiceImpl;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import java.math.BigDecimal;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;

/**
 * <p>
 * 分包完工结算表 服务实现类
 * </p>
 *
 * @author zhangwx
 * @since 2020-06-05
 */
@Service("FinishServiceImpl")
public class FinishServiceImpl extends BaseServiceImpl<FinishMapper, FinishEntity> implements IFinishService {

    private static final String SUB_FINISH_BILL_CODE = "SUB_FINISH";
    private static final String SUB_FINISH = "SUB_FINISH";
    private static final String SUB_FINISH_COST = "SUB_FINISH_COST";
    private static final String CHECK_PARAM_CODE = "P-2452lK17";

    private final Logger logger = LoggerFactory.getLogger(getClass());

    @Autowired
    private IContractService contractService;

    @Autowired
    private IBillCodeApi billCodeApi;

    @Autowired
    private IParamConfigApi paramConfigApi;

    @Autowired
    private IFinishDetailService finishDetailService;

    @Autowired
    private IFinishCostService finishCostService;

    @Autowired
    private ISettleService settleService;

    @Autowired
    private IOddjobService oddJobService;

    @Autowired
    private IOrgApi orgApi;

    @Autowired
    private ICostDetailApi costDetailApi;

    @Autowired
    private SessionManager sessionManager;

    @Autowired
    private IVoucherApi voucherApi;

    @Override
    public FinishVO insertOrUpdate(FinishVO finishVO) {
        if (checkSameContract(finishVO)) {
            throw new BusinessException("该合同在相同组织下已经做过完工结算！");
        }
        if (checkUnEffectSettle(finishVO)) {
            throw new BusinessException("该合同与组织下存在未生效过程结算单，不允许新增！");
        }
        if (checkUnEffectOddjob(finishVO)) {
            throw new BusinessException("该合同与组织下存在未生效零星用工单，不允许新增！");
        }
        if (!(finishVO.getId() != null && finishVO.getId() > 0)) { //新增
            if (checkFinishContract(finishVO)) {
                throw new BusinessException("该合同在相同组织下已经完工，不允许新增！");
            }
        }

        // 保存时校验合同version是否一致
        if (!ValidateUtil.validateUpstreamVersion(String.valueOf(finishVO.getContractId()),
                BillTypeCodeEnum.分包合同.getBillTypeCode(), finishVO.getContractVersion())) {
            throw new BusinessException("该合同已被更新，请刷新后重做！");
        }

        if (StringUtils.isEmpty(finishVO.getBillCode())) {
            CommonResponse<String> billCode = billCodeApi.getCodeBatchByRuleCode(SUB_FINISH_BILL_CODE, InvocationInfoProxy.getTenantid());
            if (billCode.isSuccess()) {
                finishVO.setBillCode(billCode.getData());
            } else {
                throw new BusinessException("网络异常， 编码生成失败， 请稍后再试");
            }
        }

        if (checkSameBillCode(finishVO)) {
            throw new BusinessException("存在相同编码，不允许保存!");
        }
        finishVO.setRelationFlag("0");
        finishVO.setProportionFlag("0");
        FinishEntity finishEntity = BeanMapper.map(finishVO, FinishEntity.class);
        //保存前清空主键和父主键，重新生成
        List<FinishDetailEntity> beforeDetails = finishEntity.getFinishDetailList();
        if (CollectionUtils.isNotEmpty(beforeDetails) && null == finishEntity.getId()) {
            for (FinishDetailEntity cdEntity : beforeDetails) {
                cdEntity.setId(null);
                cdEntity.setParentId(null);
            }
        }
        super.saveOrUpdate(finishEntity, false);

        List<FinishDetailEntity> finishDetailEntities = finishEntity.getFinishDetailList();
        if (CollectionUtils.isNotEmpty(finishDetailEntities)) {
            Map<String, Long> idMap = new HashMap<>();
            for (FinishDetailEntity cdEntity : finishDetailEntities) {
                idMap.put(cdEntity.getTid(), cdEntity.getId());
            }
            for (FinishDetailEntity cdEntity : finishDetailEntities) {
                if (StringUtils.isNotEmpty(cdEntity.getTpid())) {
                    cdEntity.setParentId(idMap.get(cdEntity.getTpid()));
                }
            }
            finishDetailService.saveOrUpdateBatch(finishDetailEntities, finishDetailEntities.size(), false);
        }

        // 回写合同完工结算状态
        this.writeBackContractFinsishFlag(finishEntity, Boolean.TRUE);

        return queryDetail(finishEntity.getId());
    }

    private Boolean checkSameBillCode(FinishVO finishVO) {
        Long tenantId = InvocationInfoProxy.getTenantid();
        LambdaQueryWrapper<FinishEntity> lambda = new LambdaQueryWrapper<>();
        lambda.eq(FinishEntity::getBillCode, finishVO.getBillCode());
        lambda.eq(FinishEntity::getTenantId, tenantId);
        if (null != finishVO.getId() && finishVO.getId() > 0) {
            lambda.ne(FinishEntity::getId, finishVO.getId());
        }
        return super.list(lambda).size() > 0;
    }

    private Boolean checkSameContract(FinishVO finishVO) {
        Long tenantId = InvocationInfoProxy.getTenantid();
        LambdaQueryWrapper<FinishEntity> lambda = new LambdaQueryWrapper<>();
        lambda.eq(FinishEntity::getContractId, finishVO.getContractId());
        lambda.eq(FinishEntity::getOrgId, finishVO.getOrgId());
        lambda.eq(FinishEntity::getTenantId, tenantId);
        if (null != finishVO.getId() && finishVO.getId() > 0) {
            lambda.ne(FinishEntity::getId, finishVO.getId());
        }
        return super.list(lambda).size() > 0;
    }

    private Boolean checkUnEffectSettle(FinishVO finishVO) {
        Long tenantId = InvocationInfoProxy.getTenantid();
        LambdaQueryWrapper<SettleEntity> lambda = new LambdaQueryWrapper<>();
        lambda.eq(SettleEntity::getContractId, finishVO.getContractId());
        lambda.eq(SettleEntity::getOrgId, finishVO.getOrgId());
        lambda.notIn(SettleEntity::getBillState, BillStateEnum.PASSED_STATE.getBillStateCode(), BillStateEnum.COMMITED_STATE.getBillStateCode());
        lambda.eq(SettleEntity::getTenantId, tenantId);
        return settleService.list(lambda).size() > 0;
    }

    private Boolean checkUnEffectOddjob(FinishVO finishVO) {
        Long tenantId = InvocationInfoProxy.getTenantid();
        LambdaQueryWrapper<OddjobEntity> lambda = new LambdaQueryWrapper<>();
        lambda.eq(OddjobEntity::getContractId, finishVO.getContractId());
        lambda.eq(OddjobEntity::getOrgId, finishVO.getOrgId());
        lambda.notIn(OddjobEntity::getBillState, BillStateEnum.PASSED_STATE.getBillStateCode(), BillStateEnum.COMMITED_STATE.getBillStateCode());
        lambda.eq(OddjobEntity::getTenantId, tenantId);
        return oddJobService.list(lambda).size() > 0;
    }

    private Boolean checkFinishContract(FinishVO finishVO) {
        Long tenantId = InvocationInfoProxy.getTenantid();
        LambdaQueryWrapper<ContractEntity> lambda = new LambdaQueryWrapper<>();
        lambda.eq(ContractEntity::getId, finishVO.getContractId());
        lambda.eq(ContractEntity::getTenantId, tenantId);
        lambda.eq(ContractEntity::getFinishFlag, true);
        return contractService.list(lambda).size() > 0;
    }

    @Override
    public FinishVO queryDetail(Long id) {
        FinishEntity finishEntity = super.selectById(id);
        FinishVO finishVO = BeanMapper.map(finishEntity, FinishVO.class);
        QueryParam queryParam = new QueryParam();
        queryParam.getParams().put("finishId", new Parameter(QueryParam.EQ, id));
        queryParam.getOrderMap().put("treeIndex", "asc");
        List<FinishDetailEntity> detailEntityList = finishDetailService.queryList(queryParam, false);
        if (CollectionUtils.isNotEmpty(detailEntityList)) {
            for (FinishDetailEntity cdEntity : detailEntityList) {
                cdEntity.setTid(cdEntity.getId().toString());
                cdEntity.setTpid(cdEntity.getParentId() != null && cdEntity.getParentId() > 0 ? cdEntity.getParentId().toString() : "");
                cdEntity.setRowState("edit");
            }
            List<FinishDetailVO> resultMapList = BeanMapper.mapList(detailEntityList, FinishDetailVO.class);
            finishVO.setFinishDetailList(TreeNodeBUtil.buildTree(resultMapList));
        }
        return finishVO;
    }

    @Override
    public CommonResponse<IPage<FinishVO>> queryListVOs(QueryParam param) {
        /** 模糊搜索配置字段示例 */
        List<String> fuzzyFields = param.getFuzzyFields();
        fuzzyFields.add("orgName");
        fuzzyFields.add("billCode");
        fuzzyFields.add("contractName");
//        fuzzyFields.add("customerName");
        fuzzyFields.add("supplierName");
        fuzzyFields.add("employeeName");
        fuzzyFields.add("memo");
        param.getParams().put("tenantId", new Parameter(QueryParam.EQ, InvocationInfoProxy.getTenantid()));
        // 组织本下
        param.getParams().put("org_id", new Parameter("in", orgApi.findChildrenByParentId(InvocationInfoProxy.getOrgId()).getData().stream().map(OrgVO::getId).collect(Collectors.toList())));
        param.getOrderMap().put("createTime", QueryParam.DESC);
        IPage<FinishEntity> page = queryPage(param, false);
        List<FinishVO> finishVOList = BeanMapper.mapList(page.getRecords(), FinishVO.class);
        IPage<FinishVO> finishVOIPage = new Page<>();
        finishVOIPage.setCurrent(page.getCurrent());
        finishVOIPage.setRecords(finishVOList);
        finishVOIPage.setSize(page.getSize());
        finishVOIPage.setTotal(page.getTotal());
        finishVOIPage.setPages(page.getPages());
        return CommonResponse.success("查询成功！", finishVOIPage);
    }

    @Override
    public CommonResponse<String> deleteByIds(List<FinishVO> vos) {
        List<Long> ids = vos.stream().map(FinishVO::getId).collect(Collectors.toList());
        // 回写合同完工结算状态
        List<FinishEntity> entityList = BeanMapper.mapList(super.listByIds(ids), FinishEntity.class);
        for (FinishEntity finishEntity : entityList) {
            this.writeBackContractFinsishFlag(finishEntity, Boolean.FALSE);
        }
        if (CollectionUtils.isNotEmpty(vos)) {
            this.removeByIds(ids, false);
        }
        return CommonResponse.success("删除成功！");
    }

    @Override
    public FinishRecordVO queryDetailRecord(Long id) {
        ContractEntity contractEntity = contractService.selectById(id);
        FinishRecordVO finishRecordVO = new FinishRecordVO();
        BigDecimal contractTaxMny = contractEntity.getContractTaxMny() == null ? BigDecimal.ZERO : contractEntity.getContractTaxMny();
        BigDecimal sumSettleTaxMny = (contractEntity.getSumSettleTaxMny() == null || !contractEntity.getFinishFlag()) ? BigDecimal.ZERO : contractEntity.getSumSettleTaxMny();
        BigDecimal sumScale = BigDecimalUtils.safeDiv(sumSettleTaxMny, contractTaxMny).multiply(new BigDecimal(100));
        finishRecordVO.setContractId(id);
        finishRecordVO.setFinishFlag(contractEntity.getFinishFlag());
        finishRecordVO.setPurchaseType(Integer.valueOf(contractEntity.getPurchaseType()));
        finishRecordVO.setContractName(contractEntity.getContractName());
        finishRecordVO.setSupplierName(contractEntity.getSupplierName());
        finishRecordVO.setContractTaxMny(contractTaxMny);
        finishRecordVO.setSumSettleTaxMny(sumSettleTaxMny);
        finishRecordVO.setSumScale(sumScale);

        LambdaQueryWrapper<FinishEntity> lambda = Wrappers.<FinishEntity>lambdaQuery();
        lambda.eq(FinishEntity::getContractId, id);
        lambda.in(FinishEntity::getBillState, BillStateEnum.PASSED_STATE.getBillStateCode(), BillStateEnum.COMMITED_STATE.getBillStateCode());
        List<FinishEntity> entities = super.list(lambda);
        List<FinishRecordDetailVO> finishRecordDetailVOList = BeanMapper.mapList(entities, FinishRecordDetailVO.class);
        if (CollectionUtils.isNotEmpty(finishRecordDetailVOList)) {
            finishRecordDetailVOList.forEach(entity -> {
                BigDecimal settleTaxMny = entity.getSettleTaxMny() == null ? BigDecimal.ZERO : entity.getSettleTaxMny();
                BigDecimal scale = contractTaxMny == BigDecimal.ZERO ? BigDecimal.ZERO : (settleTaxMny.divide(contractTaxMny, BigDecimal.ROUND_HALF_UP)).multiply(new BigDecimal(100));
                entity.setSumScale(scale);
            });
        }
        finishRecordVO.setDetailList(finishRecordDetailVOList);

        LambdaQueryWrapper<SettleEntity> lambda2 = Wrappers.<SettleEntity>lambdaQuery();
        lambda2.eq(SettleEntity::getContractId, id);
        lambda2.in(SettleEntity::getBillState, BillStateEnum.PASSED_STATE.getBillStateCode(), BillStateEnum.COMMITED_STATE.getBillStateCode());
        List<SettleEntity> settleList = settleService.list(lambda2);
        // 过程结算金额
        BigDecimal settleTaxMny = new BigDecimal(0);
        for (SettleEntity settle : settleList) {
            settleTaxMny = settleTaxMny.add(settle.getSettleTaxMny() == null ? BigDecimal.ZERO : settle.getSettleTaxMny());
        }
        finishRecordVO.setSettleTaxMny(settleTaxMny);
        finishRecordVO.setSettleBalanceMny(sumSettleTaxMny.subtract(settleTaxMny));
        return finishRecordVO;
    }

    @Override
    public FinishVO queryDetailAdd(Long contractId) {
        ContractEntity contractEntity = contractService.selectById(contractId);
        FinishVO finishVO = BeanMapper.map(contractEntity, FinishVO.class);
        finishVO.setBillCode(null);
        finishVO.setEmployeeId(Long.valueOf(InvocationInfoProxy.getEmployeeId()));
        finishVO.setEmployeeName(null);
        finishVO.setContractId(contractEntity.getId());
        finishVO.setBillState(null);
        finishVO.setCreateUserCode(null);
        finishVO.setCreateTime(null);
        finishVO.setUpdateUserCode(null);
        finishVO.setUpdateTime(null);
        finishVO.setId(null);
        finishVO.setVersion(null);
        finishVO.setTaxMny(null);
        finishVO.setContractVersion(contractEntity.getVersion());
        return finishVO;
    }

    @Override
    public List<FinishVO> queryExportList(QueryParam param) {
        param.setPageIndex(0);
        param.setPageSize(-1);
        List<FinishVO> resVos = queryListVOs(param).getData().getRecords();
        if (!resVos.isEmpty()) {
            for (int i = 0; i < resVos.size(); i++) {
                FinishVO vo = resVos.get(i);
                if (vo.getContractType() != null) {
                    if (vo.getContractType().equals(Long.valueOf("1270328729526124545"))) {
                        vo.setContractTypeName("专业分包");
                    } else if (vo.getContractType().equals(Long.valueOf("1270328674299723778"))) {
                        vo.setContractTypeName("劳务分包");
                    }
                }
                vo.setBillStateName(BillStateEnum.getEnumByStateCode(vo.getBillState()).getDescription());
            }
            ;
        }
        return resVos;
    }

    @Override
    public CommonResponse<FinishVO> pushCost(FinishVO finishVO) {
        FinishEntity finishEntity = baseMapper.selectById(finishVO.getId());
        if (CollectionUtils.isNotEmpty(finishVO.getFinishCostList())) {
            List<FinishCostEntity> costEntities = BeanMapper.mapList(finishVO.getFinishCostList(), FinishCostEntity.class);
            finishCostService.saveOrUpdateBatch(costEntities);
            finishEntity.setFinishCostList(costEntities);
        }
        //推送数据
        costPush(finishEntity);
        return CommonResponse.success(queryDetail(finishVO.getId()));
    }

    @Override
    public void costPush(FinishEntity finishEntity) {
        Boolean isPush = true;
        LambdaQueryWrapper<FinishCostEntity> costWrapper = new LambdaQueryWrapper<>();
        costWrapper.eq(FinishCostEntity::getFinishId, finishEntity.getId());
        List<FinishCostEntity> costEntities = finishCostService.list(costWrapper);
        if (CollectionUtils.isEmpty(costEntities)) {
            isPush = false;
        } else if (isPush) {
            BigDecimal totalMny = BigDecimal.ZERO;
            for (FinishCostEntity costEntity : costEntities) {
                if (null == costEntity.getSubjectId()) {
                    isPush = false;
                    break;
                } else {
                    BigDecimal money = null != costEntity.getMoney() ? costEntity.getMoney() : BigDecimal.ZERO;
                    totalMny = totalMny.add(money);
                }
            }
            if (null == finishEntity.getShouldPayTaxMny() || totalMny.compareTo(finishEntity.getShouldPayTaxMny()) != 0) {
                isPush = false;
            }
        }

        finishEntity.setFinishCostList(costEntities);

        //更新是否关联
        LambdaUpdateWrapper<FinishEntity> updateWrapper = new LambdaUpdateWrapper<>();
        updateWrapper.eq(FinishEntity::getId, finishEntity.getId());
        updateWrapper.set(FinishEntity::getRelationFlag, isPush ? "1" : "0");//(1:是，0：否)
        super.update(updateWrapper);

        //判断之前的单据是否关联
        String oldRelationFlag = finishEntity.getRelationFlag();
        //之前已关联
        if (oldRelationFlag.equals("1")) {
            if (isPush) {
                saveCost(finishEntity);
            }
            if (!isPush) {
                //删除成本中心之前的数据
                costDetailApi.deleteSubject(finishEntity.getId());
            }
        }
        //之前未关联
        if (oldRelationFlag.equals("0")) {
            if (isPush) {
                saveCost(finishEntity);
            }
        }
    }

    @Override
    public void pullCost(Long id) {
        //更新是否关联
        LambdaUpdateWrapper<FinishEntity> updateWrapper = new LambdaUpdateWrapper<>();
        updateWrapper.eq(FinishEntity::getId, id);
        updateWrapper.set(FinishEntity::getRelationFlag, "0");//(1:是，0：否)
        super.update(updateWrapper);

        costDetailApi.deleteSubject(id);
    }

    @Override
    public ParamsCheckVO checkParams(Integer purchaseType, BigDecimal settleTaxMny, BigDecimal contractTaxMny) {
        // 三种控制方式：不控制，提醒，无法保存 (默认为提醒)
        String[] paramsArray = {"none", "warn", "alert"};
        ParamsCheckVO paramsCheckVO = new ParamsCheckVO();
        List<ParamsCheckDsVO> checkDsVOS = new ArrayList<>();
        CommonResponse<BillParamVO> billParamByCode = paramConfigApi.getBillParamByCode(CHECK_PARAM_CODE);
        if (billParamByCode.isSuccess() && null != billParamByCode.getData()) {
            BillParamVO billParamVO = billParamByCode.getData();
            BigDecimal roleValue = billParamVO.getRoleValue();
            BigDecimal comMny = contractTaxMny.multiply(roleValue.divide(BigDecimal.valueOf(100)
                    , 2, BigDecimal.ROUND_HALF_UP));

            if (2 == purchaseType) {
                // 集采合同直接赋值为不控制
                paramsCheckVO.setWarnType(paramsArray[0]);
                return paramsCheckVO;
            } else {
                paramsCheckVO.setWarnType(paramsArray[billParamVO.getControlType()]);
            }

            if (settleTaxMny.compareTo(comMny) > 0) {
                ParamsCheckDsVO paramsCheckDsVO = new ParamsCheckDsVO();
                paramsCheckDsVO.setWarnItem("合同超结");
                paramsCheckDsVO.setWarnName("完工结算金额大于合同金额");
                StringBuffer stringBuffer = new StringBuffer();
                stringBuffer.append("本次结算金额：").append(settleTaxMny.toString())
                        .append("元，合同金额*").append(roleValue).append("%:")
                        .append(comMny).append("元。超出金额：")
                        .append(settleTaxMny.subtract(comMny)).append("元");
                paramsCheckDsVO.setContent(stringBuffer.toString());
                checkDsVOS.add(paramsCheckDsVO);
            }
            paramsCheckVO.setDataSource(checkDsVOS);
        } else {
            logger.info(billParamByCode.getMsg());
            throw new BusinessException("获取控制参数失败");
        }

        return paramsCheckVO;
    }

    @Override
    public CommonResponse handleVoucher(FinishEntity entity, String voucherOptFlag) {
        CommonResponse response = null;
        if(VoucherOptFlag.DEL.equals(voucherOptFlag) || VoucherOptFlag.RE_PUSH.equals(voucherOptFlag)){
            if (VoucherFlag.SUCCESS.equals(entity.getVoucherFlag())) {
                response = voucherApi.del(JSONObject.parseObject(entity.getVoucherInfo(), VoucherInfo.class));
                logger.info("{}删除凭证结果：{}", voucherOptFlag, DataConvertUtil.toPrettyFormat(response));
                if (response.isSuccess()) {
                    LambdaUpdateWrapper<FinishEntity> lambda = new LambdaUpdateWrapper<>();
                    lambda.eq(FinishEntity::getId, entity.getId());
                    lambda.set(FinishEntity::getVoucherInfo, null);
                    lambda.set(FinishEntity::getVoucherFlag, 0);
                    this.update(lambda);
                }
            }
            if(VoucherOptFlag.DEL.equals(voucherOptFlag)){
                return response;
            }
        }
        VoucherParams voucherParams = VoucherParams.newInstance("BT200630000000002", entity.getOrgId(), entity.getBillCode(), entity, SystemCodeEnum.SUB);
        response = voucherApi.save(voucherParams);
        logger.info("{}保存凭证结果：{}" ,voucherOptFlag, DataConvertUtil.toPrettyFormat(response));
        LambdaUpdateWrapper<FinishEntity> lambda = new LambdaUpdateWrapper<>();
        lambda.eq(FinishEntity::getId, entity.getId());
        if (response.isSuccess()) {
            VoucherInfo voucherInfo = (VoucherInfo) response.getData();
            lambda.set(FinishEntity::getVoucherInfo, DataConvertUtil.objToString(voucherInfo));
            lambda.set(FinishEntity::getVoucherFlag, voucherInfo.getVoucherFlag());
        } else {
            lambda.set(FinishEntity::getVoucherInfo, response.getMsg());
            lambda.set(FinishEntity::getVoucherFlag, VoucherFlag.FAILED);
        }
        this.update(lambda);
        return response;
    }

    private void saveCost(FinishEntity finishEntity) {
        //税率
        BigDecimal taxRate = finishEntity.getTaxRate().divide(new BigDecimal(100));
        BigDecimal number = taxRate.add(new BigDecimal(1));
        //明细
        List<CostDetailVO> costDetailVOList = new ArrayList<CostDetailVO>();

        //费用
        List<FinishCostEntity> finishCostList = finishEntity.getFinishCostList();
        if (CollectionUtils.isNotEmpty(finishCostList)) {
            for (FinishCostEntity costEntity : finishCostList) {
                CostDetailVO costDetailVO = new CostDetailVO();
                costDetailVO.setSubjectId(costEntity.getSubjectId());
                costDetailVO.setSourceId(costEntity.getFinishId());
                costDetailVO.setSourceDetailId(costEntity.getId());
                costDetailVO.setHappenTaxMny(costEntity.getMoney());
                //无税值计算
                BigDecimal pushSettlementMny = costEntity.getMoney();
                BigDecimal happenMny = pushSettlementMny.divide(number, 2, BigDecimal.ROUND_HALF_UP);
                costDetailVO.setHappenMny(happenMny);
                costDetailVO.setHappenDate(finishEntity.getSettleDate());
                costDetailVO.setCreateUserName(sessionManager.getUserContext().getUserName());
                costDetailVO.setSourceType(SUB_FINISH);
                costDetailVO.setSourceTabType(SUB_FINISH_COST);
                costDetailVO.setProjectId(finishEntity.getProjectId());
                costDetailVOList.add(costDetailVO);
            }
        }

        //成本中心
        if (CollectionUtils.isNotEmpty(costDetailVOList)) {
            CommonResponse<String> stringCommonResponse = costDetailApi.saveSubject(costDetailVOList);
            if (stringCommonResponse.isSuccess()) {
            } else {
                throw new BusinessException(stringCommonResponse.getMsg());
            }
        }

    }

    /**
     * 回写合同完工结算状态
     *
     * @param finishEntity
     * @param finishFlag
     */
    private void writeBackContractFinsishFlag(FinishEntity finishEntity, Boolean finishFlag) {
        LambdaUpdateWrapper<ContractEntity> updateWrapper = new LambdaUpdateWrapper<>();
        //根据完工状态回写合同状态，合同状态3已封账，2履约中
        updateWrapper.set(ContractEntity::getContractStatus, finishFlag ? "3" : "2");
        updateWrapper.set(ContractEntity::getFinishFlag, finishFlag);
        updateWrapper.eq(ContractEntity::getId, finishEntity.getContractId());
        ContractEntity contractEntity = contractService.getById(finishEntity.getContractId());
        contractService.update(contractEntity, updateWrapper, false);
    }
}
