package com.ejianc.foundation.share.service.impl;

import com.alibaba.fastjson.JSONObject;
import com.baomidou.mybatisplus.core.metadata.IPage;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.ejianc.foundation.billcode.BillCodeException;
import com.ejianc.foundation.share.bean.MaterialCategoryEntity;
import com.ejianc.foundation.share.bean.MaterialEntity;
import com.ejianc.foundation.share.enums.MaterialFuzzyMatchEnum;
import com.ejianc.foundation.share.mapper.MaterialCategoryMapper;
import com.ejianc.foundation.share.mapper.MaterialMapper;
import com.ejianc.foundation.share.service.IMaterialService;
import com.ejianc.foundation.share.vo.MaterialFuzzyMatchVO;
import com.ejianc.foundation.share.vo.MaterialPlusVO;
import com.ejianc.foundation.share.vo.MaterialVO;
import com.ejianc.foundation.support.api.IParamConfigApi;
import com.ejianc.foundation.support.service.IBillCodeGenerator;
import com.ejianc.framework.core.context.InvocationInfoProxy;
import com.ejianc.framework.core.exception.BusinessException;
import com.ejianc.framework.core.kit.collection.CollectionUtil;
import com.ejianc.framework.core.kit.collection.ListUtil;
import com.ejianc.framework.core.kit.mapper.BeanMapper;
import com.ejianc.framework.core.response.CommonResponse;
import com.ejianc.framework.core.response.Parameter;
import com.ejianc.framework.core.response.QueryParam;
import com.ejianc.support.idworker.util.IdWorker;
import com.google.common.base.Stopwatch;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.ArrayUtils;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.scheduling.concurrent.CustomizableThreadFactory;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import java.util.*;
import java.util.concurrent.*;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import java.util.stream.Collectors;

@Service
@Transactional
public class MaterialServiceImpl implements IMaterialService {

	private final Logger logger = LoggerFactory.getLogger(this.getClass());
	@Autowired
	private MaterialMapper materialMapper;
	@Autowired
	private MaterialCategoryMapper materialCategoryMapper;
	@Autowired
	private IParamConfigApi paramConfigApi;
	private static final ExecutorService executorService = new ThreadPoolExecutor(
			8,
			8,
			0L,
			TimeUnit.MILLISECONDS,
			new LinkedBlockingDeque<>(1024),
			new CustomizableThreadFactory("fuzzyMatch-pool-"),
			new ThreadPoolExecutor.AbortPolicy()
	);
	private static final String MATERIAL_BILL_CODE = "SUPPORT_MATERIAL";
	private static final String EQUIPMENT_BILL_CODE = "SUPPORT_EQUIPMENT";
	@Autowired
	private IBillCodeGenerator generator;

	@Override
	public List<MaterialVO> queryListByCategoryId(Long categoryId) {
		Long tenantId = InvocationInfoProxy.getTenantid();
		List<MaterialEntity> entities = materialMapper.queryListByCategoryId(tenantId, categoryId);
		if (entities != null && entities.size() > 0) {
			return BeanMapper.mapList(entities, MaterialVO.class);
		}
		return null;
	}

	@Override
	public MaterialVO queryDetail(Long id) {
		Long tenantId = InvocationInfoProxy.getTenantid();
		MaterialEntity entity = materialMapper.queryDetail(tenantId, id);
		if(entity != null) {
			return BeanMapper.map(entity, MaterialVO.class);
		}
		return null;
	}

	@Override
	public void update(MaterialVO uniqueBean) {
		materialMapper.update(uniqueBean);
	}

	@Override
	public void save(MaterialVO materialVo) {
		materialMapper.save(materialVo);
	}

	/**
	 * 批量保存
	 *
	 * @param materialVOList
	 */
	@Override
	public void insertBatch(List<MaterialVO> materialVOList) {
		if(ListUtil.isNotEmpty(materialVOList)){
			List<MaterialVO> toInsert = new ArrayList<>();
			for (MaterialVO materialVO : materialVOList) {
				if (toInsert.size()>0 && toInsert.size() % 1000 == 0) {
					materialMapper.insertBatch(InvocationInfoProxy.getTenantid(),toInsert);
					toInsert = new ArrayList<>();
				}
				toInsert.add(materialVO);
			}
			if(toInsert.size()>0){
				materialMapper.insertBatch(InvocationInfoProxy.getTenantid(),toInsert);
			}
		}
	}

	@Override
	public void delete(List<Long> ids) {
		Long tenantId = InvocationInfoProxy.getTenantid();
		for(Long id:ids) {
			materialMapper.delete(tenantId, id);
		}
	}

	@Override
	public IPage<MaterialEntity> queryPage(QueryParam queryParam) {
		Long tenantId = InvocationInfoProxy.getTenantid();
		Map<String, Parameter> paramMap = queryParam.getParams();

		Map<String, Object> condition = new HashMap<String, Object>();
		for(Map.Entry<String, Parameter> entry:paramMap.entrySet()) {
			if("categoryId".equals(entry.getKey())) {
				Parameter parameter = entry.getValue();
				if(StringUtils.isBlank(parameter.getValue()+"")) {
					continue;
				}
				MaterialCategoryEntity categoryEntity = materialCategoryMapper.queryDetail(tenantId, Long.parseLong(parameter.getValue().toString()));
				if(categoryEntity != null) {
					condition.put("innerCode", categoryEntity.getInnerCode());
				}
			}else{
				condition.put(entry.getKey(), entry.getValue().getValue());
			}
		}
		condition.put("pageIndex", (queryParam.getPageIndex()-1)*queryParam.getPageSize());
		condition.put("pageSize", queryParam.getPageSize());

		List<MaterialEntity> records = materialMapper.queryList(condition);
		Long count = materialMapper.queryCount(condition);

		IPage<MaterialEntity> page = new Page<>();
		page.setCurrent(queryParam.getPageIndex());
		page.setSize(queryParam.getPageSize());
		page.setTotal(count);
		page.setRecords(records);
		return page;
	}

	@Override
	public MaterialVO queryExitFlag(String name, String spec, String unitName, Long categoryId) {
		Long tenantId = InvocationInfoProxy.getTenantid();
		MaterialVO materialVo = materialMapper.queryExitFlag(tenantId, name, spec, unitName, categoryId);
		return materialVo;
	}

	/**
	 * 根据code查询物料
	 *
	 * @param code
	 * @return
	 */
	@Override
	public MaterialVO queryByCode(String code) {
		Long tenantId = InvocationInfoProxy.getTenantid();
		return materialMapper.queryByCode(tenantId, code);
	}

	@Override
	public List<MaterialEntity> queryList(QueryParam queryParam) {
		Map<String, Parameter> paramMap = queryParam.getParams();
		Long tenantId = InvocationInfoProxy.getTenantid();
		Map<String, Object> condition = new HashMap<String, Object>();
		for(Map.Entry<String, Parameter> entry:paramMap.entrySet()) {
			if("categoryId".equals(entry.getKey())) {
				Parameter parameter = entry.getValue();
				if(StringUtils.isBlank(parameter.getValue()+"")) {
					continue;
				}
				MaterialCategoryEntity categoryEntity = materialCategoryMapper.queryDetail(tenantId, Long.parseLong(parameter.getValue().toString()));
				if(categoryEntity != null) {
					condition.put("innerCode", categoryEntity.getInnerCode());
				}
			}else{
				condition.put(entry.getKey(), entry.getValue().getValue());
			}
		}
		List<MaterialEntity> records = materialMapper.queryExportList(condition);
		return records;
	}

	@Override
	public void insertMaterialListFromPlatform() {
		Long tenantId = InvocationInfoProxy.getTenantid();
		materialMapper.insertMaterialListFromPlatform(tenantId);
	}

	@Override
	public IPage<MaterialEntity> queryRefMaterialPage(Map<String, Object> params) {
		List<MaterialEntity> dataList = materialMapper.queryRefMaterialList(params);
		Long dataCount = materialMapper.queryRefMaterialCount(params);

		IPage<MaterialEntity> page = new Page<>();
		page.setRecords(dataList);
		page.setTotal(dataCount);
		return page;
	}

	@Override
	public MaterialEntity queryBySourceId(String sourceId) {
		return materialMapper.queryBySourceId(sourceId,InvocationInfoProxy.getTenantid());
	}

	@Override
	public List<MaterialEntity> queryMaterialByIds(List<Long> ids) {
		return materialMapper.queryMaterialByIds(ids,InvocationInfoProxy.getTenantid());
	}

	@Override
	public IPage<MaterialEntity> queryZjwjRefMaterialPage(Map<String, Object> params) {
		IPage<MaterialEntity> page = new Page<>();
		if (params.containsKey("sourceOrgId") && null !=params.get("sourceOrgId")) {
			List<MaterialEntity> dataList = materialMapper.queryZjwjRefMaterialList(params);
			Long dataCount = materialMapper.queryZjwjRefMaterialCount(params);
			page.setRecords(dataList);
			page.setTotal(dataCount);
		}
		return page;
	}

	@Override
	public List<MaterialEntity> queryMaterialListByNames(List<String> nameList) {
		return materialMapper.queryMaterialListByNames(InvocationInfoProxy.getTenantid(), nameList);
	}

	@Override
	public MaterialVO queryLastMaterialByTime() {
		return materialMapper.queryLastMaterialByTime(InvocationInfoProxy.getTenantid());
	}

	@Override
	public List<MaterialVO> queryMaterialList(String name, String spec, String unitName) {
		Long tenantId = InvocationInfoProxy.getTenantid();
		List<MaterialEntity> list = materialMapper.queryMaterialList(tenantId, name, spec, unitName);
		if (CollectionUtil.isNotEmpty(list)){
			List<MaterialVO> vos = BeanMapper.mapList(list, MaterialVO.class);
			for (MaterialVO materialVO : vos) {
				Long id = materialVO.getCategoryId();
				MaterialCategoryEntity entity = materialCategoryMapper.queryDetail(tenantId, id);
				materialVO.setCategoryId(entity.getId());
				materialVO.setCategoryName(entity.getName());
			}
			return vos;
		}
		return null;
	}

	/**
	 * 处理空字符
	 *
	 * @param str  str
	 * @param flag true:只去除首尾空格，false:去除空格和空白字符
	 *
	 * @return {@link String}
	 */
	private static String handleBlankString(String str, boolean flag) {
		if (StringUtils.isBlank(str)) {
			return "";
		}
		if (flag) {
			// 只去除首尾空格
			return str.trim();
		} else {
			// 去除空格和空白字符
			return str.replaceAll("\\s*", "");
		}
	}

	/**
	 * 处理特殊字符串
	 *
	 * @param str str
	 *
	 * @return {@link String}
	 */
	private static String handleSpecialString(String str) {
		// 去除特殊字符和数字，只返回英文和汉字
		Pattern p = Pattern.compile("[^(a-zA-Z\\u4e00-\\u9fa5)]");
		Matcher matcher = p.matcher(str);
		return matcher.replaceAll("");
	}

	/**
	 * 字符串分词
	 *
	 * @param oldStr 老str
	 * @param str    str
	 *
	 * @return {@link List}<{@link String}>
	 */
	private static List<String> handleSegmentation(String oldStr, String str) {
		List<String> strings = new ArrayList<>();
		strings.add(oldStr);

		if (StringUtils.isBlank(str)) {
			return strings;
		}

		// 英文单词
		List<String> english = new ArrayList<>();
		String pattern = "([a-zA-Z_]+)";
		Pattern p = Pattern.compile(pattern);
		Matcher m = p.matcher(str);
		while (m.find()) {
			english.add(m.group());
		}
		String ss = str.replaceAll(pattern, "#");

		int index = 0;
		for (int i = 0, length = ss.length(), len = length - 1; i < length; i++) {
			char c1 = ss.charAt(i);
			if (i + 1 < len) {
				char c2 = ss.charAt(i + 1);
				// 当前是#，后一个是汉字
				// 当前是汉字，后一个是汉字
				// 当前是汉字，后一个是#
				if ('#' == c1 && chineseYes(c2)) {
					String s = english.get(index) + c2;
					strings.add(s);
					index += 1;
				}
				if (chineseYes(c1) && chineseYes(c2)) {
					String s = String.valueOf(c1) + c2;
					strings.add(s);
				}
				if (chineseYes(c1) && '#' == c2) {
					String s = c1 + english.get(index);
					strings.add(s);
				}
			} else {
				// 特殊处理一下最后的数据
				String s = ss.substring(i);
				if (s.contains("#")) {
					s = s.replace("#", english.get(index));
				}
				strings.add(s);
				break;
			}
		}
		return strings;
	}

	/**
	 * 是否是中文
	 *
	 * @param ch ch
	 *
	 * @return boolean
	 */
	public static boolean chineseYes(char ch) {
		return String.valueOf(ch).matches("[\\u4e00-\\u9fa5]");
	}

	/**
	 * 拼接匹配相关参数
	 *
	 * @param materialPlusVO         材料设备增强VO
	 * @param matchStatus            匹配状态
	 * @param matchStatusDescription 匹配状态描述
	 * @param matchNumber            模糊匹配数量
	 * @param fuzzyMatchList         模糊匹配列表
	 */
	private static void spliceMaterialPlusVO(MaterialPlusVO materialPlusVO, Integer matchStatus, String matchStatusDescription, Integer matchNumber, List<MaterialPlusVO> fuzzyMatchList) {
		materialPlusVO.setMatchStatus(matchStatus);
		materialPlusVO.setMatchStatusDescription(matchStatusDescription);
		materialPlusVO.setMatchNumber(matchNumber);
		materialPlusVO.setFuzzyMatchList(fuzzyMatchList);
	}

	/**
	 * 校验档案是否存在
	 *
	 * @param materialFuzzyMatchVO 材料设备模糊匹配VO
	 *
	 * @return {@link MaterialFuzzyMatchVO}
	 */
	@Override
	public MaterialFuzzyMatchVO checkArchive(MaterialFuzzyMatchVO materialFuzzyMatchVO) {
		Stopwatch stopwatch = Stopwatch.createStarted();
		logger.info("校验档案是否存在----start");

		Long tenantId = InvocationInfoProxy.getTenantid();

		List<CompletableFuture<Map<String, Boolean>>> futures1 = new ArrayList<>();
		List<CompletableFuture<Map<String, Boolean>>> futures2 = new ArrayList<>();


		// 材料档案
		Map<String, Boolean> materialCategoryMap = materialFuzzyMatchVO.getMaterialCategoryMap();
		if (null != materialCategoryMap && !materialCategoryMap.isEmpty()) {
			for (Map.Entry<String, Boolean> entry : materialCategoryMap.entrySet()) {
				CompletableFuture<Map<String, Boolean>> materialCategoryFuture = CompletableFuture.supplyAsync(() -> {
					logger.info("材料档案，当前线程为：{}", Thread.currentThread().getName());
					HashMap<String, Boolean> hashMap = new HashMap<>();
					Long count = materialCategoryMapper.checkArchive(tenantId, 1, entry.getKey());
					hashMap.put(entry.getKey(), count.compareTo(0L) > 0);
					return hashMap;
				}, executorService);
				futures1.add(materialCategoryFuture);
			}
		}

		// 设备档案
		Map<String, Boolean> equipmentCategoryMap = materialFuzzyMatchVO.getEquipmentCategoryMap();
		if (null != equipmentCategoryMap && !equipmentCategoryMap.isEmpty()) {
			for (Map.Entry<String, Boolean> entry : equipmentCategoryMap.entrySet()) {
				CompletableFuture<Map<String, Boolean>> equipmentCategoryFuture = CompletableFuture.supplyAsync(() -> {
					logger.info("设备档案，当前线程为：{}", Thread.currentThread().getName());
					HashMap<String, Boolean> hashMap = new HashMap<>();
					Long count = materialCategoryMapper.checkArchive(tenantId, 2, entry.getKey());
					hashMap.put(entry.getKey(), count.compareTo(0L) > 0);
					return hashMap;
				}, executorService);
				futures2.add(equipmentCategoryFuture);
			}
		}

		CompletableFuture.allOf(ArrayUtils.addAll(futures1.toArray(new CompletableFuture[0]), futures2.toArray(new CompletableFuture[0])));

		Map<String, Boolean> materialCategoryResult = new HashMap<>();
		Map<String, Boolean> equipmentCategoryResult = new HashMap<>();
		futures1.forEach(future -> materialCategoryResult.putAll(future.join()));
		futures2.forEach(future -> equipmentCategoryResult.putAll(future.join()));

		MaterialFuzzyMatchVO fuzzyMatchVO = new MaterialFuzzyMatchVO();
		fuzzyMatchVO.setMaterialCategoryMap(materialCategoryResult);
		fuzzyMatchVO.setEquipmentCategoryMap(equipmentCategoryResult);

		stopwatch.stop();
		logger.info("校验档案是否存在----end：已处理材料档案【{}】条数据，共用时【{}】毫秒", materialCategoryMap == null ? 0 : materialCategoryMap.size(), stopwatch.elapsed(TimeUnit.MILLISECONDS));
		logger.info("校验档案是否存在----end：已处理设备档案【{}】条数据，共用时【{}】毫秒", equipmentCategoryMap == null ? 0 : equipmentCategoryMap.size(), stopwatch.elapsed(TimeUnit.MILLISECONDS));

		return fuzzyMatchVO;
	}

	/**
	 * 模糊匹配数量
	 *
	 * @param materialFuzzyMatchVO 材料设备模糊匹配VO
	 *
	 * @return {@link MaterialFuzzyMatchVO}
	 */
	@Override
	public MaterialFuzzyMatchVO fuzzyMatchNumber(MaterialFuzzyMatchVO materialFuzzyMatchVO) {
		Stopwatch stopwatch = Stopwatch.createStarted();
		logger.info("模糊匹配数量----start");

		MaterialFuzzyMatchVO fuzzyMatchVO = new MaterialFuzzyMatchVO();
		Long tenantId = InvocationInfoProxy.getTenantid();

		Map<String, MaterialPlusVO> fuzzyMatchMap = materialFuzzyMatchVO.getFuzzyMatchMap();
		if (null != fuzzyMatchMap && !fuzzyMatchMap.isEmpty()) {

			List<CompletableFuture<Map<String, MaterialPlusVO>>> futures = new ArrayList<>();

			for (Map.Entry<String, MaterialPlusVO> entry : fuzzyMatchMap.entrySet()) {
				CompletableFuture<Map<String, MaterialPlusVO>> future = CompletableFuture.supplyAsync(() -> {
					logger.info("模糊匹配数量，当前线程为：{}", Thread.currentThread().getName());
					HashMap<String, MaterialPlusVO> hashMap = new HashMap<>();
					MaterialPlusVO materialPlusVO = entry.getValue();

					// 清单名称为空，不处理
					String name = materialPlusVO.getName();
					if (StringUtils.isBlank(name)) {
						hashMap.put(entry.getKey(), entry.getValue());
						return hashMap;
					}

					String name1 = handleBlankString(name, false);
					String segmentation = handleSpecialString(name1);
					List<String> names = handleSegmentation(name, segmentation);

					// 费用类型 || 分类名称为空，不处理，不为空去除首尾空格
					Integer type = materialPlusVO.getType();
					if (type == null || StringUtils.isBlank(materialPlusVO.getCategoryName())) {
						hashMap.put(entry.getKey(), entry.getValue());
						return hashMap;
					}
					String categoryName = handleBlankString(materialPlusVO.getCategoryName(), true);

					// 规格型号、单位名称处理空字符串，为空按照空字符处理，不为空去除首尾空格
					String spec = handleBlankString(materialPlusVO.getSpec(), true);
					String unitName = handleBlankString(materialPlusVO.getUnitName(), true);

					List<MaterialPlusVO> materialPlusVOS = materialMapper.findFuzzyResultListByLikeOr(tenantId, name, names, type, categoryName, spec, unitName);
					if (CollectionUtils.isNotEmpty(materialPlusVOS)) {
						// 根据最高权重判断是否是完全匹配
						MaterialPlusVO plusVO = materialPlusVOS.get(0);
						if (plusVO.getWeight() == 12) {
							// 完全匹配
							MaterialPlusVO vo = BeanMapper.map(plusVO, MaterialPlusVO.class);
							spliceMaterialPlusVO(vo, MaterialFuzzyMatchEnum.MACHED.getCode(), MaterialFuzzyMatchEnum.MACHED.getDesc(), materialPlusVOS.size(), new ArrayList<>());
							entry.setValue(vo);
						} else {
							// 模糊匹配
							spliceMaterialPlusVO(materialPlusVO, MaterialFuzzyMatchEnum.FUZZY_MATCH.getCode(), MaterialFuzzyMatchEnum.FUZZY_MATCH.getDesc() + materialPlusVOS.size() + "条", materialPlusVOS.size(), new ArrayList<>());
							entry.setValue(materialPlusVO);
						}
					} else {
						// 未匹配
						spliceMaterialPlusVO(materialPlusVO, MaterialFuzzyMatchEnum.UN_MATCH.getCode(), MaterialFuzzyMatchEnum.UN_MATCH.getDesc(), 0, new ArrayList<>());
						entry.setValue(materialPlusVO);
					}

					hashMap.put(entry.getKey(), entry.getValue());

					return hashMap;
				}, executorService);
				futures.add(future);
			}

			CompletableFuture.allOf(futures.toArray(new CompletableFuture[0]));
			Map<String, MaterialPlusVO> fuzzyMatchResult = new HashMap<>();
			futures.forEach(future -> fuzzyMatchResult.putAll(future.join()));
			fuzzyMatchVO.setFuzzyMatchMap(fuzzyMatchResult);
		}

		stopwatch.stop();
		logger.info("模糊匹配数量----end：已处理【{}】条数据，共用时【{}】毫秒", fuzzyMatchMap == null ? 0 : fuzzyMatchMap.size(), stopwatch.elapsed(TimeUnit.MILLISECONDS));

		return fuzzyMatchVO;
	}

	/**
	 * 模糊匹配分页查询
	 *
	 * @param current       当前页码
	 * @param size          条数
	 * @param name          清单名称
	 * @param type          费用类型
	 * @param categoryName  分类名称
	 * @param spec          规格型号
	 * @param unitName      单位名称
	 * @param fuzzyName     清单名称模糊
	 * @param fuzzySpec     规格型号模糊
	 * @param fuzzyUnitName 单位名称模糊
	 *
	 * @return {@link IPage}<{@link MaterialPlusVO}>
	 */
	@Override
	public IPage<MaterialPlusVO> fuzzyMatchQueryPage(long current, long size, String name, Integer type, String categoryName, String spec, String unitName, String fuzzyName, String fuzzySpec, String fuzzyUnitName) {
		Long tenantId = InvocationInfoProxy.getTenantid();
		Page<MaterialPlusVO> page = new Page<>(current, size);

		// 清单名称为空，不处理
		if (StringUtils.isBlank(name)) {
			return null;
		}
		String name1 = handleBlankString(name, false);
		String segmentation = handleSpecialString(name1);
		List<String> names = handleSegmentation(name, segmentation);

		// 费用类型 || 分类名称为空，不处理，不为空去除首尾空格
		if (type == null || StringUtils.isBlank(categoryName)) {
			return null;
		}
		categoryName = handleBlankString(categoryName, true);

		// 规格型号、单位名称处理空字符串，为空按照空字符处理，不为空去除首尾空格
		spec = handleBlankString(spec, true);
		unitName = handleBlankString(unitName, true);

		List<MaterialPlusVO> materialPlusVOS = materialMapper.fuzzyMatchQueryPage(page, tenantId, name, names, type, categoryName, spec, unitName, fuzzyName, fuzzySpec, fuzzyUnitName);
		page.setRecords(materialPlusVOS);
		return page;
	}


	/**
	 * 批量插入档案
	 *
	 * @param materialVOMap 材料设备VO
	 *
	 * @param sourceBillId
	 * @param sourceBillCode
	 * @param sourceBillType
	 * @return {@link CommonResponse}<{@link Map}<{@link String}, {@link MaterialVO}>>
	 *
	 * @throws BillCodeException 单据编码异常
	 */
	@Override
	public CommonResponse<Map<String, MaterialVO>> batchInsertArchive(Map<String, MaterialVO> materialVOMap, Long sourceBillId, String sourceBillCode, String sourceProjectName, String sourceBillType) throws BillCodeException {
		Stopwatch stopwatch = Stopwatch.createStarted();
		logger.info("校验档案是否存在----start"+ JSONObject.toJSONString(materialVOMap));

		if (materialVOMap.isEmpty()) {
			throw new BusinessException("批量插入档案，数据不能为空！");
		}
		Long tenantId = InvocationInfoProxy.getTenantid();

		List<MaterialVO> materialVOS = new ArrayList<>();
		List<String> codeList = materialVOMap.values().stream().filter(e-> StringUtils.isNotBlank(e.getCode())).map(e -> e.getCode()).collect(Collectors.toList());
		if (CollectionUtils.isNotEmpty(codeList)){
			List<MaterialEntity> materialEntities = materialMapper.queryByCodes(InvocationInfoProxy.getTenantid(), codeList);
			if (CollectionUtils.isNotEmpty(materialEntities)){
				throw new BusinessException("物料档案中存在编码为"+materialEntities.get(0).getCode() + "的物料，该物料不能新增！");
			}
		}
		Map<String,String> checkMaps = new HashMap();
		Map<String,MaterialVO> checkDataMaps = new HashMap();
		for (Map.Entry<String, MaterialVO> entry : materialVOMap.entrySet()) {
			MaterialVO materialVO = entry.getValue();
			MaterialCategoryEntity materialCategory = materialCategoryMapper.queryCategoryByTypeAndName(tenantId, materialVO.getType(), materialVO.getCategoryName());
			if (materialCategory == null) {
				return CommonResponse.error("批量插入档案，档案分类:" + materialVO.getCategoryName() + "，不存在！");
			}
			Integer count = materialMapper.queryAllByTenantIdAndNameAndTypeAndSpecAndUnitName(tenantId, materialCategory.getId(), materialVO.getName(), materialVO.getType(), materialVO.getSpec(), materialVO.getUnitName());
			if (count != 1) {
				String billCode = "";
				if (StringUtils.isEmpty(materialVO.getCode())||"".equals(materialVO.getCode())){
					if (materialVO.getType() == 1) {
						billCode = generator.generateBillCodeById(MATERIAL_BILL_CODE, tenantId);
					} else if (materialVO.getType() == 2) {
						billCode = generator.generateBillCodeById(EQUIPMENT_BILL_CODE, tenantId);
					}
				}else {
					billCode = materialVO.getCode();
					//记录物资编码 和 key
				}
				if (checkMaps.containsKey(billCode)){
					if (checkMaps.get(billCode).equals(entry.getKey())){
						//物资编码相同 key相同 跳出出循环不插入数据 反写旧数据
						if (checkDataMaps.containsKey(billCode)){
							MaterialVO vo = checkDataMaps.get(billCode);
							materialVO.setId(vo.getId());
							materialVO.setEnabled(vo.getEnabled());
							materialVO.setCreateUserCode(vo.getCreateUserCode());
							materialVO.setCreateTime(vo.getCreateTime());
							materialVO.setTenantId(vo.getTenantId());
							materialVO.setCategoryId(vo.getCategoryId());
							materialVO.setCategoryName(vo.getCategoryName());
							materialVO.setCategoryCode(vo.getCategoryCode());
							materialVO.setSourceId(vo.getSourceId());
							materialVO.setSystemId(vo.getSystemId());
							materialVO.setCategorySourceId(vo.getCategorySourceId());
							materialVO.setBillState(vo.getBillState());
							entry.setValue(materialVO);
						}
						continue;
					}else {
						//物资编码相同 key不相同 说明添加了2个编码相同的物料 抛出异常
						throw new BusinessException("物料名称为:"+materialVO.getName()+"编码为:"+materialVO.getCode() + "的物料，该物料不能新增！编码重复!");
					}
				}
				materialVO.setId(IdWorker.getId());
				materialVO.setCategoryId(materialCategory.getId());
				materialVO.setCode(billCode);
				materialVO.setEnabled(1);
				materialVO.setCreateUserCode(InvocationInfoProxy.getUsercode());
				materialVO.setCreateTime(new Date());
				materialVO.setTenantId(InvocationInfoProxy.getTenantid());
				materialVO.setSourceId(materialCategory.getSourceId());
				materialVO.setSystemId(materialCategory.getSystemId());
				materialVO.setCategorySourceId(materialCategory.getSourceId());
				materialVO.setCategoryName(materialCategory.getName());
				materialVO.setCategoryCode(materialCategory.getCode());
				// 单据新增的档案默认是未审核
				materialVO.setBillState(Optional.ofNullable(materialVO.getBillState()).orElse(0));
				// 临时档案需要记录来源单据信息
				if (materialVO.getBillState().intValue() == 0) {
					materialVO.setSourceBillId(sourceBillId);
					materialVO.setSourceBillCode(sourceBillCode);
					materialVO.setSourceBillType(sourceBillType);
					materialVO.setSourceProjectName(sourceProjectName);
					materialVO.setHide(0);
				}
				materialVOS.add(materialVO);
				// 更新value值
				entry.setValue(materialVO);
				checkMaps.put(materialVO.getCode(),entry.getKey());
				checkDataMaps.put(billCode,materialVO);
			}
		}

		if (CollectionUtils.isNotEmpty(materialVOS)) {
			materialMapper.insertBatch1(tenantId, materialVOS);
		}
		stopwatch.stop();
		logger.info("校验档案是否存在----end：已处理设备档案【{}】条数据，共用时【{}】毫秒", materialVOMap.isEmpty() ? 0 : materialVOMap.size(), stopwatch.elapsed(TimeUnit.MILLISECONDS));

		return CommonResponse.success(materialVOMap);
	}

	@Override
	public List<MaterialEntity> queryMaterialListByCodes(List<String> codeList) {
		return materialMapper.queryMaterialListByCodes(InvocationInfoProxy.getTenantid(), codeList);
	}

	@Override
	public void allowIn(List<Long> ids) {
		materialMapper.allowIn(InvocationInfoProxy.getTenantid(), ids);
	}

	@Override
	public void hide(List<Long> ids) {
		materialMapper.hide(InvocationInfoProxy.getTenantid(), ids);
	}
}
