package com.ejianc.foundation.support.service.impl;

import java.text.SimpleDateFormat;
import java.util.Arrays;
import java.util.Date;
import java.util.List;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import com.ejianc.foundation.billcode.BillCodeException;
import com.ejianc.foundation.billcode.engine.persistence.IBillCodeEngineService;
import com.ejianc.foundation.billcode.engine.persistence.vo.BillCodeSNVO;
import com.ejianc.foundation.billcode.service.IBillCodeProvider;
import com.ejianc.foundation.support.service.IBillCodeGenerator;
import com.ejianc.foundation.support.service.IBillCodeRuleService;
import com.ejianc.foundation.support.vo.BillCodeParam;
import com.ejianc.foundation.support.vo.BillCodeRuleAttrVO;
import com.ejianc.foundation.support.vo.BillCodeRuleVO;
import com.ejianc.framework.cache.redis.CacheManager;
import com.ejianc.framework.core.exception.BusinessException;

@Service("generator")
public class SimpleBillCodeGeneratorImpl implements IBillCodeGenerator {
	private Logger logger = LoggerFactory.getLogger(this.getClass());
	
    private final String BILL_CODE_PREFIX = "ejcbillcode:";

    @Autowired
    private IBillCodeRuleService billCodeRuleService;

    @Autowired
    private IBillCodeProvider provider;

   @Autowired
   private CacheManager cacheManager;

    @Autowired
    private IBillCodeEngineService billCodeEngineService;

    @Override
    public String generateBillCodeById(String code, Long tenantId) throws BillCodeException {
        BillCodeRuleVO ruleVO = billCodeRuleService.queryDetailByCodeAndTenantId(code, tenantId);
        checkIsBeyondMaxSn(ruleVO, null, 1, tenantId);
        return provider.getBillCode(ruleVO, null, null, null, tenantId);
    }
    
    @Override
    public String generateBillCode(BillCodeParam billCodeParam) throws BillCodeException {
    	BillCodeRuleVO ruleVO = billCodeRuleService.queryBillCodeDetail(billCodeParam);
    	
    	Long waterBasicId = null;
    	List<BillCodeRuleAttrVO> billCodeRuleAttrList = ruleVO.getAttrs();
    	if(billCodeRuleAttrList != null && billCodeRuleAttrList.size() > 0) {
    		for(BillCodeRuleAttrVO billCodeRuleAttrVo:billCodeRuleAttrList) {
    			if("0".equals(billCodeRuleAttrVo.getElemType()) && "1".equals(billCodeRuleAttrVo.getElemIsRefer())) {
    				try {
    					waterBasicId = Long.parseLong(billCodeRuleAttrVo.getBillValue());
    				}catch(Exception e) {
    					waterBasicId = null;
    					logger.info("流水依据必须是Long类型的字段："+e);
    				}
    			}
    		}
    	}
    	if(waterBasicId != null) {
    		billCodeParam.setWaterBasicId(waterBasicId);
    	}
    	if(billCodeParam.getWaterBasicId() == null) {
    		billCodeParam.setWaterBasicId(billCodeParam.getTenantId());
    	}
    	
    	checkIsBeyondMaxSn(ruleVO, null, 1, billCodeParam.getWaterBasicId());
        return provider.getBillCode(ruleVO, null, null, null, billCodeParam.getWaterBasicId());
	}

    @Override
    public List<String> generateBillCodeById(String code, Long tenantId, int count) throws BillCodeException {
        BillCodeRuleVO ruleVO = billCodeRuleService.queryDetailByCodeAndTenantId(code, tenantId);
        checkIsBeyondMaxSn(ruleVO, null, count, tenantId);
        String[] codes = provider.getBillCode(ruleVO, null, null, null, count, tenantId);
        return Arrays.asList(codes);
    }

    /**
     * 校验是否超过流水号最大值
     *
     * @param ruleVO 编码规则vo
     * @param customSnReferMark 自定义流水依据标志
     * @param count 生成单据号的数量
     */
    private void checkIsBeyondMaxSn(BillCodeRuleVO ruleVO, String customSnReferMark, int count, Long tenantId) {
        List<BillCodeRuleAttrVO> attrs = ruleVO.getAttrs();
        StringBuffer markstr = new StringBuffer();
        int maxSn = 0;
        for(BillCodeRuleAttrVO vo : attrs) {
            maxSn = getMaxSn(maxSn, vo);
            markstr = getMarkstr(markstr, customSnReferMark, vo);
        }

        String cacheKey = getBillCodeSnReferCacheKey(ruleVO.getId().toString(), markstr.toString(), tenantId);
        BillCodeSNVO billcodesnVO = (BillCodeSNVO) cacheManager.get(cacheKey);
        if(null == billcodesnVO) {
            billcodesnVO = billCodeEngineService.querySNVO(ruleVO.getId().toString(), markstr.toString(), tenantId);
            cacheManager.set(cacheKey, billcodesnVO);
        }
        cacheKey = null;

        if(null != billcodesnVO) {
            int lastSn = Integer.parseInt(billcodesnVO.getLastsn()) + count;
            if (lastSn > maxSn) {
                throw new BusinessException("流水号超了可用流水号最大值:" + maxSn + ",请重新调整单据编码规则");
            }
        }
    }

    private String getBillCodeSnReferCacheKey(String ruleId, String billCodeSNRefer, Long tenantId) {
        return BILL_CODE_PREFIX + ruleId + billCodeSNRefer + tenantId;
    }

    private int getMaxSn(int maxSn, BillCodeRuleAttrVO elemVO) {
        if (BillCodeRuleAttrVO.ELEM_TYPE_SN.equals(elemVO.getElemType())) {
            // 流水号

            // 流水号位数
            int snLength = elemVO.getElemLength();
            int snNum = 1;
            for (int i = 0; i < snLength - 1; i++) {
                snNum += Math.pow(10, i + 1);
            }
            // 流水号最大值
            maxSn = snNum * 9;
        }
        return maxSn;
    }

    private StringBuffer getMarkstr(StringBuffer markstr, String customSnReferMark, BillCodeRuleAttrVO elemVO) {
        if(BillCodeRuleAttrVO.ELEM_TYPE_TIME.equals(elemVO.getElemType())) {
            Date currentTime = new Date();
            switch (elemVO.getElemIsRefer()) {
                // 依据年度
                case BillCodeRuleAttrVO.ELEM_ISREFER_TIME_YEAR:
                    SimpleDateFormat yearFormat = new SimpleDateFormat("yyyy");
                    markstr.append("^sysdate^" + yearFormat.format(currentTime));
                    break;
                // 依据月度
                case BillCodeRuleAttrVO.ELEM_ISREFER_TIME_MONTH:
                    SimpleDateFormat monthFormat = new SimpleDateFormat("yyyyMM");
                    markstr.append("^sysdate^" + monthFormat.format(currentTime));
                    break;
                // 依据每日
                case BillCodeRuleAttrVO.ELEM_ISREFER_TIME_DAY:
                    SimpleDateFormat dayFormat = new SimpleDateFormat("yyyyMMdd");
                    markstr.append("^sysdate^" + dayFormat.format(currentTime));
                    break;
                // 非流水依据(默认)
                default:
            }
        } else if(BillCodeRuleAttrVO.ELEM_TYPE_CUSTOM_SN.equals(elemVO.getElemType())) {
            markstr.append("^custom^" + customSnReferMark);
        }
        return markstr;
    }

}
