package com.ejianc.business.targetcost.utils;

import com.alibaba.fastjson.JSONObject;
import com.baomidou.mybatisplus.core.conditions.query.LambdaQueryWrapper;
import com.ejianc.business.procost.api.ICostDetailApi;
import com.ejianc.business.procost.vo.CostDetailVO;
import com.ejianc.business.targetcost.bean.DutyEntity;
import com.ejianc.business.targetcost.bean.RuleDetailEntity;
import com.ejianc.business.targetcost.bean.RuleEntity;
import com.ejianc.business.targetcost.enums.ControlTypeEnum;
import com.ejianc.business.targetcost.service.*;
import com.ejianc.business.targetcost.vo.CostCtrlDetailVO;
import com.ejianc.business.targetcost.vo.CostCtrlVO;
import com.ejianc.business.targetcost.vo.DutyDetailItemVO;
import com.ejianc.foundation.orgcenter.api.IOrgApi;
import com.ejianc.foundation.orgcenter.vo.OrgVO;
import com.ejianc.foundation.share.api.IShareSubjectOrgApi;
import com.ejianc.foundation.share.vo.SubjectOrgVO;
import com.ejianc.framework.core.response.BillStateEnum;
import com.ejianc.framework.core.response.CommonResponse;
import com.ejianc.framework.core.response.Parameter;
import com.ejianc.framework.core.response.QueryParam;
import com.ejianc.framework.core.util.ComputeUtil;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.redisson.api.RLock;
import org.redisson.api.RedissonClient;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import java.math.BigDecimal;
import java.util.*;
import java.util.concurrent.TimeUnit;
import java.util.function.Function;
import java.util.stream.Collectors;

/**
 * 成本控制
 */
@Component
public class SubjectCtrl {

    private final Logger logger = LoggerFactory.getLogger(getClass());

    @Autowired
    private IOrgApi orgApi;
    @Autowired
    private IRuleService ruleService;
    @Autowired
    private IRuleDetailService ruleDetailService;
    @Autowired
    private RedissonClient redissonClient;
    @Autowired
    private IDutyService dutyService;
    @Autowired
    private IDutyDetailItemService dutyDetailItemService;
    @Autowired
    private ICostDetailApi costDetailApi;
    @Autowired
    private IShareSubjectOrgApi shareSubjectOrgApi;


    private static final String TC_LOCK = "TC_LOCK::";

    public CommonResponse<JSONObject> ctrlCost(CostCtrlVO costCtrlVO) {
        JSONObject jsonObject = new JSONObject();
        //获取锁对象
        String lockKey = TC_LOCK + costCtrlVO.getProjectId();
        RLock mylock = redissonClient.getLock(lockKey);
        try {
            //加锁，并且设置锁过期时间，防止死锁的产生
            boolean lock = mylock.tryLock(5, TimeUnit.SECONDS);
            if (!lock) {
                return CommonResponse.error("获取锁失败：" + costCtrlVO.getProjectId());
            }

            LambdaQueryWrapper<DutyEntity> feeQuery = new LambdaQueryWrapper<>();
            feeQuery.eq(DutyEntity::getProjectId, costCtrlVO.getProjectId());
            feeQuery.eq(DutyEntity::getEnableState, true);
            feeQuery.eq(DutyEntity::getGatherFlag, true);
            DutyEntity dutyEntity = dutyService.getOne(feeQuery);
            if (null == dutyEntity) {
                return CommonResponse.success("该项目还没编制目标成本");
            }

            CommonResponse<List<OrgVO>> parents = orgApi.findParentsByOrgId(costCtrlVO.getOrgId());
            Set<Long> ruleIdSet = new HashSet<>();
            List<RuleEntity> ruleEntities = null;
            Map<Long, RuleEntity> ruleMap = new HashMap<>();
            if (parents.isSuccess() && parents.getData() != null && parents.getData().size() > 0) {
                List<Long> orgList = parents.getData().stream().map(OrgVO::getId).collect(Collectors.toList());
                LambdaQueryWrapper<RuleEntity> queryWrapper = new LambdaQueryWrapper<>();
                queryWrapper.eq(RuleEntity::getFeeId, dutyEntity.getFeeId());
                List<Integer> billStatusList = new ArrayList<>();
                billStatusList.add(BillStateEnum.PASSED_STATE.getBillStateCode());
                billStatusList.add(BillStateEnum.COMMITED_STATE.getBillStateCode());
                queryWrapper.eq(RuleEntity::getEnabled, 1);
                queryWrapper.in(RuleEntity::getBillState, billStatusList);
                queryWrapper.in(RuleEntity::getOrgId, orgList);
                ruleEntities = ruleService.list(queryWrapper);
                if (CollectionUtils.isNotEmpty(ruleEntities)) {
                    for (RuleEntity ruleEntity : ruleEntities) {
                        ruleIdSet.add(ruleEntity.getId());
                        ruleMap.put(ruleEntity.getId(), ruleEntity);
                    }
                }
            }

            if (CollectionUtils.isEmpty(ruleIdSet)) {
                return CommonResponse.success("该业务单据对应组织没有设置控制规则");
            }

            List<Long> longList = costCtrlVO.getDetailList().stream().map(CostCtrlDetailVO::getSubjectId).collect(Collectors.toList());

            Map<Long, String> subjectMap = getSubjectMap(longList);

            LambdaQueryWrapper<RuleDetailEntity> ruleDetailQuery = new LambdaQueryWrapper<>();
            ruleDetailQuery.eq(RuleDetailEntity::getSubjectFlag, true);
            ruleDetailQuery.in(RuleDetailEntity::getRuleId, new ArrayList<>(ruleIdSet));
//            ruleDetailQuery.in(RuleDetailEntity::getSubjectId, longList);
            List<RuleDetailEntity> ruleDetailList = ruleDetailService.list(ruleDetailQuery);

            if(CollectionUtils.isEmpty(ruleDetailList)){
                return CommonResponse.success("该业务单据对应组织没有设置成本科目控制规则");
            }

            List<RuleDetailEntity> ruleDetailEntities = new ArrayList<>();
            Set<Long> costQuerySet = new HashSet<>();
            for(Long subjectId : longList){
                String innerCode = subjectMap.get(subjectId);
                for(RuleDetailEntity ruleDetailEntity : ruleDetailList){
                    if (StringUtils.isNotBlank(innerCode) && innerCode.contains(String.valueOf(ruleDetailEntity.getSubjectId()))) {
                        ruleDetailEntities.add(ruleDetailEntity);
                        costQuerySet.add(ruleDetailEntity.getSubjectId());
                    }
                }
            }

            if(CollectionUtils.isEmpty(ruleDetailEntities)){
                return CommonResponse.success("该业务单据所包含的成本科目没有设置控制规则");
            }

            CommonResponse<List<CostDetailVO>> listCommonResponse = costDetailApi.queryDetailByProjectId(costCtrlVO.getSourceId(), costCtrlVO.getProjectId(), new ArrayList<>(costQuerySet));
            Map<Long, CostDetailVO> costMap = new HashMap<>();
            if(listCommonResponse.isSuccess() && CollectionUtils.isNotEmpty(listCommonResponse.getData())){
                costMap = listCommonResponse.getData().stream().collect(Collectors.toMap(CostDetailVO::getSubjectId, Function.identity(), (key1, key2) -> key1));
            }

            Map<Long, RuleDetailEntity> gxMap = new HashMap<>();
            Map<Long, RuleDetailEntity> rxMap = new HashMap<>();
            for (RuleDetailEntity entity : ruleDetailEntities) {
                if (ControlTypeEnum.刚性控制.getCode().equals(entity.getControlType())) {
                    Long key = entity.getSubjectId();
                    RuleDetailEntity controlScale = null;
                    if (gxMap.containsKey(key)) {
                        controlScale = ComputeUtil.isLessThan(gxMap.get(key).getControlScale(), entity.getControlScale()) ? gxMap.get(key) : entity;
                    } else {
                        controlScale = entity;
                    }
                    controlScale.setOrgName(ruleMap.get(controlScale.getRuleId()).getOrgName());
                    gxMap.put(key, controlScale);
                } else if (ControlTypeEnum.柔性控制.getCode().equals(entity.getControlType())) {
                    Long key = entity.getSubjectId();
                    RuleDetailEntity controlScale = null;
                    if (rxMap.containsKey(key)) {
                        controlScale = ComputeUtil.isLessThan(rxMap.get(key).getControlScale(), entity.getControlScale()) ? rxMap.get(key) : entity;
                    } else {
                        controlScale = entity;
                    }
                    controlScale.setOrgName(ruleMap.get(controlScale.getRuleId()).getOrgName());
                    rxMap.put(key, controlScale);
                }
            }

            Map<Long, DutyDetailItemVO> dutyCostMap = new HashMap<>();

            StringBuffer msg = new StringBuffer();
            for (CostCtrlDetailVO detailVO : costCtrlVO.getDetailList()) {
                String innerCode = subjectMap.get(detailVO.getSubjectId());
                for(Long key : gxMap.keySet()){
                    if (StringUtils.isNotBlank(innerCode) && innerCode.contains(String.valueOf(key))) {
                        Long subjectId = gxMap.get(key).getSubjectId();
                        CostDetailVO costDetailVO = costMap.containsKey(subjectId) ? costMap.get(subjectId) : new CostDetailVO();
                        DutyDetailItemVO dutyDetailItemVO;
                        if (dutyCostMap.containsKey(subjectId)) {
                            dutyDetailItemVO = dutyCostMap.get(subjectId);
                        } else {
                            dutyDetailItemVO = dutyDetailItemService.queryCostMnyBySubjectId(costCtrlVO.getProjectId(), subjectId);
                            dutyCostMap.put(subjectId, dutyDetailItemVO);
                        }
                        msg.append(ctrlCost(gxMap.get(key), costDetailVO, dutyDetailItemVO, detailVO, ControlTypeEnum.刚性控制.getCode()));
                    }
                }
            }
            if (StringUtils.isNotBlank(msg)) {
                jsonObject.put(String.valueOf(ControlTypeEnum.刚性控制.getCode()), msg);
                return CommonResponse.error("控制不通过", jsonObject);
            }

            for (CostCtrlDetailVO detailVO : costCtrlVO.getDetailList()) {
                String innerCode = subjectMap.get(detailVO.getSubjectId());
                for(Long key : rxMap.keySet()){
                    if (StringUtils.isNotBlank(innerCode) && innerCode.contains(String.valueOf(key))) {
                        Long subjectId = rxMap.get(key).getSubjectId();
                        CostDetailVO costDetailVO = costMap.containsKey(subjectId) ? costMap.get(subjectId) : new CostDetailVO();
                        DutyDetailItemVO dutyDetailItemVO;
                        if(dutyCostMap.containsKey(subjectId)){
                            dutyDetailItemVO = dutyCostMap.get(subjectId);
                        }else{
                            dutyDetailItemVO = dutyDetailItemService.queryCostMnyBySubjectId(costCtrlVO.getProjectId(), subjectId);
                            dutyCostMap.put(subjectId, dutyDetailItemVO);
                        }
                        msg.append(ctrlCost(rxMap.get(key), costDetailVO, dutyDetailItemVO, detailVO, ControlTypeEnum.柔性控制.getCode()));
                    }
                }
            }
            if (StringUtils.isNotBlank(msg)) {
                jsonObject.put(String.valueOf(ControlTypeEnum.柔性控制.getCode()), msg);
                return CommonResponse.error("控制不通过", jsonObject);
            }
        } catch (Exception e) {
            logger.error("保存前成本科目控制失败:" + JSONObject.toJSONString(e));
        } finally {
            mylock.unlock();
        }
        return CommonResponse.success("保存前成本科目控制通过");
    }

    private String ctrlCost(RuleDetailEntity ruleDetailEntity, CostDetailVO costDetailVO, DutyDetailItemVO detailItemVO, CostCtrlDetailVO ctrlDetailVO, Integer controlType) {
        StringBuffer msg = new StringBuffer();
        BigDecimal taxMny = null != detailItemVO.getTaxMny() ? detailItemVO.getTaxMny() : BigDecimal.ZERO;
        BigDecimal comTaxMny = ComputeUtil.safeAdd(costDetailVO.getHappenTaxMny(), ctrlDetailVO.getTaxMny());
        BigDecimal controlScale = ruleDetailEntity.getControlScale();
        BigDecimal comTaxScale = ComputeUtil.safeMultiply(ComputeUtil.safeDiv(comTaxMny, taxMny), BigDecimal.valueOf(100));
        if (ComputeUtil.isLessThan(controlScale, comTaxScale) || null == detailItemVO.getTaxMny()) {
            msg.append(costDetailVO.getSubjectName()).append("总额控").append("&");
            msg.append(ruleDetailEntity.getOrgName()).append("&");
            StringBuffer tmpMsg = new StringBuffer();
            tmpMsg.append(costDetailVO.getSubjectName()).append("金额(").append(ComputeUtil.scale(comTaxMny, 2)).append("元)已占目标成本中该成本科目金额");
            tmpMsg.append("(").append(ComputeUtil.scale(taxMny, 2)).append("元)").append(ComputeUtil.scale(comTaxScale, 2)).append("%");
            msg.append(getBaseMsg(tmpMsg.toString(), controlType));
        }
        return msg.toString();
    }

    private String getBaseMsg(String midMsg, Integer controlType) {
        StringBuffer msg = new StringBuffer();
        if (ControlTypeEnum.刚性控制.getCode().equals(controlType)) {
            msg.append("刚性：");
        } else if (ControlTypeEnum.柔性控制.getCode().equals(controlType)) {
            msg.append("柔性：");
        }
        msg.append(midMsg);
        if (ControlTypeEnum.刚性控制.getCode().equals(controlType)) {
            msg.append("，不能保存！/n");
        } else if (ControlTypeEnum.柔性控制.getCode().equals(controlType)) {
            msg.append("！/n");
        }
        return msg.toString();
    }

    private Map<Long, String> getSubjectMap(List<Long> subjectIdList) {
        QueryParam queryParam = new QueryParam();
        queryParam.getParams().put("id", new Parameter(QueryParam.IN, subjectIdList));
        CommonResponse<List<SubjectOrgVO>> response = shareSubjectOrgApi.querySubjectOrg(queryParam);
        if (response.isSuccess() && CollectionUtils.isNotEmpty(response.getData())) {
            List<SubjectOrgVO> subjectOrgVOS = response.getData();
            return subjectOrgVOS.stream().collect(Collectors.toMap(SubjectOrgVO::getId, SubjectOrgVO::getInnerCode, (key1, key2) -> key1));
        }
        return new HashMap<>();
    }
}
