package com.ejianc.business.targetcost.utils;

import com.alibaba.fastjson.JSONObject;
import com.baomidou.mybatisplus.core.conditions.query.LambdaQueryWrapper;
import com.ejianc.business.center.api.IWarnCenterApi;
import com.ejianc.business.center.vo.EarlyWarnTransVO;
import com.ejianc.business.procost.api.ICostDetailApi;
import com.ejianc.business.procost.vo.CostDetailVO;
import com.ejianc.business.targetcost.bean.*;
import com.ejianc.business.targetcost.enums.ControlTypeEnum;
import com.ejianc.business.targetcost.enums.CtrlMnyTypeEnum;
import com.ejianc.business.targetcost.service.*;
import com.ejianc.business.targetcost.vo.DutyDetailItemVO;
import com.ejianc.business.warn.vo.WarningReceiveVO;
import com.ejianc.foundation.orgcenter.vo.OrgVO;
import com.ejianc.foundation.share.api.IShareSubjectOrgApi;
import com.ejianc.framework.core.context.InvocationInfoProxy;
import com.ejianc.framework.core.response.BillStateEnum;
import com.ejianc.framework.core.response.CommonResponse;
import com.ejianc.framework.core.util.ComputeUtil;
import org.apache.commons.collections.CollectionUtils;
import org.redisson.api.RLock;
import org.redisson.api.RedissonClient;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.web.context.request.RequestAttributes;
import org.springframework.web.context.request.RequestContextHolder;

import java.math.BigDecimal;
import java.util.*;
import java.util.concurrent.Callable;
import java.util.concurrent.TimeUnit;
import java.util.stream.Collectors;

/**
 * 成本科目控制预警
 */
public class SubjectWarnCallable implements Callable<CommonResponse<String>> {

    private final Logger logger = LoggerFactory.getLogger(getClass());

    private static final String TC_LOCK = "TC_LOCK::";

    private static final String PC_URL = "/ejc-targetcost-frontend/#/execution?";

    private static final String WARN_ID = "584793299912114241";
    private static final String WARN_PARAM_ID = "584793299912114241";

    private String authority;
    private RequestAttributes context;
    private IWarnCenterApi warnCenterApi;
    private RedissonClient redissonClient;
    private IDutyDetailItemService dutyDetailItemService;
    private List<DutyEntity> dutyEntities;
    private IRuleService ruleService;
    private IRuleDetailService ruleDetailService;
    private IRuleReceiverService ruleReceiverService;
    private ICostDetailApi costDetailApi;
    private IShareSubjectOrgApi shareSubjectOrgApi;
    private IFeeService feeService;

    public SubjectWarnCallable(String authority, RequestAttributes context, IWarnCenterApi warnCenterApi, RedissonClient redissonClient, IDutyDetailItemService dutyDetailItemService,
                               List<DutyEntity> dutyEntities, IRuleService ruleService, IRuleDetailService ruleDetailService,
                               IRuleReceiverService ruleReceiverService, ICostDetailApi costDetailApi, IShareSubjectOrgApi shareSubjectOrgApi, IFeeService feeService) {
        this.authority = authority;
        this.context = context;
        this.warnCenterApi = warnCenterApi;
        this.redissonClient = redissonClient;
        this.dutyDetailItemService = dutyDetailItemService;
        this.dutyEntities = dutyEntities;
        this.ruleService = ruleService;
        this.ruleDetailService = ruleDetailService;
        this.ruleReceiverService = ruleReceiverService;
        this.costDetailApi = costDetailApi;
        this.shareSubjectOrgApi = shareSubjectOrgApi;
        this.feeService = feeService;
    }

    @Override
    public CommonResponse<String> call() throws Exception {
        context.setAttribute("authority", authority, RequestAttributes.SCOPE_REQUEST);
        RequestContextHolder.setRequestAttributes(context);
        InvocationInfoProxy.setExtendAttribute("authority", authority);
        List<EarlyWarnTransVO> warnList = new ArrayList<>();
        for (DutyEntity dutyEntity : dutyEntities) {
            //获取锁对象
            String lockKey = TC_LOCK + dutyEntity.getProjectId();
            RLock mylock = redissonClient.getLock(lockKey);
            try {
                //加锁，并且设置锁过期时间，防止死锁的产生
                boolean lock = mylock.tryLock(20, TimeUnit.SECONDS);
                if (!lock) {
                    logger.error("获取锁失败 is " + dutyEntity.getProjectId());
                    return CommonResponse.error("获取锁失败：" + dutyEntity.getProjectId());
                }

                FeeEntity feeEntity = feeService.selectById(dutyEntity.getFeeId());

                List<OrgVO> orgVOList = dutyEntity.getOrgList();
                Set<Long> ruleIdSet = new HashSet<>();
                List<RuleEntity> ruleEntities = null;
                Map<Long, RuleEntity> ruleMap = new HashMap<>();
                if (CollectionUtils.isNotEmpty(orgVOList)) {
                    List<Long> orgList = orgVOList.stream().map(OrgVO::getId).collect(Collectors.toList());
                    LambdaQueryWrapper<RuleEntity> queryWrapper = new LambdaQueryWrapper<>();
                    queryWrapper.eq(RuleEntity::getFeeId, dutyEntity.getFeeId());
                    List<Integer> billStatusList = new ArrayList<>();
                    billStatusList.add(BillStateEnum.PASSED_STATE.getBillStateCode());
                    billStatusList.add(BillStateEnum.COMMITED_STATE.getBillStateCode());
                    queryWrapper.in(RuleEntity::getBillState, billStatusList);
                    queryWrapper.in(RuleEntity::getOrgId, orgList);
                    ruleEntities = ruleService.list(queryWrapper);
                    if (CollectionUtils.isNotEmpty(ruleEntities)) {
                        for (RuleEntity ruleEntity : ruleEntities) {
                            ruleIdSet.add(ruleEntity.getId());
                            ruleMap.put(ruleEntity.getId(), ruleEntity);
                        }
                    }
                }

                if (CollectionUtils.isEmpty(ruleIdSet)) {
                    logger.error("该责任成本没有设置控制规则：" + JSONObject.toJSONString(dutyEntity.getProjectId()));
                    continue;
                }

                LambdaQueryWrapper<RuleDetailEntity> ruleDetailQuery = new LambdaQueryWrapper<>();
                ruleDetailQuery.eq(RuleDetailEntity::getSubjectFlag, true);
                ruleDetailQuery.in(RuleDetailEntity::getControlType, ControlTypeEnum.风险预警.getCode());
                ruleDetailQuery.in(RuleDetailEntity::getRuleId, ruleIdSet);
                List<RuleDetailEntity> ruleDetailEntities = ruleDetailService.list(ruleDetailQuery);

                if(CollectionUtils.isEmpty(ruleDetailEntities)){
                    logger.error("该责任成本没有设置成本科目控制规则" + JSONObject.toJSONString(dutyEntity.getProjectId()));
                    continue;
                }

                Map<Long, List<RuleDetailEntity>> ruleDetailMap = new HashMap<>();
                List<Long> ruleDetailList = new ArrayList<>();
                Set<Long> costQuerySet = new HashSet<>();
                for (RuleDetailEntity entity : ruleDetailEntities) {
                    Long key = entity.getSubjectId();
                    List<RuleDetailEntity> list = ruleDetailMap.containsKey(key) ? ruleDetailMap.get(key) : new ArrayList<>();
                    list.add(entity);
                    ruleDetailMap.put(key, list);
                    ruleDetailList.add(entity.getId());
                    costQuerySet.add(key);
                }

                CommonResponse<List<CostDetailVO>> listCommonResponse = costDetailApi.queryDetailByProjectId(null, dutyEntity.getProjectId(), new ArrayList<>(costQuerySet));
                if (!listCommonResponse.isSuccess() || CollectionUtils.isEmpty(listCommonResponse.getData())) {
                    logger.error(JSONObject.toJSONString(dutyEntity.getProjectId()));
                    continue;
                }
                List<CostDetailVO> costDetailVOS = listCommonResponse.getData();

                LambdaQueryWrapper<RuleReceiverEntity> query = new LambdaQueryWrapper<>();
                query.in(RuleReceiverEntity::getRuleDetailId, ruleDetailList);
                List<RuleReceiverEntity> receiverEntities = ruleReceiverService.list(query);
                Map<Long, List<RuleReceiverEntity>> receiverMap = new HashMap<>();
                if (CollectionUtils.isNotEmpty(receiverEntities)) {
                    for (RuleReceiverEntity receiverEntity : receiverEntities) {
                        List<RuleReceiverEntity> list = null;
                        if (receiverMap.containsKey(receiverEntity.getRuleDetailId())) {
                            list = receiverMap.get(receiverEntity.getRuleDetailId());
                        } else {
                            list = new ArrayList<>();
                        }
                        list.add(receiverEntity);
                        receiverMap.put(receiverEntity.getRuleDetailId(), list);
                    }
                }

                Map<Long, DutyDetailItemVO> dutyCostMap = new HashMap<>();

                List<EarlyWarnTransVO> warnTransVOS = new ArrayList<>();
                for (CostDetailVO detailVO : costDetailVOS) {
                    Long subjectId = detailVO.getSubjectId();
                    if(ruleDetailMap.containsKey(subjectId)) {
                        DutyDetailItemVO dutyDetailItemVO;
                        if (dutyCostMap.containsKey(subjectId)) {
                            dutyDetailItemVO = dutyCostMap.get(subjectId);
                        } else {
                            dutyDetailItemVO = dutyDetailItemService.queryCostMnyBySubjectId(dutyEntity.getProjectId(), subjectId);
                            dutyCostMap.put(subjectId, dutyDetailItemVO);
                        }
                        warnTransVOS.addAll(ctrlCost(dutyDetailItemVO, ruleDetailMap.get(subjectId), dutyEntity, detailVO, receiverMap, ruleMap, feeEntity.getCtrlMnyType()));
                    }
                }
                logger.error("warnTransVOS is " + JSONObject.toJSONString(warnTransVOS));
                if (CollectionUtils.isNotEmpty(warnTransVOS)) {
                    warnList.addAll(warnTransVOS);
                }
            } catch (Exception e) {
                logger.error("目标成本预警失败:" + JSONObject.toJSONString(e));
            } finally {
                mylock.unlock();
            }
            if (CollectionUtils.isNotEmpty(warnList)) {
                CommonResponse<String> warnResp = warnCenterApi.sendToWarnCenterWithReceive(warnList, Long.valueOf(WARN_ID));
                if (!warnResp.isSuccess()) {
                    logger.error("回调预警任务服务失败，{}", warnResp.getMsg());
                    logger.error("dutyEntity" + JSONObject.toJSONString(dutyEntity));
                    continue;
                }
            }
        }
        return CommonResponse.success("目标成本预警成功");
    }

    private List<EarlyWarnTransVO> ctrlCost(DutyDetailItemVO detailItemVO, List<RuleDetailEntity> ruleList, DutyEntity dutyEntity, CostDetailVO costDetailVO,
                                            Map<Long, List<RuleReceiverEntity>> receiverMap, Map<Long, RuleEntity> ruleMap, Integer ctrlMnyType) {
        List<EarlyWarnTransVO> list = new ArrayList<>();
        if (CollectionUtils.isNotEmpty(ruleList)) {
            for (RuleDetailEntity ruleDetailEntity : ruleList) {
                if (!receiverMap.containsKey(ruleDetailEntity.getId())) {
                    continue;
                }
                BigDecimal controlScale = ruleDetailEntity.getControlScale();
                EarlyWarnTransVO earlyWarnTransVO = dutyRuleToWarnVO(detailItemVO, ruleDetailEntity, controlScale, dutyEntity, costDetailVO, receiverMap.get(ruleDetailEntity.getId()), ruleMap, ctrlMnyType);
                if (null != earlyWarnTransVO) {
                    list.add(earlyWarnTransVO);
                }
            }
        }
        return list;
    }

    private EarlyWarnTransVO dutyRuleToWarnVO(DutyDetailItemVO detailItemVO, RuleDetailEntity ruleDetailEntity, BigDecimal controlScale, DutyEntity dutyEntity,
                                              CostDetailVO costDetailVO, List<RuleReceiverEntity> receiverEntities, Map<Long, RuleEntity> ruleMap, Integer ctrlMnyType) {
        List<WarningReceiveVO> receiveList = new ArrayList<>();
        if (CollectionUtils.isNotEmpty(receiverEntities)) {
            for (RuleReceiverEntity receiverEntity : receiverEntities) {
                WarningReceiveVO receiveVO = new WarningReceiveVO();
                receiveVO.setNameId(receiverEntity.getNameId());
                receiveVO.setName(receiverEntity.getName());
                receiveVO.setType(receiverEntity.getType());
                receiveVO.setWarningId(ruleDetailEntity.getId());
                receiveList.add(receiveVO);
            }
        }
        EarlyWarnTransVO vo = null;
        BigDecimal taxMny = CtrlMnyTypeEnum.无税.getCode().equals(ctrlMnyType) ? ComputeUtil.nullToZero(detailItemVO.getMny()) : ComputeUtil.nullToZero(detailItemVO.getTaxMny());
        BigDecimal comTaxMny = CtrlMnyTypeEnum.无税.getCode().equals(ctrlMnyType) ? costDetailVO.getHappenMny() : costDetailVO.getHappenTaxMny();
        BigDecimal comTaxScale = ComputeUtil.safeMultiply(ComputeUtil.safeDiv(comTaxMny, taxMny), BigDecimal.valueOf(100));
        if (ComputeUtil.isLessThan(controlScale, comTaxScale)) {
            vo = new EarlyWarnTransVO();
            vo.setBillName(dutyEntity.getProjectName());
            vo.setPcTitle("目标成本执行情况");
            vo.setPcUrl(PC_URL + "projectId=" + dutyEntity.getProjectId() + "&&" + "projectName=" + dutyEntity.getProjectName());
            RuleEntity ruleEntity = ruleMap.get(ruleDetailEntity.getRuleId());
            vo.setOrgId(ruleEntity.getOrgId());
            vo.setOrgName(ruleEntity.getOrgName());
            vo.setSourceId(String.valueOf(dutyEntity.getId()));
            vo.setTenantId(dutyEntity.getTenantId());
            vo.setWarnLevel("中");
            vo.setWarnSetId(Long.valueOf(WARN_ID));
            vo.setWarnSetParamId(Long.valueOf(WARN_ID));
            vo.setWarnType("1");
            vo.setSendOrgId(String.valueOf(ruleEntity.getOrgId()));
            vo.setDef1(String.valueOf(ruleDetailEntity.getId()));
            vo.setEarlywarnName("成本科目-[" + costDetailVO.getSubjectName() + "]总额控预警");
            vo.setReceiveList(receiveList);
            StringBuffer warnStr = new StringBuffer();
            warnStr.append("成本科目-[" + costDetailVO.getSubjectName() + "]");
            warnStr.append("金额(").append(ComputeUtil.scale(comTaxMny, 2)).append("元)已占用目标成本中该成本科目金额");
            warnStr.append("(").append(ComputeUtil.scale(taxMny, 2)).append("元)").append(ComputeUtil.scale(comTaxScale, 2)).append("%! \n");
            vo.setEarlywarnContent(warnStr.toString());
        }
        return vo;
    }
}
