package com.ejianc.business.targetcost.utils;

import com.alibaba.fastjson.JSONObject;
import com.baomidou.mybatisplus.core.conditions.query.LambdaQueryWrapper;
import com.ejianc.business.targetcost.bean.*;
import com.ejianc.business.targetcost.enums.BillCategoryEnum;
import com.ejianc.business.targetcost.enums.ControlTypeEnum;
import com.ejianc.business.targetcost.enums.DocTypeEnum;
import com.ejianc.business.targetcost.enums.FeeTypeEnum;
import com.ejianc.business.targetcost.service.*;
import com.ejianc.business.targetcost.vo.DetailExecutionVO;
import com.ejianc.business.targetcost.vo.ExecutionVO;
import com.ejianc.business.targetcost.vo.TotalExecutionVO;
import com.ejianc.foundation.orgcenter.api.IOrgApi;
import com.ejianc.foundation.orgcenter.vo.OrgVO;
import com.ejianc.framework.core.kit.mapper.BeanMapper;
import com.ejianc.framework.core.response.BillStateEnum;
import com.ejianc.framework.core.response.CommonResponse;
import com.ejianc.framework.core.util.ComputeUtil;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.BooleanUtils;
import org.apache.commons.lang3.StringUtils;
import org.redisson.api.RLock;
import org.redisson.api.RedissonClient;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.annotation.Lazy;
import org.springframework.stereotype.Component;

import java.math.BigDecimal;
import java.util.*;
import java.util.concurrent.TimeUnit;
import java.util.function.Function;
import java.util.stream.Collectors;

/**
 * 成本控制
 */
@Component
public class TargetcostCtrl {

    private final Logger logger = LoggerFactory.getLogger(getClass());

    @Autowired
    private IOrgApi orgApi;
    @Autowired
    private IRuleService ruleService;
    @Autowired
    private IRuleDetailService ruleDetailService;
    @Autowired
    private RedissonClient redissonClient;
    @Autowired
    private ITotalCacheService totalCacheService;
    @Autowired
    private IDetailCacheService detailCacheService;
    @Lazy
    @Autowired
    private IDutyService dutyService;
    @Autowired
    private IDutyDetailService dutyDetailService;
    @Autowired
    private IDutyDetailItemService dutyDetailItemService;
    @Autowired
    private IFeeDetailService feeDetailService;
    @Autowired
    private IFeeDetailScopeService feeDetailScopeService;


    private static final String TC_LOCK = "TC_LOCK::";

    public CommonResponse<JSONObject> ctrl(ExecutionVO executionVO) {
        JSONObject jsonObject = new JSONObject();
        TotalExecutionVO totalExecutionVO = executionVO.getTotalVO();
        //获取锁对象
        String lockKey = TC_LOCK + totalExecutionVO.getProjectId();
        RLock mylock = redissonClient.getLock(lockKey);
        try {
            //加锁，并且设置锁过期时间，防止死锁的产生
            boolean lock = mylock.tryLock(5, TimeUnit.SECONDS);
            if (!lock) {
                return CommonResponse.error("获取锁失败：" + totalExecutionVO.getProjectId());
            }

            LambdaQueryWrapper<DutyEntity> feeQuery = new LambdaQueryWrapper<>();
            feeQuery.eq(DutyEntity::getProjectId, totalExecutionVO.getProjectId());
            feeQuery.eq(DutyEntity::getEnableState, true);
            feeQuery.eq(DutyEntity::getGatherFlag, true);
            List<DutyEntity> dutyEntities = dutyService.list(feeQuery);
            if (CollectionUtils.isEmpty(dutyEntities)) {
                return CommonResponse.success("该项目还没编制目标成本");
            }

            DutyEntity dutyEntity = dutyEntities.get(0);
            CommonResponse<List<OrgVO>> parents = orgApi.findParentsByOrgId(totalExecutionVO.getOrgId());
            Set<Long> ruleIdSet = new HashSet<>();
            List<RuleEntity> ruleEntities = null;
            Map<Long, RuleEntity> ruleMap = new HashMap<>();
            if (parents.isSuccess() && parents.getData() != null && parents.getData().size() > 0) {
                List<Long> orgList = parents.getData().stream().map(OrgVO::getId).collect(Collectors.toList());
                LambdaQueryWrapper<RuleEntity> queryWrapper = new LambdaQueryWrapper<>();
                queryWrapper.eq(RuleEntity::getFeeId, dutyEntity.getFeeId());
                List<Integer> billStatusList = new ArrayList<>();
                billStatusList.add(BillStateEnum.PASSED_STATE.getBillStateCode());
                billStatusList.add(BillStateEnum.COMMITED_STATE.getBillStateCode());
                queryWrapper.eq(RuleEntity::getEnabled, 1);
                queryWrapper.in(RuleEntity::getBillState, billStatusList);
                queryWrapper.in(RuleEntity::getOrgId, orgList);
                ruleEntities = ruleService.list(queryWrapper);
                if (CollectionUtils.isNotEmpty(ruleEntities)) {
                    for (RuleEntity ruleEntity : ruleEntities) {
                        ruleIdSet.add(ruleEntity.getId());
                        ruleMap.put(ruleEntity.getId(), ruleEntity);
                    }
                }
            }

            if (CollectionUtils.isEmpty(ruleIdSet)) {
                return CommonResponse.success("该业务单据对应组织没有设置控制规则");
            }

            LambdaQueryWrapper<FeeDetailScopeEntity> scopeQuery = new LambdaQueryWrapper<>();
            scopeQuery.eq(FeeDetailScopeEntity::getFeeId, dutyEntity.getFeeId());
            scopeQuery.eq(FeeDetailScopeEntity::getBillCode, totalExecutionVO.getBussinessType());
            scopeQuery.eq(FeeDetailScopeEntity::getPropertyValue, totalExecutionVO.getProperty());
            List<FeeDetailScopeEntity> scopeEntities = feeDetailScopeService.list(scopeQuery);
            List<Long> feeDetailPks = new ArrayList<>();
            Map<String, String> scopeMap = new HashMap<>();
            if (CollectionUtils.isNotEmpty(scopeEntities)) {
                for (FeeDetailScopeEntity scopeEntity : scopeEntities) {
                    feeDetailPks.add(scopeEntity.getFeeDetailId());
                    scopeMap.put(scopeEntity.getBillCode(), scopeEntity.getMemoValue());
                }
            }
            if (CollectionUtils.isEmpty(feeDetailPks)) {
                return CommonResponse.success("该业务单据没有对应的费用项明细");
            }


            Map<String, List<RuleDetailEntity>> ruleDetailMap = new HashMap<>();
            LambdaQueryWrapper<RuleDetailEntity> ruleDetailQuery = new LambdaQueryWrapper<>();
            ruleDetailQuery.eq(RuleDetailEntity::getSubjectFlag, false);
            ruleDetailQuery.in(RuleDetailEntity::getRuleId, ruleIdSet);
            ruleDetailQuery.and(i -> i.isNull(RuleDetailEntity::getBillCategory).or().like(RuleDetailEntity::getBillCategory, totalExecutionVO.getBillCategory()));
            List<RuleDetailEntity> ruleDetailEntities = ruleDetailService.list(ruleDetailQuery);
            for (RuleDetailEntity entity : ruleDetailEntities) {
                String key = entity.getFeeInnerCode();
                List<RuleDetailEntity> list = ruleDetailMap.containsKey(key) ? ruleDetailMap.get(key) : new ArrayList<>();
                list.add(entity);
                ruleDetailMap.put(key, list);
            }

            List<DetailCacheEntity> detailCacheEntities = cacheDetails(dutyEntity, executionVO, feeDetailPks);
            List<TotalCacheEntity> totalCacheEntities = cacheTotal(dutyEntity, executionVO, feeDetailPks, detailCacheEntities);

            if(CollectionUtils.isEmpty(detailCacheEntities) && CollectionUtils.isEmpty(totalCacheEntities)){
                return CommonResponse.success("该业务单据未关联到项目策划数据");
            }

            String msg = ctrl(detailCacheEntities, totalCacheEntities, scopeMap, executionVO, totalExecutionVO, ruleMap, ruleDetailMap, ControlTypeEnum.刚性控制.getCode());
            if (StringUtils.isNotBlank(msg)) {
                jsonObject.put(String.valueOf(ControlTypeEnum.刚性控制.getCode()), msg);
                return CommonResponse.error("控制不通过", jsonObject);
            }

            msg = ctrl(detailCacheEntities, totalCacheEntities, scopeMap, executionVO, totalExecutionVO, ruleMap, ruleDetailMap, ControlTypeEnum.柔性控制.getCode());
            if (StringUtils.isNotBlank(msg)) {
                jsonObject.put(String.valueOf(ControlTypeEnum.柔性控制.getCode()), msg);
                return CommonResponse.error("控制不通过", jsonObject);
            }

        } catch (Exception e) {
            logger.error("保存前目标成本控制失败:" + JSONObject.toJSONString(e));
        } finally {
            mylock.unlock();
        }
        return CommonResponse.success("保存前项目策划控制通过");
    }

    private String ctrl(List<DetailCacheEntity> detailCacheEntities, List<TotalCacheEntity> totalCacheEntities, Map<String, String> scopeMap,
                        ExecutionVO executionVO, TotalExecutionVO totalExecutionVO, Map<Long, RuleEntity> ruleMainMap,
                        Map<String, List<RuleDetailEntity>> ruleDetailMap, Integer controlType) {
        String ctrlAll = scopeMap.get(totalExecutionVO.getBussinessType());
        Set<Long> feeDetailPkSet = new HashSet<>();
        for (TotalCacheEntity cacheEntity : totalCacheEntities) {
            feeDetailPkSet.add(cacheEntity.getFeeDetailId());
        }
        for (DetailCacheEntity detailCacheEntity : detailCacheEntities) {
            feeDetailPkSet.add(detailCacheEntity.getFeeDetailId());
        }
        LambdaQueryWrapper<FeeDetailEntity> queryWrapper = new LambdaQueryWrapper<>();
        queryWrapper.in(FeeDetailEntity::getId, feeDetailPkSet);
        List<FeeDetailEntity> feeDetalilist = feeDetailService.list(queryWrapper);
        Map<Long, String> innerCodeMap = feeDetalilist.stream().collect(Collectors.toMap(FeeDetailEntity::getId, FeeDetailEntity::getInnerCode, (key1, key2) -> key1));


        StringBuffer msg = new StringBuffer();
        Map<Long, List<DetailCacheEntity>> detailCacheMap = new HashMap<>();
        if (CollectionUtils.isNotEmpty(executionVO.getDetailList())) {
            for (DetailCacheEntity detailCacheEntity : detailCacheEntities) {
                Long feeDetailId = detailCacheEntity.getFeeDetailId();

                List<RuleDetailEntity> ruleDetailEntityList = getRuleDetailList(ruleDetailMap, innerCodeMap.get(feeDetailId), ctrlAll);

                if (CollectionUtils.isNotEmpty(ruleDetailEntityList)) {
                    List<Map<Integer, RuleDetailEntity>> ruleList = splitToMap(ruleDetailEntityList, controlType, ruleMainMap);
                    Map<Integer, RuleDetailEntity> ruleNullMap = ruleList.get(0); //目标为空时不控制
                    msg.append(detailCtrl(ruleNullMap, detailCacheEntity, totalExecutionVO, controlType));

                    Long key = detailCacheEntity.getFeeDetailId();
                    List<DetailCacheEntity> list = detailCacheMap.containsKey(key) ? detailCacheMap.get(key) : new ArrayList<>();
                    list.add(detailCacheEntity);
                    detailCacheMap.put(key, list);
                }
            }
        }

        for (TotalCacheEntity cacheEntity : totalCacheEntities) {
            Long feeDetailId = cacheEntity.getFeeDetailId();
            List<DetailCacheEntity> detailCacheEntityList = detailCacheMap.get(feeDetailId);
            List<RuleDetailEntity> ruleDetailEntityList = getRuleDetailList(ruleDetailMap, innerCodeMap.get(feeDetailId), ctrlAll);
            if (CollectionUtils.isNotEmpty(ruleDetailEntityList)) {
                List<Map<Integer, RuleDetailEntity>> ruleList = splitToMap(ruleDetailEntityList, controlType, ruleMainMap);
                Map<Integer, RuleDetailEntity> ruleMap = ruleList.get(0); //目标为空时不控制
                msg.append(totalCtrl(scopeMap, ruleMap, false, cacheEntity, executionVO, detailCacheEntityList, controlType));
                Map<Integer, RuleDetailEntity> ruleNullMap = ruleList.get(1); //目标为空时控制
                msg.append(totalCtrl(scopeMap, ruleNullMap, true, cacheEntity, executionVO, detailCacheEntityList, controlType));
            }
        }
        return msg.toString();
    }

    private List<RuleDetailEntity> getRuleDetailList(Map<String, List<RuleDetailEntity>> ruleDetailMap, String innerCode, String ctrlAll) {
        List<RuleDetailEntity> list = new ArrayList<>();
        for (String key : ruleDetailMap.keySet()) {
            if (innerCode.contains(key)) {
                List<RuleDetailEntity> ruleDetailEntities = ruleDetailMap.get(key);
                if (key.equals(innerCode)) {
                    for (RuleDetailEntity detailEntity : ruleDetailEntities) {
                        if (ctrlAll.contains(String.valueOf(detailEntity.getFeeType()))) {
                            list.add(detailEntity);
                        }
                    }
                } else {
                    for (RuleDetailEntity detailEntity : ruleDetailEntities) {
                        if ((FeeTypeEnum.明细量控.getCode().equals(detailEntity.getFeeType()) || FeeTypeEnum.明细金额控.getCode().equals(detailEntity.getFeeType()) || FeeTypeEnum.明细租赁时间控.getCode().equals(detailEntity.getFeeType()))
                                && ctrlAll.contains(String.valueOf(detailEntity.getFeeType()))) {
                            list.add(detailEntity);
                        }
                    }
                }
            }
        }
        return list;
    }

    private String detailCtrl(Map<Integer, RuleDetailEntity> ruleMap, DetailCacheEntity detailCacheEntity, TotalExecutionVO totalExecutionVO, Integer controlType) {
        StringBuffer msg = new StringBuffer();
        BigDecimal num = detailCacheEntity.getAmount();
        BigDecimal taxMny = detailCacheEntity.getTaxMny();
        String docTypeName = DocTypeEnum.专业分包档案.equals(detailCacheEntity.getDocType())
                || DocTypeEnum.劳务分包档案.equals(detailCacheEntity.getDocType()) ? "清单" : "档案";
        if (!ruleMap.isEmpty()) {
            for (Integer key : ruleMap.keySet()) {
                BigDecimal comNum = null;
                BigDecimal comTaxMny = null;
                BigDecimal controlScale = ruleMap.get(key).getControlScale();
                if (FeeTypeEnum.明细金额控.getCode().equals(key)) {
                    if (BillCategoryEnum.计划.getCode().equals(totalExecutionVO.getBillCategory())) {
                        comTaxMny = detailCacheEntity.getPlanTaxMoney();
                    } else if (BillCategoryEnum.合同.getCode().equals(totalExecutionVO.getBillCategory())) {
                        comTaxMny = detailCacheEntity.getContTaxMoney();
                    } else if (BillCategoryEnum.入库.getCode().equals(totalExecutionVO.getBillCategory())) {
                        comTaxMny = detailCacheEntity.getStoreTaxMoney();
                    } else if (BillCategoryEnum.其他.getCode().equals(totalExecutionVO.getBillCategory())) {
                        comTaxMny = detailCacheEntity.getOtherTaxMoney();
                    } else if (BillCategoryEnum.结算.getCode().equals(totalExecutionVO.getBillCategory())) {
                        comTaxMny = detailCacheEntity.getSettleTaxMoney();
                    }
                    BigDecimal comTaxScale = ComputeUtil.safeMultiply(ComputeUtil.safeDiv(comTaxMny, taxMny), BigDecimal.valueOf(100));
                    if (ComputeUtil.isLessThan(controlScale, comTaxScale)) {
                        msg.append(detailCacheEntity.getName()).append("明细金额控").append("&");
                        msg.append(ruleMap.get(key).getOrgName()).append("&");
                        StringBuffer tmpMsg = new StringBuffer();
                        tmpMsg.append(detailCacheEntity.getName());
                        if (StringUtils.isNotBlank(detailCacheEntity.getModel())) {
                            tmpMsg.append(detailCacheEntity.getModel());
                        }
                        tmpMsg.append("总金额(").append(ComputeUtil.scale(comTaxMny, 2)).append("元)已占用项目策划中该"+docTypeName+"金额");
                        tmpMsg.append("(").append(ComputeUtil.scale(taxMny, 2)).append("元)").append(ComputeUtil.scale(comTaxScale, 2)).append("%");
                        msg.append(getBaseMsg(tmpMsg.toString(), controlType));
                    }
                } else if (FeeTypeEnum.明细量控.getCode().equals(key)) {
                    if (BillCategoryEnum.计划.getCode().equals(totalExecutionVO.getBillCategory())) {
                        comNum = ComputeUtil.safeAdd(comNum, detailCacheEntity.getPlanNum());
                    } else if (BillCategoryEnum.合同.getCode().equals(totalExecutionVO.getBillCategory())) {
                        comNum = ComputeUtil.safeAdd(comNum, detailCacheEntity.getContNum());
                    } else if (BillCategoryEnum.入库.getCode().equals(totalExecutionVO.getBillCategory())) {
                        comNum = ComputeUtil.safeAdd(comNum, detailCacheEntity.getStoreNum());
                    } else if (BillCategoryEnum.其他.getCode().equals(totalExecutionVO.getBillCategory())) {
                        comNum = ComputeUtil.safeAdd(comNum, detailCacheEntity.getOtherNum());
                    } else if (BillCategoryEnum.结算.getCode().equals(totalExecutionVO.getBillCategory())) {
                        comNum = ComputeUtil.safeAdd(comNum, detailCacheEntity.getSettleNum());
                    }
                    BigDecimal comScale = ComputeUtil.safeMultiply(ComputeUtil.safeDiv(comNum, num), BigDecimal.valueOf(100));
                    if (ComputeUtil.isLessThan(controlScale, comScale)) {
                        msg.append(detailCacheEntity.getName()).append("明细量控").append("&");
                        msg.append(ruleMap.get(key).getOrgName()).append("&");
                        StringBuffer tmpMsg = new StringBuffer();
                        tmpMsg.append(detailCacheEntity.getName());
                        if (StringUtils.isNotBlank(detailCacheEntity.getModel())) {
                            tmpMsg.append(detailCacheEntity.getModel());
                        }
                        tmpMsg.append("总数量(").append(ComputeUtil.scale(comNum, 4)).append(")已占用项目策划中该"+docTypeName+"数量");
                        tmpMsg.append("(").append(ComputeUtil.scale(num, 4)).append(")").append(ComputeUtil.scale(comScale, 2)).append("%");
                        msg.append(getBaseMsg(tmpMsg.toString(), controlType));
                    }
                }
                /*else if (FeeTypeEnum.明细租赁时间控.getCode().equals(key)) {
                    if (BillCategoryEnum.计划.getCode().equals(totalExecutionVO.getBillCategory())) {
                        comNum = ComputeUtil.safeAdd(comNum, detailCacheEntity.getPlanNum());
                    } else if (BillCategoryEnum.合同.getCode().equals(totalExecutionVO.getBillCategory())) {
                        comNum = ComputeUtil.safeAdd(comNum, detailCacheEntity.getContNum());
                    } else if (BillCategoryEnum.入库.getCode().equals(totalExecutionVO.getBillCategory())) {
                        comNum = ComputeUtil.safeAdd(comNum, detailCacheEntity.getStoreNum());
                    } else if (BillCategoryEnum.其他.getCode().equals(totalExecutionVO.getBillCategory())) {
                        comNum = ComputeUtil.safeAdd(comNum, detailCacheEntity.getOtherNum());
                    } else if (BillCategoryEnum.结算.getCode().equals(totalExecutionVO.getBillCategory())) {
                        comNum = ComputeUtil.safeAdd(comNum, detailCacheEntity.getSettleNum());
                    }
                    BigDecimal comScale = ComputeUtil.safeMultiply(ComputeUtil.safeDiv(comNum, num), BigDecimal.valueOf(100));
                    if (ComputeUtil.isLessThan(controlScale, comScale)) {
                        msg.append(detailCacheEntity.getName()).append("明细量控").append("&");
                        msg.append(ruleMap.get(key).getOrgName()).append("&");
                        StringBuffer tmpMsg = new StringBuffer();
                        tmpMsg.append(detailCacheEntity.getName());
                        if (StringUtils.isNotBlank(detailCacheEntity.getModel())) {
                            tmpMsg.append(detailCacheEntity.getModel());
                        }
                        tmpMsg.append("总数量(").append(ComputeUtil.scale(comNum, 4)).append(")已占用项目策划中该档案数量");
                        tmpMsg.append("(").append(ComputeUtil.scale(num, 4)).append(")").append(ComputeUtil.scale(comScale, 2)).append("%");
                        msg.append(getBaseMsg(tmpMsg.toString(), controlType));
                    }
                }*/
            }
        }
        return msg.toString();
    }

    private String totalCtrl(Map<String, String> scopeMap, Map<Integer, RuleDetailEntity> ruleMap, Boolean isNullCtrl, TotalCacheEntity cacheEntity,
                             ExecutionVO executionVO, List<DetailCacheEntity> detailCacheEntities, Integer controlType) {
        StringBuffer msg = new StringBuffer();
        TotalExecutionVO totalExecutionVO = executionVO.getTotalVO();
        BigDecimal num = cacheEntity.getPreTypeAmount();
        BigDecimal taxMny = cacheEntity.getTaxMny();
        if (!ruleMap.isEmpty()) {
            List<DetailCacheEntity> itemEntityList = null;
            if (ruleMap.containsKey(FeeTypeEnum.分类量控.getCode())) {
                LambdaQueryWrapper<DetailCacheEntity> itemQuery = new LambdaQueryWrapper<>();
                itemQuery.eq(DetailCacheEntity::getDutyDetailId, cacheEntity.getDutyDetailId());
//                itemQuery.eq(DetailCacheEntity::getBillCategory, totalExecutionVO.getBillCategory());
                itemEntityList = detailCacheService.list(itemQuery);
            }
            for (Integer key : ruleMap.keySet()) {
                BigDecimal comNum = null;
                BigDecimal comTaxMny = null;
                BigDecimal controlScale = ruleMap.get(key).getControlScale();
                if (FeeTypeEnum.总额控.getCode().equals(key)) {
                    if (BillCategoryEnum.计划.getCode().equals(totalExecutionVO.getBillCategory())) {
                        comTaxMny = cacheEntity.getPlanTaxMoney();
                    } else if (BillCategoryEnum.合同.getCode().equals(totalExecutionVO.getBillCategory())) {
                        comTaxMny = cacheEntity.getContTaxMoney();
                    } else if (BillCategoryEnum.入库.getCode().equals(totalExecutionVO.getBillCategory())) {
                        comTaxMny = cacheEntity.getStoreTaxMoney();
                    } else if (BillCategoryEnum.其他.getCode().equals(totalExecutionVO.getBillCategory())) {
                        comTaxMny = cacheEntity.getOtherTaxMoney();
                    } else if (BillCategoryEnum.结算.getCode().equals(totalExecutionVO.getBillCategory())) {
                        comTaxMny = cacheEntity.getSettleTaxMoney();
                    }
                    BigDecimal comTaxScale = ComputeUtil.safeMultiply(ComputeUtil.safeDiv(comTaxMny, taxMny), BigDecimal.valueOf(100));
                    if (isNullCtrl) {
                        if (ComputeUtil.isLessThan(controlScale, comTaxScale)) {
                            msg.append(cacheEntity.getName()).append("总额控").append("&");
                            msg.append(ruleMap.get(key).getOrgName()).append("&");
                            StringBuffer tmpMsg = new StringBuffer();
                            tmpMsg.append(cacheEntity.getName()).append("金额(").append(ComputeUtil.scale(comTaxMny, 2))
                                    .append("元)已占项目策划中该费用金额");
                            tmpMsg.append("(").append(ComputeUtil.scale(taxMny, 2)).append("元)").append(ComputeUtil.scale(comTaxScale, 2)).append("%");
                            msg.append(getBaseMsg(tmpMsg.toString(), controlType));
                        }
                    } else {
                        if (null != taxMny && ComputeUtil.isLessThan(controlScale, comTaxScale)) {
                            msg.append(cacheEntity.getName()).append("总额控").append("&");
                            msg.append(ruleMap.get(key).getOrgName()).append("&");
                            StringBuffer tmpMsg = new StringBuffer();
                            tmpMsg.append(cacheEntity.getName()).append("金额(").append(ComputeUtil.scale(comTaxMny, 2)).append("元)已占项目策划中该费用金额");
                            tmpMsg.append("(").append(ComputeUtil.scale(taxMny, 2)).append("元)").append(ComputeUtil.scale(comTaxScale, 2)).append("%");
                            msg.append(getBaseMsg(tmpMsg.toString(), controlType));
                        }
                    }

                } else if (FeeTypeEnum.目标外明细总金额控.getCode().equals(key)) {
                    if (BillCategoryEnum.计划.getCode().equals(totalExecutionVO.getBillCategory())) {
                        comTaxMny = cacheEntity.getOutPlanTaxMoney();
                    } else if (BillCategoryEnum.合同.getCode().equals(totalExecutionVO.getBillCategory())) {
                        comTaxMny = cacheEntity.getOutContTaxMoney();
                    } else if (BillCategoryEnum.入库.getCode().equals(totalExecutionVO.getBillCategory())) {
                        comTaxMny = cacheEntity.getOutStoreTaxMoney();
                    } else if (BillCategoryEnum.其他.getCode().equals(totalExecutionVO.getBillCategory())) {
                        comTaxMny = cacheEntity.getOutOtherTaxMoney();
                    } else if (BillCategoryEnum.结算.getCode().equals(totalExecutionVO.getBillCategory())) {
                        comTaxMny = cacheEntity.getOutSettleTaxMoney();
                    }
                    BigDecimal comTaxScale = ComputeUtil.safeMultiply(ComputeUtil.safeDiv(comTaxMny, taxMny), BigDecimal.valueOf(100));
                    if (isNullCtrl) {
                        if (ComputeUtil.isLessThan(controlScale, comTaxScale)) {
                            msg.append(cacheEntity.getName()).append("项目策划外明细总金额控").append("&");
                            msg.append(ruleMap.get(key).getOrgName()).append("&");
                            StringBuffer tmpMsg = new StringBuffer();
                            tmpMsg.append(cacheEntity.getName()).append("项目策划外明细金额(").append(comTaxMny).append("元)已占项目策划").append(cacheEntity.getName());
                            tmpMsg.append("(").append(taxMny).append("元)").append(comTaxScale).append("%");
                            msg.append(getBaseMsg(tmpMsg.toString(), controlType));
                        }
                    } else {
                        if (null != taxMny && ComputeUtil.isLessThan(controlScale, comTaxScale)) {
                            msg.append(cacheEntity.getName()).append("项目策划外明细总金额控").append("&");
                            msg.append(ruleMap.get(key).getOrgName()).append("&");
                            StringBuffer tmpMsg = new StringBuffer();
                            tmpMsg.append(cacheEntity.getName()).append("项目策划外明细金额(").append(ComputeUtil.scale(comTaxMny, 2)).append("元)已占项目策划").append(cacheEntity.getName());
                            tmpMsg.append("(").append(ComputeUtil.scale(taxMny, 2)).append("元)").append(ComputeUtil.scale(comTaxScale, 2)).append("%");
                            msg.append(getBaseMsg(tmpMsg.toString(), controlType));
                        }
                    }

                } else if (FeeTypeEnum.分类量控.getCode().equals(key)) {
                    if (DocTypeEnum.物料档案.getCode().equals(cacheEntity.getDocType()) && cacheEntity.getLeafFlag() && checkTotal(cacheEntity, executionVO)) {
                        Set<Long> longSet = new HashSet<>();
                        if (CollectionUtils.isNotEmpty(detailCacheEntities)) {
                            for (DetailCacheEntity detailCacheEntity : detailCacheEntities) {
                                if (BillCategoryEnum.计划.getCode().equals(totalExecutionVO.getBillCategory())) {
                                    comNum = ComputeUtil.safeAdd(comNum, detailCacheEntity.getPlanNum());
                                } else if (BillCategoryEnum.合同.getCode().equals(totalExecutionVO.getBillCategory())) {
                                    comNum = ComputeUtil.safeAdd(comNum, detailCacheEntity.getContNum());
                                } else if (BillCategoryEnum.入库.getCode().equals(totalExecutionVO.getBillCategory())) {
                                    comNum = ComputeUtil.safeAdd(comNum, detailCacheEntity.getStoreNum());
                                } else if (BillCategoryEnum.其他.getCode().equals(totalExecutionVO.getBillCategory())) {
                                    comNum = ComputeUtil.safeAdd(comNum, detailCacheEntity.getOtherNum());
                                } else if (BillCategoryEnum.结算.getCode().equals(totalExecutionVO.getBillCategory())) {
                                    comNum = ComputeUtil.safeAdd(comNum, detailCacheEntity.getSettleNum());
                                }
                                longSet.add(detailCacheEntity.getId());
                            }
                        }

                        if (CollectionUtils.isNotEmpty(itemEntityList)) {
                            for (DetailCacheEntity detailCacheEntity : itemEntityList) {
                                if (!longSet.contains(detailCacheEntity.getId())) {
                                    if (BillCategoryEnum.计划.getCode().equals(totalExecutionVO.getBillCategory())) {
                                        comNum = ComputeUtil.safeAdd(comNum, detailCacheEntity.getPlanNum());
                                    } else if (BillCategoryEnum.合同.getCode().equals(totalExecutionVO.getBillCategory())) {
                                        comNum = ComputeUtil.safeAdd(comNum, detailCacheEntity.getContNum());
                                    } else if (BillCategoryEnum.入库.getCode().equals(totalExecutionVO.getBillCategory())) {
                                        comNum = ComputeUtil.safeAdd(comNum, detailCacheEntity.getStoreNum());
                                    } else if (BillCategoryEnum.其他.getCode().equals(totalExecutionVO.getBillCategory())) {
                                        comNum = ComputeUtil.safeAdd(comNum, detailCacheEntity.getOtherNum());
                                    } else if (BillCategoryEnum.结算.getCode().equals(totalExecutionVO.getBillCategory())) {
                                        comNum = ComputeUtil.safeAdd(comNum, detailCacheEntity.getSettleNum());
                                    }
                                }
                            }
                        }
                        BigDecimal comScale = ComputeUtil.safeMultiply(ComputeUtil.safeDiv(comNum, num), BigDecimal.valueOf(100));
                        if (isNullCtrl) {
                            if (ComputeUtil.isLessThan(controlScale, comScale)) {
                                msg.append(cacheEntity.getName()).append("分类量控").append("&");
                                msg.append(ruleMap.get(key).getOrgName()).append("&");
                                StringBuffer tmpMsg = new StringBuffer();
                                tmpMsg.append(cacheEntity.getName()).append("总量(").append(ComputeUtil.scale(comNum, 4)).append(")已占项目策划中该分类预估总量");
                                tmpMsg.append("(").append(ComputeUtil.scale(num, 4)).append(")").append(ComputeUtil.scale(comScale, 2)).append("%");
                                msg.append(getBaseMsg(tmpMsg.toString(), controlType));
                            }
                        } else {
                            if (null != num && ComputeUtil.isLessThan(controlScale, comScale)) {
                                msg.append(cacheEntity.getName()).append("分类量控").append("&");
                                msg.append(ruleMap.get(key).getOrgName()).append("&");
                                StringBuffer tmpMsg = new StringBuffer();
                                tmpMsg.append(cacheEntity.getName()).append("总量(").append(ComputeUtil.scale(comNum, 4)).append(")已占项目策划中该分类预估总量");
                                tmpMsg.append("(").append(ComputeUtil.scale(num, 2)).append(")").append(ComputeUtil.scale(comScale, 2)).append("%");
                                msg.append(getBaseMsg(tmpMsg.toString(), controlType));
                            }
                        }
                    }
                }
            }
        }
        return msg.toString();
    }

    private String getBaseMsg(String midMsg, Integer controlType) {
        StringBuffer msg = new StringBuffer();
        if (ControlTypeEnum.刚性控制.getCode().equals(controlType)) {
            msg.append("刚性：");
        } else if (ControlTypeEnum.柔性控制.getCode().equals(controlType)) {
            msg.append("柔性：");
        }
        msg.append(midMsg);
        if (ControlTypeEnum.刚性控制.getCode().equals(controlType)) {
            msg.append("，不能保存！/n");
        } else if (ControlTypeEnum.柔性控制.getCode().equals(controlType)) {
            msg.append("！/n");
        }
        return msg.toString();
    }

    private List<Map<Integer, RuleDetailEntity>> splitToMap(List<RuleDetailEntity> ruleDetailEntityList, int controlType, Map<Long, RuleEntity> ruleMap) {
        List<Map<Integer, RuleDetailEntity>> list = new ArrayList<>();
        Map<Integer, RuleDetailEntity> map = new HashMap<>();
        Map<Integer, RuleDetailEntity> mapNull = new HashMap<>();
        for (RuleDetailEntity entity : ruleDetailEntityList) {
            if (controlType == entity.getControlType()) {
                Integer key = entity.getFeeType();
                RuleDetailEntity controlScale = null;
                if (entity.getNullControlFlag()) {
                    if (mapNull.containsKey(key)) {
                        controlScale = ComputeUtil.isLessThan(mapNull.get(key).getControlScale(), entity.getControlScale()) ? mapNull.get(key) : entity;
                    } else {
                        controlScale = entity;
                    }
                    controlScale.setOrgName(ruleMap.get(controlScale.getRuleId()).getOrgName());
                    mapNull.put(key, controlScale);
                } else {
                    if (map.containsKey(key)) {
                        controlScale = ComputeUtil.isLessThan(map.get(key).getControlScale(), entity.getControlScale()) ? map.get(key) : entity;
                    } else {
                        controlScale = entity;
                    }
                    controlScale.setOrgName(ruleMap.get(controlScale.getRuleId()).getOrgName());
                    map.put(key, controlScale);
                }
            }
        }
        list.add(map);
        list.add(mapNull);
        return list;
    }

    private List<DetailCacheEntity> cacheDetails(DutyEntity dutyEntity, ExecutionVO executionVO, List<Long> feeDetailPks) {
        Set<DetailCacheEntity> insertSet = new HashSet<>();
        LambdaQueryWrapper<DutyDetailEntity> queryWrapper = new LambdaQueryWrapper<>();
        queryWrapper.eq(DutyDetailEntity::getDutyId, dutyEntity.getId());
        queryWrapper.in(DutyDetailEntity::getFeeDetailId, feeDetailPks);
        List<DutyDetailEntity> detailEntities = dutyDetailService.list(queryWrapper);

        List<Long> parentIdList = new ArrayList<>();
        List<Long> detailIdList = new ArrayList<>();
        for (DutyDetailEntity detailEntity : detailEntities) {
            if (detailEntity.getSelfRelatedFlag() && null == detailEntity.getCategoryId()) {
                parentIdList.add(detailEntity.getFeeDetailId());
            }
            detailIdList.add(detailEntity.getId());
        }

        Map<Long, List<FeeDetailEntity>> ctyMap = new HashMap<>();
        if (CollectionUtils.isNotEmpty(parentIdList)) {
            LambdaQueryWrapper<FeeDetailEntity> query = new LambdaQueryWrapper<>();
            query.in(FeeDetailEntity::getParentId, parentIdList);
            List<FeeDetailEntity> feeDetailEntities = feeDetailService.list(query);
            for (FeeDetailEntity detailEntity : feeDetailEntities) {
                List<FeeDetailEntity> list = ctyMap.containsKey(detailEntity.getParentId()) ? ctyMap.get(detailEntity.getParentId()) : new ArrayList<>();
                list.add(detailEntity);
                ctyMap.put(detailEntity.getParentId(), list);
            }
        }

        //所有责任成本孙表
        LambdaQueryWrapper<DutyDetailItemEntity> itemQuery = new LambdaQueryWrapper<>();
        itemQuery.in(DutyDetailItemEntity::getDutyDetailId, detailIdList);
        List<DutyDetailItemEntity> itemAllEntityList = dutyDetailItemService.list(itemQuery);
        Map<Long, List<DutyDetailItemEntity>> itemAllMap = new HashMap<>();
        for (DutyDetailItemEntity itemEntity : itemAllEntityList) {
            List<DutyDetailItemEntity> list = itemAllMap.containsKey(itemEntity.getDutyDetailId()) ? itemAllMap.get(itemEntity.getDutyDetailId()) : new ArrayList<>();
            list.add(itemEntity);
            itemAllMap.put(itemEntity.getDutyDetailId(), list);
        }

        //已生成的明细缓存
        LambdaQueryWrapper<DetailCacheEntity> detailQuery = new LambdaQueryWrapper<>();
        detailQuery.in(DetailCacheEntity::getDutyDetailId, detailIdList);
        List<DetailCacheEntity> detailCacheAllEntities = detailCacheService.list(detailQuery);
        Map<Long, List<DetailCacheEntity>> itemcacheAllMap = new HashMap<>();
        for (DetailCacheEntity cacheEntity : detailCacheAllEntities) {
            List<DetailCacheEntity> list = itemcacheAllMap.containsKey(cacheEntity.getDutyDetailId()) ? itemcacheAllMap.get(cacheEntity.getDutyDetailId()) : new ArrayList<>();
            list.add(cacheEntity);
            itemcacheAllMap.put(cacheEntity.getDutyDetailId(), list);
        }

        for (DutyDetailEntity detailEntity : detailEntities) {
            List<Long> categoryList = new ArrayList<>();
            if (detailEntity.getSelfRelatedFlag() && null == detailEntity.getCategoryId() && ctyMap.containsKey(detailEntity.getFeeDetailId())) {
                List<FeeDetailEntity> list = ctyMap.get(detailEntity.getFeeDetailId());
                categoryList = list.stream().map(FeeDetailEntity::getCategoryId).collect(Collectors.toList());
            }
            List<DutyDetailItemEntity> itemEntityList = itemAllMap.get(detailEntity.getId());
            Map<Long, DutyDetailItemEntity> itemMap = new HashMap<>();
            if (CollectionUtils.isNotEmpty(itemEntityList)) {
                itemMap = itemEntityList.stream().collect(Collectors.toMap(DutyDetailItemEntity::getDocId, Function.identity(), (key1, key2) -> key1));
            }

            List<DetailCacheEntity> detailCacheEntities = itemcacheAllMap.get(detailEntity.getId());
            Map<Long, DetailCacheEntity> itemcacheMap = new HashMap<>();
            if (CollectionUtils.isNotEmpty(detailCacheEntities)) {
                itemcacheMap = detailCacheEntities.stream().collect(Collectors.toMap(DetailCacheEntity::getDocId, Function.identity(), (key1, key2) -> key1));
            }

            for (DetailExecutionVO detailExecutionVO : executionVO.getDetailList()) {
                Long docId = detailExecutionVO.getDocId();
                if (!checkDetail(detailEntity, detailExecutionVO, categoryList)) {
                    continue;
                }
                DetailCacheEntity cacheEntity = null;
                if (itemcacheMap.containsKey(docId)) {
                    cacheEntity = itemcacheMap.get(docId);
                } else {
                    //目标内成本明细
                    if (itemMap.containsKey(docId)) {
                        DutyDetailItemEntity itemEntity = itemMap.get(docId);
                        cacheEntity = BeanMapper.map(itemEntity, DetailCacheEntity.class);
                        cacheEntity.setDutyItemId(itemEntity.getId());
                        cacheEntity.setSpec(itemEntity.getModel());
                    } else {//目标外
                        cacheEntity = BeanMapper.map(detailExecutionVO, DetailCacheEntity.class);
                        if (detailExecutionVO.getCategoryFlag()) {
                            cacheEntity.setCode(detailExecutionVO.getCategoryCode());
                            cacheEntity.setName(detailExecutionVO.getCategoryName());
                        }
                        cacheEntity.setPrice(null);
                        cacheEntity.setTaxPrice(null);
                        cacheEntity.setModel(detailExecutionVO.getSpec());
                        cacheEntity.setDutyDetailId(detailEntity.getId());
                        cacheEntity.setFeeDetailId(detailEntity.getFeeDetailId());
                        cacheEntity.setFeeDetailCode(detailEntity.getFeeDetailCode());
                        cacheEntity.setFeeDetailName(detailEntity.getFeeDetailName());
                    }
                    cacheEntity.setProjectId(dutyEntity.getProjectId());
                }
                cacheEntity.setBillCategory(executionVO.getTotalVO().getBillCategory());
                DutyCalcUtil.setDetailValue(cacheEntity, detailExecutionVO, executionVO.getTotalVO());
                insertSet.add(cacheEntity);
            }
        }

//        if(CollectionUtils.isNotEmpty(insertTmpSet)){
//            for(DetailCacheEntity cacheEntity : insertTmpSet){
//                if(null == cacheEntity.getDutyItemId()){
//                    BigDecimal planTotal = ComputeUtil.safeAdd(cacheEntity.getPlanNum(), cacheEntity.getPlanMoney(), cacheEntity.getPlanTaxMoney());
//                    BigDecimal contTotal = ComputeUtil.safeAdd(cacheEntity.getContNum(), cacheEntity.getContMoney(), cacheEntity.getContTaxMoney());
//                    BigDecimal storeTotal = ComputeUtil.safeAdd(cacheEntity.getStoreNum(), cacheEntity.getStoreMoney(), cacheEntity.getStoreTaxMoney());
//                    BigDecimal otherTotal = ComputeUtil.safeAdd(cacheEntity.getOtherNum(), cacheEntity.getOtherMoney(), cacheEntity.getOtherTaxMoney());
//                    BigDecimal total = ComputeUtil.safeAdd(planTotal, contTotal, storeTotal, otherTotal);
//                    if (null != total && 0 != total.intValue()) {
//                        insertList.add(cacheEntity);
//                    }
//                }else{
//                    insertList.add(cacheEntity);
//                }
//            }
//        }
        return insertSet.isEmpty() ? new ArrayList<>() : new ArrayList<>(insertSet);
    }

    private boolean checkDetail(DutyDetailEntity detailEntity, DetailExecutionVO detailExecutionVO, List<Long> categoryList) {
        if (null == detailEntity.getCategoryId()) {
            boolean flag = false;
            if (CollectionUtils.isNotEmpty(categoryList)) {
                for (Long category : categoryList) {
                    if (null != detailExecutionVO.getCategoryInnerCode() && detailExecutionVO.getCategoryInnerCode().contains(String.valueOf(category))) {
                        flag = true;
                    }
                }
            }
            return flag;
        } else if (!detailEntity.getLeafFlag()) {
            return false;
        } else if (detailEntity.getDocType().compareTo(detailExecutionVO.getDocType()) != 0) {
            return false;
        } else if (null == detailExecutionVO.getCategoryInnerCode() || !detailExecutionVO.getCategoryInnerCode().contains(String.valueOf(detailEntity.getCategoryId()))) {
            return false;
        }
        return true;
    }

    private List<TotalCacheEntity> cacheTotal(DutyEntity dutyEntity, ExecutionVO executionVO, List<Long> feeDetailPks, List<DetailCacheEntity> detailCacheEntities) {
        List<TotalCacheEntity> insertList = new ArrayList<>();
        LambdaQueryWrapper<FeeDetailEntity> feeQuery = new LambdaQueryWrapper<>();
        feeQuery.eq(FeeDetailEntity::getFeeId, dutyEntity.getFeeId());
        List<FeeDetailEntity> feeDetailEntities = feeDetailService.list(feeQuery);
        Map<Long, String> feeInnerMap = feeDetailEntities.stream().collect(Collectors.toMap(FeeDetailEntity::getId, FeeDetailEntity::getInnerCode, (key1, key2) -> key1));

        TotalExecutionVO totalExecutionVO = executionVO.getTotalVO();
        LambdaQueryWrapper<DutyDetailEntity> totalQuery = new LambdaQueryWrapper<>();
        totalQuery.eq(DutyDetailEntity::getDutyId, dutyEntity.getId());
        totalQuery.in(DutyDetailEntity::getFeeDetailId, feeDetailPks);
        List<DutyDetailEntity> dutyDetailEntities = dutyDetailService.list(totalQuery);

        List<Long> detailIdList = new ArrayList<>();
        for (DutyDetailEntity detailEntity : dutyDetailEntities) {
            detailIdList.add(detailEntity.getId());
        }
        LambdaQueryWrapper<TotalCacheEntity> query = new LambdaQueryWrapper<>();
        query.in(TotalCacheEntity::getDutyDetailId, detailIdList);
        List<TotalCacheEntity> totalCacheAllEntities = totalCacheService.list(query);
        Map<Long, List<TotalCacheEntity>> totalMap = new HashMap<>();
        for (TotalCacheEntity cacheEntity : totalCacheAllEntities) {
            List<TotalCacheEntity> list = totalMap.containsKey(cacheEntity.getDutyDetailId()) ? totalMap.get(cacheEntity.getDutyDetailId()) : new ArrayList<>();
            list.add(cacheEntity);
            totalMap.put(cacheEntity.getDutyDetailId(), list);
        }

        LambdaQueryWrapper<DetailCacheEntity> itemQuery = new LambdaQueryWrapper<>();
        itemQuery.in(DetailCacheEntity::getDutyDetailId, detailIdList);
//        itemQuery.eq(DetailCacheEntity::getBillCategory, totalExecutionVO.getBillCategory());
        List<DetailCacheEntity> itemEntityAllListdb = detailCacheService.list(itemQuery);
        Map<Long, List<DetailCacheEntity>> cacheAllMap = new HashMap<>();
        for (DetailCacheEntity cacheEntity : itemEntityAllListdb) {
            List<DetailCacheEntity> list = cacheAllMap.containsKey(cacheEntity.getDutyDetailId()) ? cacheAllMap.get(cacheEntity.getDutyDetailId()) : new ArrayList<>();
            list.add(cacheEntity);
            cacheAllMap.put(cacheEntity.getDutyDetailId(), list);
        }

        for (DutyDetailEntity dutyDetailEntity : dutyDetailEntities) {
            List<TotalCacheEntity> totalCacheEntities = totalMap.get(dutyDetailEntity.getId());
            TotalCacheEntity totalCacheEntity = CollectionUtils.isNotEmpty(totalCacheEntities) ? totalCacheEntities.get(0) : BeanMapper.map(dutyDetailEntity, TotalCacheEntity.class);
            totalCacheEntity.setDutyDetailId(dutyDetailEntity.getId());

            List<DetailCacheEntity> itemEntityListdb = cacheAllMap.get(dutyDetailEntity.getId());
            List<DetailCacheEntity> itemEntityList = new ArrayList<>();
            Set<Long> longSet = new HashSet<>();

            boolean isAddList = false;
            if (CollectionUtils.isNotEmpty(detailCacheEntities)) {
                for (DetailCacheEntity detailCacheEntity : detailCacheEntities) {
                    if (detailCacheEntity.getDutyDetailId().compareTo(totalCacheEntity.getDutyDetailId()) == 0) {
                        itemEntityList = CollectionUtils.isNotEmpty(itemEntityList) ? itemEntityList : new ArrayList<>();
                        itemEntityList.add(detailCacheEntity);
                        longSet.add(detailCacheEntity.getDocId());
                        isAddList = true;
                    }else if(feeInnerMap.get(detailCacheEntity.getFeeDetailId()).contains(feeInnerMap.get(dutyDetailEntity.getFeeDetailId()))){
                        isAddList = true;
                    }
                }
            }else if (CollectionUtils.isEmpty(executionVO.getDetailList()) && CollectionUtils.isEmpty(detailCacheEntities)) {
                isAddList = true;
            }
            if (CollectionUtils.isNotEmpty(itemEntityListdb)) {
                for (DetailCacheEntity detailCacheEntity : itemEntityListdb) {
                    if (!longSet.contains(detailCacheEntity.getDocId())) {
                        itemEntityList.add(detailCacheEntity);
                    }
                }
            }

            DutyCalcUtil.setValue(totalCacheEntity, itemEntityList, totalExecutionVO);

            if (isAddList) {
                insertList.add(totalCacheEntity);
            }
        }

        List<Map> resultMapList = BeanMapper.mapList(insertList, Map.class);
        List<Map<String, Object>> treeData = TreeNodeBUtil.createTreeData(resultMapList);

        gatherMnyAndNum(null, treeData);

        List<TotalCacheEntity> entities = new ArrayList<>();
        treeToList(treeData, entities);

        for (TotalCacheEntity entity : entities) {
            if (BooleanUtils.isTrue(entity.getSelfScopeFlag()) && BooleanUtils.isFalse(entity.getLeafFlag()) && (null == entity.getAllCostFlag() || BooleanUtils.isFalse(entity.getAllCostFlag()))) {
                DutyCalcUtil.setValue(entity, totalExecutionVO);
            }
        }

        return entities;
    }

    private boolean checkTotal(TotalCacheEntity cacheEntity, ExecutionVO executionVO) {
        String innerStr = "";
        List<DetailExecutionVO> detailList = executionVO.getDetailList();
        if (CollectionUtils.isNotEmpty(detailList)) {
            List<String> innerList = detailList.stream().map(DetailExecutionVO::getCategoryInnerCode).collect(Collectors.toList());
            innerStr = StringUtils.join(innerList, ",");
        }
        if (null != cacheEntity.getCategoryId()) {
            return innerStr.indexOf(String.valueOf(cacheEntity.getCategoryId())) > 0;
        } else {
            LambdaQueryWrapper<FeeDetailEntity> query = new LambdaQueryWrapper<>();
            query.eq(FeeDetailEntity::getParentId, cacheEntity.getFeeDetailId());
            List<FeeDetailEntity> feeDetailEntities = feeDetailService.list(query);
            for (FeeDetailEntity detailEntity : feeDetailEntities) {
                if (innerStr.indexOf(String.valueOf(detailEntity.getCategoryId())) > 0) {
                    return true;
                }
            }
        }
        return false;
    }

    private void treeToList(List<Map<String, Object>> treeData, List<TotalCacheEntity> entities) {
        for (Map<String, Object> ypd : treeData) {
            if (ypd.get("children") != null) {
                treeToList((List) ypd.get("children"), entities);
            }
            entities.add(BeanMapper.map(ypd, TotalCacheEntity.class));
        }
    }

    /**
     * 汇总金额和数量
     */
    private void gatherMnyAndNum(Map<String, Object> now, List<Map<String, Object>> list) {
        BigDecimal mny = null;
        BigDecimal taxMny = null;

        BigDecimal planMoney = null;
        BigDecimal planTaxMoney = null;
        BigDecimal outPlanMoney = null;
        BigDecimal outPlanTaxMoney = null;

        BigDecimal contMoney = null;
        BigDecimal contTaxMoney = null;
        BigDecimal outContMoney = null;
        BigDecimal outContTaxMoney = null;

        BigDecimal storeMoney = null;
        BigDecimal storeTaxMoney = null;
        BigDecimal outStoreMoney = null;
        BigDecimal outStoreTaxMoney = null;

        BigDecimal otherMoney = null;
        BigDecimal otherTaxMoney = null;
        BigDecimal outOtherMoney = null;
        BigDecimal outOtherTaxMoney = null;

        Boolean allCostFlag = false;

        for (Map<String, Object> ypd : list) {
            if (ypd.get("children") != null) {
                gatherMnyAndNum(ypd, (List) ypd.get("children"));
            }
            mny = ComputeUtil.safeAdd(mny, getBigDecimalValue(ypd, "mny"));
            taxMny = ComputeUtil.safeAdd(taxMny, getBigDecimalValue(ypd, "taxMny"));

            planMoney = ComputeUtil.safeAdd(planMoney, getBigDecimalValue(ypd, "planMoney"));
            planTaxMoney = ComputeUtil.safeAdd(planTaxMoney, getBigDecimalValue(ypd, "planTaxMoney"));
            outPlanMoney = ComputeUtil.safeAdd(outPlanMoney, getBigDecimalValue(ypd, "outPlanMoney"));
            outPlanTaxMoney = ComputeUtil.safeAdd(outPlanTaxMoney, getBigDecimalValue(ypd, "outPlanTaxMoney"));

            contMoney = ComputeUtil.safeAdd(contMoney, getBigDecimalValue(ypd, "contMoney"));
            contTaxMoney = ComputeUtil.safeAdd(contTaxMoney, getBigDecimalValue(ypd, "contTaxMoney"));
            outContMoney = ComputeUtil.safeAdd(outContMoney, getBigDecimalValue(ypd, "outContMoney"));
            outContTaxMoney = ComputeUtil.safeAdd(outContTaxMoney, getBigDecimalValue(ypd, "outContTaxMoney"));

            storeMoney = ComputeUtil.safeAdd(storeMoney, getBigDecimalValue(ypd, "storeMoney"));
            storeTaxMoney = ComputeUtil.safeAdd(storeTaxMoney, getBigDecimalValue(ypd, "storeTaxMoney"));
            outStoreMoney = ComputeUtil.safeAdd(outStoreMoney, getBigDecimalValue(ypd, "outStoreMoney"));
            outStoreTaxMoney = ComputeUtil.safeAdd(outStoreTaxMoney, getBigDecimalValue(ypd, "outStoreTaxMoney"));

            otherMoney = ComputeUtil.safeAdd(otherMoney, getBigDecimalValue(ypd, "otherMoney"));
            otherTaxMoney = ComputeUtil.safeAdd(otherTaxMoney, getBigDecimalValue(ypd, "otherTaxMoney"));
            outOtherMoney = ComputeUtil.safeAdd(outOtherMoney, getBigDecimalValue(ypd, "outOtherMoney"));
            outOtherTaxMoney = ComputeUtil.safeAdd(outOtherTaxMoney, getBigDecimalValue(ypd, "outOtherTaxMoney"));

            if (DocTypeEnum.成本科目.getCode().equals(ypd.get("docType"))) {
                allCostFlag = true;
            }
        }
        if (now != null) {
            now.put("allCostFlag", allCostFlag);

            now.put("outPlanMoney", outPlanMoney);
            now.put("outPlanTaxMoney", outPlanTaxMoney);
            now.put("outContMoney", outContMoney);
            now.put("outContTaxMoney", outContTaxMoney);
            now.put("outStoreMoney", outStoreMoney);
            now.put("outStoreTaxMoney", outStoreTaxMoney);
            now.put("outOtherMoney", outOtherMoney);
            now.put("outOtherTaxMoney", outOtherTaxMoney);

            if (!(Boolean) now.get("selfScopeFlag") || (!(Boolean) now.get("leafFlag") && allCostFlag)) {
                now.put("planMoney", planMoney);
                now.put("planTaxMoney", planTaxMoney);

                now.put("contMoney", contMoney);
                now.put("contTaxMoney", contTaxMoney);

                now.put("storeMoney", storeMoney);
                now.put("storeTaxMoney", storeTaxMoney);

                now.put("otherMoney", otherMoney);
                now.put("otherTaxMoney", otherTaxMoney);
            }
        }
    }

    private BigDecimal getBigDecimalValue(Map<String, Object> ypd, String code) {
        return ypd.get(code) == null ? null : (BigDecimal) ypd.get(code);
    }
}
