package com.ejianc.business.tradematerial.utils;

import com.alibaba.fastjson.JSONObject;
import com.ejianc.framework.core.exception.BusinessException;
import com.ejianc.framework.core.response.CommonResponse;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;

import java.math.BigDecimal;

public class CommonUtils {

    /** 默认异常信息 */
    public static final String defaultErrMsg = "推送数据失败！";

    /** 无权访问 */
    private static final String noPower = "无权限访问该服务，请先联系管理员进行授权！";

    private CommonUtils() {
    }

    /**
     * 校验返回的commonResponse是否是正常的，正常返回true，否则抛出异常
     *
     * @param commonResponse 接口调用结果
     * @param <T>            泛型
     * @return 校验结果  true，正常；false，异常
     */
    public static <T> boolean checkCommonResponse(CommonResponse<T> commonResponse) {
        return checkCommonResponse(commonResponse, true, null, null);
    }

    /**
     * 校验返回的commonResponse是否是正常的，正常无结果，否则抛出异常
     *
     * @param <T>            泛型
     * @param commonResponse 接口调用结果
     * @param logger         日志
     */
    public static <T> boolean checkCommonResponse(CommonResponse<T> commonResponse, Logger logger) {
        return checkCommonResponse(commonResponse, true, null, logger);
    }

    /**
     * 校验返回的commonResponse是否是正常的
     *
     * @param commonResponse 返回结果
     * @param isThrow        是否抛出异常。true抛出异常；false，不抛出异常，返回false
     * @param errMsg         抛出异常时异常信息
     * @param logger         日志
     * @param <T>            泛型
     * @return 校验结果  true，正常；false，异常
     */
    public static <T> boolean checkCommonResponse(CommonResponse<T> commonResponse, boolean isThrow, String errMsg,
                                                  Logger logger) {
        if (!commonResponse.isSuccess()) {
            if (logger != null) {
                logger.info("调用接口失败，调用结果：{}，失败原因：{}", commonResponse.isSuccess(), commonResponse.getMsg());
            }
            if (isThrow) {
                throw new BusinessException(StringUtils.isNotEmpty(errMsg) ? errMsg : commonResponse.getMsg());
            }
            return false;
        }
        try {
            T data = commonResponse.getData();
            String dataString;
            if (data instanceof String) {
                dataString = String.valueOf(data);
                if (noPower.equals(dataString)) {
                    throw new BusinessException(dataString);
                }
            }
            else {
                dataString = JSONObject.toJSONString(data);
            }
            JSONObject jsonObject = JSONObject.parseObject(dataString);
            // 如果返回的结果包含code
            if (StringUtils.isNotEmpty(dataString) && jsonObject.containsKey("code")) {
                CommonResponse check = JSONObject.toJavaObject(jsonObject, CommonResponse.class);
                if (isThrow && !check.isSuccess()) {
                    if (logger != null) {
                        logger.error("调用接口失败，返回结果：{}", check.getMsg());
                    }
                    throw new BusinessException(StringUtils.isNotEmpty(errMsg) ? errMsg : check.getMsg());
                }
                return check.isSuccess();
            }else {
                throw new BusinessException(defaultErrMsg);
            }
        }
        // 业务异常抛出
        catch (BusinessException be) {
            if (logger != null) {
                logger.error(be.getErrorMsg());
            }
            throw be;
        }
        // 其他异常做特殊处理
        catch (Exception e) {
            if (logger != null) {
                logger.error("类型转换失败！返回结果不是json或CommonResponse");
            }
            if (isThrow){
                throw new BusinessException(defaultErrMsg);
            }
            // 异常情况按照true处理
            return true;
        }
    }

    /**
     * BigDecimal默认赋值0
     *
     * @param v 需要赋值的字段
     * @return 赋值结果
     */
    public static BigDecimal setBigDecimalDefaultValue(BigDecimal v) {
        if (v == null) return BigDecimal.ZERO;
        return v;
    }

    /**
     * 求和，值可以为null
     *
     * @param v1 值1
     * @param v2 值2
     * @return 结果
     */
    public static BigDecimal addBigDecimal(BigDecimal v1, BigDecimal v2) {
        return setBigDecimalDefaultValue(v1).add(setBigDecimalDefaultValue(v2));
    }

    /**
     * 求差，v1-v2 值可以为null
     *
     * @param v1 值1
     * @param v2 值2
     * @return 差
     */
    public static BigDecimal subtractBigDecimal(BigDecimal v1, BigDecimal v2) {
        return setBigDecimalDefaultValue(v1).subtract(setBigDecimalDefaultValue(v2));
    }
}
