package com.ejianc.business.study.util;


import com.ejianc.business.study.annotation.MyAutowired;
import com.ejianc.business.study.annotation.MyService;

import java.io.IOException;
import java.lang.reflect.Field;
import java.util.List;
import java.util.Objects;
import java.util.concurrent.ConcurrentHashMap;

/**
 * ApplicationContext
 *
 * @Author: songlx
 * @Version: 1.0
 */
public class ApplicationContext {
    /**
     * 需要扫描的包
     */
    private String packageName;
    /**
     * 用来存放所有实例化的Bean对象 容器
     */
    private static ConcurrentHashMap<String, Object> beanMap;

    public ApplicationContext(String packageName) {
        this.packageName = packageName;
        beanMap = new ConcurrentHashMap<>();
        //1. 读取包中所有的class文件
        //2. 解析Class中的指定注解
        try {
            //读取所有的Class文件并实例化，存放bean对象到容器中
            initBeans();
            //读取所有被 @Autowired 注解标注的成员属性，从容器中查找相应的Bean对象进行赋值(自动装配)
            initAttributes();
        } catch (IOException exception) {
            exception.printStackTrace();
        } catch (ClassNotFoundException e) {
            e.printStackTrace();
        } catch (IllegalAccessException e) {
            e.printStackTrace();
        } catch (InstantiationException e) {
            e.printStackTrace();
        }
    }

    /**
     * 遍历容器中的所有Bean对象，为@Autowired成员赋值
     */
    private void initAttributes() {
        //key: beanId, value: bean对象
        beanMap.forEach((beanId, bean) -> {
            try {
                attributeAssign(beanId, bean);
            } catch (IllegalAccessException e) {
                e.printStackTrace();
            }
        });
    }

    private void attributeAssign(String beanId, Object bean) throws IllegalAccessException {
        //遍历所有添加Autowired注解的属性
        //获取类的所有字段
        Field[] fields = bean.getClass().getDeclaredFields();
        for (Field field : fields) {
            if (!field.isAnnotationPresent(MyAutowired.class)) continue;
            // @Autowired
            // private OrderService orderService;
            //获取要注入的beanId，默认就是成员变量名
            String autowiredBeanId = field.getName();
            //到容器中获取要注入的Bean对象
            Object autowiredBean =
                    this.getBean(autowiredBeanId, field.getType());
            if (Objects.isNull(autowiredBean)) continue;
            //下面进行属性的赋值
            field.setAccessible(true);
            field.set(bean, autowiredBean);
            field.setAccessible(false);
        }
    }

    private void initBeans() throws IOException, ClassNotFoundException, IllegalAccessException, InstantiationException {
        //1. 扫描指定的包
        List<Class<?>> classList = ClazzUtil.getClasses(packageName);
        //2. 判断遍历到的类是否被 @Service 注解所标注
        for (Class<?> clazz : classList) {
            if (!clazz.isAnnotationPresent(MyService.class)) continue;
            if (clazz.isInterface()) continue;//接口不能直接实例化，需要跳过
            //如果类中包含 Service注解，就取出这个注解
            MyService myService = clazz.getAnnotation(MyService.class);
            String beanId = myService.value();
            //有可能用户并没有给出 beanId，下面需要判断一下
            // UserServiceImpl -> userService
            if (Objects.isNull(beanId) || "".equals(beanId)) {
                //得到类的简写名称
                beanId = clazz.getSimpleName(); // -> UserServiceImpl
                //首字母小写 -> userServiceImpl
                beanId = beanId.substring(0, 1)
                        .toLowerCase() + beanId.substring(1);
                //去掉最后的Impl
                beanId = beanId.replace("Impl", "");
            }
            //3. 如果被标注，则通过反射实例化Bean对象，并放入到容器中
            beanMap.put(beanId, clazz.newInstance());
        }
    }

    /**
     * 返回容器中指定beanId的Bean对象，如果beanId不存在，就抛异常
     *
     * @param beanId
     * @param type
     * @param <T>
     * @return
     */
    public <T> T getBean(String beanId, Class<T> type) {
        if (Objects.isNull(beanId) || "".equalsIgnoreCase(beanId)) {
            throw new RuntimeException("BeanID不能为空！");
        }
        //从容器中获取beanId对应的Bean对象
        Object bean = beanMap.get(beanId);
        if (Objects.isNull(bean)) {
            throw new RuntimeException("容器中不存在BeanID为"
                    + beanId + "的对象！");
        }
        return (T) bean;
    }
}
