package com.ejianc.business.zdsmaterial.pricelib.processor;

import com.alibaba.fastjson.JSONObject;
import com.ejianc.business.zdsmaterial.pricelib.vo.MaterialPriceCalcSubConditionVO;
import com.ejianc.business.zdsmaterial.pricelib.vo.MaterialPriceSubConditionDetailVO;
import com.ejianc.business.zdsmaterial.pricelib.vo.PriceLibVO;
import com.ejianc.business.zdsmaterial.util.ComputeUtil;
import jodd.util.StringUtil;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.stereotype.Component;

import java.math.BigDecimal;

/**
 * 桥架双重基础价格计算公式处理器
 *
 * 公式1：（槽单价基准价格/0.95/槽重*1000+浮动网价*1.115-4250*1.115）*0.95*0.001*槽重
 * 公式2：（盖单价基准价格/0.95/盖重/*1000+浮动网价*1.115-4250*1.115）*0.95*0.001*盖重+运费*盖重*0.001=盖调整后单价
 * 公式3：合计调整价（调整后单价）=槽调整后单价+盖调整后单价
 *
 *
 * @Author: caojie
 * @CreateTime:2025-07-07 10:00
 * @Version: 1.0
 */
@Component
public class D2001DoublePriceProcessor implements BaseCalcProcessor {

    private Logger logger = LoggerFactory.getLogger(this.getClass());

    @Override
    public boolean calcBeforeCheck(PriceLibVO priceLibVO, MaterialPriceCalcSubConditionVO calcFormula) {
        //公共计算元素验证
        if(null == priceLibVO.getBaseMaterialNetPrice()) {
            logger.error("桥架公式：D2001DoublePriceProcessor 计算终止，原材料网价为空, {}", JSONObject.toJSONString(priceLibVO));
            priceLibVO.setCalcContent("未计算：原材料网价为空");
            return false;
        }
        if(null == priceLibVO.getBaseMaterialUnitPrice()) {
            logger.error("桥架公式：D2001DoublePriceProcessor 计算终止，原材料基础价为空, {}", JSONObject.toJSONString(priceLibVO));
            priceLibVO.setCalcContent("未计算：原材料基础价为空");
            return false;
        }
        if(StringUtil.isBlank(calcFormula.getCalcParams())) {
            logger.error("桥架公式：D2001DoublePriceProcessor 计算终止， 辅助计算参数为空, {}", JSONObject.toJSONString(priceLibVO));
            priceLibVO.setCalcContent("未计算：辅助计算参数为空");
            return false;
        }
        JSONObject calcParamJson = JSONObject.parseObject(calcFormula.getCalcParams());
        if(null == calcParamJson.get("参数1")) {
            logger.error("桥架公式：D2001DoublePriceProcessor 计算终止， 辅助计算参数【参数1】为空, {}", JSONObject.toJSONString(priceLibVO));
            priceLibVO.setCalcContent("未计算：辅助计算参数【参数1】为空");
            return false;
        }
        if(null == calcParamJson.get("参数2")) {
            logger.error("桥架公式：D2001DoublePriceProcessor 计算终止， 辅助计算参数【参数2】为空, {}", JSONObject.toJSONString(priceLibVO));
            priceLibVO.setCalcContent("未计算：辅助计算参数【参数2】为空");
            return false;
        }

        if(StringUtil.isBlank(priceLibVO.getCalcFields())) {
            logger.error("桥架公式：D2001DoublePriceProcessor 计算终止， 距离为空, {}", JSONObject.toJSONString(priceLibVO));
            priceLibVO.setCalcContent("未计算：计算因素[距离]为空");
            return false;
        }
        JSONObject paramJson = JSONObject.parseObject(priceLibVO.getCalcFields());
        if(null == paramJson.get("distance")) {
            if(StringUtil.isBlank(priceLibVO.getCalcFields())) {
                logger.error("桥架公式：D2001DoublePriceProcessor 计算终止， 距离为空, {}", JSONObject.toJSONString(priceLibVO));
                priceLibVO.setCalcContent("未计算：计算因素[距离]为空");
                return false;
            }
        }


        //第一价格计算元素验证
        if(null == priceLibVO.getBaseMaterialPrice()) {
            logger.error("桥架公式：D2001DoublePriceProcessor 计算终止，槽单价基准价格为空, {}", JSONObject.toJSONString(priceLibVO));
            priceLibVO.setCalcContent("未计算：槽单价基准价格为空");
            return false;
        }

        if(StringUtil.isBlank(priceLibVO.getExtParam1())) {
            logger.error("桥架公式：D2001DoublePriceProcessor 计算终止， 槽重为空, {}", JSONObject.toJSONString(priceLibVO));
            priceLibVO.setCalcContent("未计算：计算因素[槽重]为空");
            return false;
        }

        //第二价格计算元素验证
        if(null == priceLibVO.getBaseTwoMaterialPrice()) {
            logger.error("桥架公式：D2001DoublePriceProcessor 计算终止，盖单价基准价格为空, {}", JSONObject.toJSONString(priceLibVO));
            priceLibVO.setCalcContent("未计算：盖单价基准价格为空");
            return false;
        }

        if(StringUtil.isBlank(priceLibVO.getExtParam2())) {
            logger.error("桥架公式：D2001DoublePriceProcessor 计算终止， 盖重为空, {}", JSONObject.toJSONString(priceLibVO));
            priceLibVO.setCalcContent("未计算：计算因素[盖重]为空");
            return false;
        }

        return true;
    }

    @Override
    public BigDecimal calc(PriceLibVO priceLibVO, MaterialPriceCalcSubConditionVO calcFormula) {

         //公式1：槽调整后单价 =（槽单价基准价格 / 参数1 / 槽重 * 1000 + 浮动网价 * 参数2 - 4250 * 参数2）* 参数1 * 0.001 * 槽重
         //公式2：盖调整后单价 =（盖单价基准价格 / 参数1 / 盖重 * 1000 + 浮动网价 * 参数2 - 4250 * 参数2）* 参数1 * 0.001 * 盖重 + 运费 * 盖重 * 0.001
         //公式3：合计调整价（调整后单价）= 槽调整后单价 + 盖调整后单价

        if(!calcBeforeCheck(priceLibVO, calcFormula)) {
            return BigDecimal.ZERO;
        }
        priceLibVO.setCalcContent(calcFormula.getCalcDisplayContent());
        MaterialPriceSubConditionDetailVO calcElem = calcFormula.getDetailList().get(0);
        JSONObject calcParams = JSONObject.parseObject(calcFormula.getCalcParams());
        JSONObject calcFields = JSONObject.parseObject(priceLibVO.getCalcFields());

        BigDecimal result = BigDecimal.ZERO;
        BigDecimal resultTwo = BigDecimal.ZERO;

        //计算 槽单价基准价格 / 参数1 / 槽重 * 1000
        result = ComputeUtil.safeMultiply(ComputeUtil.safeDiv(ComputeUtil.safeDiv(priceLibVO.getBaseMaterialPrice(),new BigDecimal(calcParams.getString("参数1"))),
                new BigDecimal(priceLibVO.getExtParam1())), new BigDecimal("1000"));
        //计算 中间结果 + 浮动网价 * 参数2 - 4250 * 参数2
        result = ComputeUtil.safeAdd(result, ComputeUtil.safeSub(ComputeUtil.safeMultiply(priceLibVO.getBaseMaterialNetPrice(), new BigDecimal(calcParams.getString("参数2"))),
                ComputeUtil.safeMultiply(new BigDecimal("4250"), new BigDecimal(calcParams.getString("参数2")))));
        //计算 槽调整后单价 = 中间结果 * 参数1 * 0.001 * 槽重
        result = ComputeUtil.safeMultiply(ComputeUtil.safeMultiply(ComputeUtil.safeMultiply(result, new BigDecimal(calcParams.getString("参数1"))),
                new BigDecimal("0.001")), new BigDecimal(priceLibVO.getExtParam1()));
        priceLibVO.setAdjustedFirstPrice(result);

        //计算 盖单价基准价格 / 参数1 / 盖重 * 1000
        resultTwo = ComputeUtil.safeMultiply(ComputeUtil.safeDiv(ComputeUtil.safeDiv(priceLibVO.getBaseTwoMaterialPrice(),new BigDecimal(calcParams.getString("参数1"))),
                new BigDecimal(priceLibVO.getExtParam1())), new BigDecimal("1000"));
        //计算 中间结果 + 浮动网价 * 参数2 - 4250 * 参数2
        resultTwo = ComputeUtil.safeAdd(resultTwo, ComputeUtil.safeSub(ComputeUtil.safeMultiply(priceLibVO.getBaseMaterialNetPrice(), new BigDecimal(calcParams.getString("参数2"))),
                ComputeUtil.safeMultiply(new BigDecimal("4250"), new BigDecimal(calcParams.getString("参数2")))));
        //计算 中间结果 * 参数1 * 0.001 * 盖重
        resultTwo = ComputeUtil.safeMultiply(ComputeUtil.safeMultiply(ComputeUtil.safeMultiply(resultTwo, new BigDecimal(calcParams.getString("参数1"))),
                new BigDecimal("0.001")), new BigDecimal(priceLibVO.getExtParam1()));
        // 盖调整后单价 = 计算中间结果 + 运费(距离 * 0.36) * 盖重 * 0.001
        resultTwo = ComputeUtil.safeAdd(resultTwo, ComputeUtil.safeMultiply(ComputeUtil.safeMultiply(ComputeUtil.safeMultiply(new BigDecimal(calcFields.getString("distance")), new BigDecimal("0.36")),
                new BigDecimal(priceLibVO.getExtParam2())), new BigDecimal("0.001")));
        priceLibVO.setAdjustedSecPrice(resultTwo);

        //合计调整价（调整后单价）= 槽调整后单价 + 盖调整后单价
        priceLibVO.setAdjustedPrice(ComputeUtil.safeAdd(result, resultTwo));
        //计算金额
        priceLibVO.setCalcMny(ComputeUtil.safeMultiply(priceLibVO.getAdjustedPrice(), priceLibVO.getSourceNum()));

        return priceLibVO.getAdjustedPrice();
    }
}
