package com.ejianc.business.zdsmaterial.util;

import com.ejianc.framework.core.util.JsonUtils;
import com.fasterxml.jackson.core.JsonGenerator;
import com.fasterxml.jackson.core.type.TypeReference;
import com.fasterxml.jackson.databind.DeserializationFeature;
import com.fasterxml.jackson.databind.JsonNode;
import com.fasterxml.jackson.databind.ObjectMapper;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.io.IOException;
import java.io.StringWriter;
import java.util.Arrays;
import java.util.List;

/**
 * @author CJ
 * @Description:
 * @date 2024/11/14 18:57
 */
public class DataJsonUtil {

    private static final Logger log = LoggerFactory.getLogger(DataJsonUtil.class);

    // 可以全局共享，所以设计成static类型
    private static final ObjectMapper MAPPER;

    static {
        MAPPER = new ObjectMapper();
        MAPPER.configure(DeserializationFeature.FAIL_ON_UNKNOWN_PROPERTIES, false);
    }

    /**
     *
     * @param pojo 支持JavaBean/Map/List/Array等
     * @return JSON字符串
     * @throws IOException
     */
    public static String object2JsonStr(Object pojo) {
        if (pojo == null) {
            return null;
        }

        String json = null;
        JsonGenerator jsonGenerator = null;
        try {
            StringWriter sw = new StringWriter();
            jsonGenerator = MAPPER.getFactory().createGenerator(sw);
            jsonGenerator.writeObject(pojo);
            json = sw.toString();
        } catch (IOException e) {
            log.error("Convert to json failure.", e);
        } finally {
            if (jsonGenerator != null) {
                try {
                    jsonGenerator.close();
                } catch (IOException e) {
                    ;
                }
            }
        }

        return json;
    }

    /**
     * 说明：json转对象
     *
     * @param json
     * @param cls
     * @return
     * @throws Exception
     */
    public static <T> T jsonStr2Object(String json, Class<T> cls) {
        if (StringUtils.isEmpty(json)) {
            return null;
        }

        T obj = null;
        try {
            obj = MAPPER.readValue(json, cls);
        } catch (Exception e) {
            if (!JsonUtils.isJson(json)) {
                log.error("content is not json: " + json + " --> " + e.getMessage());
            } else {
                log.error("Convert to object failure: " + json, e);
            }
        }

        return obj;
    }

    // string转泛型类
    public static <T> T jsonStr2Object(String json, TypeReference<T> typeReference) {
        if (StringUtils.isEmpty(json)) {
            return null;
        }

        T obj = null;
        try {
            obj = MAPPER.readValue(json, typeReference);
        } catch (Exception e) {
            if (!JsonUtils.isJson(json)) {
                log.error("content is not json: " + json + " --> " + e.getMessage());
            } else {
                log.error("Convert to object failure: " + json, e);
            }
        }

        return obj;
    }

    // string 转list
    public static <T> List<T> json2List(String json, Class<T[]> cls) {
        T[] objArr = JsonUtils.jsonStr2Object(json, cls);
        return Arrays.asList(objArr);
    }

    public static JsonNode json2Tree(String json) {
        if (StringUtils.isEmpty(json)) {
            return null;
        } else if (!JsonUtils.isJson(json)) {
            log.error("content is not json: " + json);
            return null;
        }

        JsonNode jsonTree = null;
        try {
            jsonTree = MAPPER.readTree(json);
        } catch (Exception e) {
            log.error("Convert to JsonNode failure: " + json, e);
        }

        return jsonTree;
    }

    public static boolean isJson(String json) {
        return (json.trim().startsWith("{") || json.trim().startsWith("["));
    }
}
