package com.ejianc.business.zdsmaterial.erp.controller;

import com.alibaba.fastjson.JSONArray;
import com.alibaba.fastjson.JSONObject;
import com.alibaba.fastjson.serializer.SerializerFeature;
import com.baomidou.mybatisplus.core.metadata.IPage;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.ejianc.business.zdsmaterial.asynchandler.QueueUtils;
import com.ejianc.business.zdsmaterial.cons.PlanConstant;
import com.ejianc.business.zdsmaterial.erp.bean.BrandEntity;
import com.ejianc.business.zdsmaterial.erp.bean.BrandRelationEntity;
import com.ejianc.business.zdsmaterial.erp.service.IBrandRelationService;
import com.ejianc.business.zdsmaterial.erp.service.IBrandService;
import com.ejianc.business.zdsmaterial.erp.vo.BrandVO;
import com.ejianc.business.zdsmaterial.erp.vo.SyncJobExecRecordsVO;
import com.ejianc.business.zdsmaterial.material.bean.MaterialCategoryEntity;
import com.ejianc.business.zdsmaterial.material.service.IMaterialCategoryService;
import com.ejianc.business.zdsmaterial.util.ZDSInterfaceCommonUtil;
import com.ejianc.framework.core.context.InvocationInfoProxy;
import com.ejianc.framework.core.exception.BusinessException;
import com.ejianc.framework.core.kit.mapper.BeanMapper;
import com.ejianc.framework.core.kit.time.DateFormatUtil;
import com.ejianc.framework.core.response.CommonResponse;
import com.ejianc.framework.core.response.Parameter;
import com.ejianc.framework.core.response.QueryParam;
import com.ejianc.framework.core.util.HttpTookit;
import com.google.common.base.Stopwatch;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.web.bind.annotation.*;

import java.security.MessageDigest;
import java.util.*;
import java.util.stream.Collectors;

/**
 * @author CJ
 * @Description:
 * @date 2023/12/11 16:39
 */
@RestController
@RequestMapping(value = "/brand/")
public class BrandController {

    private Logger logger = LoggerFactory.getLogger(this.getClass());

    @Autowired
    private IBrandService service;

    @Value("${erp.appId}")
    private String appId;

    @Value("${erp.secret}")
    private String secret;

    @Value("${erp.reqHost}")
    private String reqHost;

    @Value("${erp.batchSize:100}")
    private String batchSize;

    @Autowired
    private QueueUtils queueUtils;

    @Autowired
    private IBrandRelationService relationService;

    @Autowired
    private IMaterialCategoryService categoryService;

    private final String BRAND_ERP_REQ_URL = "/cefoc/yql/getBrandList";

    @Value("${spring.cloud.config.profile}")
    private String profile;

    @Autowired
    private IBrandRelationService brandRelationService;

    /**
     * @Description 参照
     * @Return void
     */
    @GetMapping(value = "refBrandData")
    public CommonResponse<IPage<BrandVO>> refDeliveryData(@RequestParam Integer pageNumber, @RequestParam Integer pageSize,
                                                          String condition,
                                                          String searchObject,
                                                          String searchText) {
        QueryParam param = new QueryParam();
        List<String> fuzzyFields = param.getFuzzyFields();
        fuzzyFields.add("brandName");

        param.setPageSize(pageSize);
        param.setPageIndex(pageNumber);
        param.setSearchText(searchText);
        param.setSearchObject(searchObject);

        Map<String, Parameter> params = param.getParams();
        params.put("tenantId", new Parameter(QueryParam.EQ, InvocationInfoProxy.getTenantid()));

        //查询未验收量量大于零的
//        params.put("enabled", new Parameter(QueryParam.EQ, ZDSMaterialCommonEnums.停启用_启用.getCode()));
        param.getOrderMap().put("sequence", "asc");

        IPage<BrandEntity> page = service.queryPage(param,false);
        IPage<BrandVO> pageData = new Page<>(page.getCurrent(), page.getSize(), page.getTotal());
        pageData.setRecords(BeanMapper.mapList(page.getRecords(), BrandVO.class));
        return CommonResponse.success("查询参照数据成功！",pageData);
    }


    @GetMapping(value = "syncErpBrands")
    public CommonResponse<String> syncErpBrands() {
        logger.info("*********************中电四-品牌信息同步任务 开始*********************");
        Integer pageNum = 0;
        Integer pageSize = Integer.valueOf(batchSize);
        Map<String, Integer> param = new HashMap<>();
        String reqUrl = reqHost + BRAND_ERP_REQ_URL;
        boolean hasNext = true;
        String reqResp = null;

        try {
            Map<String, String> headers = new HashMap<>();
            String nowStr = DateFormatUtil.formatDate("yyyy-MM-dd HH:mm:ss", new Date());
            headers.put("appid", toMD5(appId));
            headers.put("ticket", nowStr);
            headers.put("sign", toMD5(secret + toMD5(appId) + toMD5(nowStr)));
            param.put("PageSize", pageSize);
            JSONObject reqJson = null;
            JSONArray pageData =null;

            param.put("PageNum", ++pageNum);
            reqResp = HttpTookit.postByJson(reqUrl, JSONObject.toJSONString(param), headers,
                    ZDSInterfaceCommonUtil.CONN_TIME_OUT, ZDSInterfaceCommonUtil.READ_TIME_OUT);
            reqJson = JSONObject.parseObject(reqResp);

            if("ok".equals(reqJson.getString("status"))) {
                pageData = reqJson.getJSONArray("data");
                if(null != pageData && pageData.size() > 0) {
                    logger.info("中电四品牌同步，处理第{}页, {}条数据", pageNum, pageData.size());
                    handleBrandErpPage(pageData);
                } else {
                    logger.info("中电四品牌同步 处理数据完成！！！总共{}页数据！", pageNum);
                }
            } else {
                logger.error("请求中电四获取品牌结果返回失败：请求地址-{},参数-{},header-{},结果-{}", reqUrl, JSONObject.toJSONString(param), JSONObject.toJSONString(headers), reqResp);
                queueUtils.sendMq(SyncJobExecRecordsVO.QUEUE_NAME+"_1"+profile, JSONObject.toJSONString(
                        new SyncJobExecRecordsVO(reqUrl, JSONObject.toJSONString(param), PlanConstant.STRING_YES, reqResp, null)));
            }
        } catch (Exception e) {
            logger.error("获取中电四品牌信息异常, 请求地址：{}, 请求参数：{}", reqUrl, JSONObject.toJSONString(param, SerializerFeature.PrettyFormat));
            queueUtils.sendMq(SyncJobExecRecordsVO.QUEUE_NAME+"_1"+profile, JSONObject.toJSONString(
                    new SyncJobExecRecordsVO(reqUrl, JSONObject.toJSONString(param), PlanConstant.STRING_YES, reqResp, e.getMessage())));
            return CommonResponse.error("同步中电四品牌信息异常");
        }

        logger.info("*********************中电四-品牌信息同步任务 结束*********************");
        queueUtils.sendMq(SyncJobExecRecordsVO.QUEUE_NAME+"_1"+profile, JSONObject.toJSONString(new SyncJobExecRecordsVO(reqUrl, JSONObject.toJSONString(param), PlanConstant.STRING_YES)));
        return CommonResponse.success("中电四品牌同步完成！");
    }

    @PostMapping(value = "pageBrandPool")
    public CommonResponse<IPage<BrandVO>> pageBrandPool(@RequestBody QueryParam param) {
        Map<String, Object> params = new HashMap<>();
        if(null != param.getSearchText()) {
            params.put("searchText", param.getSearchText());
        }

        if(param.getParams().containsKey("categoryId")) {
            //查询末级分类信息
            List<Long> categoryIds = new ArrayList<>();
            categoryIds.add(Long.valueOf(param.getParams().get("categoryId").getValue().toString()));
            List<MaterialCategoryEntity> leafCategorys = categoryService.getAllLeafByPids(categoryIds, PlanConstant.INTEGER_YES);

            List<Long> leafCategoryIds = new ArrayList<>();
            if(CollectionUtils.isNotEmpty(leafCategorys)) {
                leafCategoryIds = leafCategorys.stream().map(MaterialCategoryEntity::getId).collect(Collectors.toList());
            } else {
                leafCategoryIds.add(-99L);
            }
            params.put("categoryIds", leafCategoryIds);
            param.getParams().remove("categoryId");
        }

        for(String key: param.getParams().keySet()) {
            params.put(key, param.getParams().get(key).getValue());
        }

        params.put("startLine", (param.getPageIndex() - 1) * param.getPageSize());
        params.put("pageSize", param.getPageSize());

        long total = service.countBrandPool(params);
        if(total == 0) {
            IPage<BrandVO> pageData = new Page<>(param.getPageIndex(), param.getPageSize(), 0);
            pageData.setRecords(new ArrayList<>());

            return CommonResponse.success("查询列表数据成功！",pageData);
        }
        List<BrandVO> brandList = service.pageBrandPool(params);
        IPage<BrandVO> pageData = new Page<>(param.getPageIndex(), param.getPageSize(), total);
        pageData.setRecords(brandList);

        return CommonResponse.success("查询列表数据成功！",pageData);
    }

    /**
     * @Description queryList 查询列表
     * @param param
     * @Return com.ejianc.framework.core.response.CommonResponse<java.lang.String>
     */
    @RequestMapping(value = "/queryList", method = RequestMethod.POST)
    @ResponseBody
    public CommonResponse<IPage<BrandVO>> queryList(@RequestBody QueryParam param) {
        List<String> fuzzyFields = param.getFuzzyFields();
        fuzzyFields.add("brandName");
        fuzzyFields.add("label");
        /** 租户隔离 */
        param.getParams().put("tenantId", new Parameter(QueryParam.EQ, InvocationInfoProxy.getTenantid()));

        if(param.getParams().containsKey("categoryId")) {
            //查询品牌关系
            List<BrandRelationEntity> relations = relationService.getAllByCategoryId(Long.valueOf(param.getParams().get("categoryId").getValue().toString()));
            if(CollectionUtils.isNotEmpty(relations)) {
                param.getParams().put("id", new Parameter(QueryParam.IN, relations.stream().map(BrandRelationEntity::getBrandId).collect(Collectors.toList())));
            } else {
                param.getParams().put("id", new Parameter(QueryParam.EQ, -999L));
            }
            param.getParams().remove("categoryId");
        }

        IPage<BrandEntity> page = service.queryPage(param,false);
        IPage<BrandVO> pageData = new Page<>(page.getCurrent(), page.getSize(), page.getTotal());
        pageData.setRecords(BeanMapper.mapList(page.getRecords(), BrandVO.class));

        return CommonResponse.success("查询列表数据成功！",pageData);
    }

    private void handleBrandErpPage(JSONArray data) {
        JSONObject tmp = null;
        BrandEntity tmpBrand = null;
        Map<String, BrandEntity> brandSourceIdMap = new HashMap<>();
        List<BrandEntity> saveList = new ArrayList<>();

        for(Object obj : data) {
            tmp = (JSONObject) obj;
            tmpBrand = new BrandEntity();
            tmpBrand.setBrandName(tmp.getString("Title"));
            tmpBrand.setSourceId(tmp.getString("C_PS_BrandLibraryOID"));
            tmpBrand.setEnabled(StringUtils.isNotBlank(tmp.getString("IsEnable")) ? "true".equals(tmp.getString("IsEnable").toLowerCase()) ? 1 : 0 : 1);
            tmpBrand.setSequence(null != tmp.get("Sort_ID") ? tmp.getInteger("Sort_ID") : 1);
            tmpBrand.setMemo(null != tmp.getString("Memo") ? tmp.getString("Memo") : null);

            brandSourceIdMap.put(tmpBrand.getSourceId(), tmpBrand);
        }

        List<BrandEntity> dbListBySourceId = service.getAllBySourceIds(new ArrayList<>(brandSourceIdMap.keySet()));
        if(CollectionUtils.isNotEmpty(dbListBySourceId)) {
            //更新列表
            for(BrandEntity dbEntity : dbListBySourceId) {
                tmpBrand = brandSourceIdMap.get(dbEntity.getSourceId());
                dbEntity.setBrandName(tmpBrand.getBrandName());
                dbEntity.setEnabled(tmpBrand.getEnabled());
                dbEntity.setSequence(tmpBrand.getSequence());
                dbEntity.setMemo(tmpBrand.getMemo());
                saveList.add(dbEntity);

                brandSourceIdMap.remove(dbEntity.getSourceId());
            }
        }

        if(!brandSourceIdMap.isEmpty()) {
            saveList.addAll(brandSourceIdMap.values());
        }

        logger.info("本次保存数据-{}条", saveList.size());
        service.saveOrUpdateBatch(saveList, saveList.size(), false);
    }

    public static String toMD5(String plainText) throws Exception {
        MessageDigest digest = MessageDigest.getInstance("MD5");
        byte[] hashBytes = digest.digest(plainText.getBytes());

        StringBuilder sb = new StringBuilder();
        for (byte b : hashBytes) {
            sb.append(Integer.toString((b & 0xff) + 0x100, 16).substring(1));
        }

        return sb.toString().toLowerCase();
    }

    @PostMapping(value = "updateBrandEnable")
    public CommonResponse<String> updateBrandEnable(@RequestBody BrandVO brand) {
        service.updateRelationEnable(brand);
        return CommonResponse.success("操作成功！");
    }

    @PostMapping(value = "pushAllToErp")
    public CommonResponse<String> pushAllToErp(@RequestBody List<Long> ids) {
        logger.info("品牌信息推送ERP-------------开始----------------");
        List<BrandVO> dbList = service.getAllByIds(ids);

        List<JSONObject> pushParams = new ArrayList<>();
        JSONObject syncItem = null;
        for(BrandVO brand : dbList) {
            syncItem = new JSONObject();
            syncItem.put("C_PS_BrandLibraryOID", brand.getSourceId()); //品牌ERP sid
            syncItem.put("Title_ID", brand.getId()); //品牌EL主键
            syncItem.put("Title", brand.getBrandName()); //品牌名称
            syncItem.put("Sort_ID", brand.getSequence()); //序号
            syncItem.put("IsEnable", null != brand.getEnabled() ? 1 == brand.getEnabled() : null); //停启用
//            syncItem.put("Memo", brand.getMemo().trim());//备注
            pushParams.add(syncItem);
        }

        //推送ERP
        Map<String, String> headers = null;
        String reqUrl = reqHost + PlanConstant.品牌更新推送ERP地址;
        boolean haxNext = true;
        List<JSONObject> p = new ArrayList<>();
        for(Iterator<JSONObject> it =pushParams.iterator();it.hasNext();) {
            p.add(it.next());
            it.remove();
            if(p.size() == 1) {
                try {
                    Stopwatch start = Stopwatch.createStarted();
                    headers = ZDSInterfaceCommonUtil.getErpHeaders();
                    String reqResp = HttpTookit.postByJson(reqUrl, JSONObject.toJSONString(p),
                            headers, ZDSInterfaceCommonUtil.CONN_TIME_OUT, ZDSInterfaceCommonUtil.READ_TIME_OUT);

                    logger.info("品牌信息推送ERP：url-{}, 参数：{}，结果：{}", reqUrl,
                            JSONObject.toJSONString(p, SerializerFeature.PrettyFormat), reqResp);
                    Stopwatch stop = start.stop();


                    JSONObject reqJson = JSONObject.parseObject(reqResp);

                    p.clear();
                } catch (Exception e) {
                    throw new BusinessException("品牌信息同步ERP异常，", e);
                }
            }
        }

            return CommonResponse.success("处理成功！");
    }
}
