package com.ejianc.business.zdsmaterial.erp.service.impl;

import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.JSONObject;
import com.alibaba.fastjson.serializer.SerializerFeature;
import com.baomidou.mybatisplus.core.conditions.query.QueryWrapper;
import com.baomidou.mybatisplus.core.conditions.update.UpdateWrapper;
import com.ejianc.business.zdsmaterial.asynchandler.QueueUtils;
import com.ejianc.business.zdsmaterial.asynchandler.bean.MaPushErpRecordsEntity;
import com.ejianc.business.zdsmaterial.asynchandler.consumer.PushErpRecordListener;
import com.ejianc.business.zdsmaterial.cons.PlanConstant;
import com.ejianc.business.zdsmaterial.erp.bean.BrandEntity;
import com.ejianc.business.zdsmaterial.erp.bean.BrandRelationEntity;
import com.ejianc.business.zdsmaterial.erp.mapper.BrandMapper;
import com.ejianc.business.zdsmaterial.erp.service.IBrandRelationService;
import com.ejianc.business.zdsmaterial.erp.service.IBrandService;
import com.ejianc.business.zdsmaterial.erp.vo.BrandRelationVO;
import com.ejianc.business.zdsmaterial.erp.vo.BrandVO;
import com.ejianc.business.zdsmaterial.erp.vo.DataPushErpParam;
import com.ejianc.business.zdsmaterial.util.ZDSInterfaceCommonUtil;
import com.ejianc.foundation.orgcenter.api.IUserApi;
import com.ejianc.foundation.usercenter.vo.UserVO;
import com.ejianc.framework.core.context.InvocationInfoProxy;
import com.ejianc.framework.core.exception.BusinessException;
import com.ejianc.framework.core.http.HttpClientUtils;
import com.ejianc.framework.core.kit.mapper.BeanMapper;
import com.ejianc.framework.core.response.CommonResponse;
import com.ejianc.framework.core.util.EnvironmentTools;
import com.ejianc.framework.core.util.HttpTookit;
import com.ejianc.framework.skeleton.refer.constants.ReferConstant;
import com.ejianc.framework.skeleton.template.BaseServiceImpl;
import com.google.common.base.Stopwatch;
import org.apache.commons.collections.CollectionUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.scheduling.annotation.Async;
import org.springframework.stereotype.Service;

import java.util.*;
import java.util.concurrent.TimeUnit;
import java.util.stream.Collectors;

/**
 * @author CJ
 * @Description:
 * @date 2023/12/11 16:38
 */
@Service
public class BrandServiceImpl extends BaseServiceImpl<BrandMapper, BrandEntity> implements IBrandService {

    private Logger logger = LoggerFactory.getLogger(this.getClass());

    @Autowired
    private BrandMapper mapper;

    @Autowired
    private EnvironmentTools environmentTools;

    @Autowired
    private QueueUtils queueUtils;

    @Autowired
    private IBrandRelationService brandRelationService;

    @Value("${erp.reqHost}")
    private String reqHost;

    @Autowired
    private IUserApi userApi;

    @Value("${spring.cloud.config.profile}")
    private String profile;

    @Override
    public Map<String, BrandEntity> getAllByNames(List<String> brandNames) {
        Map<String, BrandEntity> resp = new HashMap<>();
        QueryWrapper<BrandEntity> query = new QueryWrapper<>();
        query.in("brand_name", brandNames);
        List<BrandEntity> brandList = super.list(query);

        if(CollectionUtils.isNotEmpty(brandList)) {
            resp = brandList.stream().collect(Collectors.toMap(item -> item.getBrandName(), item -> item));
        }

        return resp;
    }

    @Override
    public Map<String, BrandVO> getAllVoMapByNames(List<String> names) {
        Map<String, BrandVO> resp = new HashMap<>();
        QueryWrapper<BrandEntity> query = new QueryWrapper<>();
        query.in("brand_name", names);
        List<BrandEntity> brandList = super.list(query);

        if(CollectionUtils.isNotEmpty(brandList)) {
            List<BrandVO> vos = BeanMapper.mapList(brandList, BrandVO.class);
            resp = vos.stream().collect(Collectors.toMap(item -> item.getBrandName(), item -> item));
        }

        return resp;
    }

    @Override
    public BrandEntity getOneByName(String name) {
        QueryWrapper<BrandEntity> query = new QueryWrapper<>();
        query.eq("brand_name", name);
        List<BrandEntity> brandList = super.list(query);
        if(CollectionUtils.isNotEmpty(brandList)) {
            BrandEntity brandEntity = brandList.get(0);
            return brandEntity;
        }
        return null;
    }

    @Override
    public List<BrandEntity> getAllBySourceIds(List<String> sourceIds) {
        QueryWrapper<BrandEntity> query = new QueryWrapper<>();
        query.in("source_id", sourceIds);
        query.eq("tenant_id", InvocationInfoProxy.getTenantid());

        return super.list(query);
    }

    @Override
    public List<BrandVO> getAllByIds(List<Long> ids) {
        List<BrandVO> resp = new ArrayList<>();
        QueryWrapper<BrandEntity> query = new QueryWrapper<>();
        query.in("id", ids);

        List<BrandEntity> dbList = super.list(query);
        if(CollectionUtils.isNotEmpty(dbList)) {
            resp = BeanMapper.mapList(dbList, BrandVO.class);
        }
        return resp;
    }

    @Override
    @Async(value = "maContextCommonTask")
    public void pushBrandToErp(DataPushErpParam pushParam) {
        logger.info("品牌信息推送ERP-------------开始----------------");
        logger.info("本次推送ERP品牌信息：{}, 推送类型: {}", JSONObject.toJSONString(pushParam.getPushData()), pushParam.getBillType());
        //上下文准备
        if(null == InvocationInfoProxy.getUserid() && null != pushParam.getOperatorId()) {
            initContext(pushParam.getOperatorId());
        }

        String pushResult = null;
        //推送地址
        String reqUrl = reqHost + (PlanConstant.品牌新增.equals(pushParam.getPushType()) ? PlanConstant.品牌新增推送ERP地址 : PlanConstant.品牌更新推送ERP地址);

        CommonResponse<UserVO> userResp = userApi.findUserByUserId(pushParam.getOperatorId());
        if(!userResp.isSuccess() || null == userResp.getData()) {
            logger.error("查询人员id-{}信息失败，{}", pushParam.getOperatorId(), JSONObject.toJSONString(userResp));
            throw new BusinessException("查询人员信息失败");
        }
        UserVO user = userResp.getData();
        //推送参数记录
        MaPushErpRecordsEntity record = new MaPushErpRecordsEntity();
        record.setData(JSONObject.toJSONString(pushParam.getPushData()));
        record.setOperatorId(pushParam.getOperatorId());
        record.setOperatorName(user.getUserName());
        record.setBillType(pushParam.getBillType());
        record.setPushDate(new Date());
        record.setUrl(reqUrl);

        //组装推送参数
        List<JSONObject> pushParams = new ArrayList<>();
        JSONObject syncItem = null;
        List<Long> brandIds = new ArrayList<>();
        for(BrandRelationVO brand : pushParam.getPushData()) {
            if(brandIds.contains(brand.getBrandId())) {
                continue;
            }
            syncItem = new JSONObject();
            syncItem.put("C_PS_BrandLibraryOID", brand.getBrandSid()); //品牌ERP sid
            syncItem.put("Title_ID", brand.getBrandId()); //品牌EL主键
            syncItem.put("Title", brand.getBrandName()); //品牌名称
            syncItem.put("Sort_ID", brand.getBrandSequence()); //序号
            syncItem.put("IsEnable", brand.getBrandEnabled()); //停启用
            syncItem.put("Memo", brand.getMemo());//备注
            syncItem.put("BrandLabel", brand.getBrandLabel());//标签
            syncItem.put("Factory", brand.getSupplierName());//标签
            syncItem.put("CategoryName", brand.getMaterialTypeName()); //物料分类
            syncItem.put("CategorySid", brand.getMaterialTypeSid()); //物料分类
            brandIds.add(brand.getBrandId());
            pushParams.add(syncItem);
        }

        //推送ERP
        Map<String, String> headers = null;
        try {
            Stopwatch start = Stopwatch.createStarted();
            headers = ZDSInterfaceCommonUtil.getErpHeaders();
            String reqResp = HttpTookit.postByJson(reqUrl, JSONObject.toJSONString(pushParams),
                    headers, ZDSInterfaceCommonUtil.CONN_TIME_OUT, ZDSInterfaceCommonUtil.READ_TIME_OUT);

            logger.info("品牌信息推送ERP：url-{}, 参数：{}，结果：{}", reqUrl,
                    JSONObject.toJSONString(pushParams, SerializerFeature.PrettyFormat), reqResp);
            Stopwatch stop = start.stop();

            record.setErpResult(reqResp);
            record.setErpHandleSeconds(stop.elapsed(TimeUnit.SECONDS));
            logger.info("品牌信息推送ERP耗时：{}秒", record.getErpHandleSeconds());

            JSONObject reqJson = JSONObject.parseObject(reqResp);
            if("true".equals(reqJson.getString("status"))) {
                pushResult = PlanConstant.STRING_YES;
            } else {
                pushResult = PlanConstant.STRING_NO;
                record.setFailType("ERP数据处理失败！");
            }
            record.setSucFlag(pushResult);

        } catch (Exception e) {
            logger.error("品牌信息同步ERP异常，", e);
            record.setSucFlag(PlanConstant.STRING_NO);
            pushResult = PlanConstant.STRING_NO;
            record.setFailType("品牌信息同步ERP，请求异常！");
        }

        //更新品牌推送ERP结果
        this.updateBrandPushFlag(new ArrayList<>(pushParam.getPushData().stream().map(BrandRelationVO::getBrandId).collect(Collectors.toSet())), pushResult);

        //推送记录放入待处理队列中
        queueUtils.sendMq(PushErpRecordListener.PUSH_ERP_RECORD_QUEUE+"_1"+profile, JSONObject.toJSONString(record));
        logger.info("品牌信息推送ERP-------------结束----------------");
    }

    private void updateBrandPushFlag(List<Long> ids, String pushResult) {
        UpdateWrapper<BrandEntity> update = new UpdateWrapper<>();
        update.in("id", ids);
        update.set("push_erp_flag", pushResult);

        super.update(update);
    }

    private void initContext(Long userId) {
        String initContextUrl = environmentTools.getBaseHost() + "ejc-idm-web/user/context/getBytenantid?tenantId=999999";
        initContextUrl = initContextUrl + "&userId="+userId.toString();
        String responseStr = HttpClientUtils.getInstance().getSync(initContextUrl);

        CommonResponse<JSONObject> userContextResponse = JSON.parseObject(responseStr, CommonResponse.class);
        try {
            if(userContextResponse.isSuccess()) {
                JSONObject userContext =userContextResponse.getData().getJSONObject("userContext");
                StringBuilder authtoken = new StringBuilder();
                if(null != userContext) {
                    authtoken.append("userType=").append(userContext.get("userType"))
                            .append(";userCode=").append(userContext.get("userCode"))
                            .append(";orgId=").append(userContext.get("orgId"))
                            .append(";tenantid=").append(userContext.get("tenantid"))
                            .append(";token=").append(userContext.get("token"))
                            .append(";u_logints=").append(userContext.get("u_logints"))
                            .append(";u_usercode=").append(userContext.get("u_usercode"))
                            .append(";userId=").append(userContext.get("userId"));
                }
                if(authtoken.length() > 0) {
                    InvocationInfoProxy.setExtendAttribute("authority", authtoken.toString());
                    InvocationInfoProxy.setParameter(ReferConstant.HEAD_authority, authtoken.toString());
                }
                if(null != userContext) {
                    InvocationInfoProxy.setTenantid(userContext.getLong("tenantid"));
                    InvocationInfoProxy.setUserid(userContext.getLong("userId"));
                    InvocationInfoProxy.setUsercode(userContext.getString("userCode"));
                    InvocationInfoProxy.setOrgId(userContext.getLong("orgId"));
                }
            }
        } catch (Exception e) {
            logger.error("初始化异步任务上下文异常：",e);
        }

    }

    @Override
    public void updateBrandEnable(BrandVO brand) {
        BrandEntity entity = super.selectById(brand.getId());
        entity.setEnabled(brand.getEnabled());
        super.saveOrUpdate(entity, false);
    }

    @Override
    public long countBrandPool(Map<String, Object> param) {
        return mapper.countBrandPool(param);
    }

    @Override
    public List<BrandVO> pageBrandPool(Map<String, Object> param) {
        return mapper.pageBrandPool(param);
    }

    @Override
    public void updateRelationEnable(BrandVO brand) {
        BrandEntity entity = super.selectById(brand.getId());
        entity.setRelationEnabled(brand.getRelationEnabled());
        super.saveOrUpdate(entity, false);

        //推送ERP
        List<BrandRelationEntity> pushList = brandRelationService.getByBrandId(brand.getId());
        if(CollectionUtils.isNotEmpty(pushList)) {
            for(BrandRelationEntity relation : pushList) {
                relation.setBrandSequence(entity.getSequence());
                relation.setBrandSid(entity.getSourceId());
                relation.setBrandName(entity.getBrandName());
                relation.setBrandId(entity.getId());
                relation.setBrandEnabled(brand.getRelationEnabled());
            }

            brandRelationService.syncRelationToErp(new DataPushErpParam(BeanMapper.mapList(pushList, BrandRelationVO.class),
                    InvocationInfoProxy.getUserid(), PlanConstant.品牌关系更新, PlanConstant.BRAND_POOL_BILL_TYPE));
        } else {
            logger.info("品牌【id-{}，name-{}下无品牌供应商关系，不推送ERP~】");
        }
    }
}
