package com.ejianc.business.zdsmaterial.erp.service.impl;

import com.baomidou.mybatisplus.core.conditions.query.QueryWrapper;
import com.ejianc.business.zdsmaterial.erp.bean.ExpenseAccountDetailEntity;
import com.ejianc.business.zdsmaterial.erp.mapper.ExpenseAccountDetailMapper;
import com.ejianc.business.zdsmaterial.erp.service.IExpenseAccountDetailService;
import com.ejianc.business.zdsmaterial.util.ComputeUtil;
import com.ejianc.framework.cache.utils.RedisTool;
import com.ejianc.framework.skeleton.template.BaseServiceImpl;
import com.ejianc.framework.skeleton.template.BaseVO;
import org.apache.commons.collections.CollectionUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import redis.clients.jedis.Jedis;
import redis.clients.jedis.JedisPool;

import java.math.BigDecimal;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.concurrent.TimeUnit;

/**
 * @author CJ
 * @Description:
 * @date 2024/10/22 17:19
 */
@Service("expenseAccountDetailService")
public class ExpenseAccountDetailServiceImpl extends BaseServiceImpl<ExpenseAccountDetailMapper, ExpenseAccountDetailEntity>
        implements IExpenseAccountDetailService {

    private static final String BILL_TYPE_CODE = "EJCBT202312000001";
    private final String OPERATE = "EXPENSE_ACCOUNT_ACCEPT";
    private Logger logger = LoggerFactory.getLogger(this.getClass());
    @Autowired
    private JedisPool jedisPool;

    @Override
    public List<ExpenseAccountDetailEntity> getAllBySourceDetailIds(List<String> sourceDetailIds) {
        QueryWrapper<ExpenseAccountDetailEntity> query = new QueryWrapper<>();
        query.eq("dr", BaseVO.DR_UNDELETE);
        query.in("source_detail_id", sourceDetailIds);

        return super.list(query);
    }

    @Override
    public String handleAcceptNum(Long expenseAccountId, Map<Long, BigDecimal> nums, boolean addFlag) {
        String msg = null;
        Jedis jedis = null;
        boolean locked = false;
        String key = BILL_TYPE_CODE + "::" + expenseAccountId.toString();
        int retryTime = 0;
        try {
            jedis = jedisPool.getResource();
            List<ExpenseAccountDetailEntity> updateList = new ArrayList<>();

            locked = RedisTool.tryLock(jedis, key, OPERATE, 600);
            while (!locked && retryTime <= 3) {
                retryTime++;
                logger.info("报销单id:{}明细入库量更新，获取锁失败，{}s后重试", expenseAccountId, retryTime * 5);
                TimeUnit.SECONDS.sleep(retryTime * 5);
                locked = RedisTool.tryLock(jedis, key, OPERATE, 600);
            }

            //加锁失败
            if (!locked) {
                releaseLock(jedis, false, key, OPERATE);
                return "零星材料回写报销单失败，获取锁失败！";
            }

            List<Long> detailIds = new ArrayList<>(nums.keySet());
            List<ExpenseAccountDetailEntity> dbDetailList = this.getAllByIds(detailIds);
            for(ExpenseAccountDetailEntity dbDetail : dbDetailList) {
                if(nums.containsKey(dbDetail.getId())) {
                    if(addFlag) {
                        if(dbDetail.getPurchaseNum().compareTo(ComputeUtil.safeAdd(dbDetail.getAcceptedNum(), nums.get(dbDetail.getId())))  < 0) {
                            return "操作失败，编码为【"+dbDetail.getMaterialCode()+
                                    "】，名称为【"+dbDetail.getMaterialName()+
                                    "】的材料本次可验收量为【" + ComputeUtil.safeSub(dbDetail.getPurchaseNum(), dbDetail.getAcceptedNum()).stripTrailingZeros().toPlainString() + "】，小于本次验收量【"+
                                    nums.get(dbDetail.getId()).stripTrailingZeros().toPlainString()+"】";
                        }
                        dbDetail.setAcceptedNum(ComputeUtil.safeAdd(dbDetail.getAcceptedNum(), nums.get(dbDetail.getId())));
                    } else {
                        dbDetail.setAcceptedNum(ComputeUtil.safeSub(dbDetail.getAcceptedNum(), nums.get(dbDetail.getId())));
                    }
                    updateList.add(dbDetail);
                }
            }

            if(CollectionUtils.isNotEmpty(updateList)) {
                super.saveOrUpdateBatch(updateList,updateList.size(), false);
            }

            return null;
        } catch (Exception e) {
            logger.error("零星材料回写报销单:{}异常，",expenseAccountId, e);
        } finally {
            releaseLock(jedis, false, key, OPERATE);
        }
        return null;
    }

    @Override
    public String validateAcceptNum(Map<Long, BigDecimal> nums) {
        List<ExpenseAccountDetailEntity> dbDetailList = this.getAllByIds(new ArrayList<>(nums.keySet()));
        for(ExpenseAccountDetailEntity dbDetail : dbDetailList) {
            if(nums.containsKey(dbDetail.getId())) {
                if(dbDetail.getPurchaseNum().compareTo(ComputeUtil.safeAdd(dbDetail.getAcceptedNum(), nums.get(dbDetail.getId())))  < 0) {
                    return "编码为【"+dbDetail.getMaterialCode()+
                            "】，名称为【"+dbDetail.getMaterialName()+
                            "】的材料本次可验收量为【" + ComputeUtil.safeSub(dbDetail.getPurchaseNum(), dbDetail.getAcceptedNum()).stripTrailingZeros().toPlainString() + "】，小于本次验收量【"+
                            nums.get(dbDetail.getId()).stripTrailingZeros().toPlainString()+"】";
                }
            }
        }
        return null;
    }

    private List<ExpenseAccountDetailEntity> getAllByIds(List<Long> detailIds) {
        QueryWrapper<ExpenseAccountDetailEntity> query = new QueryWrapper<>();
        query.in("id", detailIds);
        query.eq("dr", BaseVO.DR_UNDELETE);

        return super.list(query);
    }

    public void releaseLock(Jedis jedis, boolean locked, String key, String OPERATE) {
        try {
            if (locked) {
                RedisTool.releaseLock(jedis, key, OPERATE);
            }
        } finally {
            if (null != jedis) {
                jedis.close();
            }
        }
    }
}
