package com.ejianc.business.zdsmaterial.pricelib.processor;

import com.alibaba.fastjson.JSONObject;
import com.ejianc.business.zdsmaterial.material.vo.MaterialCategoryVO;
import com.ejianc.business.zdsmaterial.pricelib.vo.MaterialPriceCalcSubConditionVO;
import com.ejianc.business.zdsmaterial.pricelib.vo.MaterialPriceSubConditionDetailVO;
import com.ejianc.business.zdsmaterial.pricelib.vo.PriceLibVO;
import com.ejianc.business.zdsmaterial.util.ComputeUtil;
import jodd.util.StringUtil;
import org.apache.commons.collections.CollectionUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.stereotype.Component;

import java.math.BigDecimal;

/**
 * 不锈钢无缝管单品牌计算公式处理器
 *
 * 公式：（基准价格/重量*1000-基准原材料网价+浮动网价）*重量*0.001
 *
 *
 * @Author: caojie
 * @CreateTime:2025-07-07 10:00
 * @Version: 1.0
 */
@Component
public class C0301BrandAndPropertyProcessor implements BaseCalcProcessor {

    private Logger logger = LoggerFactory.getLogger(this.getClass());

    @Override
    public boolean calcBeforeCheck(PriceLibVO priceLibVO, MaterialPriceCalcSubConditionVO calcFormula) {
        MaterialCategoryVO categoryVO = priceLibVO.getCategoryVO();
        String categoryName = "";
        if(null != categoryVO) {
            categoryName = categoryVO.getName();
        }
        if(null == priceLibVO.getBaseMaterialNetPrice()) {
            logger.error("【{}】公式：C0301BrandAndPropertyProcessor 计算终止，原材料网价为空, {}", categoryName, JSONObject.toJSONString(priceLibVO));
            priceLibVO.setCalcContent("未计算：原材料网价为空");
            return false;
        }
        if(null == priceLibVO.getBaseMaterialUnitPrice()) {
            logger.error("【{}】公式：C0301BrandAndPropertyProcessor 计算终止，原材料基础价为空, {}", categoryName, JSONObject.toJSONString(priceLibVO));
            priceLibVO.setCalcContent("未计算：原材料基础价为空");
            return false;
        }
        if(null == priceLibVO.getBaseMaterialPrice()) {
            logger.error("【{}】公式：C0301BrandAndPropertyProcessor 计算终止，基础单价为空, {}", categoryName, JSONObject.toJSONString(priceLibVO));
            priceLibVO.setCalcContent("未计算：基础单价为空");
            return false;
        }
        if(CollectionUtils.isEmpty(calcFormula.getDetailList())) {
            logger.error("【{}】公式：C0301BrandAndPropertyProcessor 计算终止， 计算规则元素清单为空, {}", categoryName, JSONObject.toJSONString(priceLibVO));
            priceLibVO.setCalcContent("未计算：计算规则元素清单为空");
            return false;
        }
        if(StringUtil.isBlank(priceLibVO.getExtParam1())) {
            logger.error("【{}】公式：C0301BrandAndPropertyProcessor 计算终止， 重量为空, {}", categoryName, JSONObject.toJSONString(priceLibVO));
            priceLibVO.setCalcContent("未计算：计算因素[重量]为空");
            return false;
        }

        return true;
    }

    @Override
    public BigDecimal calc(PriceLibVO priceLibVO, MaterialPriceCalcSubConditionVO calcFormula) {

        //公式：（基准价格/重量*1000-基准原材料网价+浮动网价）*重量*0.001

        if(!calcBeforeCheck(priceLibVO, calcFormula)) {
            return BigDecimal.ZERO;
        }
        priceLibVO.setCalcContent(calcFormula.getCalcDisplayContent());
        MaterialPriceSubConditionDetailVO calcElem = calcFormula.getDetailList().get(0);
        JSONObject params = JSONObject.parseObject(calcFormula.getCalcParams());

        BigDecimal result = BigDecimal.ZERO;

        //计算 基准价格/重量*1000
        result = ComputeUtil.safeMultiply(ComputeUtil.safeDiv(priceLibVO.getBaseMaterialPrice(),new BigDecimal(priceLibVO.getExtParam1())), new BigDecimal("1000"));
        //计算 中间结果 -基准原材料网价+浮动网价
        result = ComputeUtil.safeAdd(ComputeUtil.safeSub(result,priceLibVO.getBaseMaterialUnitPrice()), priceLibVO.getBaseMaterialNetPrice());
        //计算 中间结果 * 重量*0.001
        result = ComputeUtil.safeMultiply(result, ComputeUtil.safeMultiply(new BigDecimal(priceLibVO.getExtParam1()), new BigDecimal("0.001")));
        priceLibVO.setAdjustedPrice(result);
        priceLibVO.setAdjustedFirstPrice(result);
        //计算金额
        priceLibVO.setCalcMny(ComputeUtil.safeMultiply(result, priceLibVO.getSourceNum()));

        return result;
    }
}
