package com.ejianc.business.zdsmaterial.erp.service.impl;

import com.alibaba.fastjson.JSONObject;
import com.alibaba.fastjson.serializer.SerializerFeature;
import com.baomidou.mybatisplus.core.conditions.query.QueryWrapper;
import com.baomidou.mybatisplus.core.conditions.update.UpdateWrapper;
import com.ejianc.business.zdsmaterial.accept.bean.AcceptEntity;
import com.ejianc.business.zdsmaterial.cons.PlanConstant;
import com.ejianc.business.zdsmaterial.cons.enums.ZDSMaterialCommonEnums;
import com.ejianc.business.zdsmaterial.erp.bean.*;
import com.ejianc.business.zdsmaterial.erp.enums.OrderDeliveryStatusEnums;
import com.ejianc.business.zdsmaterial.erp.mapper.OrderMapper;
import com.ejianc.business.zdsmaterial.erp.service.IOrderDetailService;
import com.ejianc.business.zdsmaterial.erp.service.IOrderService;
import com.ejianc.business.zdsmaterial.erp.vo.OrderDetailVO;
import com.ejianc.business.zdsmaterial.erp.vo.OrderVO;
import com.ejianc.business.zdsmaterial.material.service.IMaterialCategoryService;
import com.ejianc.business.zdsmaterial.material.vo.MaterialCategoryVO;
import com.ejianc.business.zdssupplier.material.api.IMatSupplierApi;
import com.ejianc.business.zdssupplier.material.vo.MatSupplierVO;
import com.ejianc.foundation.message.api.IPushMessageApi;
import com.ejianc.foundation.message.vo.PushMsgParameter;
import com.ejianc.foundation.orgcenter.api.IEmployeeApi;
import com.ejianc.foundation.orgcenter.vo.EmployeeVO;
import com.ejianc.foundation.share.api.IProSupplierApi;
import com.ejianc.foundation.share.api.IProjectPoolApi;
import com.ejianc.foundation.share.vo.ProjectPoolSetVO;
import com.ejianc.foundation.support.api.IBillCodeApi;
import com.ejianc.foundation.support.api.IParamConfigApi;
import com.ejianc.foundation.support.vo.BillCodeParam;
import com.ejianc.foundation.support.vo.ParamRegisterSetVO;
import com.ejianc.framework.cache.utils.RedisTool;
import com.ejianc.framework.core.context.InvocationInfoProxy;
import com.ejianc.framework.core.exception.BusinessException;
import com.ejianc.framework.core.kit.mapper.BeanMapper;
import com.ejianc.framework.core.response.CommonResponse;
import com.ejianc.framework.core.util.ComputeUtil;
import com.ejianc.framework.skeleton.dataPush.ISystemDataPushService;
import com.ejianc.framework.skeleton.template.BaseServiceImpl;
import com.ejianc.framework.skeleton.template.BaseVO;
import com.ejianc.support.idworker.util.IdWorker;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import redis.clients.jedis.Jedis;
import redis.clients.jedis.JedisPool;

import java.io.InputStream;
import java.math.BigDecimal;
import java.util.*;
import java.util.stream.Collectors;

/**
 * 订单
 * 
 * @author generator
 * 
 */
@Service("orderService")
public class OrderServiceImpl extends BaseServiceImpl<OrderMapper, OrderEntity> implements IOrderService{

    private Logger logger = LoggerFactory.getLogger(this.getClass());
    private static final String BILL_CODE = "ZDS-ORD";

    @Autowired
    private JedisPool jedisPool;

    @Autowired
    private IOrderDetailService detailService;

    @Autowired
    private IBillCodeApi billCodeApi;

    @Autowired
    private ISystemDataPushService systemDataPushService;

    @Autowired
    private IProSupplierApi proSupplierApi;

    @Autowired
    private IMaterialCategoryService categoryService;

    private final String OPERATE = "orderBill";

    @Autowired
    private IParamConfigApi paramConfigApi;

    @Autowired
    private OrderMapper mapper;

    @Autowired
    private IEmployeeApi employeeApi;

    private final String PUSH_BILL_SERVER_URL = "/ejc-zdssupbusiness-web/openapi/supOrder/syncOrder";
    private final String UPDATE_STATUS_BILL_SERVER_URL = "/ejc-zdssupbusiness-web/openapi/supOrder/syncUpdateFreezeStatus";

    private final String ORDER_CHECK_CODE = "P-lk3to00003";

    @Autowired
    private IPushMessageApi pushMessageApi;

    @Autowired
    private IProjectPoolApi projectPoolApi;

    @Autowired
    private IMatSupplierApi matSupplierApi;

    @Override
    public OrderVO getOneBySourceContractId(String sourceOrderId) {
        OrderVO resp = new OrderVO();

        QueryWrapper<OrderEntity> query = new QueryWrapper<>();
        query.eq("source_order_id", sourceOrderId);
        OrderEntity e = super.getOne(query);

        if(null != e) {
            e.setDetailList(detailService.getAllByOrderId(e.getId()));
            resp = BeanMapper.map(e, OrderVO.class);
        }

        return resp;
    }

    @Override
    public void saveByContracts(List<ContractEntity> saveList) {
        List<OrderEntity> orderList = new ArrayList<>();
        List<OrderDetailEntity> detailList = new ArrayList<>();
        OrderEntity tmp = null;
        for(ContractEntity c : saveList) {
            if(ZDSMaterialCommonEnums.总价合同.getCode().equals(c.getPriceType())) {
                tmp = generateOrder(c);
                if(null != tmp) {
                    orderList.add(tmp);
                    detailList.addAll(tmp.getDetailList());
                }
            }
        }

        if(CollectionUtils.isNotEmpty(orderList)) {
            super.saveOrUpdateBatch(orderList, orderList.size(), false);
        }
        if(CollectionUtils.isNotEmpty(detailList)) {
            detailService.saveOrUpdateBatch(detailList, detailList.size(), false);
        }
    }

    @Override
    public OrderEntity updateFreezeStatus(OrderVO saveOrUpdateVO) {
        OrderEntity orderEntity = super.selectById(saveOrUpdateVO.getId());
        orderEntity.setFreezeStatus(saveOrUpdateVO.getFreezeStatus());
        super.updateById(orderEntity);
        OrderVO orderVO = BeanMapper.map(orderEntity, OrderVO.class);

        //同步供方
        syncUpdateFreezeStatus(orderVO);
        return orderEntity;
    }

    @Override
    public List<Long> getCanSyncOrderIdsBySupplierId(Long supplierId) {
        return mapper.getCanSyncOrderIdsBySupplierId(supplierId);
    }

    @Override
    public void cancelOrderByContractId(List<Long> mainContractIds) {

        //查询订单信息
        QueryWrapper<OrderEntity> update = new QueryWrapper<>();
        update.in("contract_id", mainContractIds);
        update.eq("dr", BaseVO.DR_UNDELETE);
        List<OrderEntity> dbList = super.list(update);
        if(CollectionUtils.isNotEmpty(dbList)) {
            this.cancelOrderList(dbList.stream().map(OrderEntity::getId).collect(Collectors.toList()));
        }
    }

    @Override
    public void cancelOrderList(List<Long> orderIds) {
        UpdateWrapper<OrderEntity> update = new UpdateWrapper<>();
        update.in("id", orderIds);
        update.eq("dr", BaseVO.DR_UNDELETE);
        update.set("dr", 1);
        update.set("update_time", new Date());
        update.set("memo", "ERP-合同作废删除");
        super.update(update);

        //删除子表
        detailService.cancelByOrderIds(orderIds);
    }

    //删除供方业务单据数据
    private String syncUpdateFreezeStatus(OrderVO orderVO) {
        String msg = null;
        Jedis jedis = null;
        boolean locked = false;
        String key = OPERATE + "::" + orderVO.getId().toString();

        //设置单据当前系统信息
        CommonResponse<String> ejcCloudSystemCode = proSupplierApi.getEjcCloudSystemCode();
        if (!ejcCloudSystemCode.isSuccess()) {
            logger.error("推送订单暂停状态-{}失败，获取当前系统编码失败,{}", orderVO.getId(), ejcCloudSystemCode.getMsg());
            return "推送订单暂停状态失败，获取当前系统编码失败";
        }

        //设置当前系统ID
        orderVO.setSourceSystemId(ejcCloudSystemCode.getData());
        orderVO.setSourceId(orderVO.getId().toString());
        Map<String, String> paramMap = new HashMap<>();
        paramMap.put("transData", JSONObject.toJSONString(orderVO));

        Map<String, Map<String, InputStream>> files = new HashMap<>();

        try {
            jedis = jedisPool.getResource();
            //对单据进行加锁
            locked = RedisTool.tryLock(jedis, key, OPERATE, 600);

            //推送单据到指定的供方
            CommonResponse<String> syncReqResp = systemDataPushService.exchangeDataAndFilesWithEachLinkSystem(UPDATE_STATUS_BILL_SERVER_URL,
                    paramMap,
                    orderVO.getSupplierId().toString(),
                    files);
            if (syncReqResp.isSuccess()) {
                CommonResponse<String> billPushResp = JSONObject.parseObject(syncReqResp.getData(), CommonResponse.class);
                if (billPushResp.isSuccess()) {
                } else {
                    logger.error("供方id-{}推送订单暂停状态id-{}失败, {}", orderVO.getSupplierId(), orderVO.getId(), billPushResp.getMsg());
                    msg = "订单暂停状态失败，供方处理订单暂停状态失败";
                }
            } else {
                logger.error("发送请求订单暂停状态id-{}给供方id-{}失败, {}", orderVO.getId(), orderVO.getSupplierId(), syncReqResp.getMsg());
                msg = "发送请求订单暂停状态失败";
            }
        } catch (Exception e) {
            logger.error("订单暂停状态id-{}推送供方订单暂停状态，", orderVO.getId(), e);
            msg = "操作失败，订单暂停状态失败！";
        } finally {
            releaseLock(jedis, locked, key, OPERATE);
        }

        return msg;
    }

    private OrderEntity generateOrder(ContractEntity contract) {
        //新增订单
        OrderEntity orderEntity = new OrderEntity();
        orderEntity.setId(IdWorker.getId());
        orderEntity.setFreezeStatus(1);
        orderEntity.setApprovedTaxMny(BigDecimal.ZERO); //批复金额
        orderEntity.setBalanceTaxMny(contract.getBalanceTaxMny()); //合同余额
        orderEntity.setContractPriceType(ZDSMaterialCommonEnums.总价合同.getCode());
        orderEntity.setSourceOrderCode(contract.getContractCode()); //取合同编码作为订单编码

        //合同、供应商信息
        orderEntity.setSourceContractId(contract.getSourceContractId());
        orderEntity.setContractId(contract.getId());
        orderEntity.setContractCode(contract.getContractCode());
        orderEntity.setContractName(contract.getContractName());
        orderEntity.setContractPriceType(contract.getPriceType());
        orderEntity.setSupplierId(contract.getSupplierId());
        orderEntity.setSupplierCode(contract.getSupplierCode());
        orderEntity.setSupplierName(contract.getSupplierName());
        orderEntity.setSupplierManagerSupId(contract.getSupplierManagerSupId());

        //供应商联系人、供应商联系人电话
        orderEntity.setSupplierLinkerId(contract.getSupplierManagerId());
        orderEntity.setSupplierLinkerSid(contract.getSupplierManagerSid());
        orderEntity.setSupplierLinkName(contract.getSupplierManagerName());
        orderEntity.setSupplierLinkPhone(contract.getSupplierManagerPhone());

        //项目信息
        orderEntity.setProjectId(contract.getProjectId());
        orderEntity.setProjectCode(contract.getProjectCode());
        orderEntity.setProjectName(contract.getProjectName());
        orderEntity.setOrgId(contract.getOrgId());
        orderEntity.setOrgCode(contract.getOrgCode());
        orderEntity.setOrgName(contract.getOrgName());
        orderEntity.setParentOrgId(contract.getParentOrgId());
        orderEntity.setParentOrgCode(contract.getParentOrgCode());
        orderEntity.setParentOrgName(contract.getParentOrgName());

        //停启用、发货日期、接货人、接货人联系方式
        orderEntity.setEnableStatus(contract.getEnableStatus());
        orderEntity.setDeliveryDate(contract.getDeliveryDate());
        orderEntity.setAcceptorId(contract.getMasterPorterId());
        orderEntity.setAcceptorCode(contract.getMasterPorterCode());
        orderEntity.setAcceptorName(contract.getMasterPorterName());
        orderEntity.setAcceptorLinkPhone(contract.getLinkPhone());

        //发货地址、要求
        orderEntity.setDeliveryAddress(contract.getDeliveryAddress());
        orderEntity.setDeliveryRemark(contract.getDeliveryRemark());
        orderEntity.setRemark(contract.getRemark());

        //采购人信息
        orderEntity.setPurchaseUserId(contract.getEmployeeId());
        orderEntity.setPurchaseUserCode(contract.getEmployeeCode());
        orderEntity.setPurchaseUserName(contract.getEmployeeName());
        orderEntity.setPurchaseUserLinkPhone(contract.getEmployeePhone());

        //订单金额
        orderEntity.setOrderMny(contract.getMny());
        orderEntity.setOrderTaxMny(contract.getTaxMny());
        //已发货金额
        orderEntity.setDeliveredMny(BigDecimal.ZERO);
        orderEntity.setDeliveredTaxMny(BigDecimal.ZERO);
        //验收金额
        orderEntity.setHandAcceptMny(BigDecimal.ZERO);
        orderEntity.setHandAcceptTaxMny(BigDecimal.ZERO);
        orderEntity.setTotalAcceptMny(BigDecimal.ZERO);
        orderEntity.setTotalAcceptTaxMny(BigDecimal.ZERO);

        orderEntity.setDeliveryStatus(OrderDeliveryStatusEnums.待发货.getCode());
        orderEntity.setSignatureStatus(ZDSMaterialCommonEnums.签章确认状态_未签字.getCode());
        OrderDetailEntity orderDetail = null;
        for(ContractDetailEntity contractDetail : contract.getDetailList()) {
            if(StringUtils.isBlank(contractDetail.getMaterialSerialNo()) || !contractDetail.getMaterialSerialNo().contains("pd-")) {
                logger.info("合同-【id-{}，code-{}】对应明细[{}]物资流水号为空或非来源于采购计划，跳过生成订单明细！", contract.getId(), contract.getContractCode(), contractDetail);
                continue;
            }
            if(null == contractDetail.getPurNum() || BigDecimal.ZERO.compareTo(contractDetail.getPurNum()) == 1) {
                logger.info("合同-【id-{}，code-{}】对应明细[{}]数量为空或为0，跳过生成订单明细！", contract.getId(), contract.getContractCode(), contractDetail);
                continue;
            }
            if(null == contractDetail.getBrandId()) {
                logger.info("合同-【id-{}，code-{}】对应明细[{}]品牌为空，跳过生成订单明细！", contract.getId(), contract.getContractCode(), contractDetail);
                continue;
            }

            orderDetail = new OrderDetailEntity();
            //来源合同信息
            orderDetail.setContractDetailId(contractDetail.getId());
            orderDetail.setSourceContractDetailId(contractDetail.getSourceDetailId());
            orderDetail.setSourceContractId(contract.getSourceContractId());

            //合同量、订单量
            orderDetail.setContractNum(contractDetail.getPurNum());
            orderDetail.setDetailNum(contractDetail.getPurNum());
            orderDetail.setHandAcceptNum(BigDecimal.ZERO); //手动验收量
            orderDetail.setDeliveredNum(BigDecimal.ZERO); //已发货量
            orderDetail.setAcceptedNum(BigDecimal.ZERO); //发货单验收量

            //物资信息
            orderDetail.setMaterialSerialNo(contractDetail.getMaterialSerialNo());
            orderDetail.setMaterialTypeId(contractDetail.getMaterialTypeId());
            orderDetail.setMaterialTypeName(contractDetail.getMaterialTypeName());
            orderDetail.setMaterialTypeCode(contractDetail.getMaterialTypeCode());
            orderDetail.setMaterialId(contractDetail.getMaterialId());
            orderDetail.setMaterialCode(contractDetail.getMaterialCode());
            orderDetail.setMaterialName(contractDetail.getMaterialName());
            orderDetail.setDetailPrice(contractDetail.getDetailPrice());
            orderDetail.setDetailTaxPrice(contractDetail.getDetailTaxPrice());
            orderDetail.setDetailMny(contractDetail.getDetailMny());
            orderDetail.setDetailTaxMny(contractDetail.getDetailTaxMny());
            orderDetail.setDetailTaxRate(contractDetail.getDetailTaxRate());
            orderDetail.setDetailTax(contractDetail.getDetailTax());
            orderDetail.setDetailUnitId(contractDetail.getDetailUnitId());
            orderDetail.setDetailUnitName(contractDetail.getDetailUnitName());
            orderDetail.setMemo(contractDetail.getDetailRemark());
            orderDetail.setProductCode(contractDetail.getProductCode());
            orderDetail.setPropertyValue(contractDetail.getPropertyValue());
            orderDetail.setBrandId(contractDetail.getBrandId());
            orderDetail.setBrandName(contractDetail.getBrandName());
            orderDetail.setId(IdWorker.getId());
            orderDetail.setOrderId(orderEntity.getId());
            orderDetail.setRowState("add");

            orderEntity.getDetailList().add(orderDetail);
        }

        if(CollectionUtils.isEmpty(orderEntity.getDetailList())) {
            logger.info("合同-[id-{},code-{}]可生成订单明细为空，不生成采购订单！！！！！", contract.getId(), contract.getContractCode());
            return null;
        }

        BillCodeParam billCodeParam = BillCodeParam.build(BILL_CODE, InvocationInfoProxy.getTenantid(), BeanMapper.map(orderEntity, OrderVO.class));
        CommonResponse<String> billCode = billCodeApi.generateBillCode(billCodeParam);
        if(billCode.isSuccess()) {
            orderEntity.setBillCode(billCode.getData());
        }else{
            throw new BusinessException("网络异常， 编码生成失败， 请稍后再试");
        }

        //推送供方
        String msg = pushToSupBusiness(BeanMapper.map(orderEntity, OrderVO.class));
        orderEntity.setBillPushFlag(StringUtils.isNotBlank(msg) ? "sucPushed" : "unSucPush");

        return orderEntity;
    }

    @Override
    public void saveByContract(ContractEntity contract, boolean addFlag) {
        OrderEntity orderEntity = null;
        if(addFlag) {
            //新增订单
            orderEntity = new OrderEntity();
            orderEntity.setId(IdWorker.getId());

            //合同、供应商信息
            orderEntity.setSourceContractId(contract.getSourceContractId());
            orderEntity.setContractId(contract.getId());
            orderEntity.setContractCode(contract.getBillCode());
            orderEntity.setContractName(contract.getContractName());
            orderEntity.setContractPriceType(contract.getPriceType());
            orderEntity.setSupplierId(contract.getSupplierId());
            orderEntity.setSupplierCode(contract.getSupplierCode());
            orderEntity.setSupplierName(contract.getSupplierName());

            //供应商联系人、供应商联系人电话
            orderEntity.setSupplierLinkName(contract.getSupplierManagerName());
            orderEntity.setSupplierLinkPhone(contract.getSupplierManagerPhone());

            //项目信息
            orderEntity.setProjectId(contract.getProjectId());
            orderEntity.setProjectCode(contract.getProjectCode());
            orderEntity.setProjectName(contract.getProjectName());
            orderEntity.setOrgId(contract.getOrgId());
            orderEntity.setOrgCode(contract.getOrgCode());
            orderEntity.setOrgName(contract.getOrgName());
            orderEntity.setParentOrgId(contract.getParentOrgId());
            orderEntity.setParentOrgCode(contract.getParentOrgCode());
            orderEntity.setParentOrgName(contract.getParentOrgName());

            //停启用、发货日期、接货人、接货人联系方式
            orderEntity.setEnableStatus(contract.getEnableStatus());
            orderEntity.setDeliveryDate(contract.getDeliveryDate());
            orderEntity.setAcceptorId(contract.getMasterPorterId());
            orderEntity.setAcceptorCode(contract.getMasterPorterCode());
            orderEntity.setAcceptorName(contract.getMasterPorterName());
            orderEntity.setAcceptorLinkPhone(contract.getLinkPhone());

            //发货地址、要求
            orderEntity.setDeliveryAddress(contract.getDeliveryAddress());
            orderEntity.setDeliveryRemark(contract.getDeliveryRemark());
            orderEntity.setRemark(contract.getRemark());

            //采购人信息
            orderEntity.setPurchaseUserId(contract.getEmployeeId());
            orderEntity.setPurchaseUserCode(contract.getEmployeeCode());
            orderEntity.setPurchaseUserName(contract.getEmployeeName());
            orderEntity.setPurchaseUserLinkPhone(contract.getEmployeePhone());

            //订单金额
            orderEntity.setOrderMny(contract.getMny());
            orderEntity.setOrderTaxMny(contract.getTaxMny());
            //已发货金额
            orderEntity.setDeliveredMny(BigDecimal.ZERO);
            orderEntity.setDeliveredTaxMny(BigDecimal.ZERO);
            //验收金额
            orderEntity.setHandAcceptMny(BigDecimal.ZERO);
            orderEntity.setHandAcceptTaxMny(BigDecimal.ZERO);
            orderEntity.setTotalAcceptMny(BigDecimal.ZERO);
            orderEntity.setTotalAcceptTaxMny(BigDecimal.ZERO);

            orderEntity.setDeliveryStatus(OrderDeliveryStatusEnums.待发货.getCode());
            orderEntity.setSignatureStatus(ZDSMaterialCommonEnums.签章确认状态_未签字.getCode());
            OrderDetailEntity orderDetail = null;
            for(ContractDetailEntity contractDetail : contract.getDetailList()) {
                orderDetail = new OrderDetailEntity();
                //来源合同信息
                orderDetail.setContractDetailId(contractDetail.getId());
                orderDetail.setSourceContractDetailId(contractDetail.getSourceDetailId());
                orderDetail.setSourceContractId(contract.getSourceContractId());

                //合同量、订单量
                orderDetail.setContractNum(contractDetail.getPurNum());
                orderDetail.setDetailNum(contractDetail.getPurNum());
                orderDetail.setHandAcceptNum(BigDecimal.ZERO); //手动验收量
                orderDetail.setDeliveredNum(BigDecimal.ZERO); //已发货量
                orderDetail.setAcceptedNum(BigDecimal.ZERO); //发货单验收量

                //物资信息
                orderDetail.setMaterialSerialNo(contractDetail.getMaterialSerialNo());
                orderDetail.setMaterialTypeId(contractDetail.getMaterialTypeId());
                orderDetail.setMaterialTypeName(contractDetail.getMaterialTypeName());
                orderDetail.setMaterialTypeCode(contractDetail.getMaterialTypeCode());
                orderDetail.setMaterialId(contractDetail.getMaterialId());
                orderDetail.setMaterialCode(contractDetail.getMaterialCode());
                orderDetail.setMaterialName(contractDetail.getMaterialName());
                orderDetail.setDetailPrice(contractDetail.getDetailPrice());
                orderDetail.setDetailTaxPrice(contractDetail.getDetailTaxPrice());
                orderDetail.setDetailMny(contractDetail.getDetailMny());
                orderDetail.setDetailTaxMny(contractDetail.getDetailTaxMny());
                orderDetail.setDetailTaxRate(contractDetail.getDetailTaxRate());
                orderDetail.setDetailTax(contractDetail.getDetailTax());
                orderDetail.setDetailUnitId(contractDetail.getDetailUnitId());
                orderDetail.setDetailUnitName(contractDetail.getDetailUnitName());
                orderDetail.setMemo(contractDetail.getDetailRemark());
                orderDetail.setProductCode(contractDetail.getProductCode());
                orderDetail.setPropertyValue(contractDetail.getPropertyValue());
                orderDetail.setBrandId(contractDetail.getBrandId());
                orderDetail.setBrandName(contractDetail.getBrandName());
                orderDetail.setId(IdWorker.getId());
                orderDetail.setOrderId(orderEntity.getId());
                orderDetail.setRowState("add");

                orderEntity.getDetailList().add(orderDetail);
            }

            BillCodeParam billCodeParam = BillCodeParam.build(BILL_CODE, InvocationInfoProxy.getTenantid(), BeanMapper.map(orderEntity, OrderVO.class));
            CommonResponse<String> billCode = billCodeApi.generateBillCode(billCodeParam);
            if(billCode.isSuccess()) {
                orderEntity.setBillCode(billCode.getData());
            }else{
                throw new BusinessException("网络异常， 编码生成失败， 请稍后再试");
            }

        } else {
            //更新订单
            orderEntity = selectByContractId(contract.getId());
            if(null == orderEntity) {
                logger.error("根据合同Id-[{}]找不到匹配的订单信息！", contract.getId());
                throw new BusinessException("不存在合同对应的订单信息");
            }
        }

        //推送供方
        String msg = pushToSupBusiness(BeanMapper.map(orderEntity, OrderVO.class));
        orderEntity.setBillPushFlag(StringUtils.isNotBlank(msg) ? "sucPushed" : "unSucPush");

        super.saveOrUpdate(orderEntity, false);
    }

    /**
     *
     * 订单量达总控量X%预警，发送、工程师、采购、项目经理
     * 消息模板：材料分类（末级）+材料编码+材料名称 总控计划量X 订单量y  已达到z% 请知悉
     * @param orderEntity
     */
    public void orderCheck(OrderEntity orderEntity) {
        CommonResponse<ParamRegisterSetVO> response = paramConfigApi.getByCode(ORDER_CHECK_CODE);
        if (!response.isSuccess() || response.getData() == null) {
            logger.error("获取订单量达总控量百分比预警参数请求失败，{}，跳过预警任务执行。。。", JSONObject.toJSONString(response, SerializerFeature.PrettyFormat));
            return;
        }

        //控制参数为1时：全部章可选，为0时，仅部分章可选
        BigDecimal percent = new BigDecimal(response.getData().getValueData());
        logger.info("当前订单量达总控量{}%时发送消息", response.getData().getValueData());
        //查询当前订单明细占比总控量百分比
        List<Map<String, Object>> detailPercent = mapper.getOrderDetailPercent(orderEntity.getProjectId());
        if(CollectionUtils.isNotEmpty(detailPercent)) {
            detailPercent = detailPercent.stream().filter(item -> BigDecimal.ZERO.compareTo(new BigDecimal(item.get("controlNum").toString())) < 1
                    && percent.compareTo(new BigDecimal(item.get("percent").toString())) >=0).collect(Collectors.toList());
            StringBuffer sp = new StringBuffer();
            if(CollectionUtils.isNotEmpty(detailPercent)) {
                detailPercent.stream().forEach(item -> {
                    sp.append("材料分类【").append(item.get("material_type_name"))
                            .append("】，材料编码【").append(item.get("material_code")).append("】，材料名称【")
                            .append(item.get("material_name")).append("】，总控计划量【").append(item.get("controlNum"))
                            .append("】，订单量【").append(item.get("orderNum")).append("】，已达").append(new BigDecimal(item.get("orderNum").toString())
                            .setScale(2, BigDecimal.ROUND_HALF_UP)).append("%；");
                });


                //TODO 查询项目下：工程师、采购、项目经理
                CommonResponse<ProjectPoolSetVO> prjResp = projectPoolApi.getById(orderEntity.getProjectId());
                if(!prjResp.isSuccess() || null == prjResp.getData()) {
                    logger.error("订单超量预警信息发送，获取项目id-{}信息失败, {}",
                            orderEntity.getProjectId(), JSONObject.toJSONString(prjResp, SerializerFeature.PrettyFormat));
                }

                List<String> recUserIds= new ArrayList<>();
                ProjectPoolSetVO project = prjResp.getData();
                if(null != project.getProjectManagementId()) {
                    recUserIds.add(project.getProjectManagementId().toString());
                }

                List<String> postNames = new ArrayList<>();
                postNames.add("采购工程师");
                CommonResponse<List<EmployeeVO>> empResp = employeeApi.queryByPostNamesAndOrgIds(postNames, Arrays.asList(new Long[]{orderEntity.getOrgId()}));
                if(!empResp.isSuccess()) {
                    logger.error("订单超量预警信息发送，根据岗位names-[{}]获取组织id-{}获取人员信息失败, {}",
                            StringUtils.join(postNames, ","), orderEntity.getOrgId(),
                            JSONObject.toJSONString(prjResp, SerializerFeature.PrettyFormat));
                }
                List<EmployeeVO> empList = empResp.getData();
                if(CollectionUtils.isEmpty(empList)) {
                    logger.error("订单超量预警信息发送，根据岗位names-[{}]获取组织id-{}获取人员信息为空",
                            StringUtils.join(postNames, ","), orderEntity.getOrgId());
                } else {
                    empList.stream().forEach(emp -> {
                        recUserIds.add(emp.getUserId().toString());
                    });
                }

                if(CollectionUtils.isEmpty(recUserIds)) {
                    logger.error("订单超量预警信息发送失败，项目下采购工程师、项目经理对应人员列表为空");
                    return;
                }

                //调拨申请生效 向项目经理、物资发布人
                PushMsgParameter parameter = new PushMsgParameter();

                parameter.setContent(sp.substring(0, sp.length()-1) + " 请知悉！");
                parameter.setSubject("订单超量预警");
                parameter.setReceivers(recUserIds.toArray(new String[recUserIds.size()]));
                sendMsg(parameter, orderEntity.getId(), "订单量达总控量X%预警，发送、工程师、采购、项目经理发送消息");
            }
        }
    }

    public void sendMsg(PushMsgParameter parameter, Long allotOrderId, String oprMsg) {
        parameter.setSaveFlag(true);
        parameter.setTenantId(InvocationInfoProxy.getTenantid().toString());
        parameter.setMsgType("notice");
        parameter.setChannel(new String[]{PushMsgParameter.CHANNEL_TYPE_SYS, PushMsgParameter.CHANNEL_TYPE_EMAIL});

        CommonResponse<String> sendResp = pushMessageApi.pushMessage(parameter);
        if (!sendResp.isSuccess()) {
            logger.error("订单id-{}超量预警发送消息失败，{}", allotOrderId, oprMsg, JSONObject.toJSONString(sendResp, SerializerFeature.PrettyFormat,
                    SerializerFeature.WriteMapNullValue));
        } else {
            logger.info("订单id-{}超量预警发送消息成功！", allotOrderId, oprMsg);
        }
    }

    @Override
    public String pushToSupBusiness(OrderVO orderVO) {
        String msg = null;
        Jedis jedis = null;
        boolean locked = false;
        String key = OPERATE + "::" + orderVO.getId().toString();

        //验证是否需要推送供方：订单存在需要协同验收的明细
        if(CollectionUtils.isEmpty(orderVO.getDetailList())) {
            msg = "推送订单失败，订单字表为空！";
        }

        List<Long> categoryIds = orderVO.getDetailList().stream().map(OrderDetailVO::getMaterialTypeId).collect(Collectors.toList());
        List<MaterialCategoryVO> categoryEntities =categoryService.queryCategoryListByIds(categoryIds);
        if(categoryEntities.stream().filter(item -> ZDSMaterialCommonEnums.协同验收_是.getCode().equals(item.getCoordinationFlag())).count() == 0) {
            logger.info("订单【id-{}】不存在需要协同验收单的明细，不推送供方！", orderVO.getId().toString());
            return msg;
        }

        //设置单据当前系统信息
        CommonResponse<String> ejcCloudSystemCode = proSupplierApi.getEjcCloudSystemCode();
        if (!ejcCloudSystemCode.isSuccess()) {
            logger.error("推送订单-{}失败，获取当前系统编码失败,{}", orderVO.getId(), ejcCloudSystemCode.getMsg());
            return "推送订单失败，获取当前系统编码失败";
        }

        //查询当前订单对应供应商是否已协同
        CommonResponse<MatSupplierVO> supplierResp = matSupplierApi.getOneById(orderVO.getSupplierId());
        if(!supplierResp.isSuccess()) {
            logger.error("推送订单-{}失败，根据id-{}获取供应商信息失败,{}", orderVO.getId(), orderVO.getSupplierId(), JSONObject.toJSONString(supplierResp));
            return "推送订单失败，获取供方信息失败";
        }
        MatSupplierVO supplier = supplierResp.getData();
        if(PlanConstant.STRING_NO.equals(supplier.getCoordination())) {
            logger.error("推送订单-{}失败，供应商id-{}未开通协同账号", orderVO.getId(), orderVO.getSupplierId());
            return "推送订单失败，订单对应供应商联系人未开通协同账号！";
        }

        //设置当前系统ID
        orderVO.setSourceSystemId(ejcCloudSystemCode.getData());

        Map<String, String> paramMap = new HashMap<>();
        paramMap.put("transData", JSONObject.toJSONString(orderVO));

        Map<String, Map<String, InputStream>> files = new HashMap<>();

        try {
            jedis = jedisPool.getResource();
            //对单据进行加锁
            locked = RedisTool.tryLock(jedis, key, OPERATE, 600);

            //推送单据到指定的供方
            CommonResponse<String> syncReqResp = systemDataPushService.exchangeDataAndFilesWithEachLinkSystem(PUSH_BILL_SERVER_URL,
                    paramMap,
                    orderVO.getSupplierId().toString(),
                    files);
            if (syncReqResp.isSuccess()) {
                CommonResponse<String> billPushResp = JSONObject.parseObject(syncReqResp.getData(), CommonResponse.class);
                if (billPushResp.isSuccess()) {
                    return msg;
                } else {
                    logger.error("供方id-{}处理推送订单id-{}失败, {}", orderVO.getSupplierId(), orderVO.getId(), billPushResp.getMsg());
                    msg = "推送订单失败，供方处理推送订单失败";
                }
            } else {
                logger.error("发送请求推送结算单据id-{}给供方id-{}失败, {}", orderVO.getId(), orderVO.getSupplierId(), syncReqResp.getMsg());
                msg = "发送请求推送订单失败";
            }
        } catch (Exception e) {
            logger.error("订单id-{}推送供方失败，", orderVO.getId(), e);
            msg = "操作失败，订单推送供方失败！";
        } finally {
            releaseLock(jedis, locked, key, OPERATE);
        }

        return msg;
    }

    @Override
    public void supOrderSync(OrderVO orderVO) {
        Long orderId = Long.valueOf(orderVO.getSourceId());

        Jedis jedis = null;
        boolean locked = false;
        String key = OPERATE + "::" + orderId.toString();

        try {
            jedis = jedisPool.getResource();
            //对单据进行加锁
            locked = RedisTool.tryLock(jedis, key, OPERATE, 600);

            OrderEntity order = super.selectById(orderId);

            if(!order.getSignatureStatus().equals(orderVO.getSignatureStatus())) {
                //更新确认信息
                order.setSignatureStatus(orderVO.getSignatureStatus());
                order.setSignTime(orderVO.getSignTime());
                order.setSignUserId(orderVO.getSignUserId());
                order.setSignUserName(orderVO.getSignUserName());
                order.setSignUserPhone(orderVO.getSignUserPhone());
            } else {
                //更新发货信息
                order.setDeliveryStatus(orderVO.getDeliveryStatus());
                order.setDeliveredMny(orderVO.getDeliveredMny());
                order.setDeliveredTaxMny(orderVO.getDeliveredTaxMny());

                Map<String, OrderDetailVO> orderDetailVOMap = orderVO.getDetailList().stream().collect(Collectors.toMap(item -> item.getSourceId(), item -> item));
                //子表已发货量更新
                order.getDetailList().stream().filter(item -> orderDetailVOMap.containsKey(item.getId().toString())).forEach(detail -> {
                    detail.setDeliveredNum(orderDetailVOMap.get(detail.getId().toString()).getDeliveredNum());
                });
            }
            order.setFreezeStatus(1);
            super.saveOrUpdate(order, false);
        } catch (Exception e) {
            logger.error("订单id-{}确认状态同步异常，", orderId, e);
        } finally {
            releaseLock(jedis, locked, key, OPERATE);
        }

    }

    @Override
    public void returnHandAcceptNum(AcceptEntity entity, Boolean status) {
        detailService.returnHandAcceptNum(entity,status);
        totalAcceptMny(entity.getOrderBillId());
    }
    public void totalAcceptMny(Long orderBillId){
        OrderEntity order = super.selectById(orderBillId);
        List<OrderDetailEntity> detailList = order.getDetailList();
        BigDecimal totalMny = BigDecimal.ZERO;
        BigDecimal totalTaxMny = BigDecimal.ZERO ;
        BigDecimal totalHandMny = BigDecimal.ZERO;
        BigDecimal totalHandTaxMny = BigDecimal.ZERO ;
        for (OrderDetailEntity detail :detailList){
            BigDecimal handAcceptNum = detail.getHandAcceptNum();
            BigDecimal acceptedNum = detail.getAcceptedNum();
            BigDecimal detailTaxPrice = detail.getDetailTaxPrice();
            BigDecimal detailPrice = detail.getDetailPrice();
            BigDecimal mny = ComputeUtil.safeMultiply(acceptedNum, detailPrice); //发货单验收金额
            BigDecimal taxMny = ComputeUtil.safeMultiply(acceptedNum, detailTaxPrice);//发货单验收金额含税

            BigDecimal mny1 = ComputeUtil.safeMultiply(handAcceptNum, detailPrice); //订单验收金额
            BigDecimal taxMny1 = ComputeUtil.safeMultiply(handAcceptNum, detailTaxPrice); //订单验收金额

            totalHandMny = ComputeUtil.safeAdd(totalHandMny, mny1);
            totalHandTaxMny = ComputeUtil.safeAdd(totalHandTaxMny, taxMny1);

            totalMny = ComputeUtil.safeAdd(totalMny, ComputeUtil.safeAdd(mny, mny1)); //总的验收金额
            totalTaxMny = ComputeUtil.safeAdd(totalTaxMny, ComputeUtil.safeAdd(taxMny, taxMny1)); //总的验收金额
        }
        order.setTotalAcceptMny(totalMny);
        order.setTotalAcceptTaxMny(totalTaxMny);
        order.setHandAcceptMny(totalHandMny);
        order.setHandAcceptTaxMny(totalHandTaxMny);

        super.saveOrUpdate(order);
    }

    @Override
    public void returnAcceptedNum(DeliveryEntity deliveryEntity, boolean addFlag) {
        detailService.returnAcceptedNum(deliveryEntity.getDetailList(),addFlag);
        totalAcceptMny(deliveryEntity.getOrderId());
    }

    @Override
    public List<OrderEntity> getBySourceIds(List<String> sourceIds) {
        QueryWrapper<OrderEntity> query = new QueryWrapper<>();
        query.eq("dr", BaseVO.DR_UNDELETE);
        query.in("source_id", sourceIds);
        return super.list(query);
    }

    public void releaseLock(Jedis jedis, boolean locked, String key, String OPERATE) {
        try {
            if(locked) {
                RedisTool.releaseLock(jedis, key, OPERATE);
            }
        } finally {
            if(null != jedis) {
                jedis.close();
            }
        }
    }

    private OrderEntity selectByContractId(Long contractId) {
        QueryWrapper<OrderEntity> query = new QueryWrapper<>();
        query.eq("contract_id", contractId);
        return super.getOne(query);
    }
}
