package com.ejianc.business.plan.utils;

import com.ejianc.framework.core.util.ComputeUtil;
import org.apache.commons.lang3.StringUtils;

import java.math.BigDecimal;
import java.math.RoundingMode;

public class NumberUtil {
    private NumberUtil() {
    }

    /**
     * 中文数字
     */
    private static final String[] CN_NUM = {"零", "一", "二", "三", "四", "五", "六", "七", "八", "九"};

    /**
     * 中文数字单位
     */
    private static final String[] CN_UNIT = {"", "十", "百", "千", "万", "十", "百", "千", "亿", "十", "百", "千"};

    /**
     * 特殊字符：负
     */
    private static final String CN_NEGATIVE = "负";

    /**
     * 特殊字符：点
     */
    private static final String CN_POINT = "点";


    /**
     * int 转 中文数字
     * 支持到int最大值
     */
    public static String int2chineseNum(int intNum) {
        if (intNum == 0)
            return CN_NUM[0];
        StringBuffer sb = new StringBuffer();
        boolean isNegative = false;
        if (intNum < 0) {
            isNegative = true;
            intNum *= -1;
        }
        int count = 0;
        while (intNum > 0) {
            sb.insert(0, CN_NUM[intNum % 10] + CN_UNIT[count]);
            intNum = intNum / 10;
            count++;
        }

        if (isNegative)
            sb.insert(0, CN_NEGATIVE);

        String res = sb.toString().replaceAll("零[千百十]", "零").replaceAll("零+万", "万")
                .replaceAll("零+亿", "亿").replaceAll("亿万", "亿零")
                .replaceAll("零+", "零").replaceAll("零$", "");
        if (StringUtils.startsWith(res, "一十")) {
            res = res.substring(1);
        }
        if (StringUtils.startsWith(res, CN_NEGATIVE + "一十")) {
            res = CN_NEGATIVE + res.substring(2);
        }
        return res;
    }

    /**
     * bigDecimal 转 中文数字
     * 整数部分只支持到int的最大值
     */
    public static String bigDecimal2chineseNum(BigDecimal bigDecimalNum) {
        if (bigDecimalNum == null)
            return CN_NUM[0];

        StringBuffer sb = new StringBuffer();

        //将小数点后面的零给去除
        String numStr = bigDecimalNum.abs().stripTrailingZeros().toPlainString();

        String[] split = numStr.split("\\.");
        String integerStr = int2chineseNum(Integer.parseInt(split[0]));

        sb.append(integerStr);

        //如果传入的数有小数，则进行切割，将整数与小数部分分离
        if (split.length == 2) {
            //有小数部分
            sb.append(CN_POINT);
            String decimalStr = split[1];
            char[] chars = decimalStr.toCharArray();
            for (int i = 0; i < chars.length; i++) {
                int index = Integer.parseInt(String.valueOf(chars[i]));
                sb.append(CN_NUM[index]);
            }
        }

        //判断传入数字为正数还是负数
        int signum = bigDecimalNum.signum();
        if (signum == -1) {
            sb.insert(0, CN_NEGATIVE);
        }

        return sb.toString();
    }

    /**
     * 对象类型转为BigDecimal
     *
     * @param x(可为null)
     * @return BigDecimal (不可为null)
     */
    public static BigDecimal toBigDecimal(Object x) {
        if (x == null) {
            return BigDecimal.ZERO;
        }
        BigDecimal result = null;
        try {
            result = new BigDecimal(x.toString());
        } catch (Exception e) {
            e.printStackTrace();
        }
        return result;
    }

    public static Integer safeSub(Integer num1, Integer num2) {
        if (num1 == null || num2 == null) {
            return 0;
        }
        if (num2.equals(Integer.valueOf(0))) {
            return 0;
        }
        BigDecimal decimal1 = new BigDecimal(num1);
        BigDecimal decimal2 = new BigDecimal(num2);
        return decimal1.subtract(decimal2).intValue();
    }

    public static Integer safeSub(Integer num1, Integer... subNums) {
        for (Integer temp : subNums) {
            num1 = safeSub(num1, temp);
        }
        return num1;
    }

    /**
     * 计算 num1 / num2 的百分比
     * getPercentage(10, 100, 2) = 10.00
     *
     * @param num1
     * @param num2
     * @param point 保留几位小数
     * @return BigDecimal
     */
    public static BigDecimal bigDecimalPercent(Integer num1, Integer num2, int point) {
        if (num1 == null || num2 == null) {
            return BigDecimal.ZERO;
        }
        if (num2.equals(Integer.valueOf(0))) {
            return BigDecimal.ZERO;
        }
        BigDecimal bigDecimalNum1 = new BigDecimal(num1);
        BigDecimal bigDecimalNum2 = new BigDecimal(num2);
        return bigDecimalPercent(bigDecimalNum1, bigDecimalNum2, point);
    }

    /**
     * 计算 num1 / num2 的百分比
     * getPercentage(10, 100, 2) = 10.00
     *
     * @param num1  10
     * @param num2  100
     * @param point 保留几位小数 2
     * @return BigDecimal 10.00
     */
    public static BigDecimal bigDecimalPercent(BigDecimal num1, BigDecimal num2, int point) {
        if (num1 == null || num2 == null) {
            return BigDecimal.ZERO;
        }
        if (ComputeUtil.equals(BigDecimal.ZERO, num2)) {
            return BigDecimal.ZERO;
        }
        BigDecimal percent = num1.divide(num2, point + 2, RoundingMode.HALF_UP);
        BigDecimal percent100 = percent.multiply(new BigDecimal(100)).setScale(point);
        // 百分比不能大于100
        if(ComputeUtil.isGreaterThan(percent100, new BigDecimal("100"))){
            percent100 = new BigDecimal("100");
        }
        // 百分比不能小于0
        if(ComputeUtil.isLessThan(percent100, new BigDecimal("0"))){
            percent100 = new BigDecimal("0");
        }
        return percent100;
    }
}
