package com.ejianc.business.bidprice.material.controller;

import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.JSONArray;
import com.alibaba.fastjson.JSONObject;
import com.baomidou.mybatisplus.core.metadata.IPage;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.ejianc.business.bidprice.material.bean.MaterialInquiryEntity;
import com.ejianc.business.bidprice.material.bean.MaterialQuoteEntity;
import com.ejianc.business.bidprice.material.service.IMaterialInquiryService;
import com.ejianc.business.bidprice.material.service.IMaterialQuoteService;
import com.ejianc.business.bidprice.material.vo.*;
import com.ejianc.business.utils.ComputeUtil;
import com.ejianc.foundation.file.api.IAttachmentApi;
import com.ejianc.foundation.file.vo.AttachmentVO;
import com.ejianc.foundation.orgcenter.api.IOrgApi;
import com.ejianc.foundation.support.api.IBillCodeApi;
import com.ejianc.foundation.support.api.IBillTypeApi;
import com.ejianc.foundation.support.vo.BillCodeParam;
import com.ejianc.framework.auth.session.SessionManager;
import com.ejianc.framework.core.context.InvocationInfoProxy;
import com.ejianc.framework.core.exception.BusinessException;
import com.ejianc.framework.core.kit.collection.ListUtil;
import com.ejianc.framework.core.kit.mapper.BeanMapper;
import com.ejianc.framework.core.response.CommonResponse;
import com.ejianc.framework.core.response.QueryParam;
import com.ejianc.framework.core.util.ExcelExport;
import com.ejianc.framework.core.util.ExcelReader;
import com.ejianc.framework.core.util.FileUtils;
import com.ejianc.framework.core.util.ImportTemplate;
import io.swagger.annotations.Api;
import io.swagger.annotations.ApiOperation;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.web.bind.annotation.*;
import org.springframework.web.multipart.MultipartFile;
import org.springframework.web.multipart.MultipartHttpServletRequest;

import javax.annotation.Resource;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import java.io.Serializable;
import java.math.BigDecimal;
import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.util.*;
import java.util.function.Function;
import java.util.stream.Collectors;

/**
 * 报价记录
 *
 * @author baipengyan
 */
@RestController
@RequestMapping("materialQuote")
@Api(tags = {"材料报价记录"})
public class MaterialQuoteController implements Serializable {
    private static final long serialVersionUID = -6476701269896286249L;
    private static final String RULE_CODE = "ZYDX_MATERIAL_QUOTE";
    private static final String BILL_CODE = "EJCBT202502000017";
    private final Logger logger = LoggerFactory.getLogger(this.getClass());
    @Resource
    private IBillTypeApi billTypeApi;
    @Resource
    private IBillCodeApi billCodeApi;
    @Resource
    private IOrgApi iOrgApi;
    @Resource
    private IMaterialQuoteService service;
    @Autowired
    private SessionManager sessionManager;
    @Resource
    private IAttachmentApi attachmentApi;
    @Resource
    private IMaterialInquiryService materialInquiryService;


    /**
     * 保存或更新
     *
     * @param materialQuoteVO 报价记录
     * @return {@link CommonResponse}<{@link MaterialQuoteVO}>
     */
    @ApiOperation(value = "保存或更新")
    @PostMapping(value = "/saveOrUpdate")
    public CommonResponse<MaterialQuoteVO> saveOrUpdate(@RequestBody MaterialQuoteVO materialQuoteVO) {
        MaterialQuoteEntity entity = BeanMapper.map(materialQuoteVO, MaterialQuoteEntity.class);
        // 校验当前时间是否超过询价单报价截止时间
        service.checkQuoteEndDate(entity.getInquiryId());
        if (entity.getId() == null || entity.getId() == 0) {
            BillCodeParam billCodeParam = BillCodeParam.build(RULE_CODE, InvocationInfoProxy.getTenantid(), materialQuoteVO);
            CommonResponse<String> billCode = billCodeApi.generateBillCode(billCodeParam);
            if (billCode.isSuccess()) {
                entity.setBillCode(billCode.getData());//此处需要根据实际修改 删除本行或者上一行
            } else {
                throw new BusinessException("网络异常， 编码生成失败， 请稍后再试");
            }
        }
        service.saveOrUpdate(entity, false);
        MaterialQuoteVO vo = BeanMapper.map(entity, MaterialQuoteVO.class);
        return CommonResponse.success("保存或修改单据成功！", vo);
    }


    /**
     * 查询详情
     *
     * @param id id
     * @return {@link CommonResponse}<{@link MaterialQuoteVO}>
     */
    @ApiOperation(value = "查询详情")
    @GetMapping(value = "/queryDetail")
    public CommonResponse<MaterialQuoteVO> queryDetail(Long id) {
        MaterialQuoteEntity entity = service.selectById(id);
        if (entity == null) {
            return CommonResponse.error("未查询到报价记录数据！");
        }
        MaterialQuoteVO vo = BeanMapper.map(entity, MaterialQuoteVO.class);
        return CommonResponse.success("查询详情数据成功！", vo);
    }


    /**
     * 批量删除单据
     *
     * @param vos vos
     * @return {@link CommonResponse}<{@link String}>
     */
    @ApiOperation(value = "批量删除单据")
    @PostMapping(value = "/delete")
    public CommonResponse<String> delete(@RequestBody List<MaterialQuoteVO> vos) {
        if (ListUtil.isNotEmpty(vos)) {
            for (MaterialQuoteVO vo : vos) {
                CommonResponse<String> resp = billTypeApi.checkQuote(BILL_CODE, vo.getId());
                if (!resp.isSuccess()) {
                    return CommonResponse.error("删除失败！" + resp.getMsg());
                }
            }
        }
        service.removeByIds(vos.stream().map(MaterialQuoteVO::getId).collect(Collectors.toList()), true);
        return CommonResponse.success("删除成功！");
    }


    /**
     * 查询列表
     *
     * @param param 参数
     * @return {@link CommonResponse}<{@link IPage}<{@link MaterialQuoteVO}>>
     */
    @ApiOperation(value = "查询列表")
    @PostMapping(value = "/queryList")
    public CommonResponse<IPage<MaterialQuoteVO>> queryList(@RequestBody QueryParam param) {
        LinkedHashMap<String, String> orderMap = new LinkedHashMap<>();
        orderMap.put("create_time", "desc");
        param.getOrderMap().putAll(orderMap);
        IPage<MaterialQuoteEntity> page = service.queryPage(param, false);
        IPage<MaterialQuoteVO> pageData = new Page<>(page.getCurrent(), page.getSize(), page.getTotal());
        List<MaterialQuoteVO> materialQuoteVOS = BeanMapper.mapList(page.getRecords(), MaterialQuoteVO.class);
        if (CollectionUtils.isNotEmpty(materialQuoteVOS)) {
            List<Long> quoteIds = materialQuoteVOS.stream().map(MaterialQuoteVO::getId).collect(Collectors.toList());
            JSONObject fileParam = new JSONObject();
            fileParam.put("sourceIds", quoteIds);
            JSONArray sourceTypeArr = new JSONArray();
            sourceTypeArr.add("quote");
            fileParam.put("sourceTypeList", JSON.toJSONString(sourceTypeArr));
            CommonResponse<List<AttachmentVO>> listCommonResponse = attachmentApi.queryAllBySourceIdList(fileParam);
            List<AttachmentVO> fileData = listCommonResponse.getData();
            Map<Long, AttachmentVO> fileMap = new HashMap<>();
            if (listCommonResponse.isSuccess() && CollectionUtils.isNotEmpty(fileData)) {
                fileMap = fileData.stream().collect(Collectors.toMap(AttachmentVO::getSourceId, Function.identity(), (e1, e2) -> e1));
            }

            for (MaterialQuoteVO materialQuoteVO : materialQuoteVOS) {
                if (fileMap.containsKey(materialQuoteVO.getId())) {
                    AttachmentVO attachmentVO = fileMap.get(materialQuoteVO.getId());
                    materialQuoteVO.setQuoteFileId(attachmentVO.getId());
                    materialQuoteVO.setQuoteFileName(attachmentVO.getFileName());
                }
            }
        }
        pageData.setRecords(materialQuoteVOS);
        return CommonResponse.success("查询列表数据成功！", pageData);
    }

    /**
     * 指定供应商登录、所有供应商登录校验查看权限
     *
     * @param inquiryId   询价单id
     * @param phoneNumber 电话号码
     * @return {@link CommonResponse}<{@link Void}>
     */
    @GetMapping(value = "/checkViewAuthority")
    public CommonResponse<Void> checkViewAuthority(@RequestParam(value = "inquiryId") Long inquiryId,
                                                   @RequestParam(value = "phoneNumber") String phoneNumber) {
        service.checkViewAuthority(inquiryId, phoneNumber);
        return CommonResponse.success();
    }

    /**
     * 指定供应商报价校验报价权限
     *
     * @param inquiryId   询价单id
     * @param phoneNumber 电话号码
     * @return {@link CommonResponse}<{@link Void}>
     */
    @GetMapping(value = "/checkQuotationAuthority")
    public CommonResponse<Void> checkQuotationAuthority(@RequestParam(value = "inquiryId") Long inquiryId,
                                                        @RequestParam(value = "phoneNumber") String phoneNumber) {
        service.checkQuotationAuthority(inquiryId, phoneNumber);
        return CommonResponse.success();
    }


    /**
     * 获取验证码
     *
     * @param inquiryId   询价单id
     * @param phoneNumber 电话号码
     * @return {@link CommonResponse}<{@link Void}>
     * @throws Exception 异常
     */
    @GetMapping(value = "/fetchVerifyCode")
    public CommonResponse<Void> fetchVerifyCode(@RequestParam(value = "inquiryId") Long inquiryId,
                                                @RequestParam(value = "phoneNumber") String phoneNumber) throws Exception {
        service.fetchVerifyCode(inquiryId, phoneNumber);
        return CommonResponse.success("验证码发送成功！");
    }


    /**
     * 登录
     *
     * @param phoneNumber 电话号码
     * @param validate    验证码
     * @return {@link CommonResponse}<{@link String}>
     */
    @ApiOperation(value = "登录")
    @GetMapping(value = "/login")
    public CommonResponse<String> login(@RequestParam(value = "phoneNumber") String phoneNumber,
                                        @RequestParam(value = "validate") String validate) {
        service.login(phoneNumber, validate);
        return CommonResponse.success("登录成功！", phoneNumber);
    }

    /**
     * 收回
     *
     * @param id 报价单id
     * @return {@link CommonResponse}<{@link String}>
     */
    @ApiOperation(value = "收回")
    @GetMapping(value = "/withdraw")
    public CommonResponse<String> withdraw(@RequestParam(value = "id") Long id) {
        service.withdraw(id);
        return CommonResponse.success("收回成功！");
    }

    /**
     * 修改单据状态
     *
     * @param id        id
     * @param billState 单据状态
     * @return {@link CommonResponse}<{@link String}>
     */
    @ApiOperation(value = "修改单据状态")
    @GetMapping(value = "/changeStatus")
    public CommonResponse<String> changeStatus(@RequestParam(value = "id") Long id,
                                               @RequestParam(value = "billState") Integer billState) {
        service.changeStatus(id, billState);
        return CommonResponse.success("单据状态修改成功！");
    }

    /**
     * 最近一次的报价记录
     *
     * @param phoneNumber 报价手机号码（当前登录的手机号码）
     * @return {@link CommonResponse}<{@link MaterialQuoteVO}>
     */
    @ApiOperation(value = "最近一次的报价记录")
    @GetMapping(value = "/recentQuote")
    public CommonResponse<MaterialQuoteVO> recentQuote(@RequestParam(value = "phoneNumber") String phoneNumber) {
        return CommonResponse.success("单据状态修改成功！", service.recentQuote(phoneNumber));
    }

    /**
     * 供应商报价清单导出
     *
     * @param response
     */
    @PostMapping("excelExportPurchaseList")
    public void excelExportPurchaseList(@RequestBody MaterialQuoteVO materialQuoteVO, HttpServletResponse response) {
        List list = new ArrayList<>();
        Map<String, Object> beans = new HashMap<>();
        Long materialQuoteVOId = materialQuoteVO.getId();
        if (materialQuoteVOId != null) {
            MaterialInquiryEntity entity = materialInquiryService.selectById(materialQuoteVOId);
            List data = entity.getMaterialInquiryPurchaseDetailList();
            if (CollectionUtils.isNotEmpty(data)) {
                data.forEach(materialSub -> {
                    MaterialInquiryPurchaseDetailVO vo = BeanMapper.map(materialSub, MaterialInquiryPurchaseDetailVO.class);
                    list.add(vo);
                });
            }
            beans.put("records", list);
            ExcelExport.getInstance().export("Materialpurchase-List-export.xlsx", beans, response);


        }
    }

    /**
     * 供应商报价清单导出
     *
     * @param response
     */
    @PostMapping("excelExportRentList")
    public void excelExportRentList(@RequestBody MaterialQuoteVO materialQuoteVO, HttpServletResponse response) {
        List list = new ArrayList<>();
        Map<String, Object> beans = new HashMap<>();
        Long materialQuoteVOId = materialQuoteVO.getId();
        if (materialQuoteVOId != null) {
            MaterialInquiryEntity entity = materialInquiryService.selectById(materialQuoteVOId);
            List data = entity.getMaterialInquiryRentDetailList();
            if (CollectionUtils.isNotEmpty(data)) {
                data.forEach(materialSub -> {
                    MaterialInquiryRentDetailVO vo = BeanMapper.map(materialSub, MaterialInquiryRentDetailVO.class);
                    list.add(vo);
                });
            }
            beans.put("records", list);
            ExcelExport.getInstance().export("Materialrent-List-export.xlsx", beans, response);
        }
    }

    /**
     * 导入模板下载
     *
     * @param request
     * @param response
     */
    @RequestMapping(value = "/purchaseDownload")
    @ResponseBody
    public void purchaseDownload(HttpServletRequest request, HttpServletResponse response) {
        ImportTemplate.initialize(response);
        ImportTemplate.templetdownload(request, "Materialpurchase-List-import.xlsx", "清单导入模板");
    }

    /**
     * excel导入
     *
     * @param request
     * @return
     */
    @RequestMapping(value = "/excelPurchaseImport", method = RequestMethod.POST)
    @ResponseBody
    public CommonResponse<JSONObject> excelPurchaseImport(HttpServletRequest request, HttpServletResponse response) {
        MultipartHttpServletRequest multipartRequest = (MultipartHttpServletRequest) request;
        Map<String, MultipartFile> fileMap = multipartRequest.getFileMap();
        boolean isFailed = false;
        MultipartFile mf = null;
        for (Map.Entry<String, MultipartFile> entity : fileMap.entrySet()) {
            mf = entity.getValue();
            String originalFileName = mf.getOriginalFilename();
            String extName = null;
            originalFileName = originalFileName.replaceAll("\\/|\\/|\\||:|\\?|\\*|\"|<|>|\\p{Cntrl}", "_");
            originalFileName.replaceAll("00.", "");
            extName = FileUtils.getFileExt(originalFileName, false);
            if (!"xls".equals(extName) && !"xlsx".equals(extName)) {
                isFailed = true;
                break;
            }
        }

        if (isFailed) {
            return CommonResponse.error("文件格式不合法");
        } else {
            List<List<String>> result = ExcelReader.readExcel(mf);
            List<MaterialQuotePurchaseDetailVO> successList = new ArrayList<>();
            List<MaterialQuotePurchaseDetailVO> errorList = new ArrayList<>();
            if (result != null && result.size() > 0) {
                if (result.get(1).size() != 12) {
                    throw new BusinessException("请按照导入模板导入数据");
                }
                for (int i = 1; i < result.size(); i++) {
                    List<String> datas = result.get(i);
                    MaterialQuotePurchaseDetailVO materialQuotePurchaseDetailVO = new MaterialQuotePurchaseDetailVO();
                    if (StringUtils.isBlank(datas.get(0))) {
                        materialQuotePurchaseDetailVO.setErrorMessage("物料分类为必填项");
                    } else {
                        String materialCategoryName = datas.get(0).trim();
                        materialQuotePurchaseDetailVO.setMaterialCategoryName(materialCategoryName);
                    }
                    if (StringUtils.isBlank(datas.get(1))) {
                        materialQuotePurchaseDetailVO.setErrorMessage("物资名称为必填项");
                    } else {
                        materialQuotePurchaseDetailVO.setMaterialName(datas.get(1).trim());
                    }
                    if (StringUtils.isBlank(datas.get(2))) {
                        materialQuotePurchaseDetailVO.setErrorMessage("规格型号为必填项");
                    } else {
                        materialQuotePurchaseDetailVO.setSpec(datas.get(2).trim());
                    }
                    if (!Objects.isNull(datas.get(3))) {
                        materialQuotePurchaseDetailVO.setUnit(datas.get(3).trim());
                    }
                    if (StringUtils.isNotBlank(datas.get(4))) {
                        SimpleDateFormat format = new SimpleDateFormat("yyyy-MM-dd");
                        try {
                            materialQuotePurchaseDetailVO.setPlanUseDate(format.parse(datas.get(4)));
                        } catch (ParseException e) {
                            e.printStackTrace();
                        }
                    }
                    if (StringUtils.isNotBlank(datas.get(6))) {
                        try {
                            materialQuotePurchaseDetailVO.setPurchaseQuotePriceNoTax(new BigDecimal(datas.get(6)));
                        } catch (Exception e) {
                            materialQuotePurchaseDetailVO.setErrorMessage("报价(无税)必须为数字");
                        }
                    }
                    if (StringUtils.isNotBlank(datas.get(7))) {
                        try {
                            materialQuotePurchaseDetailVO.setPurchaseRate(ComputeUtil.toBigDecimal(datas.get(7)));
                        } catch (Exception e) {
                            materialQuotePurchaseDetailVO.setErrorMessage("税率必须为数字");
                        }
                    }
                    if (StringUtils.isNotBlank(datas.get(8))) {
                        try {
                            materialQuotePurchaseDetailVO.setPurchaseQuotePrice(ComputeUtil.toBigDecimal(datas.get(8)));
                        } catch (Exception e) {
                            materialQuotePurchaseDetailVO.setErrorMessage("报价必须为数字");
                        }
                    }
                    if (StringUtils.isBlank(materialQuotePurchaseDetailVO.getErrorMessage())) {
                        successList.add(materialQuotePurchaseDetailVO);
                    } else {
                        errorList.add(materialQuotePurchaseDetailVO);
                    }
                    if (StringUtils.isNotBlank(datas.get(11))) {
                        materialQuotePurchaseDetailVO.setPurchaseMemo(datas.get(11));
                    }
                }
            }
            JSONObject json = new JSONObject();
            json.put("successList", successList);
            json.put("errorList", errorList);
            return CommonResponse.success(json);
        }
    }

    /**
     * 导入模板下载
     *
     * @param request
     * @param response
     */
    @RequestMapping(value = "/rentDownload")
    @ResponseBody
    public void rentDownload(HttpServletRequest request, HttpServletResponse response) {
        ImportTemplate.initialize(response);
        ImportTemplate.templetdownload(request, "Materialrent-List-import.xlsx", "物料导入模板");
    }

    /**
     * excel导入
     *
     * @param request
     * @return
     */
    @RequestMapping(value = "/excelRentImport", method = RequestMethod.POST)
    @ResponseBody
    public CommonResponse<JSONObject> excelRentImport(HttpServletRequest request, HttpServletResponse response) {
        MultipartHttpServletRequest multipartRequest = (MultipartHttpServletRequest) request;
        Map<String, MultipartFile> fileMap = multipartRequest.getFileMap();
        boolean isFailed = false;
        MultipartFile mf = null;
        for (Map.Entry<String, MultipartFile> entity : fileMap.entrySet()) {
            mf = entity.getValue();
            String originalFileName = mf.getOriginalFilename();
            String extName = null;
            originalFileName = originalFileName.replaceAll("\\/|\\/|\\||:|\\?|\\*|\"|<|>|\\p{Cntrl}", "_");
            originalFileName.replaceAll("00.", "");
            extName = FileUtils.getFileExt(originalFileName, false);
            if (!"xls".equals(extName) && !"xlsx".equals(extName)) {
                isFailed = true;
                break;
            }
        }

        if (isFailed) {
            return CommonResponse.error("文件格式不合法");
        } else {
            List<List<String>> result = ExcelReader.readExcel(mf);
            List<MaterialQuoteRentDetailVO> successList = new ArrayList<>();
            List<MaterialQuoteRentDetailVO> errorList = new ArrayList<>();
            if (result != null && result.size() > 0) {
                if (result.get(1).size() != 14) {
                    throw new BusinessException("请按照导入模板导入数据");
                }
                for (int i = 1; i < result.size(); i++) {
                    List<String> datas = result.get(i);
                    MaterialQuoteRentDetailVO materialQuoteRentDetailVO = new MaterialQuoteRentDetailVO();
                    if (StringUtils.isBlank(datas.get(0))) {
                        materialQuoteRentDetailVO.setErrorMessage("物料分类为必填项");
                    } else {
                        String materialCategoryName = datas.get(0).trim();
                        materialQuoteRentDetailVO.setMaterialCategoryName(materialCategoryName);
                    }
                    if (StringUtils.isBlank(datas.get(1))) {
                        materialQuoteRentDetailVO.setErrorMessage("物资名称为必填项");
                    } else {
                        materialQuoteRentDetailVO.setMaterialName(datas.get(1).trim());
                    }
                    if (StringUtils.isBlank(datas.get(2))) {
                        materialQuoteRentDetailVO.setErrorMessage("规格型号为必填项");
                    } else {
                        materialQuoteRentDetailVO.setSpec(datas.get(2).trim());
                    }
                    if (!Objects.isNull(datas.get(3))) {
                        materialQuoteRentDetailVO.setUnit(datas.get(3).trim());
                    }
                    if (StringUtils.isNotBlank(datas.get(5))) {
                        SimpleDateFormat format = new SimpleDateFormat("yyyy-MM-dd");
                        try {
                            materialQuoteRentDetailVO.setPlanUseDate(format.parse(datas.get(5)));
                        } catch (ParseException e) {
                            e.printStackTrace();
                        }
                    }
                    if (StringUtils.isNotBlank(datas.get(8))) {
                        try {
                            materialQuoteRentDetailVO.setRentQuotePriceNoTax(new BigDecimal(datas.get(8)));
                        } catch (Exception e) {
                            materialQuoteRentDetailVO.setErrorMessage("报价(无税)必须为数字");
                        }
                    }
                    if (StringUtils.isNotBlank(datas.get(9))) {
                        try {
                            materialQuoteRentDetailVO.setRentRate(ComputeUtil.toBigDecimal(datas.get(9)));
                        } catch (Exception e) {
                            materialQuoteRentDetailVO.setErrorMessage("税率必须为数字");
                        }
                    }
                    if (StringUtils.isNotBlank(datas.get(10))) {
                        try {
                            materialQuoteRentDetailVO.setRentQuotePrice(ComputeUtil.toBigDecimal(datas.get(10)));
                        } catch (Exception e) {
                            materialQuoteRentDetailVO.setErrorMessage("报价必须为数字");
                        }
                    }
                    if (StringUtils.isBlank(materialQuoteRentDetailVO.getErrorMessage())) {
                        successList.add(materialQuoteRentDetailVO);
                    } else {
                        errorList.add(materialQuoteRentDetailVO);
                    }
                    if (StringUtils.isNotBlank(datas.get(13))) {
                        materialQuoteRentDetailVO.setRentMemo(datas.get(13));
                    }
                }
            }
            JSONObject json = new JSONObject();
            json.put("successList", successList);
            json.put("errorList", errorList);
            return CommonResponse.success(json);
        }
    }


}
