package com.ejianc.business.sub.service.impl;

import com.baomidou.mybatisplus.core.conditions.query.LambdaQueryWrapper;
import com.baomidou.mybatisplus.core.conditions.update.LambdaUpdateWrapper;
import com.baomidou.mybatisplus.core.metadata.IPage;
import com.baomidou.mybatisplus.core.toolkit.Wrappers;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.ejianc.business.sub.bean.SubContractEntity;
import com.ejianc.business.sub.bean.SubFinishDetailEntity;
import com.ejianc.business.sub.bean.SubFinishEntity;
import com.ejianc.business.sub.bean.SubSettleEntity;
import com.ejianc.business.sub.mapper.SubFinishMapper;
import com.ejianc.business.sub.service.IContractService;
import com.ejianc.business.sub.service.ISubFinishDetailService;
import com.ejianc.business.sub.service.ISubFinishService;
import com.ejianc.business.sub.service.ISubSettleService;
import com.ejianc.business.sub.utils.BigDecimalUtils;
import com.ejianc.business.sub.utils.TreeNodeBUtil;
import com.ejianc.business.sub.vo.*;
import com.ejianc.business.utils.ComputeUtil;
import com.ejianc.foundation.orgcenter.api.IOrgApi;
import com.ejianc.foundation.orgcenter.vo.OrgVO;
import com.ejianc.foundation.support.api.IBillCodeApi;
import com.ejianc.foundation.support.api.IParamConfigApi;
import com.ejianc.foundation.support.vo.BillParamVO;
import com.ejianc.framework.auth.session.SessionManager;
import com.ejianc.framework.auth.session.UserContext;
import com.ejianc.framework.core.context.InvocationInfoProxy;
import com.ejianc.framework.core.exception.BusinessException;
import com.ejianc.framework.core.kit.mapper.BeanMapper;
import com.ejianc.framework.core.response.BillStateEnum;
import com.ejianc.framework.core.response.CommonResponse;
import com.ejianc.framework.core.response.Parameter;
import com.ejianc.framework.core.response.QueryParam;
import com.ejianc.framework.skeleton.template.BaseServiceImpl;
import com.fasterxml.jackson.databind.ObjectMapper;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import java.math.BigDecimal;
import java.util.*;
import java.util.stream.Collectors;

/**
 * <p>
 * 分包完工结算表 服务实现类
 * </p>
 *
 * @author zhangwx
 * @since 2020-06-05
 */
@Service("SubFinishServiceImpl")
public class SubFinishServiceImpl extends BaseServiceImpl<SubFinishMapper, SubFinishEntity> implements ISubFinishService {

    private static final String SUB_FINISH_BILL_CODE = "SUB_FINISH";
    private static final String SUB_FINISH = "SUB_FINISH";
    private static final String SUB_FINISH_COST = "SUB_FINISH_COST";
    private static final String CHECK_PARAM_CODE = "P-2452lK17";

    private final Logger logger = LoggerFactory.getLogger(getClass());
    @Autowired
    private IContractService contractService;

    @Autowired
    private IBillCodeApi billCodeApi;

    @Autowired
    private IParamConfigApi paramConfigApi;

    @Autowired
    private ISubFinishDetailService finishDetailService;

    @Autowired
    private ISubSettleService settleService;

    @Autowired
    private IOrgApi orgApi;

    @Autowired
    private SessionManager sessionManager;

    @Override
    public SubFinishVO insertOrUpdate(SubFinishVO finishVO) {
        if (checkSameContract(finishVO)) {
            throw new BusinessException("该合同在相同组织下已经做过完工结算！");
        }
        if (checkUnEffectSettle(finishVO)) {
            throw new BusinessException("该合同与组织下存在未生效过程结算单，不允许新增！");
        }
        if (!(finishVO.getId() != null && finishVO.getId() > 0)) { //新增
            if (checkFinishContract(finishVO)) {
                throw new BusinessException("该合同在相同组织下已经完工，不允许新增！");
            }
        }


        if (StringUtils.isEmpty(finishVO.getBillCode())) {
            CommonResponse<String> billCode = billCodeApi.getCodeBatchByRuleCode(SUB_FINISH_BILL_CODE, InvocationInfoProxy.getTenantid());
            if (billCode.isSuccess()) {
                finishVO.setBillCode(billCode.getData());
            } else {
                throw new BusinessException("网络异常， 编码生成失败， 请稍后再试");
            }
        }

        if (checkSameBillCode(finishVO)) {
            throw new BusinessException("存在相同编码，不允许保存!");
        }
        finishVO.setRelationFlag("0");
        finishVO.setProportionFlag("0");
        SubFinishEntity finishEntity = BeanMapper.map(finishVO, SubFinishEntity.class);

        //保存前清空主键和父主键，重新生成
        List<SubFinishDetailEntity> beforeDetails = finishEntity.getFinishDetailList();
        if (CollectionUtils.isNotEmpty(beforeDetails) && null == finishEntity.getId()) {
            for (SubFinishDetailEntity cdEntity : beforeDetails) {
                cdEntity.setId(null);
                cdEntity.setParentId(null);
            }
        }

        //保存修改合同，原合同完工状态要恢复过来
        if (finishEntity.getId() != null) {
            SubFinishEntity oldFinish = this.getById(finishEntity.getId());
            Long oldContractId = oldFinish.getContractId();
            if (oldContractId != null && !oldContractId.equals(finishEntity.getContractId())) {
                this.writeBackContractFinsishFlag(oldFinish, Boolean.FALSE);
            }
        }

        super.saveOrUpdate(finishEntity, false);

        List<SubFinishDetailEntity> finishDetailEntities = finishEntity.getFinishDetailList();
        if (CollectionUtils.isNotEmpty(finishDetailEntities)) {
            Map<String, Long> idMap = new HashMap<>();
            for (SubFinishDetailEntity cdEntity : finishDetailEntities) {
                idMap.put(cdEntity.getTid(), cdEntity.getId());
            }
            for (SubFinishDetailEntity cdEntity : finishDetailEntities) {
                if (StringUtils.isNotEmpty(cdEntity.getTpid())) {
                    cdEntity.setParentId(idMap.get(cdEntity.getTpid()));
                }
            }
            finishDetailService.saveOrUpdateBatch(finishDetailEntities, finishDetailEntities.size(), false);
        }

        // 回写合同完工结算状态
        this.writeBackContractFinsishFlag(finishEntity, Boolean.TRUE);
        return queryDetail(finishEntity.getId());
    }

    private Boolean checkSameBillCode(SubFinishVO finishVO) {
        Long tenantId = InvocationInfoProxy.getTenantid();
        LambdaQueryWrapper<SubFinishEntity> lambda = new LambdaQueryWrapper<>();
        lambda.eq(SubFinishEntity::getBillCode, finishVO.getBillCode());
        lambda.eq(SubFinishEntity::getTenantId, tenantId);
        if (null != finishVO.getId() && finishVO.getId() > 0) {
            lambda.ne(SubFinishEntity::getId, finishVO.getId());
        }
        return super.list(lambda).size() > 0;
    }

    private Boolean checkSameContract(SubFinishVO finishVO) {
        Long tenantId = InvocationInfoProxy.getTenantid();
        LambdaQueryWrapper<SubFinishEntity> lambda = new LambdaQueryWrapper<>();
        lambda.eq(SubFinishEntity::getContractId, finishVO.getContractId());
        lambda.eq(SubFinishEntity::getOrgId, finishVO.getOrgId());
        lambda.eq(SubFinishEntity::getTenantId, tenantId);
        if (null != finishVO.getId() && finishVO.getId() > 0) {
            lambda.ne(SubFinishEntity::getId, finishVO.getId());
        }
        return super.list(lambda).size() > 0;
    }

    private Boolean checkUnEffectSettle(SubFinishVO finishVO) {
        Long tenantId = InvocationInfoProxy.getTenantid();
        LambdaQueryWrapper<SubSettleEntity> lambda = new LambdaQueryWrapper<>();
        lambda.eq(SubSettleEntity::getContractId, finishVO.getContractId());
        lambda.eq(SubSettleEntity::getOrgId, finishVO.getOrgId());
        lambda.notIn(SubSettleEntity::getBillState, BillStateEnum.PASSED_STATE.getBillStateCode(), BillStateEnum.COMMITED_STATE.getBillStateCode());
        lambda.eq(SubSettleEntity::getTenantId, tenantId);
        return settleService.list(lambda).size() > 0;
    }

    private Boolean checkFinishContract(SubFinishVO finishVO) {
        Long tenantId = InvocationInfoProxy.getTenantid();
        LambdaQueryWrapper<SubContractEntity> lambda = new LambdaQueryWrapper<>();
        lambda.eq(SubContractEntity::getId, finishVO.getContractId());
        lambda.eq(SubContractEntity::getTenantId, tenantId);
        lambda.eq(SubContractEntity::getFinishFlag, true);
        return contractService.list(lambda).size() > 0;
    }

    @Override
    public SubFinishVO queryDetail(Long id) {
        SubFinishEntity finishEntity = super.selectById(id);
        SubFinishVO finishVO = BeanMapper.map(finishEntity, SubFinishVO.class);
        QueryParam queryParam = new QueryParam();
        queryParam.getParams().put("finishId", new Parameter(QueryParam.EQ, id));
        queryParam.getOrderMap().put("treeIndex", "asc");
        List<SubFinishDetailEntity> detailEntityList = finishDetailService.queryList(queryParam, false);
        if (CollectionUtils.isNotEmpty(detailEntityList)) {

            for (SubFinishDetailEntity cdEntity : detailEntityList) {
                cdEntity.setTid(cdEntity.getId().toString());
                cdEntity.setTpid(cdEntity.getParentId() != null && cdEntity.getParentId() > 0 ? cdEntity.getParentId().toString() : "");
                cdEntity.setRowState("edit");
            }
            List<SubFinishDetailVO> resultMapList = BeanMapper.mapList(detailEntityList, SubFinishDetailVO.class);
            sortIntMethod(resultMapList);
            List<SubFinishDetailVO> finishDetailVOS = TreeNodeBUtil.buildTree(resultMapList);
            finishVO.setFinishDetailList(finishDetailVOS);
        }
        // 【未付金额】=【完工结算金额】-【已付金额】
        finishVO.setUnPaidMoney(ComputeUtil.safeSub(finishEntity.getSettleTaxMny(), finishEntity.getPaidMoney()));
        return finishVO;
    }
    private static void sortIntMethod(List<SubFinishDetailVO> list){
        Collections.sort(list, new Comparator(){
            @Override
            public int compare(Object o1, Object o2) {
                ObjectMapper objectMapper = new ObjectMapper();
                SubFinishDetailVO o1Detail = objectMapper.convertValue(o1, SubFinishDetailVO.class);
                SubFinishDetailVO o2Detail = objectMapper.convertValue(o2, SubFinishDetailVO.class);
                String[] strs1 = o1Detail.getTreeIndex().split("\\.");
                String[] strs2 = o2Detail.getTreeIndex().split("\\.");
                int length = strs1.length > strs2.length ? strs1.length : strs2.length;
                for (int i = 0; i < length; i++) {
                    int num1 = 0;
                    int num2 = 0;
                    try {
                        num1 = Integer.parseInt(strs1[i]);
                        num2 = Integer.parseInt(strs2[i]);
                    } catch (Exception e) {

                    }
                    if (num1 < num2)
                        return -1;
                    if (num1 > num2)
                        return 1;
                }
                return 0;
            }
        });
    }
    @Override
    public CommonResponse<IPage<SubFinishVO>> queryListVOs(QueryParam param) {
        /** 模糊搜索配置字段示例 */
        List<String> fuzzyFields = param.getFuzzyFields();
        fuzzyFields.add("orgName");
        fuzzyFields.add("billCode");
        fuzzyFields.add("contractName");
//        fuzzyFields.add("customerName");
        fuzzyFields.add("supplierName");
        fuzzyFields.add("employeeName");
        fuzzyFields.add("memo");
        param.getParams().put("tenantId", new Parameter(QueryParam.EQ, InvocationInfoProxy.getTenantid()));
        // 组织本下
        UserContext userContext = sessionManager.getUserContext();
        String authOrgIds = userContext.getAuthOrgIds();
        if (StringUtils.isNotEmpty(authOrgIds)) {
            CommonResponse<List<OrgVO>> authResponse =
                    orgApi.findChildrenByParentIds(Arrays.stream(authOrgIds.split(",")).map(Long::parseLong).
                            collect(Collectors.toList()));
            param.getParams().put("orgId", new Parameter(QueryParam.IN, authResponse.getData().stream().map(OrgVO::getId).collect(Collectors.toList())));
        } else {
            param.getParams().put("orgId", new Parameter(QueryParam.IN, orgApi.findChildrenByParentId(InvocationInfoProxy.getOrgId()).getData().stream().map(OrgVO::getId).collect(Collectors.toList())));
        }
        param.getOrderMap().put("createTime", QueryParam.DESC);
        // 结算跳转付款申请制单标识
        boolean payApplyFlagHasFree = param.getParams().containsKey("payApplyFlagHasFree");
        param.getParams().remove("payApplyFlagHasFree");

        IPage<SubFinishEntity> page = queryPage(param, false);
        List<SubFinishVO> finishVOList = BeanMapper.mapList(page.getRecords(), SubFinishVO.class);
        /*
        // 结算跳转付款申请制单标识，已申请金额需要包含在途
        if (payApplyFlagHasFree && CollectionUtils.isNotEmpty(finishVOList)) {
            List<Long> settleIds = finishVOList.stream().map(e -> e.getId()).collect(Collectors.toList());
            CommonResponse<Map<Long, BigDecimal>> applyMnyHasFreeRes = payContractApi.getApplyMnyHasFree(settleIds);
            Map<Long, BigDecimal> settleApplyData = applyMnyHasFreeRes.getData();
            for (FinishVO subSettleVO : finishVOList) {
                BigDecimal sumApplyMnyHasFree = settleApplyData.get(subSettleVO.getId());
                subSettleVO.setSumApplyMny(sumApplyMnyHasFree);
                BigDecimal surplusApplyMny = ComputeUtil.safeSub(subSettleVO.getSettleTaxMny(), sumApplyMnyHasFree);
                subSettleVO.setSurplusApplyMny(surplusApplyMny);
            }
        }
        */
        IPage<SubFinishVO> finishVOIPage = new Page<>();
        finishVOIPage.setCurrent(page.getCurrent());
        finishVOIPage.setRecords(finishVOList);
        finishVOIPage.setSize(page.getSize());
        finishVOIPage.setTotal(page.getTotal());
        finishVOIPage.setPages(page.getPages());
        return CommonResponse.success("查询成功！", finishVOIPage);
    }

    @Override
    public CommonResponse<String> deleteByIds(List<SubFinishVO> vos) {
        List<Long> ids = vos.stream().map(SubFinishVO::getId).collect(Collectors.toList());
        // 回写合同完工结算状态
        List<SubFinishEntity> entityList = BeanMapper.mapList(super.listByIds(ids), SubFinishEntity.class);
        for (SubFinishEntity finishEntity : entityList) {
            this.writeBackContractFinsishFlag(finishEntity, Boolean.FALSE);
        }
        if (CollectionUtils.isNotEmpty(vos)) {
            this.removeByIds(ids, false);
        }
        return CommonResponse.success("删除成功！");
    }

    @Override
    public SubFinishRecordVO queryDetailRecord(Long id) {
        SubContractEntity contractEntity = contractService.selectById(id);
        SubFinishRecordVO finishRecordVO = new SubFinishRecordVO();
        BigDecimal contractTaxMny = contractEntity.getContractTaxMny() == null ? BigDecimal.ZERO : contractEntity.getContractTaxMny();
        BigDecimal sumSettleTaxMny = (contractEntity.getSumSettleTaxMny() == null || !contractEntity.getFinishFlag()) ? BigDecimal.ZERO : contractEntity.getSumSettleTaxMny();
        BigDecimal sumScale = BigDecimalUtils.safeDiv(sumSettleTaxMny, contractTaxMny).multiply(new BigDecimal(100));
        finishRecordVO.setContractId(id);
        finishRecordVO.setFinishFlag(contractEntity.getFinishFlag());
        finishRecordVO.setContractStatus(Integer.valueOf(contractEntity.getContractStatus()));
        finishRecordVO.setPurchaseType(Integer.valueOf(contractEntity.getPurchaseType()));
        finishRecordVO.setContractName(contractEntity.getContractName());
        finishRecordVO.setSupplierName(contractEntity.getSupplierName());
        finishRecordVO.setContractTaxMny(contractTaxMny);
        finishRecordVO.setSumSettleTaxMny(sumSettleTaxMny);
        finishRecordVO.setSumScale(sumScale);

        LambdaQueryWrapper<SubFinishEntity> lambda = Wrappers.<SubFinishEntity>lambdaQuery();
        lambda.eq(SubFinishEntity::getContractId, id);
        List<SubFinishEntity> entitiesAllState = super.list(lambda);
        lambda.in(SubFinishEntity::getBillState, BillStateEnum.PASSED_STATE.getBillStateCode(), BillStateEnum.COMMITED_STATE.getBillStateCode());
        List<SubFinishEntity> entities = super.list(lambda);
        List<SubFinishRecordDetailVO> finishRecordDetailVOList = BeanMapper.mapList(entities, SubFinishRecordDetailVO.class);
        if (CollectionUtils.isNotEmpty(finishRecordDetailVOList)) {
            finishRecordDetailVOList.forEach(entity -> {
                BigDecimal settleTaxMny = entity.getSettleTaxMny() == null ? BigDecimal.ZERO : entity.getSettleTaxMny();
                BigDecimal scale = contractTaxMny == BigDecimal.ZERO ? BigDecimal.ZERO : (ComputeUtil.safeDiv(settleTaxMny,contractTaxMny)).multiply(new BigDecimal(100));
                entity.setSumScale(scale);
            });
        }
        finishRecordVO.setDetailList(BeanMapper.mapList(entitiesAllState, SubFinishRecordDetailVO.class));

        LambdaQueryWrapper<SubSettleEntity> lambda2 = Wrappers.<SubSettleEntity>lambdaQuery();
        lambda2.eq(SubSettleEntity::getContractId, id);
        lambda2.in(SubSettleEntity::getBillState, BillStateEnum.PASSED_STATE.getBillStateCode(), BillStateEnum.COMMITED_STATE.getBillStateCode());
        List<SubSettleEntity> settleList = settleService.list(lambda2);
        // 过程结算金额
        BigDecimal settleTaxMny = new BigDecimal(0);
        for (SubSettleEntity settle : settleList) {
            settleTaxMny = settleTaxMny.add(settle.getSettleTaxMny() == null ? BigDecimal.ZERO : settle.getSettleTaxMny());
        }
        finishRecordVO.setSettleTaxMny(settleTaxMny);
        finishRecordVO.setSettleBalanceMny(sumSettleTaxMny.subtract(settleTaxMny));
        return finishRecordVO;
    }

    @Override
    public SubFinishVO queryDetailAdd(Long contractId) {
        SubContractEntity contractEntity = contractService.selectById(contractId);
        SubFinishVO finishVO = BeanMapper.map(contractEntity, SubFinishVO.class);
        finishVO.setBillCode(null);
        finishVO.setEmployeeId(Long.valueOf(InvocationInfoProxy.getEmployeeId()));
        finishVO.setEmployeeName(null);
        finishVO.setContractId(contractEntity.getId());
        finishVO.setBillState(null);
        finishVO.setCreateUserCode(null);
        finishVO.setCreateTime(null);
        finishVO.setUpdateUserCode(null);
        finishVO.setUpdateTime(null);
        finishVO.setId(null);
        finishVO.setVersion(null);
        finishVO.setTaxMny(null);
        finishVO.setSumInvoiceTaxMny(null);
        finishVO.setContractVersion(contractEntity.getVersion());
        return finishVO;
    }

    @Override
    public List<SubFinishVO> queryExportList(QueryParam param) {
        param.setPageIndex(0);
        param.setPageSize(-1);
        List<SubFinishVO> resVos = queryListVOs(param).getData().getRecords();
        if (!resVos.isEmpty()) {
            for (int i = 0; i < resVos.size(); i++) {
                SubFinishVO vo = resVos.get(i);
                if (vo.getContractType() != null) {
                    if (vo.getContractType().equals(Long.valueOf("1270328729526124545"))) {
                        vo.setContractTypeName("专业分包");
                    } else if (vo.getContractType().equals(Long.valueOf("1270328674299723778"))) {
                        vo.setContractTypeName("劳务分包");
                    }
                }
                vo.setBillStateName(BillStateEnum.getEnumByStateCode(vo.getBillState()).getDescription());
            }
            ;
        }
        return resVos;
    }

    @Override
    public SubParamsCheckVO checkParams(Integer purchaseType, BigDecimal settleTaxMny, BigDecimal contractTaxMny, Long orgId) {
        Long curOrgId = Optional.ofNullable(orgId).orElse(InvocationInfoProxy.getOrgId());
        // 存放预警结果
        Map<String, List<SubParamsCheckDsVO>> paramsCheckVOMap = new HashMap<>();
        paramsCheckVOMap.put("alert",new ArrayList<>());
        paramsCheckVOMap.put("warn",new ArrayList<>());
        // 三种控制方式：不控制，提醒，无法保存 (默认为提醒)
        String[] paramsArray = {"none", "warn", "alert"};
        SubParamsCheckVO paramsCheckVO = new SubParamsCheckVO();
        List<SubParamsCheckDsVO> checkDsVOS = new ArrayList<>();
        CommonResponse<List<BillParamVO>> billParamByCode = paramConfigApi.getBillParamByCodeAndOrgId(CHECK_PARAM_CODE, curOrgId);
        if (billParamByCode.isSuccess() && null != billParamByCode.getData()) {
            List<BillParamVO> data = billParamByCode.getData();
            for (BillParamVO datum : data) {
                BigDecimal roleValue = datum.getRoleValue();
                BigDecimal comMny = contractTaxMny.multiply(roleValue.divide(BigDecimal.valueOf(100)
                        , 2, BigDecimal.ROUND_HALF_UP));

                if (2 == purchaseType) {
                    // 集采合同直接赋值为不控制
                    paramsCheckVO.setWarnType(paramsArray[0]);
                    return paramsCheckVO;
                } else {
                    paramsCheckVO.setWarnType(paramsArray[datum.getControlType()]);
                }

                if (settleTaxMny.compareTo(comMny) > 0) {
                    SubParamsCheckDsVO paramsCheckDsVO = new SubParamsCheckDsVO();
                    paramsCheckDsVO.setOrgName(datum.getOrgName());
                    paramsCheckDsVO.setWarnItem("合同超结");
                    paramsCheckDsVO.setWarnName("完工结算金额大于合同金额");
                    StringBuffer stringBuffer = new StringBuffer();
                    stringBuffer.append("本次结算金额：").append(settleTaxMny.setScale(2, BigDecimal.ROUND_HALF_UP))
                            .append("元，合同金额*").append(roleValue.setScale(2, BigDecimal.ROUND_HALF_UP)).append("%:")
                            .append(comMny.setScale(2, BigDecimal.ROUND_HALF_UP)).append("元。超出金额：")
                            .append((settleTaxMny.subtract(comMny)).setScale(2, BigDecimal.ROUND_HALF_UP)).append("元");
                    paramsCheckDsVO.setContent(stringBuffer.toString());
                    checkDsVOS.add(paramsCheckDsVO);
                    updateParamsCheckVOMap(paramsArray, paramsCheckVOMap, datum, paramsCheckDsVO);
                }
            }
        } else {
            logger.info(billParamByCode.getMsg());
            throw new BusinessException("获取控制参数失败");
        }

        //处理刚性、柔性预警
        SubParamsCheckVO pc = new SubParamsCheckVO();
        if (CollectionUtils.isNotEmpty(paramsCheckVOMap.get("alert"))) {
            pc.setWarnType("alert");
            pc.setDataSource(paramsCheckVOMap.get("alert"));
        } else if (CollectionUtils.isNotEmpty(paramsCheckVOMap.get("warn"))) {
            pc.setWarnType("warn");
            pc.setDataSource(paramsCheckVOMap.get("warn"));
        } else {
            pc.setWarnType("none");
            pc.setDataSource(null);
        }
        return pc;
    }

    /**
     * 更新参数控制结果
     *
     * @param paramsArray      参数数组
     * @param paramsCheckVOMap 预警结果map
     * @param billParamVO      控制参数
     * @param paramsCheckDsVO  预警信息
     */
    private static void updateParamsCheckVOMap(String[] paramsArray, Map<String, List<SubParamsCheckDsVO>> paramsCheckVOMap, BillParamVO billParamVO, SubParamsCheckDsVO paramsCheckDsVO) {
        if ("alert".equals(paramsArray[billParamVO.getControlType()])) {
            List<SubParamsCheckDsVO> alert = paramsCheckVOMap.get("alert");
            alert.add(paramsCheckDsVO);
        }
        if ("warn".equals(paramsArray[billParamVO.getControlType()])) {
            List<SubParamsCheckDsVO> warn = paramsCheckVOMap.get("warn");
            warn.add(paramsCheckDsVO);
        }
    }

    /**
     * 回写合同完工结算状态
     *
     * @param finishEntity
     * @param finishFlag
     */
    private void writeBackContractFinsishFlag(SubFinishEntity finishEntity, Boolean finishFlag) {
        LambdaUpdateWrapper<SubContractEntity> updateWrapper = new LambdaUpdateWrapper<>();
        //根据完工状态回写合同状态，合同状态3已封账，2履约中
        updateWrapper.set(SubContractEntity::getContractStatus, finishFlag ? "3" : "2");
        updateWrapper.set(SubContractEntity::getFinishFlag, finishFlag);
        updateWrapper.eq(SubContractEntity::getId, finishEntity.getContractId());
        SubContractEntity contractEntity = contractService.getById(finishEntity.getContractId());
        contractService.update(updateWrapper);
    }
}
