package com.ejianc.business.material.controller;

import com.alibaba.fastjson.JSONObject;
import com.baomidou.mybatisplus.core.metadata.IPage;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.ejianc.business.material.bean.ComparePriceEntity;
import com.ejianc.business.material.bean.ComparePriceSelectedSupplierEntity;
import com.ejianc.business.material.bean.ComparePriceSubEntity;
import com.ejianc.business.material.service.IComparePriceSelectedSupplierService;
import com.ejianc.business.material.service.IComparePriceService;
import com.ejianc.business.material.service.IComparePriceSubService;
import com.ejianc.business.material.vo.ComparePriceExportVO;
import com.ejianc.business.material.vo.ComparePriceSubVO;
import com.ejianc.business.material.vo.ComparePriceVO;
import com.ejianc.foundation.orgcenter.api.IOrgApi;
import com.ejianc.foundation.orgcenter.vo.OrgVO;
import com.ejianc.foundation.support.api.IBillCodeApi;
import com.ejianc.framework.core.context.InvocationInfoProxy;
import com.ejianc.framework.core.exception.BusinessException;
import com.ejianc.framework.core.kit.collection.ListUtil;
import com.ejianc.framework.core.kit.mapper.BeanMapper;
import com.ejianc.framework.core.kit.time.DateFormatUtil;
import com.ejianc.framework.core.response.BillStateEnum;
import com.ejianc.framework.core.response.CommonResponse;
import com.ejianc.framework.core.response.Parameter;
import com.ejianc.framework.core.response.QueryParam;
import com.ejianc.framework.core.util.ExcelExport;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Controller;
import org.springframework.web.bind.annotation.*;

import javax.servlet.http.HttpServletResponse;
import java.math.BigDecimal;
import java.util.*;
import java.util.stream.Collectors;

/**
 * @Author mrsir_wxp
 * @Date 2020/6/6 物资询比价控制器
 * @Description ComparePriceController
 */

@Controller
@RequestMapping("/comparePrice/")
public class ComparePriceController {

    private static final String BILL_CODE = "comparePrice";
    @Autowired
    private IComparePriceService iComparePriceService;
    @Autowired
    private IComparePriceSubService iComparePriceSubService;
    @Autowired
    private IComparePriceSelectedSupplierService iComparePriceSelectedSupplierService;
    @Autowired
    private IBillCodeApi billCodeApi;
    @Autowired
    private IOrgApi iOrgApi;


    @RequestMapping(value = "/saveOrUpdate", method = RequestMethod.POST)
    @ResponseBody
    public CommonResponse<ComparePriceVO> saveOrUpdate(@RequestBody ComparePriceVO vo) {
        ComparePriceEntity entity = BeanMapper.map(vo, ComparePriceEntity.class);
        if(vo.getId()==null){
            CommonResponse<String> billCode = billCodeApi.getCodeBatchByRuleCode(BILL_CODE, InvocationInfoProxy.getTenantid());
            if(billCode.isSuccess()) {
                entity.setBillCode(billCode.getData());
            }else{
                throw new BusinessException("网络异常， 编码生成失败， 请稍后再试");
            }
        }
        iComparePriceService.insertOrUpdate(entity);
        vo = BeanMapper.map(entity,ComparePriceVO.class);
        return CommonResponse.success("保存或修改单据成功！",vo);
    }

    @RequestMapping(value = "/queryDetail", method = RequestMethod.GET)
    @ResponseBody
    public CommonResponse<ComparePriceVO> queryDetail(Long id) {
        ComparePriceEntity entity = iComparePriceService.selectById(id);
        if(!ListUtil.isEmpty(entity.getPriceSubEntities())){
            entity.getPriceSubEntities().forEach(sub->{
                QueryParam param = new QueryParam();
                param.getParams().put("comparePriceSubId",new Parameter(QueryParam.EQ,sub.getId()));
                sub.setSupplierEntities(iComparePriceSelectedSupplierService.queryList(param,false));
            });
        }
        ComparePriceVO vo = BeanMapper.map(entity, ComparePriceVO.class);
        return CommonResponse.success("查询详情数据成功！",vo);
    }

    @RequestMapping(value = "/queryList", method = RequestMethod.POST)
    @ResponseBody
    public CommonResponse<IPage<ComparePriceVO>> queryList(@RequestBody QueryParam param) {
        /** 模糊搜索配置字段示例 */
        List<String> fuzzyFields = param.getFuzzyFields();
        fuzzyFields.add("billCode");
        fuzzyFields.add("name");
        fuzzyFields.add("projectName");
        fuzzyFields.add("materialTypeName");
        fuzzyFields.add("inquiryPerson");
        fuzzyFields.add("orgName");
        /** 数据隔离 本下 */
        param.getParams().put("orgId",new Parameter(QueryParam.IN,iOrgApi.findChildrenByParentId(InvocationInfoProxy.getOrgId()).getData().stream().map(OrgVO::getId).collect(Collectors.toList())));

        return CommonResponse.success("查询列表数据成功！",iComparePriceService.queryForList(param));
    }

    @RequestMapping(value = "/queryPrint", method = RequestMethod.GET)
    @ResponseBody
    public CommonResponse<JSONObject> queryPrint(Long id) {
        JSONObject json = new JSONObject();
        json.put("ComparePrice",iComparePriceService.selectById(id));
        return CommonResponse.success("查询打印数据成功！",json);
    }

    @RequestMapping(value = "/delete", method = RequestMethod.POST)
    @ResponseBody
    public CommonResponse<String> delete(@RequestBody List<ComparePriceVO> vos) {
        iComparePriceService.removeByIds(vos.stream().map(ComparePriceVO::getId).collect(Collectors.toList()),true);
        return CommonResponse.success("删除成功！");
    }

    @RequestMapping(value = "/excelExport", method = RequestMethod.POST)
    @ResponseBody
    public void excelExport(@RequestBody QueryParam queryParam, HttpServletResponse response) {
        queryParam.getParams().put("tenantId", new Parameter(QueryParam.EQ, InvocationInfoProxy.getTenantid()));
        queryParam.setPageIndex(0);
        queryParam.setPageSize(10000);
        List<String> fuzzyFields = queryParam.getFuzzyFields();
        fuzzyFields.add("billCode");
        fuzzyFields.add("name");
        fuzzyFields.add("projectName");
        fuzzyFields.add("materialTypeName");
        fuzzyFields.add("inquiryPerson");
        fuzzyFields.add("orgName");

        /** 数据隔离 本下 */
        queryParam.getParams().put("orgId",new Parameter(QueryParam.IN,iOrgApi.findChildrenByParentId(InvocationInfoProxy.getOrgId()).getData().stream().map(OrgVO::getId).collect(Collectors.toList())));

        IPage<ComparePriceVO>  voPage = iComparePriceService.queryForList(queryParam);
        List<ComparePriceVO> list = voPage.getRecords();
        List<ComparePriceExportVO> exportVOS = new ArrayList<>();
        list.forEach(vo -> {
            ComparePriceExportVO exportVO = BeanMapper.map(vo, ComparePriceExportVO.class);
            exportVO.setCreateTime(DateFormatUtil.formatDate("yyyy-MM-dd HH:mm:ss",vo.getCreateTime()));
            if(vo.getInquiryDate()!=null){
                exportVO.setInquiryDate(DateFormatUtil.formatDate("yyyy-MM-dd",vo.getInquiryDate()));
            }
            if(vo.getRecommendPrice()!=null){
                exportVO.setRecommendPrice(String.valueOf(vo.getRecommendPrice().setScale(2,BigDecimal.ROUND_HALF_UP)));
            }else{
                exportVO.setRecommendPrice(String.valueOf(new BigDecimal("0")));
            }
            exportVOS.add(exportVO);
        });
        Map<String, Object> beans = new HashMap<>();
        beans.put("records", exportVOS);
        ExcelExport.getInstance().export("comparePriceListExport.xlsx", beans, response);
    }

    /**
     * @Author mrsir_wxp
     * @Date 2020/6/22 参照询价单
     * @Description refComparePrice
     * @Param [pageNumber, pageSize, searchText]
     * @Return com.ejianc.framework.core.response.CommonResponse<com.baomidou.mybatisplus.core.metadata.IPage<com.ejianc.business.material.vo.ComparePriceVO>>
     */
    @RequestMapping(value = "/refComparePrice", method = RequestMethod.GET)
    @ResponseBody
    public CommonResponse<IPage<ComparePriceVO>> refComparePrice(@RequestParam Integer pageNumber,
                                                                 @RequestParam Integer pageSize,
                                                                 @RequestParam(required = false) String condition,
                                                                 String searchText) {
        QueryParam param = new QueryParam();
        param.setPageIndex(pageNumber);
        param.setPageSize(pageSize);
        param.setSearchText(searchText);
        /** 模糊搜索配置字段示例 */
        List<String> fuzzyFields = param.getFuzzyFields();
        fuzzyFields.add("billCode");
        fuzzyFields.add("name");
        fuzzyFields.add("projectName");
        fuzzyFields.add("materialTypeName");
        fuzzyFields.add("inquiryPerson");
        fuzzyFields.add("orgName");

        //按照创建时间倒序排列
        param.getOrderMap().put("createTime", QueryParam.DESC);

        Long orgId = InvocationInfoProxy.getOrgId();

        if(StringUtils.isNotBlank(condition)) {
            Map<String, Object> conditionMap = JSONObject.parseObject(condition, Map.class);
            if(null != conditionMap.get("orgId")) {
                orgId = Long.valueOf(conditionMap.get("orgId").toString());
            }
        }

        /** 数据隔离 本下 */
        param.getParams().put("orgId",new Parameter(QueryParam.IN,iOrgApi.findChildrenByParentId(orgId).getData().stream().map(OrgVO::getId).collect(Collectors.toList())));
        param.getParams().put("billState",new Parameter(QueryParam.IN, Arrays.asList(BillStateEnum.COMMITED_STATE.getBillStateCode(),BillStateEnum.PASSED_STATE.getBillStateCode())));

        return CommonResponse.success("查询询价单参照数据成功！",iComparePriceService.queryForList(param));
    }


    /**
     * @Author mrsir_wxp
     * @Date 2020/6/22 参照询价单子表
     * @Description refComparePrice
     * @Param [pageNumber, pageSize, searchText]
     * @Return com.ejianc.framework.core.response.CommonResponse<com.baomidou.mybatisplus.core.metadata.IPage<com.ejianc.business.material.vo.ComparePriceVO>>
     */
    @RequestMapping(value = "/refComparePriceSubs", method = RequestMethod.GET)
    @ResponseBody
    public CommonResponse<IPage<ComparePriceSubVO>> refComparePriceSubs(@RequestParam Integer pageNumber,
                                                                 @RequestParam Integer pageSize,
                                                                 String condition ,
                                                                 @RequestParam(value = "searchText", required = false) String searchText) {
        if(StringUtils.isBlank(condition)){
            return CommonResponse.error("查询条件condition不能为空");
        }
        JSONObject con = JSONObject.parseObject(condition);
        if(StringUtils.isBlank(con.getString("comparePriceId"))){
            return CommonResponse.error("查询条件比价单Id【comparePriceId】不能为空");
        }
        ComparePriceEntity entity = iComparePriceService.selectById(con.getLong("comparePriceId"));
        String selectedSupplierId = StringUtils.isNotBlank(entity.getRecommendSupplier())?entity.getRecommendSupplier().split(",")[0]:null;
        QueryParam queryParam = new QueryParam();
        queryParam.getParams().put("pid", new Parameter(QueryParam.EQ, con.getLong("comparePriceId")));
        queryParam.setSearchText(searchText);
        queryParam.setPageIndex(pageNumber);
        queryParam.setPageSize(pageSize);
        queryParam.getFuzzyFields().add("materialName");
        queryParam.getFuzzyFields().add("materialTypeName");
        queryParam.getFuzzyFields().add("model");
        queryParam.getFuzzyFields().add("unit");

        queryParam.getOrderMap().put("createTime", QueryParam.DESC);

        IPage<ComparePriceSubEntity> pageData = iComparePriceSubService.queryPage(queryParam, false);
        List<ComparePriceSubVO> voList = BeanMapper.mapList(pageData.getRecords(), ComparePriceSubVO.class);
        voList.stream().forEach(vo -> {
            if(StringUtils.isNotBlank(selectedSupplierId)){
                QueryParam param = new QueryParam();
                param.getParams().put("comparePriceSubId",new Parameter(QueryParam.EQ,vo.getId()));
                param.getParams().put("supplierId",new Parameter(QueryParam.EQ,selectedSupplierId));
                List<ComparePriceSelectedSupplierEntity> supplierEntities = iComparePriceSelectedSupplierService.queryList(param,false);
                if(ListUtil.isNotEmpty(supplierEntities)){
                    ComparePriceSelectedSupplierEntity supplierEntity = supplierEntities.get(0);
                    vo.setRecommendPrice(supplierEntity.getPrice()==null?"0.00":supplierEntity.getPrice().toString());
                    vo.setRecommendMoney(supplierEntity.getMoney()==null?"0.00":supplierEntity.getMoney().toString());
                }else {
                    vo.setRecommendPrice("0.00");
                    vo.setRecommendMoney("0.00");
                }
            }else {
                vo.setRecommendPrice("0.00");
                vo.setRecommendMoney("0.00");
            }
        });
        IPage<ComparePriceSubVO> data = new Page<>();
        data.setRecords(voList);
        data.setTotal(pageData.getTotal());
        data.setSize(pageData.getSize());
        data.setCurrent(pageData.getCurrent());
        data.setPages(pageData.getPages());

        return CommonResponse.success("查询询价单参照数据成功！",data);
    }

}
