package com.ejianc.business.income.controller;


import com.alibaba.fastjson.JSONObject;
import com.baomidou.mybatisplus.core.metadata.IPage;
import com.ejianc.business.income.bean.OutProductionEntity;
import com.ejianc.business.income.bean.QuoteEntity;
import com.ejianc.business.income.history.ProductionHistoryVo;
import com.ejianc.business.income.service.IOutProductionService;
import com.ejianc.business.income.service.IProductionService;
import com.ejianc.business.income.service.IQuoteService;
import com.ejianc.business.income.utils.ComputeUtil;
import com.ejianc.business.income.vo.OutProductionDetailVo;
import com.ejianc.business.income.vo.OutProductionVo;
import com.ejianc.business.income.vo.report.ProductionReportVO;
import com.ejianc.foundation.orgcenter.api.IOrgApi;
import com.ejianc.foundation.orgcenter.vo.OrgVO;
import com.ejianc.framework.auth.session.SessionManager;
import com.ejianc.framework.auth.session.UserContext;
import com.ejianc.framework.core.context.InvocationInfoProxy;
import com.ejianc.framework.core.kit.mapper.BeanMapper;
import com.ejianc.framework.core.response.BillStateEnum;
import com.ejianc.framework.core.response.CommonResponse;
import com.ejianc.framework.core.response.Parameter;
import com.ejianc.framework.core.response.QueryParam;
import com.ejianc.framework.core.util.ExcelExport;
import com.ejianc.framework.core.util.ImportTemplate;
import org.apache.commons.collections.CollectionUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.web.bind.annotation.*;

import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import java.math.BigDecimal;
import java.util.*;
import java.util.stream.Collectors;

/**
 * <p>
 * 产值进度 前端控制器
 * </p>
 *
 * @author yuezx
 * @since 2020-06-04
 */
@RestController
@RequestMapping("/outProduction")
public class OutProductionController {
    @Autowired
    private IOutProductionService productionService;
    @Autowired
    private IOrgApi orgApi;

    @Autowired
    private IQuoteService quoteService;
    @Autowired
    private SessionManager sessionManager;

    /***
     * @description: 处理历史数据
     * 甲方报量、产值进度原有的累计进度和累计甲方都需要加上本期
     * @return: com.ejianc.framework.core.response.CommonResponse<java.lang.String>
     * @author songlx
     * @date: 2021/8/7
     */
    @RequestMapping(value = "/init", method = RequestMethod.GET)
    @ResponseBody
    public CommonResponse<String> init() {
        long startTime = System.currentTimeMillis();    //获取开始时间
        List<OutProductionEntity> list = productionService.list();
        for (OutProductionEntity p : list) {
            BigDecimal contractTaxMny = p.getContractTaxMny();
            BigDecimal sumProductionTaxMny = p.getSumProductionTaxMny();
            BigDecimal sumProductionMny = p.getSumProductionMny();
            BigDecimal finishTaxMny = p.getFinishTaxMny();
            BigDecimal finishMny = p.getFinishMny();

            BigDecimal newSumProTaxMny = ComputeUtil.safeAdd(sumProductionTaxMny, finishTaxMny);
            BigDecimal newSumProMny = ComputeUtil.safeAdd(sumProductionMny, finishMny);
            BigDecimal percent = ComputeUtil.bigDecimalPercent(newSumProTaxMny, contractTaxMny, 2);
            p.setSumProductionTaxMny(newSumProTaxMny);
            p.setSumProductionMny(newSumProMny);
            p.setSumImageProgress(percent);
        }
        productionService.updateBatchById(list);

        List<QuoteEntity> entityList = quoteService.list();
        for (QuoteEntity p : entityList) {
            BigDecimal contractTaxMny = p.getContractTaxMny();
            BigDecimal sumQuoteTaxMny = p.getSumQuoteTaxMny();
            BigDecimal sumQuoteMny = p.getSumQuoteMny();
            BigDecimal quoteTaxMny = p.getQuoteTaxMny();
            BigDecimal quoteMny = p.getQuoteMny();

            BigDecimal newSumProTaxMny = ComputeUtil.safeAdd(sumQuoteTaxMny, quoteTaxMny);
            BigDecimal newSumProMny = ComputeUtil.safeAdd(sumQuoteMny, quoteMny);
            BigDecimal percent = ComputeUtil.bigDecimalPercent(newSumProTaxMny, contractTaxMny, 8);
            p.setSumQuoteTaxMny(newSumProTaxMny);
            p.setSumQuoteMny(newSumProMny);
            p.setSumQuoteRate(percent);
        }
        quoteService.updateBatchById(entityList);

        long endTime = System.currentTimeMillis();    //获取结束时间

        String msg = "更新产值数量：" + list.size() + "，更新甲方数量：" + entityList.size() + "，程序运行时间：" + (endTime - startTime) + "ms";

        return CommonResponse.success(msg);
    }


    /**
     * 新增或编辑
     *
     * @param productionVo
     * @return
     */
    @RequestMapping(value = "/saveOrUpdate", method = RequestMethod.POST)
    @ResponseBody
    public CommonResponse<OutProductionVo> saveOrUpdate(@RequestBody OutProductionVo productionVo) {
        CommonResponse<OutProductionVo> response = productionService.saveOrUpdate(productionVo);
        return response;
    }

    /**
     * 根据主键ID查询详情
     *
     * @param id
     * @return
     */
    @RequestMapping(value = "/queryDetail", method = RequestMethod.GET)
    @ResponseBody
    public CommonResponse<OutProductionVo> queryDetail(@RequestParam Long id) {
        OutProductionVo vo = productionService.queryDetail(id);
        return CommonResponse.success(vo);
    }
    /**
     * 导入模板下载
     *
     * @param request
     * @param response
     */
    @RequestMapping(value = "/download")
    @ResponseBody
    public void download(HttpServletRequest request, HttpServletResponse response) {
        ImportTemplate.initialize(response);
        ImportTemplate.templetdownload(request, "production-detail-import.xlsx", "产值明细导入模板");
    }
    /**
     * excel导入
     *
     * @param request
     * @return
     */
    @RequestMapping(value = "/excelImport", method = RequestMethod.POST)
    @ResponseBody
    public CommonResponse<JSONObject> excelImport(HttpServletRequest request, HttpServletResponse response) {
        return productionService.excelImport(request,response);
    }
    /**
     * 查询分页列表
     *
     * @param param
     * @return
     */
    @RequestMapping(value = "/pageList", method = RequestMethod.POST)
    @ResponseBody
    public CommonResponse<IPage<OutProductionEntity>> pageList(@RequestBody QueryParam param) {
        /** 模糊搜索配置字段示例 */
        List<String> fuzzyFields = param.getFuzzyFields();
        //项目名称、合同名称、变更摘要、变更签证编号、经办人
        fuzzyFields.add("projectName");
        fuzzyFields.add("billCode");
        fuzzyFields.add("contractName");
        fuzzyFields.add("employeeName");
        param.getParams().put("tenantId", new Parameter(QueryParam.EQ, InvocationInfoProxy.getTenantid()));
        UserContext userContext = sessionManager.getUserContext();
        String authOrgIds = userContext.getAuthOrgIds();
        if (org.apache.commons.lang.StringUtils.isNotEmpty(authOrgIds)) {
            CommonResponse<List<OrgVO>> authResponse =
                    orgApi.findChildrenByParentIds(Arrays.stream(authOrgIds.split(",")).map(Long::parseLong).
                            collect(Collectors.toList()));
            param.getParams().put("orgId", new Parameter(QueryParam.IN, authResponse.getData().stream().map(OrgVO::getId).collect(Collectors.toList())));
        } else {
            param.getParams().put("orgId", new Parameter(QueryParam.IN, orgApi.findChildrenByParentId(InvocationInfoProxy.getOrgId()).getData().stream().map(OrgVO::getId).collect(Collectors.toList())));
        }
        param.getOrderMap().put("createTime", "desc");
        return CommonResponse.success("查询列表数据成功！", productionService.queryPage(param, false));
    }

    /**
     * 删除
     *
     * @param
     * @return
     */
    @RequestMapping(value = "/delete", method = RequestMethod.POST)
    @ResponseBody
    public CommonResponse<String> delete(@RequestBody List<OutProductionVo> vos) {
        productionService.deleteProduction(vos);
        return CommonResponse.success("删除成功");
    }

    /**
     * 产值进度历史记录
     *
     * @param id
     * @return
     */
    @RequestMapping(value = "/queryProductionHistory", method = RequestMethod.GET)
    @ResponseBody
    public CommonResponse<ProductionHistoryVo> queryProductionHistory(@RequestParam Long id) {
        ProductionHistoryVo vo = productionService.queryProductionHistory(id);
        return CommonResponse.success(vo);
    }

    /**
     * 查询产值进度参照分页列表
     *
     * @param
     * @return
     */
    @RequestMapping(value = "/pageRef", method = RequestMethod.GET)
    @ResponseBody
    public CommonResponse<List<OutProductionEntity>> pageRef(@RequestParam(value = "contractId") String contractId,
                                                          @RequestParam(value = "searchText", required = false) String searchText,
                                                          @RequestParam(value = "productionDate", required = false) String productionDate) {
        QueryParam queryParam = new QueryParam();
        queryParam.setSearchText(searchText);
        /** 模糊搜索配置字段示例 */
        queryParam.getFuzzyFields().add("projectName");
        queryParam.getFuzzyFields().add("billCode");
        queryParam.getFuzzyFields().add("contractName");
        queryParam.getFuzzyFields().add("employeeName");
        queryParam.getParams().put("contractId", new Parameter(QueryParam.EQ, contractId));
        queryParam.getParams().put("quoteFlag", new Parameter(QueryParam.EQ, 0));
        if (null != productionDate && productionDate.length() > 0) {
            queryParam.getParams().put("productionDate", new Parameter(QueryParam.LE, productionDate));
        }
        List<Integer> billStatus = new ArrayList<>();
        billStatus.add(1);
        billStatus.add(3);
        queryParam.getParams().put("bill_state", new Parameter(QueryParam.IN, billStatus));
        queryParam.getOrderMap().put("productionDate", "desc");
        queryParam.getOrderMap().put("createTime", "desc");
        return CommonResponse.success("查询产值进度参照成功！", productionService.queryList(queryParam, false));
    }

    /**
     * 产值进度明细参照
     *
     * @param productionId
     * @return
     */
    @RequestMapping(value = "/detailRef", method = RequestMethod.GET)
    @ResponseBody
    public CommonResponse<List<OutProductionDetailVo>> detailRef(@RequestParam Long productionId) {
        return CommonResponse.success("查询产值进度明细成功！", productionService.detailRef(productionId));
    }

    /**
     * 导出
     *
     * @param param
     * @return
     */
    @PostMapping("excelExport")
    public void excelExport(@RequestBody QueryParam param, HttpServletResponse response) {
        /** 模糊搜索配置字段示例 */
        List<String> fuzzyFields = param.getFuzzyFields();
        //项目名称、合同名称、变更摘要、变更签证编号、经办人
        fuzzyFields.add("projectName");
        fuzzyFields.add("billCode");
        fuzzyFields.add("contractName");
        fuzzyFields.add("employeeName");
        param.getParams().put("tenantId", new Parameter(QueryParam.EQ, InvocationInfoProxy.getTenantid()));
        param.getParams().put("org_id", new Parameter("in", orgApi.findChildrenByParentId(InvocationInfoProxy.getOrgId()).getData().stream().map(OrgVO::getId).collect(Collectors.toList())));
        param.getOrderMap().put("createTime", "desc");
        param.setPageIndex(1);
        param.setPageSize(-1);
        IPage<OutProductionEntity> pageData = productionService.queryPage(param, false);

        Map<String, Object> beans = new HashMap<String, Object>();
        List<OutProductionVo> list = new ArrayList<>();
        if (null != pageData.getRecords() && CollectionUtils.isNotEmpty(pageData.getRecords())) {
            list = BeanMapper.mapList(pageData.getRecords(), OutProductionVo.class);
            list.forEach(vo -> {
                vo.setBillStateName(BillStateEnum.getEnumByStateCode(vo.getBillState()).getDescription());
            });

        }
        beans.put("records", list);
        ExcelExport.getInstance().export("production-export.xlsx", beans, response);
    }

    /**
     * 根据来源主键（合同清单）查询历史数据的含本期累计完成量，完成产值
     * @param
     * @param
     * @return
     */
    @RequestMapping(value = "/queryProductionDetailHistory" , method = RequestMethod.POST)
    @ResponseBody
    public CommonResponse<Map<Long, OutProductionDetailVo>> queryProductionDetailHistory(@RequestBody OutProductionDetailVo productionDetailVo){
        Map<Long, OutProductionDetailVo> page = productionService.queryProductionDetailHistory(productionDetailVo);
        return CommonResponse.success("查询成功",page);
    }

    /***
     * 根据项目id 查询 累计完成产值
     * @param projectId
     * @return
     */
    @RequestMapping(value = "/queryInfoProjectId", method = RequestMethod.GET)
    @ResponseBody
    public  CommonResponse<OutProductionVo> queryInfoProjectId(Long projectId){
        OutProductionVo productionVo = productionService.queryInfoProjectId(projectId);
        return CommonResponse.success("查询打印数据成功！",productionVo);
    }
    /***
     * 项目产值统计表
     */
    @RequestMapping(value = "/queryProductionReport", method = RequestMethod.POST)
    @ResponseBody
    public  CommonResponse<JSONObject> queryProductionReport(@RequestBody QueryParam param){
        return CommonResponse.success("查询项目产值统计表成功！",productionService.queryProductionReport(param,Boolean.FALSE));
    }

    @RequestMapping(value = "/excelExportProductionReport", method = RequestMethod.POST)
    @ResponseBody
    public void excelExportProductionReport(@RequestBody QueryParam queryParam, HttpServletResponse response) {
        List<ProductionReportVO> records = productionService.excelExportProductionReport(queryParam);

        Map<String, Object> beans = new HashMap<>();
        beans.put("records", records);
        ExcelExport.getInstance().export("production-report-export.xlsx", beans, response);
    }
}
