package com.ejianc.business.labor.service.impl;

import cn.hutool.core.date.DateField;
import cn.hutool.core.date.DateRange;
import cn.hutool.core.date.DateTime;
import cn.hutool.core.date.DateUtil;
import com.alibaba.fastjson.JSONObject;
import com.baomidou.mybatisplus.core.conditions.query.QueryWrapper;
import com.baomidou.mybatisplus.core.metadata.IPage;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.ejianc.business.desktop.vo.AttendancePerLineVO;
import com.ejianc.business.labor.bean.AttendanceEntity;
import com.ejianc.business.labor.bean.AttendanceLogEntity;
import com.ejianc.business.labor.bean.WorkRecordEntity;
import com.ejianc.business.labor.mapper.AttendanceMapper;
import com.ejianc.business.labor.service.IAttendanceService;
import com.ejianc.business.labor.service.IWorkRecordService;
import com.ejianc.business.labor.vo.AttendanceDailyExportVO;
import com.ejianc.business.labor.vo.AttendanceVO;
import com.ejianc.business.market.api.IProjectApi;
import com.ejianc.framework.core.exception.BusinessException;
import com.ejianc.framework.core.kit.collection.ListUtil;
import com.ejianc.framework.core.response.Parameter;
import com.ejianc.framework.core.response.QueryParam;
import com.ejianc.framework.skeleton.template.BaseServiceImpl;
import org.apache.tools.ant.util.DateUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import java.util.*;
import java.util.stream.Collectors;


/**
 * 考勤记录
 *
 * @author baipengyan
 * @version 1.0
 * @since JDK 1.8
 */
@Service("attendanceService")
public class AttendanceServiceImpl extends BaseServiceImpl<AttendanceMapper, AttendanceEntity> implements IAttendanceService {

    @Autowired
    private IProjectApi projectApi;
    @Autowired
    private IWorkRecordService workRecordService;


    private final Logger logger = LoggerFactory.getLogger(this.getClass());

    @Override
    public int totalAttendanceDays(Date enterDate, Date exitDate, Long workerId,Long projectId) {
        QueryWrapper<AttendanceEntity> queryWrapper = new QueryWrapper<>();

        queryWrapper.eq("project_id",projectId);

        queryWrapper.eq("worker_id",workerId);
//        queryWrapper.eq("tenant_id", InvocationInfoProxy.getTenantid());
        queryWrapper.between("punch_card_date",enterDate,exitDate);
        int count = super.count(queryWrapper);
        return count;
    }


    /**
     * 按个人统计分页查询
     *
     * @param p       分页
     * @param wrapper 条件构造器
     *
     * @param startDate
     * @param endDate
     * @return {@link IPage}<{@link AttendanceEntity}>
     */
    @Override
    public IPage<AttendanceEntity> queryPageByPerson(Page<AttendanceEntity> p, QueryWrapper<AttendanceEntity> wrapper, String startDate, String endDate) {
        return baseMapper.queryPageByPerson(p, wrapper, startDate, endDate);
    }

    @Override
    public Integer queryAttendanceOrgIds(String orgIds){
        return baseMapper.queryAttendanceOrgIds(orgIds);
    }

    @Override
    public IPage<AttendanceVO> queryDailyAttendancePage(Page<AttendanceVO> p, QueryWrapper<AttendanceVO> wrapper) {
        return baseMapper.queryDailyAttendancePage(p, wrapper);
    }

    @Override
    public List<AttendanceVO> queryDailyAttendanceList(QueryWrapper<AttendanceVO> wrapper) {
        return baseMapper.queryDailyAttendanceList(wrapper);
    }
    @Override
    public IPage<AttendanceVO> queryAllAttendanceRecord(Page<AttendanceVO> p,QueryWrapper<AttendanceVO> wrapper) {
        return baseMapper.queryAllAttendanceRecord(p,wrapper);
    }

    @Override
    public List<AttendancePerLineVO> queryAttendancePerLine(QueryWrapper<AttendanceVO> wrapper) {
        return baseMapper.queryAttendancePerLine(wrapper);
    }

    @Override
    public JSONObject queryDailyAttendanceExport(QueryParam param) {


        // 拷贝参数用于考勤查询
        QueryParam copyParam = new QueryParam();

        // 保存打卡日期参数
        if (param.getParams().containsKey("punchCardDate")) {
            copyParam.getParams().put("punchCardDate",param.getParams().get("punchCardDate"));
            param.getParams().remove("punchCardDate");
        }
        param.getOrderMap().putIfAbsent("enter_date", QueryParam.DESC);

        // 1. 先查询进场花名册（分页）
        IPage<WorkRecordEntity> page = workRecordService.queryPage(param, false);
        List<WorkRecordEntity> workRecordEntities = page.getRecords();


        if (ListUtil.isEmpty(workRecordEntities)) {
            JSONObject pageData = new JSONObject();
            pageData.put("records", Collections.emptyList());
            pageData.put("total", page.getTotal());
            pageData.put("current", page.getCurrent());
            pageData.put("size", page.getSize());
            return pageData;
        }


        // 2. 构建考勤查询条件（根据进场人员）
        List<Long> workerIds = workRecordEntities.stream()
                .map(WorkRecordEntity::getWorkerId)
                .distinct()
                .collect(Collectors.toList());

        if (ListUtil.isNotEmpty(workerIds)) {
            copyParam.getParams().put("workerId", new Parameter(QueryParam.IN, workerIds));
        }

        QueryWrapper<AttendanceDailyExportVO> queryWrapper = new QueryWrapper<>();

        // 处理打卡日期条件,支持按月查询或者按范围查询
        if(copyParam.getParams().get("punchCardDate") != null){
            String punchCardDate = copyParam.getParams().get("punchCardDate").getValue().toString();
            if(punchCardDate.split(",").length == 1){
                copyParam.getParams().remove("punchCardDate");

                queryWrapper = changeToQueryWrapper(copyParam);
                queryWrapper.apply("date_format(punch_card_date,'%Y-%m') = {0}", punchCardDate);
            } else {
                queryWrapper = changeToQueryWrapper(copyParam);
            }
        } else {
            queryWrapper = changeToQueryWrapper(copyParam);
        }

        // 3. 查询考勤数据
        List<AttendanceDailyExportVO> records = baseMapper.queryDailyAttendanceExport(queryWrapper);
        logger.info("1---------查询原始导出数据为{}",JSONObject.toJSONString(records));


//        // 4、转换数据
//        // 使用LinkedHashMap保持插入顺序
//        Map<String, JSONObject> resultMap = new LinkedHashMap<>();
//        for (WorkRecordEntity workRecordEntity : workRecordEntities) {
//            JSONObject employeeJson = new JSONObject();
//            employeeJson.put("name", workRecordEntity.getName());
//            resultMap.put(workRecordEntity.getName(), employeeJson);
//
//            for (AttendanceDailyExportVO record : records) {
//                if(workRecordEntity.getName().equals(record.getName())){
//                    String name = record.getName();
//                    String dateKey = DateUtils.format(record.getWorkDate(),"yyyy-MM-dd");
//                    double workHours = record.getWorkHours();
//
//                    // 获取或创建员工的JSONObject
//                    employeeJson = resultMap.get(name);
//                    // 添加日期-工时对
//                    employeeJson.put(dateKey, workHours);
//                }
//            }
//        }

        // 4. 数据转换 - 核心逻辑
        // 4.1 构建员工ID到姓名的映射（用于快速查找）
        Map<Long, String> workerIdToNameMap = new HashMap<>();
        for (WorkRecordEntity workRecord : workRecordEntities) {
            workerIdToNameMap.put(workRecord.getWorkerId(), workRecord.getName());
        }

        // 4.2 按员工worker_id分组考勤数据
        Map<String, Map<String, Double>> employeeAttendanceMap = new HashMap<>();

        for (AttendanceDailyExportVO record : records) {
            Long workerId = record.getId();
            String employeeName = workerIdToNameMap.get(workerId);

            // 只处理在分页查询结果中的员工
            if (employeeName != null) {
                String dateKey = DateUtils.format(record.getWorkDate(), "yyyy-MM-dd");
                double workHours = record.getWorkHours();

                // 获取或创建员工的考勤Map
                Map<String, Double> dateMap = employeeAttendanceMap.computeIfAbsent(
                        employeeName, k -> new HashMap<>());
                dateMap.put(dateKey, workHours);
            }
        }
        // 4.3 按分页顺序构建结果
        List<JSONObject> resultList = new ArrayList<>();

        for (WorkRecordEntity workRecordEntity : workRecordEntities) {
            String employeeName = workRecordEntity.getName();

            // 创建员工JSON对象
            JSONObject employeeJson = new JSONObject();
            employeeJson.put("name", employeeName);

            // 可选：添加员工其他信息
             employeeJson.put("idCard", workRecordEntity.getIdCard());
             employeeJson.put("phone", workRecordEntity.getPhone());
             employeeJson.put("teamName", workRecordEntity.getTeamName());
             employeeJson.put("projectName", workRecordEntity.getProjectName());

            // 获取该员工的考勤数据
            Map<String, Double> dateMap = employeeAttendanceMap.get(employeeName);
            if (dateMap != null && !dateMap.isEmpty()) {
                // 按日期排序（可选）
                List<String> sortedDates = dateMap.keySet().stream()
                        .sorted()
                        .collect(Collectors.toList());

                for (String date : sortedDates) {
                    employeeJson.put(date, dateMap.get(date).toString());
                }
            }
            // 如果没有考勤数据，JSON对象只包含name字段

            resultList.add(employeeJson);
        }

        //5、 封装数据
        JSONObject pageData = new JSONObject();
        pageData.put("records", resultList);
        pageData.put("total", page.getTotal());
        pageData.put("current", page.getCurrent());
        pageData.put("size", page.getSize());

        return pageData;
    }




    @Override
    public List<AttendanceDailyExportVO> queryDailyAttendanceExportExcel(QueryParam param) {


        // 拷贝参数用于考勤查询
        QueryParam copyParam = new QueryParam();

        // 保存打卡日期参数
        if (param.getParams().containsKey("punchCardDate")) {
            copyParam.getParams().put("punchCardDate",param.getParams().get("punchCardDate"));
            param.getParams().remove("punchCardDate");
        }
        param.getOrderMap().putIfAbsent("enter_date", QueryParam.DESC);

        // 1. 先查询进场花名册（分页）
        IPage<WorkRecordEntity> page = workRecordService.queryPage(param, false);
        List<WorkRecordEntity> workRecordEntities = page.getRecords();


        if (ListUtil.isEmpty(workRecordEntities)) {

            return new ArrayList<>();
        }


        // 2. 构建考勤查询条件（根据进场人员）
        List<Long> workerIds = workRecordEntities.stream()
                .map(WorkRecordEntity::getWorkerId)
                .distinct()
                .collect(Collectors.toList());

        if (ListUtil.isNotEmpty(workerIds)) {
            copyParam.getParams().put("workerId", new Parameter(QueryParam.IN, workerIds));
        }

        QueryWrapper<AttendanceDailyExportVO> queryWrapper = new QueryWrapper<>();

        // 处理打卡日期条件,支持按月查询或者按范围查询
        if(copyParam.getParams().get("punchCardDate") != null){
            String punchCardDate = copyParam.getParams().get("punchCardDate").getValue().toString();
            if(punchCardDate.split(",").length == 1){
                copyParam.getParams().remove("punchCardDate");

                queryWrapper = changeToQueryWrapper(copyParam);
                queryWrapper.apply("date_format(punch_card_date,'%Y-%m') = {0}", punchCardDate);
            } else {
                queryWrapper = changeToQueryWrapper(copyParam);
            }
        } else {
            queryWrapper = changeToQueryWrapper(copyParam);
        }

        // 3. 查询考勤数据
        List<AttendanceDailyExportVO> records = baseMapper.queryDailyAttendanceExport(queryWrapper);
        logger.info("1---------查询原始导出数据为{}",JSONObject.toJSONString(records));


        // 4. 数据转换 - 核心逻辑
        // 4.1 构建员工ID到姓名的映射（用于快速查找）
        Map<Long, String> workerIdToNameMap = new HashMap<>();
        // 生成完整的日期列表

        String punchCardDate = copyParam.getParams().get("punchCardDate").getValue().toString();
        String[] dates = punchCardDate.split(",");
        if (dates.length != 2) {
            throw new BusinessException("日期格式不正确");
        }
        DateTime startDate = DateUtil.parseDate(dates[0]);
        DateTime endDate = DateUtil.parseDate(dates[1]);
        List<String> allDateKeys = generateDateRangeKeys(startDate, endDate);

        for (WorkRecordEntity workRecord : workRecordEntities) {
            workerIdToNameMap.put(workRecord.getWorkerId(), workRecord.getName());
        }


        // 存储员工考勤数据
        Map<String, Map<String, Double>> employeeAttendanceMap = new LinkedHashMap<>();
        Map<String, List<Double>> employeeAttendanceList = new LinkedHashMap<>();
        // 第一步：初始化所有员工的数据结构
        for (String employeeName : workerIdToNameMap.values()) {
            // 使用LinkedHashMap保持日期顺序
            Map<String, Double> dateMap = new LinkedHashMap<>();
            List<Double> hourList = new ArrayList<>();

            // 为每个日期初始化默认值0
            for (String dateKey : allDateKeys) {
                dateMap.put(dateKey, 0.0);
                hourList.add(0.0);
            }

            employeeAttendanceMap.put(employeeName, dateMap);
            employeeAttendanceList.put(employeeName, hourList);
        }

        // 第二步：填充实际考勤数据
        for (AttendanceDailyExportVO record : records) {
            Long workerId = record.getId();
            String employeeName = workerIdToNameMap.get(workerId);

            if (employeeName != null) {
                String dateKey = DateUtils.format(record.getWorkDate(), "yyyy-MM-dd");
                double workHours = record.getWorkHours();

                // 更新对应日期的工时
                Map<String, Double> dateMap = employeeAttendanceMap.get(employeeName);
                if (dateMap != null) {
                    dateMap.put(dateKey, workHours);
                }

                // 同时更新列表格式的数据
                List<Double> hourList = employeeAttendanceList.get(employeeName);
                if (hourList != null) {
                    int index = allDateKeys.indexOf(dateKey);
                    if (index != -1) {
                        hourList.set(index, workHours);
                    }
                }
            }
        }

        // 4.3 按分页顺序构建结果
        List<JSONObject> resultListJSON = new ArrayList<>();
        List<AttendanceDailyExportVO> resultList = new ArrayList<>();

        for (WorkRecordEntity workRecordEntity : workRecordEntities) {
            String employeeName = workRecordEntity.getName();

            // 创建员工JSON对象
            JSONObject employeeJson = new JSONObject();
            employeeJson.put("name", employeeName);

            // 可选：添加员工其他信息
            employeeJson.put("idCard", workRecordEntity.getIdCard());
            employeeJson.put("phone", workRecordEntity.getPhone());
            employeeJson.put("teamName", workRecordEntity.getTeamName());
            employeeJson.put("projectName", workRecordEntity.getProjectName());




            // 获取该员工的考勤数据
            Map<String, Double> dateMap = employeeAttendanceMap.get(employeeName);
            if (dateMap != null && !dateMap.isEmpty()) {
                // 按日期排序（可选）
                List<String> sortedDates = dateMap.keySet().stream()
                        .sorted()
                        .collect(Collectors.toList());
                Map<String, Object> dynamicData = new LinkedHashMap<>(); // 用LinkedHashMap保持顺序
                List<String> dynamicValues = new ArrayList<>();
                for (String date : sortedDates) {
                    employeeJson.put(date, dateMap.get(date).toString());

                    dynamicValues.add(dateMap.get(date).toString());
                }
                employeeJson.put("dynamicValues",dynamicValues);
            }
            // 如果没有考勤数据，JSON对象只包含name字段

            resultListJSON.add(employeeJson);
        }
        for (JSONObject jsonObject : resultListJSON) {
            AttendanceDailyExportVO employeeVO = new AttendanceDailyExportVO();
            // 可选：添加员工其他信息
            employeeVO.setName(jsonObject.getString("name"));
            employeeVO.setIdCard(jsonObject.getString("idCard"));
            employeeVO.setPhone(jsonObject.getString("phone"));
            employeeVO.setTeamName(jsonObject.getString("teamName"));
            employeeVO.setProjectName(jsonObject.getString("projectName"));
            employeeVO.setDynamicValues(jsonObject.getJSONArray("dynamicValues").toJavaList(String.class));

            resultList.add(employeeVO);
        }


        return resultList;
    }

    // 生成日期范围内的所有日期key
    public List<String> generateDateRangeKeys(Date startDate, Date endDate) {
        List<String> dateKeys = new ArrayList<>();

        // 创建日期范围，按天递增
        DateRange range = DateUtil.range(startDate, endDate, DateField.DAY_OF_YEAR);

        // 遍历日期范围
        for (DateTime dateTime : range) {
            String dateKey = dateTime.toString("yyyy-MM-dd");
            dateKeys.add(dateKey);
        }
        return dateKeys;
    }
}
